import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Modal,
  Animated,
  FlatList,
  ScrollView,
  Clipboard,
  Alert,
  Share,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { MongoAPI, USERS_API } from '../../ConfigMongo/MongoDBConfig';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const TeamMemberHome = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [passwordModalVisible, setPasswordModalVisible] = useState(false);
  const [isLoadingPassword, setIsLoadingPassword] = useState(false);
  const [selectedMember, setSelectedMember] = useState(null);
  const [teamMembers, setTeamMembers] = useState([]);
  const [dropdownVisible, setDropdownVisible] = useState(false);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0 });
  const [isOnline, setIsOnline] = useState(true);
  const [forceOfflineMode, setForceOfflineMode] = useState(false);
  const [hasShownOfflineToast, setHasShownOfflineToast] = useState(false);
  const [hasShownOnlineToast, setHasShownOnlineToast] = useState(false);
  const [pendingActions, setPendingActions] = useState([]);
  const [lastSyncTime, setLastSyncTime] = useState(null);
  const [selectedCountry, setSelectedCountry] = useState('All');
  const [selectedProject, setSelectedProject] = useState('All');
  const [selectedRole, setSelectedRole] = useState('All');
  const [countryDropdownVisible, setCountryDropdownVisible] = useState(false);
  const [projectDropdownVisible, setProjectDropdownVisible] = useState(false);
  const [roleDropdownVisible, setRoleDropdownVisible] = useState(false);
  const [countryDropdownPosition, setCountryDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [projectDropdownPosition, setProjectDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [roleDropdownPosition, setRoleDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [isLoading, setIsLoading] = useState(true);
  const [adminProfile, setAdminProfile] = useState(null);
  const [pwQueueStatusMap, setPwQueueStatusMap] = useState({}); // userId -> 'queued' | 'done' | 'error'
  const [userCountry, setUserCountry] = useState([]);
  const [userProject, setUserProject] = useState([]);
  const [userPermissions, setUserPermissions] = useState([]);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [profileModalVisible, setProfileModalVisible] = useState(false);
  const [totalUsers, setTotalUsers] = useState(0);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(0);
  const [isLoadingMore, setIsLoadingMore] = useState(false);
  const [hasMoreData, setHasMoreData] = useState(true);
  const [allUsers, setAllUsers] = useState([]); // Store all loaded users
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const isMounted = useRef(true);
  const countryButtonRef = useRef(null);
  const projectButtonRef = useRef(null);
  const roleButtonRef = useRef(null);

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const showToast = (type, title, message) => {
    if (!isMounted.current) return;
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  const filterTeamMembersByPermissions = (members, userCountry, userProject, userPermissions, adminUid) => {
    let filteredMembers = [...members];
    
    // Debug logging
    console.log('🔍 Filtering debug:', {
      totalMembers: members.length,
      userCountry: userCountry,
      userProject: userProject,
      userPermissions: userPermissions,
      adminUid: adminUid
    });
    
    if (userPermissions.includes('OnlyMineView')) {
      filteredMembers = filteredMembers.filter(member => member.id === adminUid);
      console.log('🔍 OnlyMineView filter applied:', filteredMembers.length);
    } else {
      // Show ALL members - no restrictions
      // This ensures all users are loaded and displayed
      filteredMembers = filteredMembers;
      console.log('🔍 No restrictions - showing all members:', filteredMembers.length);
    }
    
    // Sort to put admin profile at the top
    filteredMembers.sort((a, b) => {
      if (a.id === adminUid) return -1; // Admin profile first
      if (b.id === adminUid) return 1;
      return 0; // Keep original order for others
    });
    
    console.log('🔍 After sorting - admin profile at top:', filteredMembers.length);
    
    return filteredMembers;
  };

  const getFilterOptions = () => {
    // Use allUsers instead of paginatedMembers to get complete data
    const allMembers = allUsers.length > 0 ? allUsers : paginatedMembers;
    
    // Get all unique countries and projects from all loaded users
    const allCountries = Array.from(new Set(allMembers.flatMap(member => member.countries || [])));
    const allProjects = Array.from(new Set(allMembers.flatMap(member => member.projects || [])));
    const allRoles = Array.from(new Set(allMembers.map(member => member.role || 'Operator')));
    
    // Sort them alphabetically for better UX
    const sortedCountries = allCountries.sort();
    const sortedProjects = allProjects.sort();
    const sortedRoles = allRoles.sort();
    
    const countries = ['All', ...sortedCountries];
    const projects = ['All', ...sortedProjects];
    const roles = ['All', ...sortedRoles];
    
    // Calculate counts for each filter option - show all countries/projects
    const countryCounts = {};
    const projectCounts = {};
    
    // Calculate country counts - show all countries even with 0 users
    countries.forEach(country => {
      if (country === 'All') {
        countryCounts[country] = allMembers.length;
      } else {
        const count = allMembers.filter(member => 
          member.countries && member.countries.includes(country)
        ).length;
        countryCounts[country] = count; // Show all countries, even with 0 users
      }
    });
    
    // Calculate project counts - show all projects even with 0 users
    projects.forEach(project => {
      if (project === 'All') {
        projectCounts[project] = allMembers.length;
      } else {
        const count = allMembers.filter(member => 
          member.projects && member.projects.includes(project)
        ).length;
        projectCounts[project] = count; // Show all projects, even with 0 users
      }
    });
    
    // Show all countries and projects - no filtering
    const filteredCountries = countries;
    const filteredProjects = projects;
    
    const roleCounts = roles.reduce((acc, role) => {
      if (role === 'All') {
        acc[role] = allMembers.length;
      } else {
        acc[role] = allMembers.filter(member => member.role === role).length;
      }
      return acc;
    }, {});
    
    // Generate project-country combinations for detailed analysis
    const projectCountryCombinations = {};
    allMembers.forEach(member => {
      const memberCountries = member.countries || [];
      const memberProjects = member.projects || [];
      
      memberCountries.forEach(country => {
        memberProjects.forEach(project => {
          const key = `${project} - ${country}`;
          projectCountryCombinations[key] = (projectCountryCombinations[key] || 0) + 1;
        });
      });
    });
    
    console.log('🔍 Filter options generated:', {
      totalMembers: allMembers.length,
      countries: filteredCountries,
      projects: filteredProjects,
      roles: sortedRoles,
      countryCounts,
      projectCounts,
      roleCounts,
      projectCountryCombinations
    });
    
    // No auto-reset - let users select any country/project
    
    return { countries: filteredCountries, projects: filteredProjects, roles, countryCounts, projectCounts, roleCounts };
  };

  useEffect(() => {
    const initialize = async () => {
      setIsLoading(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      try {
        console.log('═══════════════════════════════════════');
        console.log('🚀 INITIALIZING TEAM EMPLOYEE MODULE');
        
        // Load force offline mode preference
        const savedOfflineMode = await AsyncStorage.getItem('forceOfflineMode_team');
        if (savedOfflineMode) {
          setForceOfflineMode(JSON.parse(savedOfflineMode));
          console.log('📴 Loaded offline mode:', JSON.parse(savedOfflineMode));
        }
        
        // Load last sync time
        const savedSyncTime = await AsyncStorage.getItem('teamMembersLastSync');
        if (savedSyncTime) {
          setLastSyncTime(savedSyncTime);
          console.log('⏰ Last sync:', savedSyncTime);
        }
        
        const myUid = await AsyncStorage.getItem('userUid');
        console.log('👤 User ID:', myUid);
        
        // ===== OFFLINE-FIRST: Load from AsyncStorage first =====
        const cachedAdminData = await AsyncStorage.getItem('adminProfile');
        const cachedPermissions = await AsyncStorage.getItem('teamEmployPermissions');
        
        if (cachedAdminData) {
          try {
            const cached = JSON.parse(cachedAdminData);
            setAdminProfile(cached);
            setUserCountry(cached.countries || []);
            setUserProject(cached.projects || []);
            
            // Load permissions from cache
            if (cachedPermissions) {
              const perms = JSON.parse(cachedPermissions);
              setUserPermissions(perms);
              console.log('✅ LOADED FROM CACHE - Permissions:', perms);
            } else if (cached.permissions) {
              setUserPermissions(cached.permissions);
              console.log('✅ LOADED FROM CACHED PROFILE - Permissions:', cached.permissions);
            }
            
            console.log('✅ Admin profile loaded from cache');
          } catch (err) {
            console.warn('⚠️ Error parsing cached admin data:', err);
          }
        }
        
        // ===== ONLINE: Fetch from MongoDB to update permissions =====
        let adminProfile = null;
        const online = await checkNetworkStatus();
        
        if (myUid && (online || !cachedAdminData)) {
          try {
            const adminResponse = await MongoAPI.getUserById(myUid);
            if (adminResponse.success && adminResponse.data) {
              adminProfile = adminResponse.data;
              
              console.log('✅ FETCHED FROM MONGODB:', {
                name: adminProfile.name,
                hasModules: !!adminProfile.modules,
                modulesCount: adminProfile.modules?.length || 0
              });
            }
          } catch (err) {
            console.warn('⚠️ Failed to fetch from MongoDB, using cache:', err);
          }
        }

        // Set admin profile if fetched from MongoDB
        if (adminProfile) {
          const profileData = {
            ...adminProfile,
            id: myUid,
            fullName: adminProfile.name || 'N/A',
            employeeNumber: adminProfile.employeeNumber || 'N/A',
            projects: adminProfile.projects || [],
            countries: adminProfile.countries || [],
            role: adminProfile.role || 'Operator',
            isBlocked: adminProfile.isBlocked || false,
            password: adminProfile.password || 'N/A',
            profilePic: adminProfile.profilePic || null,
          };
          
          setAdminProfile(profileData);
          setUserCountry(adminProfile.countries || []);
          setUserProject(adminProfile.projects || []);
          
          // Extract teamEmploy permissions
          const teamEmployModule = adminProfile.modules?.find(m => m.module === 'teamEmploy');
          const permissions = teamEmployModule?.permissions || [];
          setUserPermissions(permissions);
          
          // ===== STORE IN ASYNCSTORAGE FOR OFFLINE ACCESS =====
          await AsyncStorage.setItem('adminProfile', JSON.stringify({
            ...profileData,
            permissions: permissions
          }));
          await AsyncStorage.setItem('teamEmployPermissions', JSON.stringify(permissions));
          
          console.log('💾 STORED ADMIN PROFILE & PERMISSIONS:', {
            name: profileData.fullName,
            countries: profileData.countries,
            projects: profileData.projects,
            permissions: permissions
          });
          
          // Compare with cached permissions
          if (cachedPermissions) {
            try {
              const oldPerms = JSON.parse(cachedPermissions).sort().join(',');
              const newPerms = permissions.sort().join(',');
              if (oldPerms !== newPerms) {
                console.log('🔄 PERMISSIONS CHANGED!');
                console.log('Old:', JSON.parse(cachedPermissions));
                console.log('New:', permissions);
                showToast('info', 'Permissions Updated', 'Your access has been updated');
              }
            } catch (err) {
              console.warn('⚠️ Error comparing permissions:', err);
            }
          }
        } else if (!cachedAdminData) {
          // No admin profile at all - set defaults
          setUserCountry([]);
          setUserProject([]);
          setUserPermissions([]);
          console.log('⚠️ No admin profile available');
        }
        
        console.log('✅ INITIALIZATION COMPLETE');
        console.log('═══════════════════════════════════════');

        // Load all users at once
        await loadTeamMembers(myUid);

        await loadPendingActions();

        const unsubscribeNetInfo = NetInfo.addEventListener(async (state) => {
          if (!isMounted.current || forceOfflineMode) return;
          
          // More robust network status checking
          const isConnected = state.isConnected && state.isInternetReachable !== false;
          const wasOnline = isOnline;
          
          console.log('🌐 Network state change:', {
            isConnected: state.isConnected,
            isInternetReachable: state.isInternetReachable,
            type: state.type,
            wasOnline,
            willBeOnline: isConnected,
            forceOffline: forceOfflineMode
          });
          
          // Only update if status actually changed
          if (wasOnline !== isConnected) {
            setIsOnline(isConnected);
            
            if (!isConnected && !hasShownOfflineToast) {
              showToast('info', 'Offline Mode', 'Working offline - changes will sync when online');
              setHasShownOfflineToast(true);
              setHasShownOnlineToast(false);
            } else if (isConnected && !hasShownOnlineToast) {
              showToast('success', 'Online', 'Connection restored, syncing...');
              setHasShownOnlineToast(true);
              setHasShownOfflineToast(false);
              
              // Sync pending actions when coming back online
              if (pendingActions.length > 0) {
                console.log('🔄 Auto-syncing pending actions...');
                try {
                  await syncPendingActions();
                } catch (error) {
                  console.error('Error syncing pending actions:', error);
                }
              }
            }
          }
        });

        // Set up periodic network check to maintain connection
        const networkCheckInterval = setInterval(async () => {
          if (!isMounted.current) return;
          
          const currentOnline = await checkNetworkStatus();
          if (currentOnline !== isOnline) {
            console.log('🔄 Periodic network check detected change:', { was: isOnline, now: currentOnline });
            setIsOnline(currentOnline);
          }
        }, 10000); // Check every 10 seconds

        return () => {
          isMounted.current = false;
          unsubscribeNetInfo();
          clearInterval(networkCheckInterval);
        };
       } catch (error) {
         console.error('❌ Error initializing team management:', error);
         showToast('error', 'Error', 'Failed to load team data.');
       } finally {
        setIsLoading(false);
        rotateAnim.setValue(0);
      }
    };

    initialize();
  }, []);

  // Auto-clear done/error banners after a short delay
  useEffect(() => {
    const timers = [];
    Object.entries(pwQueueStatusMap).forEach(([userId, status]) => {
      if (status === 'done' || status === 'error') {
        const t = setTimeout(() => {
          setPwQueueStatusMap((prev) => {
            const next = { ...prev };
            delete next[userId];
            return next;
          });
        }, 6000);
        timers.push(t);
      }
    });
    return () => timers.forEach(clearTimeout);
  }, [pwQueueStatusMap]);

  const loadPendingActions = async () => {
    try {
      const storedPending = await AsyncStorage.getItem('pendingActionsTeamMemberHome');
      if (storedPending) {
        const actions = JSON.parse(storedPending);
        setPendingActions(actions);
        console.log('Loaded pending actions:', actions);
      }
    } catch (error) {
      console.error('Error loading pending actions:', error);
      showToast('error', 'Error', 'Failed to load pending actions.');
    }
  };

   const loadTeamMembers = async (adminUid, retryCount = 0) => {
     try {
       if (!isMounted.current) return;

       const online = await checkNetworkStatus();
       setIsOnline(online);

       if (!online && retryCount < 3) {
         // Retry network check after a short delay
         console.log(`🔄 Network offline, retrying in 2 seconds... (attempt ${retryCount + 1}/3)`);
         setTimeout(() => {
           loadTeamMembers(adminUid, retryCount + 1);
         }, 2000);
         return;
       }

       if (!online) {
         // Load from cache for offline mode after retries failed
         console.log('📱 Loading from cache - network unavailable after retries');
         const cachedMembers = await AsyncStorage.getItem('teamMembers');
         if (cachedMembers) {
           const members = JSON.parse(cachedMembers);
           
           // Check how many are pending sync
           const pendingCount = members.filter(m => m.pendingSync).length;
           
           const filtered = filterTeamMembersByPermissions(members, userCountry, userProject, userPermissions, adminUid || '');
           setTeamMembers(filtered);
           setAllUsers(members);
           setTotalUsers(members.length);
           
           if (pendingCount > 0) {
             showToast('info', 'Offline Mode', `Loaded cached data. ${pendingCount} user(s) waiting to sync.`);
           } else {
             showToast('info', 'Offline Mode', 'Loaded cached team members.');
           }
         }
         setIsLoading(false);
         rotateAnim.setValue(0);
         return;
       }

      // Load team members from MongoDB using fast endpoint with pagination
      let teamResponse;
      try {
        teamResponse = await MongoAPI.getUsersFast(1, 8000);
        console.log('📊 MongoDB Response:', {
          success: teamResponse.success,
          dataLength: teamResponse.data?.length,
          firstUser: teamResponse.data?.[0] ? {
            id: teamResponse.data[0]._id,
            name: teamResponse.data[0].name,
            password: teamResponse.data[0].password,
            hasPassword: !!teamResponse.data[0].password
          } : 'No users'
        });
      } catch (apiError) {
        console.error('❌ MongoDB API Error:', apiError);
        if (retryCount < 2) {
          console.log(`🔄 API Error, retrying in 3 seconds... (attempt ${retryCount + 1}/2)`);
          setTimeout(() => {
            loadTeamMembers(adminUid, retryCount + 1);
          }, 3000);
          return;
        } else {
          throw apiError;
        }
      }
      
      if (teamResponse.success && teamResponse.data) {
        const membersArray = teamResponse.data.map((user) => ({
          // Essential fields only for better performance
          id: user._id,
          fullName: user.name || 'N/A',
          employeeNumber: user.employeeNumber || 'N/A',
          email: user.email || user.employeeNumber || 'N/A',
          projects: user.projects || [],
          countries: user.countries || [],
          role: user.role || 'Operator',
          isBlocked: user.isBlocked || false,
          password: user.password || user.employeeNumber || user.email || 'N/A',
          profilePic: user.profilePic || null,
          pendingSync: false, // From server, not pending
          // Additional fields only if needed
          driverLicenseNumber: user.driverLicenseNumber || '',
          driverLicenseCode: user.driverLicenseCode || '',
          driverLicenseExpiry: user.driverLicenseExpiry || '',
          dateCompletedFirstAid: user.dateCompletedFirstAid || '',
          expiryDateFirstAid: user.expiryDateFirstAid || '',
          fireFightingDateAttended: user.fireFightingDateAttended || '',
          blastingLicenseDateIssued: user.blastingLicenseDateIssued || '',
        }));
        
        // Check if there are offline users in cache that need to be preserved
        const cachedMembers = await AsyncStorage.getItem('teamMembers');
        let offlineUsers = [];
        if (cachedMembers) {
          const cached = JSON.parse(cachedMembers);
          offlineUsers = cached.filter(user => user.pendingSync === true);
          console.log(`📤 Found ${offlineUsers.length} offline users to preserve`);
        }
        
        // Merge offline users with server users
        const mergedUsers = [...offlineUsers, ...membersArray];
        
        // Store all users and pagination info
        setAllUsers(mergedUsers);
        setTotalUsers(teamResponse.totalUsers || membersArray.length);
        setTotalPages(teamResponse.totalPages || 1);
        setCurrentPage(1);
        setHasMoreData(teamResponse.hasNext || false);
        
        // Apply admin permissions filter
        console.log('🔍 Filtering debug:', {
          totalUsers: mergedUsers.length,
          serverUsers: membersArray.length,
          offlineUsers: offlineUsers.length,
          userCountry: userCountry,
          userProject: userProject,
          userPermissions: userPermissions,
          adminUid: adminUid
        });
        
        const filteredMembers = filterTeamMembersByPermissions(mergedUsers, userCountry, userProject, userPermissions, adminUid || '');
        console.log('🔍 Filtered result:', {
          filteredCount: filteredMembers.length,
          originalCount: mergedUsers.length,
          offlineUsers: offlineUsers.length,
          serverUsers: membersArray.length
        });
        
        setTeamMembers(filteredMembers);
        
        // Cache merged data and save sync time
        await AsyncStorage.setItem('teamMembers', JSON.stringify(mergedUsers));
        const syncTime = new Date().toISOString();
        await AsyncStorage.setItem('teamMembersLastSync', syncTime);
        setLastSyncTime(syncTime);
        
        if (offlineUsers.length > 0) {
          showToast('success', 'Success', `Loaded ${membersArray.length} users. ${offlineUsers.length} offline user(s) pending sync.`);
        } else {
          showToast('success', 'Success', `Loaded ${membersArray.length} of ${teamResponse.totalUsers || membersArray.length} team members`);
        }
        
        // Debug: Show all projects and countries with counts
        const allProjects = Array.from(new Set(membersArray.flatMap(member => member.projects || [])));
        const allCountries = Array.from(new Set(membersArray.flatMap(member => member.countries || [])));
        
        console.log('📊 Initial Projects with counts:', allProjects.map(project => {
          const count = membersArray.filter(member => 
            member.projects && member.projects.includes(project)
          ).length;
          return `${project}: ${count}`;
        }));
        
        console.log('🌍 Initial Countries with counts:', allCountries.map(country => {
          const count = membersArray.filter(member => 
            member.countries && member.countries.includes(country)
          ).length;
          return `${country}: ${count}`;
        }));
        
        // Auto-load more users if there are more pages available
        if (teamResponse.hasNext && teamResponse.totalUsers > 8000) {
          setTimeout(() => {
            loadMoreUsers();
          }, 1000); // Start loading more after 1 second
        }
         
       } else {
         setTeamMembers([]);
         setAllUsers([]);
         setTotalUsers(0);
         await AsyncStorage.setItem('teamMembers', JSON.stringify([]));
       }
       
       setIsLoading(false);
       rotateAnim.setValue(0);
     } catch (error) {
       console.error('Error loading team members:', error);
       showToast('error', 'Error', 'Failed to load team members.');
       
       // Try to load from cache as fallback
       try {
         const cachedMembers = await AsyncStorage.getItem('teamMembers');
         if (cachedMembers) {
           const members = JSON.parse(cachedMembers);
           const filtered = filterTeamMembersByPermissions(members, userCountry, userProject, userPermissions, adminUid || '');
           setTeamMembers(filtered);
           setTotalUsers(members.length);
           showToast('info', 'Using Cached Data', 'Showing cached team members.');
         }
       } catch (cacheError) {
         console.error('Error loading cached data:', cacheError);
       }
       
       setIsLoading(false);
       rotateAnim.setValue(0);
     }
  };

  const checkNetworkStatus = async () => {
    try {
      if (forceOfflineMode) {
        console.log('📴 Force offline mode enabled');
        return false;
      }
      const netInfo = await NetInfo.fetch();
      const isConnected = netInfo.isConnected && netInfo.isInternetReachable !== false;
      console.log('🌐 Network status check:', {
        isConnected: netInfo.isConnected,
        isInternetReachable: netInfo.isInternetReachable,
        type: netInfo.type,
        forceOffline: forceOfflineMode,
        finalResult: isConnected
      });
      return isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const loadMoreUsers = async () => {
    if (!hasMoreData || isLoadingMore) return;
    
    try {
      setIsLoadingMore(true);
      const nextPage = currentPage + 1;
      
      console.log(`📄 Loading more users - Page ${nextPage}`);
      
      const teamResponse = await MongoAPI.getUsersFast(nextPage, 8000);
      if (teamResponse.success && teamResponse.data) {
        const newMembers = teamResponse.data.map((user) => ({
          id: user._id,
          fullName: user.name || 'N/A',
          employeeNumber: user.employeeNumber || 'N/A',
          email: user.email || user.employeeNumber || 'N/A',
          projects: user.projects || [],
          countries: user.countries || [],
          role: user.role || 'Operator',
          isBlocked: user.isBlocked || false,
          password: user.password || user.employeeNumber || user.email || 'N/A',
          profilePic: user.profilePic || null,
          driverLicenseNumber: user.driverLicenseNumber || '',
          driverLicenseCode: user.driverLicenseCode || '',
          driverLicenseExpiry: user.driverLicenseExpiry || '',
          dateCompletedFirstAid: user.dateCompletedFirstAid || '',
          expiryDateFirstAid: user.expiryDateFirstAid || '',
          fireFightingDateAttended: user.fireFightingDateAttended || '',
          blastingLicenseDateIssued: user.blastingLicenseDateIssued || '',
        }));
        
        // Preserve offline users (those with pendingSync = true)
        const offlineUsers = allUsers.filter(user => user.pendingSync === true);
        
        // Remove offline users from current list before adding new server users
        const serverUsersOnly = allUsers.filter(user => !user.pendingSync);
        
        // Add new members to server users
        const updatedServerUsers = [...serverUsersOnly, ...newMembers];
        
        // Merge offline users at the beginning
        const updatedAllUsers = [...offlineUsers, ...updatedServerUsers];
        
        setAllUsers(updatedAllUsers);
        setCurrentPage(nextPage);
        setHasMoreData(teamResponse.hasNext || false);
        
        // Apply filters to all users
        const adminUid = await AsyncStorage.getItem('userUid');
        const filteredMembers = filterTeamMembersByPermissions(updatedAllUsers, userCountry, userProject, userPermissions, adminUid || '');
        setTeamMembers(filteredMembers);
        
        // Update cache
        await AsyncStorage.setItem('teamMembers', JSON.stringify(updatedAllUsers));
        
        console.log(`📤 Preserved ${offlineUsers.length} offline users in updated list`);
        
        console.log(`✅ Loaded ${newMembers.length} more users. Total: ${updatedAllUsers.length}`);
        
        // Debug: Show all projects and countries with counts
        const allProjects = Array.from(new Set(updatedAllUsers.flatMap(member => member.projects || [])));
        const allCountries = Array.from(new Set(updatedAllUsers.flatMap(member => member.countries || [])));
        
        console.log('📊 All Projects with counts:', allProjects.map(project => {
          const count = updatedAllUsers.filter(member => 
            member.projects && member.projects.includes(project)
          ).length;
          return `${project}: ${count}`;
        }));
        
        console.log('🌍 All Countries with counts:', allCountries.map(country => {
          const count = updatedAllUsers.filter(member => 
            member.countries && member.countries.includes(country)
          ).length;
          return `${country}: ${count}`;
        }));
        
        // Auto-load next page if there's more data and we haven't loaded all users yet
        if (teamResponse.hasNext && updatedAllUsers.length < totalUsers) {
          setTimeout(() => {
            loadMoreUsers();
          }, 500); // Small delay to prevent overwhelming the server
        }
      }
    } catch (error) {
      console.error('Error loading more users:', error);
      showToast('error', 'Error', 'Failed to load more users.');
    } finally {
      setIsLoadingMore(false);
    }
  };

  const syncPendingActions = async () => {
    if (!isOnline) return;
    try {
      const storedPending = await AsyncStorage.getItem('pendingActionsTeamMemberHome');
      const actions = storedPending ? JSON.parse(storedPending) : [];
      if (actions.length === 0) return;
      setSyncModalVisible(true);
      const adminUid = await AsyncStorage.getItem('userUid');
      for (const action of actions) {
        if (action.type === 'addTeamMember') {
          await handleAddTeamMemberAction(action.teamMember, adminUid);
        } else if (action.type === 'deleteTeamMember') {
          await handleDeleteTeamMemberAction(adminUid, action.memberId);
        }
      }
      setPendingActions([]);
      await AsyncStorage.setItem('pendingActionsTeamMemberHome', JSON.stringify([]));
      await loadTeamMembers(adminUid);
    } catch (error) {
      console.error('Error syncing pending actions:', error);
      showToast('error', 'Error', 'Failed to sync pending actions');
    } finally {
      setSyncModalVisible(false);
    }
  };

  const handleAddTeamMemberAction = async (teamMember, adminUid) => {
    try {
      console.log('🔄 Syncing offline team member to MongoDB:', teamMember.fullName || teamMember.name);
      
      // Prepare user data for MongoDB
      const mongoUserData = {
        _id: teamMember.id || teamMember.uid || `user_${Date.now()}`,
        uid: teamMember.id || teamMember.uid || `user_${Date.now()}`,
        name: teamMember.name || teamMember.fullName,
        email: teamMember.email,
        password: teamMember.password,
        mobile: teamMember.mobile,
        role: teamMember.role,
        loginStatus: false,
        lastLogin: null,
        countries: teamMember.countries || [],
        projects: teamMember.projects || [],
        modules: teamMember.modules || [],
        department: teamMember.department,
        nrcIdNumber: teamMember.nrcIdNumber,
        employeeNumber: teamMember.employeeNumber,
        engagementDate: teamMember.engagementDate,
        titanInductionDate: teamMember.titanInductionDate,
        clientInduction: teamMember.clientInduction,
        dateExamined: teamMember.dateExamined,
        expiryMedicals: teamMember.expiryMedicals,
        driverLicenseNumber: teamMember.driverLicenseNumber || '',
        driverLicenseCode: teamMember.driverLicenseCode || '',
        driverLicenseExpiry: teamMember.driverLicenseExpiry || '',
        clientDrivingDate: teamMember.clientDrivingDate || '',
        dateCompletedFirstAid: teamMember.dateCompletedFirstAid || '',
        levelFirstAid: teamMember.levelFirstAid || '',
        expiryDateFirstAid: teamMember.expiryDateFirstAid || '',
        profilePic: teamMember.profilePic || null,
        medicalDocs: teamMember.medicalDocs || [],
        driverLicenseDocs: teamMember.driverLicenseDocs || [],
        defensiveDrivingDocs: teamMember.defensiveDrivingDocs || [],
        uploadTimestamp: new Date(),
        authCreated: false,
        originalUserId: teamMember.id || teamMember.uid,
      };
      
      // Create user in MongoDB
      const response = await MongoAPI.createUser(mongoUserData._id, mongoUserData);
      
      if (response.success) {
        console.log('✅ Synced team member to MongoDB:', mongoUserData.name);
        
        // Update local storage - remove pendingSync flag
        const storedTeam = await AsyncStorage.getItem('teamMembers');
        let teamMembers = storedTeam ? JSON.parse(storedTeam) : [];
        
        // Find and update the team member
        teamMembers = teamMembers.map(member => 
          member.id === teamMember.id 
            ? { ...member, pendingSync: false }
            : member
        );
        
        const filtered = filterTeamMembersByPermissions(teamMembers, userCountry, userProject, userPermissions, adminUid);
        setTeamMembers(filtered);
        await AsyncStorage.setItem('teamMembers', JSON.stringify(teamMembers));
        
        showToast('success', 'Synced', `${mongoUserData.name} synced to server`);
      } else {
        throw new Error(response.error || 'Failed to create team member in MongoDB');
      }
    } catch (error) {
      console.error('❌ Error syncing add team member action:', error);
      showToast('error', 'Sync Failed', `Failed to sync ${teamMember.fullName || teamMember.name}`);
      throw error;
    }
  };

  const handleDeleteTeamMemberAction = async (adminUid, memberId) => {
    try {
      // Delete team member from MongoDB
      const response = await MongoAPI.deleteUser(memberId);
      if (response.success) {
        const storedTeam = await AsyncStorage.getItem('teamMembers');
        let teamMembers = storedTeam ? JSON.parse(storedTeam) : [];
        teamMembers = teamMembers.filter((member) => member.id !== memberId);
        const filtered = filterTeamMembersByPermissions(teamMembers, userCountry, userProject, userPermissions, adminUid);
        setTeamMembers(filtered);
        await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
        console.log('Synced delete team member action for ID:', memberId);
      } else {
        throw new Error('Failed to delete team member from MongoDB');
      }
    } catch (error) {
      console.error('Error syncing delete team member action:', error);
      throw error;
    }
  };

  // Toggle online/offline mode
  const toggleOfflineMode = async () => {
    const newOfflineMode = !forceOfflineMode;
    
    if (newOfflineMode) {
      // Switching to offline
      setForceOfflineMode(true);
      await AsyncStorage.setItem('forceOfflineMode_team', JSON.stringify(true));
      setIsOnline(false);
      showToast('info', 'Offline Mode', 'Working offline - using cached permissions');
      console.log('📴 Forced offline mode enabled for team management');
    } else {
      // Trying to switch to online - check actual network first
      const netInfo = await NetInfo.fetch();
      const actuallyOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      
      if (actuallyOnline) {
        // Network is available - switch to online
        setForceOfflineMode(false);
        await AsyncStorage.setItem('forceOfflineMode_team', JSON.stringify(false));
        setIsOnline(true);
        showToast('success', 'Online Mode', 'Syncing permissions & data...');
        console.log('🌐 Online mode enabled - Syncing permissions & team data...');
        
        // Auto-sync admin profile (to get latest permissions)
        const adminUid = await AsyncStorage.getItem('userUid');
        if (adminUid) {
          try {
            const adminResponse = await MongoAPI.getUserById(adminUid);
            if (adminResponse.success && adminResponse.data) {
              const adminProfile = adminResponse.data;
              
              // Extract teamEmploy permissions
              const teamEmployModule = adminProfile.modules?.find(m => m.module === 'teamEmploy');
              const newPermissions = teamEmployModule?.permissions || [];
              
              // Update state
              setUserPermissions(newPermissions);
              
              // Update cached data
              const profileData = {
                ...adminProfile,
                id: adminUid,
                fullName: adminProfile.name || 'N/A',
                employeeNumber: adminProfile.employeeNumber || 'N/A',
                projects: adminProfile.projects || [],
                countries: adminProfile.countries || [],
                permissions: newPermissions
              };
              
              await AsyncStorage.setItem('adminProfile', JSON.stringify(profileData));
              await AsyncStorage.setItem('teamEmployPermissions', JSON.stringify(newPermissions));
              
              console.log('✅ Permissions refreshed:', newPermissions);
            }
          } catch (err) {
            console.warn('⚠️ Failed to refresh permissions:', err);
          }
          
          // Load team members
          await loadTeamMembers(adminUid);
        }
        
        // Sync pending actions
        if (pendingActions.length > 0) {
          await syncPendingActions();
        }
      } else {
        // Network not available - stay offline
        setForceOfflineMode(true);
        setIsOnline(false);
        showToast('error', 'No Internet', 'No network. Staying offline.');
        console.log('❌ Cannot switch to online - no internet connection');
      }
    }
  };

  const handleCloudSync = async () => {
    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();
    try {
      const online = await checkNetworkStatus();
      if (!online) {
        showToast('info', 'Offline Mode', 'Cannot sync while offline - check your connection');
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        return;
      }
      
      console.log('☁️ CLOUD SYNC STARTED');
      
      // Sync pending actions first
      if (pendingActions.length > 0) {
        console.log('🔄 Syncing pending actions...');
        await syncPendingActions();
      }
      
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        throw new Error('Admin not authenticated.');
      }
      
      // ===== REFRESH ADMIN PERMISSIONS =====
      try {
        console.log('🔄 Refreshing admin permissions...');
        const adminResponse = await MongoAPI.getUserById(adminUid);
        if (adminResponse.success && adminResponse.data) {
          const adminData = adminResponse.data;
          
          // Extract teamEmploy permissions
          const teamEmployModule = adminData.modules?.find(m => m.module === 'teamEmploy');
          const newPermissions = teamEmployModule?.permissions || [];
          
          // Update state
          setUserPermissions(newPermissions);
          setUserCountry(adminData.countries || []);
          setUserProject(adminData.projects || []);
          
          // Update cached profile
          const profileData = {
            ...adminData,
            id: adminUid,
            fullName: adminData.name || 'N/A',
            employeeNumber: adminData.employeeNumber || 'N/A',
            projects: adminData.projects || [],
            countries: adminData.countries || [],
            role: adminData.role || 'Operator',
            isBlocked: adminData.isBlocked || false,
            password: adminData.password || 'N/A',
            profilePic: adminData.profilePic || null,
            permissions: newPermissions
          };
          
          setAdminProfile(profileData);
          await AsyncStorage.setItem('adminProfile', JSON.stringify(profileData));
          await AsyncStorage.setItem('teamEmployPermissions', JSON.stringify(newPermissions));
          
          console.log('✅ Admin permissions refreshed:', newPermissions);
        }
      } catch (err) {
        console.warn('⚠️ Failed to refresh admin permissions:', err);
      }
      
      // Load team members from MongoDB using fast endpoint with pagination
      const teamResponse = await MongoAPI.getUsersFast(1, 8000);
      if (teamResponse.success && teamResponse.data) {
        const membersArray = teamResponse.data.map((user) => ({
          // Essential fields only for better performance
          id: user._id,
          fullName: user.name || 'N/A',
          employeeNumber: user.employeeNumber || 'N/A',
          email: user.email || user.employeeNumber || 'N/A',
          projects: user.projects || [],
          countries: user.countries || [],
          role: user.role || 'Operator',
          isBlocked: user.isBlocked || false,
          password: user.password || user.employeeNumber || user.email || 'N/A',
          profilePic: user.profilePic || null,
          // Additional fields only if needed
          driverLicenseNumber: user.driverLicenseNumber || '',
          driverLicenseCode: user.driverLicenseCode || '',
          driverLicenseExpiry: user.driverLicenseExpiry || '',
          dateCompletedFirstAid: user.dateCompletedFirstAid || '',
          expiryDateFirstAid: user.expiryDateFirstAid || '',
          fireFightingDateAttended: user.fireFightingDateAttended || '',
          blastingLicenseDateIssued: user.blastingLicenseDateIssued || '',
        }));
        
        // Update state with new data
        setAllUsers(membersArray);
        setTotalUsers(teamResponse.totalUsers || membersArray.length);
        setTotalPages(teamResponse.totalPages || 1);
        setCurrentPage(1);
        setHasMoreData(teamResponse.hasNext || false);
        
        // Check if there are offline users in cache that need to be preserved
        const cachedMembers = await AsyncStorage.getItem('teamMembers');
        let offlineUsers = [];
        if (cachedMembers) {
          const cached = JSON.parse(cachedMembers);
          offlineUsers = cached.filter(user => user.pendingSync === true);
          console.log(`📤 Found ${offlineUsers.length} offline users to preserve during sync`);
        }
        
        // Merge offline users with server users
        const mergedUsers = [...offlineUsers, ...membersArray];
        
        const filtered = filterTeamMembersByPermissions(mergedUsers, userCountry, userProject, userPermissions, adminUid);
        setTeamMembers(filtered);
        setAllUsers(mergedUsers);
        
        // Save sync time
        const syncTime = new Date().toISOString();
        await AsyncStorage.setItem('teamMembers', JSON.stringify(mergedUsers));
        await AsyncStorage.setItem('teamMembersLastSync', syncTime);
        setLastSyncTime(syncTime);
        
        if (offlineUsers.length > 0) {
          showToast('success', 'Success', `Synced ${membersArray.length} users. ${offlineUsers.length} offline user(s) still pending.`);
        } else {
          showToast('success', 'Success', `Synced ${membersArray.length} of ${teamResponse.totalUsers || membersArray.length} users.`);
        }
      } else {
        throw new Error('Failed to load team members from MongoDB');
      }
    } catch (error) {
      console.error('Error syncing data:', error);
      showToast('error', 'Error', 'Failed to sync data with MongoDB.');
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  const handleMemberPress = (item, event) => {
    setSelectedMember(item);
    setDropdownVisible(true);
    const screenWidth = Dimensions.get('window').width;
    const screenHeight = Dimensions.get('window').height;
    const dropdownWidth = 200;
    const dropdownHeight = 300;
    let x = event.nativeEvent.pageX - dropdownWidth / 2;
    let y = event.nativeEvent.pageY - 50;
    if (x < 10) x = 10;
    else if (x + dropdownWidth > screenWidth - 10) x = screenWidth - dropdownWidth - 10;
    if (y < 10) y = event.nativeEvent.pageY + 30;
    else if (y + dropdownHeight > screenHeight - 10) y = screenHeight - dropdownHeight - 10;
    setDropdownPosition({ x, y });
  };

  const handleDropdownAction = async (action) => {
    if (action === 'view-profile') {
      setProfileModalVisible(true);
    } else if (action === 'view-password') {
      console.log('🔐 View Password - Selected Member:', {
        id: selectedMember?.id,
        name: selectedMember?.fullName,
        email: selectedMember?.email,
        password: selectedMember?.password,
        employeeNumber: selectedMember?.employeeNumber
      });
      
      // Always try to fetch fresh data to ensure we have the latest password
      console.log('🔐 Fetching fresh user data for password...');
      setIsLoadingPassword(true);
      try {
        const userResponse = await MongoAPI.getUserById(selectedMember.id);
        if (userResponse.success && userResponse.data) {
          console.log('🔐 Fresh user data received:', {
            id: userResponse.data._id,
            name: userResponse.data.name,
            email: userResponse.data.email,
            employeeNumber: userResponse.data.employeeNumber,
            password: userResponse.data.password,
            hasPassword: !!userResponse.data.password,
            hasEmail: !!userResponse.data.email,
            passwordType: typeof userResponse.data.password,
            emailType: typeof userResponse.data.email
          });
          
          // Use employeeNumber as password if password is missing
          const finalPassword = userResponse.data.password || 
                               userResponse.data.employeeNumber || 
                               userResponse.data.email || 
                               'Password not available';
          
          // Use email or employeeNumber as email if email is missing
          const finalEmail = userResponse.data.email || 
                            userResponse.data.employeeNumber || 
                            'N/A';
          
          // Update the selected member with fresh data
          const updatedMember = {
            ...selectedMember,
            password: finalPassword,
            email: finalEmail
          };
          setSelectedMember(updatedMember);
          
          // Also update in the allUsers array
          setAllUsers(prevUsers => 
            prevUsers.map(user => 
              user.id === selectedMember.id 
                ? { ...user, password: finalPassword, email: finalEmail }
                : user
            )
          );
          
          console.log('🔐 Password updated successfully:', finalPassword);
        } else {
          console.error('🔐 Failed to fetch user data:', userResponse);
          showToast('error', 'Error', 'Failed to fetch user data');
        }
      } catch (error) {
        console.error('🔐 Error fetching fresh user data:', error);
        showToast('error', 'Error', 'Failed to fetch password');
      } finally {
        setIsLoadingPassword(false);
      }
      
      setPasswordModalVisible(true);
    } else if (action === 'edit') {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Editing requires internet connection.');
        setDropdownVisible(false);
        return;
      }
      navigation.navigate('Edit_team_member', { teamData: selectedMember });
    } else if (action === 'share') {
      if (!selectedMember) return;
      
      try {
        const name = selectedMember.fullName || 'N/A';
        const username = selectedMember.email || selectedMember.employeeNumber || 'N/A';
        const password = selectedMember.password || 'N/A';
        
        // Create an attractive formatted message
        const shareText = `🔐 **User Credentials** 🔐\n\n` +
          `👤 **Name:** ${name}\n` +
          `📧 **Username:** ${username}\n` +
          `🔑 **Password:** ${password}\n\n` +
          `📱 *Shared from Titan Drilling App*\n` +
          `⏰ *${new Date().toLocaleDateString()} ${new Date().toLocaleTimeString()}*`;
        
        // Use React Native's Share API
        Share.share({
          message: shareText,
          title: `User Credentials - ${name}`,
        }, {
          dialogTitle: `Share ${name}'s Credentials`,
          excludedActivityTypes: [],
          tintColor: '#3481BC',
        });
      } catch (error) {
        console.error('Error sharing user credentials:', error);
        showToast('error', 'Error', 'Failed to share user credentials.');
      }
    } else if (action === 'block') {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Blocking requires internet connection.');
        setDropdownVisible(false);
        return;
      }
      try {
        const adminUid = await AsyncStorage.getItem('userUid');
        const newBlockedStatus = !selectedMember.isBlocked;
        
        // Update user in MongoDB
        const updateData = { isBlocked: newBlockedStatus };
        const response = await MongoAPI.updateUser(selectedMember.id, updateData);
        
        if (response.success) {
          showToast('success', 'Success', `Team member ${newBlockedStatus ? 'blocked' : 'unblocked'} successfully.`);
          const updatedMembers = teamMembers.map((member) =>
            member.id === selectedMember.id ? { ...member, isBlocked: newBlockedStatus } : member
          );
          const filtered = filterTeamMembersByPermissions(updatedMembers, userCountry, userProject, userPermissions, adminUid);
          setTeamMembers(filtered);
          await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
        } else {
          throw new Error('Failed to update user in MongoDB');
        }
      } catch (error) {
        console.error('Error updating block status:', error);
        showToast('error', 'Error', 'Failed to update team member status.');
      }
    } else if (action === 'delete') {
      try {
        const confirmDelete = await new Promise((resolve) => {
          Alert.alert(
            'Confirm Delete',
            `Are you sure you want to delete ${selectedMember?.fullName || 'this user'}?`,
            [
              { text: 'Cancel', style: 'cancel', onPress: () => resolve(false) },
              { text: 'Delete', style: 'destructive', onPress: () => resolve(true) },
            ],
            { cancelable: true }
          );
        });
        if (!confirmDelete) return;

        const adminUid = await AsyncStorage.getItem('userUid');
        if (!isOnline) {
          const newPendingActions = [...pendingActions, { type: 'deleteTeamMember', adminUid, memberId: selectedMember.id }];
          setPendingActions(newPendingActions);
          await AsyncStorage.setItem('pendingActionsTeamMemberHome', JSON.stringify(newPendingActions));
          const updatedMembers = teamMembers.filter((member) => member.id !== selectedMember.id);
          const filtered = filterTeamMembersByPermissions(updatedMembers, userCountry, userProject, userPermissions, adminUid);
          setTeamMembers(filtered);
          await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
          showToast('info', 'Offline Mode', 'Deletion will sync when online.');
        } else {
          // Delete from MongoDB
          const response = await MongoAPI.deleteUser(selectedMember.id);
          if (response.success) {
            const updatedMembers = teamMembers.filter((member) => member.id !== selectedMember.id);
            const filtered = filterTeamMembersByPermissions(updatedMembers, userCountry, userProject, userPermissions, adminUid);
            setTeamMembers(filtered);
            await AsyncStorage.setItem('teamMembers', JSON.stringify(filtered));
            showToast('success', 'Success', 'Team member deleted successfully.');
          } else {
            throw new Error('Failed to delete user from MongoDB');
          }
        }
      } catch (error) {
        console.error('Error deleting team member:', error);
        showToast('error', 'Error', 'Failed to delete team member.');
      }
    } else if (action === 'give-permission') {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Assigning permissions requires internet connection.');
        setDropdownVisible(false);
        return;
      }
      navigation.navigate('TEpremissionR', { member: selectedMember });
    }
    setDropdownVisible(false);
  };

  const handleResetToDefault = async () => {
    if (!selectedMember) return;
    
    try {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Password reset requires internet connection.');
        return;
      }

      const newPassword = selectedMember.employeeNumber || 'default123';
      
      // Update password in MongoDB
      const updateData = { password: newPassword };
      const response = await MongoAPI.updateUser(selectedMember.id, updateData);
      
      if (response.success) {
        // Update local state
        const updatedMembers = teamMembers.map((member) =>
          member.id === selectedMember.id ? { ...member, password: newPassword } : member
        );
        setTeamMembers(updatedMembers);
        await AsyncStorage.setItem('teamMembers', JSON.stringify(updatedMembers));

        // Update selected member for modal display
        setSelectedMember({ ...selectedMember, password: newPassword });

        showToast('success', 'Success', 'Password reset to default successfully.');
      } else {
        throw new Error('Failed to update password in MongoDB');
      }
    } catch (error) {
      console.error('Error resetting password:', error);
      showToast('error', 'Error', 'Failed to reset password.');
    }
  };

  const handleGenerateComplex = async () => {
    if (!selectedMember) return;
    try {
      if (!isOnline) {
        showToast('info', 'Offline Mode', 'Password generation requires internet connection.');
        return;
      }
      // Simple 8+ char generator: letters and numbers only
      const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
      let newPassword = '';
      for (let i = 0; i < 8; i++) {
        newPassword += chars.charAt(Math.floor(Math.random() * chars.length));
      }
      
      // Update password in MongoDB
      const updateData = { password: newPassword };
      const response = await MongoAPI.updateUser(selectedMember.id, updateData);
      
      if (response.success) {
        const updatedMembers = teamMembers.map((member) => member.id === selectedMember.id ? { ...member, password: newPassword } : member);
        setTeamMembers(updatedMembers);
        await AsyncStorage.setItem('teamMembers', JSON.stringify(updatedMembers));
        setSelectedMember({ ...selectedMember, password: newPassword });
        showToast('success', 'Success', 'Password generated.');
      } else {
        throw new Error('Failed to update password in MongoDB');
      }
    } catch (error) {
      console.error('Error generating password:', error);
      showToast('error', 'Error', 'Failed to generate password.');
    }
  };

  const handleCopyToClipboard = async () => {
    if (!selectedMember) return;
    
    try {
      const employeeId = selectedMember.employeeNumber || 'N/A';
      const email = selectedMember.email || 'N/A';
      const password = selectedMember.password || 'N/A';
      
      const clipboardText = `Employee ID: ${employeeId}\nEmail: ${email}\nPassword: ${password}`;
      
      await Clipboard.setString(clipboardText);
      showToast('success', 'Copied!', 'User information copied to clipboard.');
    } catch (error) {
      console.error('Error copying to clipboard:', error);
      showToast('error', 'Error', 'Failed to copy to clipboard.');
    }
  };

  const filteredTeamMembers = teamMembers.filter((member) => {
    // Search filter
    const matchesSearch = searchQuery === '' || 
      (member.fullName?.toLowerCase().includes(searchQuery.toLowerCase()) ||
       member.employeeNumber?.toLowerCase().includes(searchQuery.toLowerCase()) ||
       member.role?.toLowerCase().includes(searchQuery.toLowerCase()));
    
    // Country filter
    const matchesCountry = selectedCountry === 'All' || 
      (member.countries && member.countries.includes(selectedCountry));
    
    // Project filter  
    const matchesProject = selectedProject === 'All' || 
      (member.projects && member.projects.includes(selectedProject));
    
    // Role filter
    const matchesRole = selectedRole === 'All' || 
      member.role === selectedRole;
    
    const result = matchesSearch && matchesCountry && matchesProject && matchesRole;
    
    // Debug logging for filtering
    if (!result && searchQuery !== '') {
      console.log('🔍 Member filtered out by search:', {
        name: member.fullName,
        employeeNumber: member.employeeNumber,
        searchQuery: searchQuery,
        matchesSearch: matchesSearch
      });
    }
    
    return result;
  });

  // Use filtered team members directly - admin profile is already included in the data
  const paginatedMembers = filteredTeamMembers.map(member => ({
    ...member,
    isSelf: member.id === adminProfile?.id
  }));

  // Apply filters when they change - simplified logic
  useEffect(() => {
    const applyFilters = async () => {
      try {
        // Use allUsers if available, otherwise load from cache
        let allMembers = allUsers;
        if (allMembers.length === 0) {
          const cachedMembers = await AsyncStorage.getItem('teamMembers');
          if (cachedMembers) {
            allMembers = JSON.parse(cachedMembers);
          }
        }
        
        if (allMembers.length > 0) {
          // Apply admin permissions filter first - but be more permissive
          const myUid = await AsyncStorage.getItem('userUid');
          const permissionFiltered = filterTeamMembersByPermissions(allMembers, userCountry, userProject, userPermissions, myUid || '');
          
          console.log('🔍 Filter application result:', {
            allMembers: allMembers.length,
            afterPermissionFilter: permissionFiltered.length,
            selectedCountry,
            selectedProject,
            selectedRole,
            searchQuery
          });
          
          setTeamMembers(permissionFiltered);
        }
      } catch (error) {
        console.error('Error applying filters:', error);
      }
    };
    applyFilters();
  }, [allUsers]); // Only depend on allUsers, not on filter changes

  // Footer showing pagination info and load more button
  const renderFooter = () => {
    if (paginatedMembers.length > 0) {
      return (
        <View style={styles.paginationInfo}>
          <Text style={styles.paginationText}>
            Showing {paginatedMembers.length} of {totalUsers} team members
          </Text>
          {isLoadingMore && (
            <View style={styles.loadingContainer}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.loadMoreText}>Loading more users...</Text>
            </View>
          )}
          {hasMoreData && !isLoadingMore && (
            <TouchableOpacity
              style={styles.loadMoreButton}
              onPress={loadMoreUsers}
            >
              <Text style={styles.loadMoreText}>Load More Users</Text>
            </TouchableOpacity>
          )}
          {!hasMoreData && allUsers.length > 0 && (
            <Text style={[styles.loadMoreText, { color: '#4CAF50', fontWeight: 'bold' }]}>
              All {totalUsers} users loaded!
            </Text>
          )}
        </View>
      );
    }
    return null;
  };

  // Render a team member (admin or not)
  const renderTeamMember = ({ item }) => {
    const isSelf = item.isSelf;
    const isPendingSync = item.pendingSync === true;
    return (
      <View style={[
        styles.teamMemberContainer,
        isSelf ? { borderWidth: 1, borderColor: '#4CAF50', marginBottom: 8 } : {},
        isPendingSync ? { borderLeftWidth: 4, borderLeftColor: '#FF9800' } : {},
      ]}>
        <View style={styles.teamMemberLeft}>
          {item.profilePic && (
            <Image
              source={{ uri: item.profilePic.url }}
              style={styles.profileImage}
            />
          )}
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Employee Number: </Text>
            {item.employeeNumber || 'N/A'}
            {isPendingSync && <Text style={{ color: '#FF9800', fontWeight: 'bold' }}> 📤 Offline</Text>}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Name: </Text>
            {item.fullName || 'N/A'}
            {isSelf && <Text style={{ color: '#4CAF50', fontWeight: 'bold' }}> (My Profile)</Text>}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Role: </Text>
            {item.role || 'Operator'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Projects: </Text>
            {item.projects?.length > 0 ? item.projects.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Country: </Text>
            {item.countries?.length > 0 ? item.countries.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Status: </Text>
            {item.isBlocked ? 'Blocked' : 'Active'}
          </Text>
          {pwQueueStatusMap[item.id] === 'queued' && (
            <Text style={[styles.teamMemberText, { color: '#ff9800', fontWeight: '600' }]}>Password update queued…</Text>
          )}
          {pwQueueStatusMap[item.id] === 'done' && (
            <Text style={[styles.teamMemberText, { color: '#2e7d32', fontWeight: '600' }]}>Password updated</Text>
          )}
          {pwQueueStatusMap[item.id] === 'error' && (
            <Text style={[styles.teamMemberText, { color: '#d32f2f', fontWeight: '600' }]}>Password update failed</Text>
          )}
        </View>
        <View style={styles.teamMemberRight}>
          <View style={styles.roleBadge}>
            <Text style={styles.teamMemberRole}>{item.role || 'Operator'}</Text>
          </View>
          <TouchableOpacity 
            style={styles.menuButton}
            onPress={(event) => handleMemberPress(item, event)}
          >
            <MaterialIcons name="more-vert" size={width * 0.06} color="#666" />
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  // Dropdown logic: show only allowed options for self, all for others
  const renderDropdown = () => (
    <Modal visible={dropdownVisible} transparent animationType="none">
      <TouchableOpacity
        style={styles.dropdownOverlay}
        onPress={() => setDropdownVisible(false)}
      >
        <View
          style={[
            styles.dropdownContainer,
            { 
              top: dropdownPosition.y, 
              left: dropdownPosition.x,
              position: 'absolute',
            },
          ]}
        >
          {selectedMember?.isSelf ? (
            // My profile: show Edit (if allowed), View Password, Share, and Give Permission
            <>
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('view-profile')}
              >
                <MaterialIcons name="person" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>View Profile</Text>
              </TouchableOpacity>
              {userPermissions.includes('edit') && (
                <TouchableOpacity
                  style={styles.dropdownItem}
                  onPress={() => handleDropdownAction('edit')}
                >
                  <MaterialIcons name="edit" size={width * 0.05} color="#333" />
                  <Text style={styles.dropdownText}>Edit</Text>
                </TouchableOpacity>
              )}
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('share')}
              >
                <MaterialIcons name="share" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>Share</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('view-password')}
              >
                <MaterialIcons name="lock" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>View Password</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.dropdownItem}
                onPress={() => handleDropdownAction('give-permission')}
              >
                <MaterialIcons name="security" size={width * 0.05} color="#333" />
                <Text style={styles.dropdownText}>Give Permission</Text>
              </TouchableOpacity>
            </>
          ) : (
            // Other profiles: show all options as per permissions
            [
              { icon: 'person', text: 'View Profile', action: 'view-profile' },
              { icon: 'edit', text: 'Edit', action: 'edit', permission: 'edit' },
              { icon: 'share', text: 'Share', action: 'share' },
              { icon: 'lock', text: 'View Password', action: 'view-password' },
              { icon: 'security', text: 'Give Permission', action: 'give-permission' },
              { icon: 'block', text: selectedMember?.isBlocked ? 'Unblock Team Member' : 'Block Team Member', action: 'block', permission: 'edit' },
              { icon: 'delete', text: 'Delete', action: 'delete' },
            ].map((option) => {
              if (option.permission && !userPermissions.includes(option.permission)) {
                return null;
              }
              return (
                <TouchableOpacity
                  key={option.action}
                  style={styles.dropdownItem}
                  onPress={() => handleDropdownAction(option.action)}
                >
                  <MaterialIcons name={option.icon} size={width * 0.05} color="#333" />
                  <Text style={styles.dropdownText}>{option.text}</Text>
                </TouchableOpacity>
              );
            })
          )}
        </View>
      </TouchableOpacity>
    </Modal>
  );

  const handleFilterPress = (type) => {
    let ref;
    if (type === 'country') ref = countryButtonRef;
    else if (type === 'project') ref = projectButtonRef;
    else if (type === 'role') ref = roleButtonRef;

    if (ref && ref.current) {
      ref.current.measureInWindow((x, y, width, height) => {
        const left = x;
        const top = y + height + 4;
        if (type === 'country') {
          setCountryDropdownPosition({ x: left, y: top, width });
          setCountryDropdownVisible(true);
          setProjectDropdownVisible(false);
          setRoleDropdownVisible(false);
        } else if (type === 'project') {
          setProjectDropdownPosition({ x: left, y: top, width });
          setProjectDropdownVisible(true);
          setCountryDropdownVisible(false);
          setRoleDropdownVisible(false);
        } else if (type === 'role') {
          setRoleDropdownPosition({ x: left, y: top, width });
          setRoleDropdownVisible(true);
          setCountryDropdownVisible(false);
          setProjectDropdownVisible(false);
        }
      });
    }
  };

  const renderFilterDropdown = (type, options, counts, selected, setSelected, visible, setVisible) => {
    let position = { x: 0, y: 0, width: 0 };
    if (type === 'country') position = countryDropdownPosition;
    else if (type === 'project') position = projectDropdownPosition;
    else if (type === 'role') position = roleDropdownPosition;

    return (
      <Modal visible={visible} transparent animationType="none">
        <TouchableOpacity
          style={styles.dropdownOverlay}
          onPress={() => setVisible(false)}
          activeOpacity={1}
        >
          <View
            style={[
              styles.dropdownContainer,
              {
                top: position.y,
                left: position.x,
                width: position.width,
                minWidth: 80,
                maxWidth: 300,
              },
            ]}
          >
            <ScrollView
              nestedScrollEnabled={true}
              showsVerticalScrollIndicator={true}
              style={{ maxHeight: 200 }}
            >
              {options.map((option) => (
                <TouchableOpacity
                  key={option}
                  style={[
                    styles.dropdownItem,
                    selected === option && styles.dropdownItemActive,
                  ]}
                  onPress={() => {
                    setSelected(option);
                    setVisible(false);
                  }}
                >
                  <MaterialIcons
                    name={type === 'country' ? 'public' : type === 'project' ? 'work' : 'person'}
                    size={width * 0.05}
                    color={selected === option ? '#3481BC' : '#333'}
                  />
                  <Text
                    style={[
                      styles.dropdownText,
                      selected === option && styles.dropdownTextActive,
                    ]}
                  >
                    {option} ({counts[option] || 0})
                  </Text>
                </TouchableOpacity>
              ))}
            </ScrollView>
          </View>
        </TouchableOpacity>
      </Modal>
    );
  };

  const renderFilterBar = () => {
    const { countries, projects, roles, countryCounts, projectCounts, roleCounts } = getFilterOptions();
    return (
      <View style={{ backgroundColor: '#f5f5f5' }}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.filterBar}
        >
          <TouchableOpacity
            ref={countryButtonRef}
            style={styles.filterButton}
            onPress={() => handleFilterPress('country')}
          >
            <MaterialIcons name="public" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Country: {selectedCountry} ({countryCounts[selectedCountry] || 0})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          <TouchableOpacity
            ref={projectButtonRef}
            style={styles.filterButton}
            onPress={() => handleFilterPress('project')}
          >
            <MaterialIcons name="work" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Project: {selectedProject} ({projectCounts[selectedProject] || 0})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          <TouchableOpacity
            ref={roleButtonRef}
            style={styles.filterButton}
            onPress={() => handleFilterPress('role')}
          >
            <MaterialIcons name="person" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Role: {selectedRole} ({roleCounts[selectedRole] || 0})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setPermissionModalVisible(true)}
          >
            <MaterialIcons name="verified-user" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>Permissions</Text>
          </TouchableOpacity>
        </ScrollView>
        {renderFilterDropdown(
          'country',
          countries,
          countryCounts,
          selectedCountry,
          setSelectedCountry,
          countryDropdownVisible,
          setCountryDropdownVisible
        )}
        {renderFilterDropdown(
          'project',
          projects,
          projectCounts,
          selectedProject,
          setSelectedProject,
          projectDropdownVisible,
          setProjectDropdownVisible
        )}
        {renderFilterDropdown(
          'role',
          roles,
          roleCounts,
          selectedRole,
          setSelectedRole,
          roleDropdownVisible,
          setRoleDropdownVisible
        )}
      </View>
    );
  };

  const renderAdminProfile = () => {
    if (!adminProfile) return null;
    return (
      <View style={[styles.teamMemberContainer, { borderWidth: 1, borderColor: '#4CAF50', marginBottom: 8 }]}> 
        <View style={styles.teamMemberLeft}>
          {adminProfile.profilePic && (
            <Image
              source={{ uri: adminProfile.profilePic.url }}
              style={styles.profileImage}
            />
          )}
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Employee Number: </Text>
            {adminProfile.employeeNumber || 'N/A'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Name: </Text>
            {adminProfile.fullName || 'N/A'} <Text style={{ color: '#4CAF50', fontWeight: 'bold' }}>(My Profile)</Text>
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Role: </Text>
            {adminProfile.role || 'Operator'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Projects: </Text>
            {adminProfile.projects?.length > 0 ? adminProfile.projects.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Country: </Text>
            {adminProfile.countries?.length > 0 ? adminProfile.countries.join(', ') : 'None'}
          </Text>
          <Text style={styles.teamMemberText}>
            <Text style={styles.labelText}>Status: </Text>
            {adminProfile.isBlocked ? 'Blocked' : 'Active'}
          </Text>
          {pwQueueStatusMap[adminProfile.id] === 'queued' && (
            <Text style={[styles.teamMemberText, { color: '#ff9800', fontWeight: '600' }]}>Password update queued…</Text>
          )}
          {pwQueueStatusMap[adminProfile.id] === 'done' && (
            <Text style={[styles.teamMemberText, { color: '#2e7d32', fontWeight: '600' }]}>Password updated</Text>
          )}
          {pwQueueStatusMap[adminProfile.id] === 'error' && (
            <Text style={[styles.teamMemberText, { color: '#d32f2f', fontWeight: '600' }]}>Password update failed</Text>
          )}
        </View>
        <View style={styles.teamMemberRight}>
          <View style={styles.roleBadge}>
            <Text style={styles.teamMemberRole}>{adminProfile.role || 'Operator'}</Text>
          </View>
          <TouchableOpacity 
            style={styles.menuButton}
            onPress={(event) => handleMemberPress(adminProfile, event)}
          >
            <MaterialIcons name="more-vert" size={width * 0.06} color="#666" />
          </TouchableOpacity>
        </View>
      </View>
    );
  };

  const renderPermissionModal = () => (
    <Modal visible={permissionModalVisible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={[styles.syncModalContent, { maxWidth: '90%', minWidth: '70%' }]}>
          <Text style={styles.syncModalText}>User Permissions</Text>
          <View style={styles.syncIcondiv}>
            <View style={{ flex: 1 }}>
              <Text style={styles.permissionLabel}>Country:</Text>
              <Text style={styles.permissionValue}>{userCountry.length > 0 ? userCountry.join(', ') : 'None'}</Text>
              <Text style={styles.permissionLabel}>Project:</Text>
              <Text style={styles.permissionValue}>{userProject.length > 0 ? userProject.join(', ') : 'None'}</Text>
              <Text style={styles.permissionLabel}>Permissions:</Text>
              <Text style={styles.permissionValue}>{userPermissions.length > 0 ? userPermissions.join(', ') : 'None'}</Text>
            </View>
          </View>
          <TouchableOpacity
            style={styles.closeButton}
            onPress={() => setPermissionModalVisible(false)}
          >
            <Text style={styles.closeButtonText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  const renderProfileModal = () => (
    <Modal visible={profileModalVisible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={[styles.syncModalContent, { maxWidth: '95%', minWidth: '80%', maxHeight: '80%' }]}>
          <Text style={styles.syncModalText}>Team Member Profile</Text>
          <ScrollView style={{ width: '100%' }} showsVerticalScrollIndicator={false}>
            <View style={styles.syncIcondiv}>
              <View style={{ width: '100%' }}>
                {/* Profile Picture */}
                {selectedMember?.profilePic && (
                  <View style={styles.profileImageContainer}>
                    <Image
                      source={{ uri: selectedMember.profilePic.url }}
                      style={styles.profileModalImage}
                    />
                  </View>
                )}
                
                {/* Basic Information */}
                <View style={styles.profileSection}>
                  <Text style={styles.profileSectionTitle}>Basic Information</Text>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="badge" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Employee ID:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.employeeNumber || 'N/A'}</Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="person" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Full Name:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.fullName || 'N/A'}</Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="email" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Email:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.email || 'N/A'}</Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="work" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Role:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.role || 'N/A'}</Text>
                  </View>
                </View>

                {/* Projects & Countries */}
                <View style={styles.profileSection}>
                  <Text style={styles.profileSectionTitle}>Projects & Countries</Text>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="business" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Projects:</Text>
                    <Text style={styles.profileInfoValue}>
                      {selectedMember?.projects?.length > 0 ? selectedMember.projects.join(', ') : 'None'}
                    </Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="public" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Countries:</Text>
                    <Text style={styles.profileInfoValue}>
                      {selectedMember?.countries?.length > 0 ? selectedMember.countries.join(', ') : 'None'}
                    </Text>
                  </View>
                </View>

                {/* Status & Security */}
                <View style={styles.profileSection}>
                  <Text style={styles.profileSectionTitle}>Status & Security</Text>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="security" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Status:</Text>
                    <Text style={[styles.profileInfoValue, { 
                      color: selectedMember?.isBlocked ? '#d32f2f' : '#4CAF50',
                      fontWeight: 'bold'
                    }]}>
                      {selectedMember?.isBlocked ? 'Blocked' : 'Active'}
                    </Text>
                  </View>
                  <View style={styles.profileInfoRow}>
                    <MaterialIcons name="lock" size={width * 0.05} color="#3481BC" />
                    <Text style={styles.profileInfoLabel}>Password:</Text>
                    <Text style={styles.profileInfoValue}>{selectedMember?.password || 'N/A'}</Text>
                  </View>
                </View>

                {/* Additional Information */}
                {selectedMember?.driverLicenseNumber && (
                  <View style={styles.profileSection}>
                    <Text style={styles.profileSectionTitle}>Additional Information</Text>
                    <View style={styles.profileInfoRow}>
                      <MaterialIcons name="drive-eta" size={width * 0.05} color="#3481BC" />
                      <Text style={styles.profileInfoLabel}>Driver License:</Text>
                      <Text style={styles.profileInfoValue}>{selectedMember.driverLicenseNumber}</Text>
                    </View>
                    {selectedMember.driverLicenseCode && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="code" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>License Code:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.driverLicenseCode}</Text>
                      </View>
                    )}
                    {selectedMember.driverLicenseExpiry && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="event" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>License Expiry:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.driverLicenseExpiry}</Text>
                      </View>
                    )}
                  </View>
                )}

                {/* Training & Certifications */}
                {(selectedMember?.dateCompletedFirstAid || selectedMember?.fireFightingDateAttended || selectedMember?.blastingLicenseDateIssued) && (
                  <View style={styles.profileSection}>
                    <Text style={styles.profileSectionTitle}>Training & Certifications</Text>
                    {selectedMember.dateCompletedFirstAid && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="healing" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>First Aid Completed:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.dateCompletedFirstAid}</Text>
                      </View>
                    )}
                    {selectedMember.expiryDateFirstAid && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="schedule" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>First Aid Expiry:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.expiryDateFirstAid}</Text>
                      </View>
                    )}
                    {selectedMember.fireFightingDateAttended && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="local-fire-department" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>Fire Fighting Training:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.fireFightingDateAttended}</Text>
                      </View>
                    )}
                    {selectedMember.blastingLicenseDateIssued && (
                      <View style={styles.profileInfoRow}>
                        <MaterialIcons name="explosion" size={width * 0.05} color="#3481BC" />
                        <Text style={styles.profileInfoLabel}>Blasting License:</Text>
                        <Text style={styles.profileInfoValue}>{selectedMember.blastingLicenseDateIssued}</Text>
                      </View>
                    )}
                  </View>
                )}
              </View>
            </View>
          </ScrollView>
          <TouchableOpacity
            style={styles.closeButton}
            onPress={() => setProfileModalVisible(false)}
          >
            <Text style={styles.closeButtonText}>Close</Text>
          </TouchableOpacity>
        </View>
      </View>
    </Modal>
  );

  return (
    <View style={styles.container}>
      {isLoading && (
        <Modal visible={isLoading} transparent animationType="fade">
          <View style={styles.loadingModalContainer}>
            <View style={styles.loadingModalContent}>
              <Text style={styles.loadingModalText}>Please Wait</Text>
              <View style={styles.loadingIcondiv}>
                <Animated.Image
                  source={require('../../../Images/adminlogin/sync.png')}
                  style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
                />
                <Text style={styles.loadingModalTextadd}>Processing data syncing</Text>
              </View>
            </View>
          </View>
        </Modal>
      )}
      
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Team Management</Text>
          </View>
          <View style={styles.headerRight}>
            {/* Online/Offline Toggle */}
            <TouchableOpacity 
              onPress={toggleOfflineMode}
              style={[styles.networkToggle, { backgroundColor: isOnline && !forceOfflineMode ? '#4CAF50' : '#757575' }]}
            >
              <MaterialIcons 
                name={isOnline && !forceOfflineMode ? 'wifi' : 'wifi-off'} 
                size={width * 0.045} 
                color="#fff" 
              />
            </TouchableOpacity>
            
            {/* Cloud Sync Button */}
            <TouchableOpacity onPress={handleCloudSync} style={styles.refreshButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
              {pendingActions.length > 0 && (
                <View style={styles.pendingBadge}>
                  <Text style={styles.pendingBadgeText}>{pendingActions.length}</Text>
                </View>
              )}
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search by name, employee number, or role..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={setSearchQuery}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons name="filter-list" size={width * 0.06} color="black" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {renderFilterBar()}

      <FlatList
        data={paginatedMembers}
        renderItem={renderTeamMember}
        keyExtractor={(item) => item.id + (item.isSelf ? '_self' : '')}
        style={styles.teamList}
        ListFooterComponent={renderFooter}
        ListEmptyComponent={<Text style={styles.emptyText}>No team members found.</Text>}
        removeClippedSubviews={true}
        maxToRenderPerBatch={10}
        updateCellsBatchingPeriod={100}
        initialNumToRender={10}
        windowSize={10}
      />

      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          {userPermissions.includes('create') && (
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('Team_memberHeader')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.buttonText}>Add Team Member</Text>
            </TouchableOpacity>
          )}
        </LinearGradient>
      </View>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={passwordModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>User Information</Text>
            <View style={styles.syncIcondiv}>
              <View style={{ width: '100%' }}>
                <View style={styles.infoRow}>
                  <MaterialIcons name="badge" size={width * 0.05} color="#3481BC" style={{ marginRight: width * 0.02 }} />
                  <Text style={styles.infoLabel}>Employee ID:</Text>
                  <Text style={styles.infoValue}>{selectedMember?.employeeNumber || 'N/A'}</Text>
                </View>
                <View style={styles.infoRow}>
                  <MaterialIcons name="email" size={width * 0.05} color="#3481BC" style={{ marginRight: width * 0.02 }} />
                  <Text style={styles.infoLabel}>Email:</Text>
                  <Text style={styles.infoValue}>{selectedMember?.email || 'N/A'}</Text>
                </View>
                <View style={styles.infoRow}>
                  <MaterialIcons name="lock" size={width * 0.05} color="#3481BC" style={{ marginRight: width * 0.02 }} />
                  <Text style={styles.infoLabel}>Password:</Text>
                  {isLoadingPassword ? (
                    <View style={{ flexDirection: 'row', alignItems: 'center' }}>
                      <Animated.Image
                        source={require('../../../Images/adminlogin/sync.png')}
                        style={[styles.syncIcon, { 
                          transform: [{ rotate: rotateAnim.interpolate({ 
                            inputRange: [0, 1], 
                            outputRange: ['0deg', '360deg'] 
                          }) }] 
                        }]}
                      />
                      <Text style={[styles.infoValue, { marginLeft: 10 }]}>Loading...</Text>
                    </View>
                  ) : (
                    <Text style={[styles.infoValue, { 
                      color: selectedMember?.password && selectedMember?.password !== 'N/A' ? '#2ecc71' : '#e74c3c',
                      fontWeight: 'bold'
                    }]}>
                      {selectedMember?.password || 'Password not available'}
                    </Text>
                  )}
                </View>
                
                {/* Refresh Password Button */}
                {(!selectedMember?.password || selectedMember?.password === 'N/A') && !isLoadingPassword && (
                  <View style={styles.passwordButtonsContainer}>
                    <TouchableOpacity
                      style={[styles.passwordButton, { backgroundColor: '#3498db' }]}
                      onPress={async () => {
                        setIsLoadingPassword(true);
                        try {
                          const userResponse = await MongoAPI.getUserById(selectedMember.id);
                          if (userResponse.success && userResponse.data) {
                            // Use employeeNumber as password if password is missing
                            const finalPassword = userResponse.data.password || 
                                                 userResponse.data.employeeNumber || 
                                                 userResponse.data.email || 
                                                 'Password not available';
                            
                            // Use email or employeeNumber as email if email is missing
                            const finalEmail = userResponse.data.email || 
                                              userResponse.data.employeeNumber || 
                                              'N/A';
                            
                            const updatedMember = {
                              ...selectedMember,
                              password: finalPassword,
                              email: finalEmail
                            };
                            setSelectedMember(updatedMember);
                            
                            // Also update in the allUsers array
                            setAllUsers(prevUsers => 
                              prevUsers.map(user => 
                                user.id === selectedMember.id 
                                  ? { ...user, password: finalPassword, email: finalEmail }
                                  : user
                              )
                            );
                            
                            showToast('success', 'Success', 'Password refreshed');
                          }
                        } catch (error) {
                          showToast('error', 'Error', 'Failed to refresh password');
                        } finally {
                          setIsLoadingPassword(false);
                        }
                      }}
                    >
                      <MaterialIcons name="refresh" size={16} color="#fff" />
                      <Text style={styles.passwordButtonText}>Refresh Password</Text>
                    </TouchableOpacity>
                  </View>
                )}
                
                {/* <View style={styles.passwordButtonsContainer}>
                  <TouchableOpacity
                    style={styles.passwordButton}
                    onPress={() => handleResetToDefault()}
                  >
                    <MaterialIcons name="refresh" size={width * 0.04} color="#fff" />
                    <Text style={styles.passwordButtonText}>Reset to Default</Text>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={styles.passwordButton}
                    onPress={() => handleGenerateComplex()}
                  >
                    <MaterialIcons name="security" size={width * 0.04} color="#fff" />
                    <Text style={styles.passwordButtonText}>Generate Complex</Text>
                  </TouchableOpacity>
                </View> */}
                <TouchableOpacity
                  style={styles.copyButton}
                  onPress={() => handleCopyToClipboard()}
                >
                  <MaterialIcons name="content-copy" size={width * 0.04} color="#fff" />
                  <Text style={styles.copyButtonText}>Copy All to Clipboard</Text>
                </TouchableOpacity>
              </View>
            </View>
            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => setPasswordModalVisible(false)}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal visible={filterModalVisible} transparent animationType="slide">
        <View style={styles.filterModalContainer}>
          <View style={styles.filterModalContent}>
            <Text style={styles.filterModalTitle}>Filter Team Members</Text>
            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => setFilterModalVisible(false)}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {renderDropdown()}
      {renderPermissionModal()}
      {renderProfileModal()}
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  networkStatus: {
    width: width * 0.08,
    height: width * 0.08,
    borderRadius: width * 0.04,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: width * 0.02,
  },
  networkToggle: {
    width: width * 0.09,
    height: width * 0.09,
    borderRadius: width * 0.045,
    justifyContent: 'center',
    alignItems: 'center',
    marginRight: width * 0.02,
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 3,
  },
  pendingBadge: {
    position: 'absolute',
    top: -4,
    right: -4,
    backgroundColor: '#FF5722',
    borderRadius: 10,
    minWidth: 20,
    height: 20,
    justifyContent: 'center',
    alignItems: 'center',
    paddingHorizontal: 4,
  },
  pendingBadgeText: {
    color: '#fff',
    fontSize: 10,
    fontWeight: 'bold',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginHorizontal: width * 0.01,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  filterButtonText: {
    fontSize: width * 0.035,
    color: '#333',
    marginLeft: width * 0.015,
    marginRight: width * 0.01,
  },
  filterIcon: {
    marginRight: width * 0.01,
  },
  filterBar: {
    flexDirection: 'row',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    maxWidth: 400,
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    justifyContent: 'center',
    width: '100%',
    marginBottom: 10,
  },
  permissionLabel: {
    fontWeight: 'bold',
    color: '#333',
    fontSize: width * 0.037,
    marginTop: 6,
    marginBottom: 2,
  },
  permissionValue: {
    color: '#333',
    fontSize: width * 0.037,
    marginBottom: 4,
    flexWrap: 'wrap',
    width: '100%',
    lineHeight: width * 0.045,
  },
  teamList: {
    flex: 1,
    marginTop: width * 0.02,
  },
  teamMemberContainer: {
    flexDirection: 'row',
    padding: width * 0.04,
    borderBottomWidth: 1,
    borderBottomColor: '#E0E0E0',
    backgroundColor: '#fff',
    marginHorizontal: width * 0.02,
    marginVertical: width * 0.01,
    borderRadius: 8,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  teamMemberLeft: {
    flex: 2,
    paddingRight: width * 0.02,
  },
  teamMemberRight: {
    flex: 1,
    alignItems: 'flex-end',
    justifyContent: 'space-between',
    flexDirection: 'column',
  },
  teamMemberText: {
    fontSize: width * 0.035,
    color: '#333',
    marginBottom: width * 0.015,
    lineHeight: width * 0.05,
  },
  labelText: {
    fontWeight: 'bold',
    color: '#333',
  },
  teamMemberRole: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#fff',
  },
  roleBadge: {
    backgroundColor: '#4CAF50',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: 12,
    alignItems: 'center',
    marginBottom: width * 0.02,
  },
  menuButton: {
    padding: width * 0.01,
    borderRadius: width * 0.02,
    backgroundColor: '#f5f5f5',
    borderWidth: 1,
    borderColor: '#ddd',
  },
  emptyText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'transparent',
  },
  dropdownContainer: {
    position: 'absolute',
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 5,
    width: 200,
    maxWidth: width * 0.8,
    minWidth: 150,
    zIndex: 1000,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownItemActive: {
    backgroundColor: '#e6f0ff',
  },
  dropdownText: {
    marginLeft: width * 0.025,
    fontSize: width * 0.04,
    color: '#333',
  },
  dropdownTextActive: {
    color: '#3481BC',
    fontWeight: 'bold',
  },
  closeButton: {
    backgroundColor: '#3481BC',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.012,
    marginTop: 10,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  filterModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  filterModalContent: {
    width: '80%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  filterModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
  },
  profileImage: {
    width: width * 0.1,
    height: width * 0.1,
    borderRadius: width * 0.05,
    marginBottom: width * 0.015,
  },
  loadingModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  loadingModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  loadingIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  loadingModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  loadingIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  infoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.025,
    paddingVertical: width * 0.01,
  },
  infoLabel: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#333',
    marginRight: width * 0.02,
    minWidth: 80,
  },
  infoValue: {
    fontSize: width * 0.035,
    color: '#666',
    flex: 1,
  },
  passwordButtonsContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginTop: width * 0.02,
    gap: width * 0.02,
  },
  passwordButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#3481BC',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    flex: 1,
    justifyContent: 'center',
  },
  passwordButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  copyButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#4CAF50',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    justifyContent: 'center',
    marginTop: width * 0.015,
  },
  copyButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: 'bold',
    marginLeft: width * 0.01,
  },
  // Profile Modal Styles
  profileImageContainer: {
    alignItems: 'center',
    marginBottom: width * 0.03,
  },
  profileModalImage: {
    width: width * 0.15,
    height: width * 0.15,
    borderRadius: width * 0.075,
    borderWidth: 3,
    borderColor: '#3481BC',
  },
  profileSection: {
    marginBottom: width * 0.04,
    paddingHorizontal: width * 0.02,
  },
  profileSectionTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#3481BC',
    marginBottom: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    paddingBottom: width * 0.01,
  },
  profileInfoRow: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: width * 0.015,
    paddingVertical: width * 0.01,
  },
  profileInfoLabel: {
    fontSize: width * 0.035,
    fontWeight: 'bold',
    color: '#333',
    marginLeft: width * 0.02,
    marginRight: width * 0.02,
    minWidth: 120,
  },
  profileInfoValue: {
    fontSize: width * 0.035,
    color: '#666',
    flex: 1,
    lineHeight: width * 0.045,
  },
  // Pagination Loading Styles
  loadingFooter: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.04,
  },
  loadingFooterIcon: {
    width: width * 0.05,
    height: width * 0.05,
    marginRight: width * 0.02,
  },
  loadingFooterText: {
    fontSize: width * 0.035,
    color: '#666',
    fontStyle: 'italic',
  },
  // Pagination Info Styles
  paginationInfo: {
    alignItems: 'center',
    paddingVertical: width * 0.03,
    paddingHorizontal: width * 0.04,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
    backgroundColor: '#f8f9fa',
  },
  paginationText: {
    fontSize: width * 0.035,
    color: '#666',
    fontWeight: '500',
    marginBottom: width * 0.02,
  },
  loadMoreButton: {
    backgroundColor: '#3481BC',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
    borderRadius: width * 0.02,
    alignItems: 'center',
    marginTop: width * 0.01,
  },
  loadMoreText: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: 'bold',
  },
  loadingContainer: {
    flexDirection: 'row',
    alignItems: 'center',
  },
});

export default TeamMemberHome;