import { View, Text, TouchableOpacity, ScrollView, StyleSheet, StatusBar, Dimensions, ActivityIndicator } from 'react-native';
import React, { useState, useEffect } from 'react';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation, useRoute } from '@react-navigation/native';
import DateTimePickerModal from 'react-native-modal-datetime-picker';
import API_ENDPOINTS from '../../../../config/mongodb.endpoints';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.32;

const TABS = [
  { key: 'inspection', label: 'Inspection' },
  { key: 'jobCard', label: 'Job Card' },
  { key: 'maintenance', label: 'Maintenance' },
  { key: 'serviceOrder', label: 'Service Order' },
];

const TrackEquipmentStatus = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const equipment = route.params?.equipment || {};
  const [activeTab, setActiveTab] = useState('inspection');
  const [tabData, setTabData] = useState([]);
  const [loading, setLoading] = useState(false);
  const [dateRange, setDateRange] = useState({ from: null, to: null });
  const [datePickerVisible, setDatePickerVisible] = useState(false);
  const [datePickerMode, setDatePickerMode] = useState('from');

  useEffect(() => {
    if (!equipment.equipmentNumber) {
      setTabData([]);
      return;
    }
    setLoading(true);
    const fetchTabData = async () => {
      let results = [];
      try {
        if (activeTab === 'inspection') {
          // Fetch all inspections from MongoDB
          const inspectionsResponse = await fetch(`${API_ENDPOINTS.INSPECTIONS.GET_ALL}?page=1&limit=10000`);
          const inspectionsData = await inspectionsResponse.json();
          const allInspectionsData = inspectionsData.success ? inspectionsData.data : [];

          allInspectionsData.forEach(r => {
            if (r?.step1?.selectedEquipment?.equipmentNumber === equipment.equipmentNumber) {
              const createdAt = r?.step3?.createdAt || r?.createdAt;
              if (createdAt) {
                const reportDate = new Date(createdAt);
                if (dateRange.from && reportDate < dateRange.from) return;
                if (dateRange.to && reportDate > dateRange.to) return;
                results.push(r);
              }
            }
          });
        } else if (activeTab === 'jobCard') {
          // Fetch all job cards from MongoDB
          const jobCardsResponse = await fetch(`${API_ENDPOINTS.JOB_CARDS.GET_ALL}?page=1&limit=10000`);
          const jobCardsData = await jobCardsResponse.json();
          const allJobCardsData = jobCardsData.success ? jobCardsData.data : [];

          allJobCardsData.forEach(jc => {
            if (jc?.equipment?.equipmentNumber === equipment.equipmentNumber) {
              const createdAt = jc?.createdAt;
              if (createdAt) {
                const cardDate = new Date(createdAt);
                if (dateRange.from && cardDate < dateRange.from) return;
                if (dateRange.to && cardDate > dateRange.to) return;
                results.push(jc);
              }
            }
          });
        } else if (activeTab === 'maintenance') {
          // Fetch all maintenance reports from MongoDB
          const maintenanceResponse = await fetch(API_ENDPOINTS.MAINTENANCE.GET_ALL);
          const maintenanceData = await maintenanceResponse.json();
          const allMaintenanceData = maintenanceData.success ? maintenanceData.data : [];

          allMaintenanceData.forEach(m => {
            if (m?.equipment?.equipmentNumber === equipment.equipmentNumber) {
              const createdAt = m?.createdAt;
              if (createdAt) {
                const reportDate = new Date(createdAt);
                if (dateRange.from && reportDate < dateRange.from) return;
                if (dateRange.to && reportDate > dateRange.to) return;
                results.push(m);
              }
            }
          });
        } else if (activeTab === 'serviceOrder') {
          // Fetch all service orders from MongoDB
          const serviceResponse = await fetch(API_ENDPOINTS.SERVICE_SCHEDULES.GET_ALL);
          const serviceData = await serviceResponse.json();
          const allServiceData = serviceData.success ? serviceData.data : [];

          allServiceData.forEach(so => {
            const eq = so?.equipment;
            if (eq?.equipmentNumber && eq.equipmentNumber === equipment.equipmentNumber) {
              const createdAt = so?.createdAt;
              if (createdAt) {
                const orderDate = new Date(createdAt);
                if (dateRange.from && orderDate < dateRange.from) return;
                if (dateRange.to && orderDate > dateRange.to) return;
                results.push(so);
              }
            }
          });
        }
      } catch (e) {
        console.error('Error fetching tab data:', e);
        results = [];
      }
      setTabData(results.sort((a, b) => (b.createdAt ? new Date(b.createdAt) : 0) - (a.createdAt ? new Date(a.createdAt) : 0)));
      setLoading(false);
    };
    fetchTabData();
  }, [activeTab, equipment.equipmentNumber, dateRange]);

  return (
    <View style={{ flex: 1, backgroundColor: '#fff' }}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity onPress={() => navigation.goBack()} style={styles.headerLeftIcon}>
            <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Equipment Details</Text>
          <View style={styles.headerLeftIcon} />
        </View>
      </LinearGradient>
      <ScrollView style={{ flex: 1, padding: 16 }}>
        {/* Equipment Details Card */}
        <View style={styles.equipmentCard}>
          <Text style={styles.equipmentName}>{equipment.equipmentName || '-'}</Text>
          <Text style={styles.equipmentField}>Number: {equipment.equipmentNumber || '-'}</Text>
          <Text style={styles.equipmentField}>Category: {equipment.mainCategory || '-'}</Text>
          <Text style={styles.equipmentField}>Model: {equipment.model || '-'}</Text>
          <Text style={styles.equipmentField}>Meter Reading: {equipment.meterReading || '-'} {equipment.meterUnit || ''}</Text>
          <Text style={styles.equipmentField}>Status: {equipment.status || '-'}</Text>
        </View>
        {/* Date Range Picker UI */}
        <View style={{ flexDirection: 'row', alignItems: 'center', justifyContent: 'center', marginTop: 10, marginBottom: 8 }}>
          <TouchableOpacity
            style={{ backgroundColor: '#F0F0F0', borderRadius: 8, padding: 10, marginRight: 8 }}
            onPress={() => { setDatePickerMode('from'); setDatePickerVisible(true); }}
          >
            <Text style={{ color: '#015185', fontWeight: '600' }}>{dateRange.from ? `From: ${dateRange.from.toLocaleDateString()}` : 'Select From Date'}</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={{ backgroundColor: '#F0F0F0', borderRadius: 8, padding: 10, marginRight: 8 }}
            onPress={() => { setDatePickerMode('to'); setDatePickerVisible(true); }}
          >
            <Text style={{ color: '#015185', fontWeight: '600' }}>{dateRange.to ? `To: ${dateRange.to.toLocaleDateString()}` : 'Select To Date'}</Text>
          </TouchableOpacity>
          {(dateRange.from || dateRange.to) && (
            <TouchableOpacity style={{ backgroundColor: '#D81B60', borderRadius: 8, padding: 10 }} onPress={() => setDateRange({ from: null, to: null })}>
              <Text style={{ color: '#fff', fontWeight: '600' }}>Clear</Text>
            </TouchableOpacity>
          )}
        </View>
        <DateTimePickerModal
          isVisible={datePickerVisible}
          mode="date"
          onConfirm={date => {
            if (datePickerMode === 'from') {
              setDateRange(r => ({ ...r, from: date, to: null }));
            } else {
              setDateRange(r => ({ ...r, to: date }));
            }
            setDatePickerVisible(false);
          }}
          onCancel={() => setDatePickerVisible(false)}
        />
        {/* Tabs */}
        <View style={styles.tabsRow}>
          {TABS.map(tab => (
            <TouchableOpacity
              key={tab.key}
              style={[styles.tabButton, activeTab === tab.key && styles.tabButtonActive]}
              onPress={() => setActiveTab(tab.key)}
            >
              <Text style={[styles.tabText, activeTab === tab.key && styles.tabTextActive]}>{tab.label}</Text>
            </TouchableOpacity>
          ))}
        </View>
        {/* Tab Content */}
        <View style={styles.tabContent}>
          {loading ? (
            <ActivityIndicator size="large" color="#015185" style={{ marginTop: 16 }} />
          ) : tabData.length === 0 ? (
            <Text style={styles.tabContentText}>No data found for this equipment.</Text>
          ) : (
            tabData.map((item, idx) => (
              <View key={item.id || idx} style={styles.dataCard}>
                {activeTab === 'inspection' && (
                  <>
                    <Text style={styles.dataTitle}>Inspection #{item.step1?.requestNumber || item.id || '-'}</Text>
                    <Text style={styles.dataField}>Date: {item.step3?.createdAt ? new Date(item.step3.createdAt).toLocaleString() : '-'}</Text>
                    <Text style={styles.dataField}>Inspector: {item.step1?.inspector || '-'}</Text>
                    <Text style={styles.dataField}>Status: {item.step3?.status || '-'}</Text>
                  </>
                )}
                {activeTab === 'jobCard' && (
                  <>
                    <Text style={styles.dataTitle}>Job Card #{item.jobCardNumber || item.requestNumber || item.id || '-'}</Text>
                    <Text style={styles.dataField}>Date: {item.createdAt ? new Date(item.createdAt).toLocaleString() : '-'}</Text>
                    <Text style={styles.dataField}>Status: {item.status || '-'}</Text>
                    <Text style={styles.dataField}>Priority: {item.priority || '-'}</Text>
                  </>
                )}
                {activeTab === 'maintenance' && (
                  <>
                    <Text style={styles.dataTitle}>Maintenance #{item.requestNumber || item.id || '-'}</Text>
                    <Text style={styles.dataField}>Date: {item.createdAt ? new Date(item.createdAt).toLocaleString() : '-'}</Text>
                    <Text style={styles.dataField}>Status: {item.status || '-'}</Text>
                    <Text style={styles.dataField}>Priority: {item.priority || '-'}</Text>
                  </>
                )}
                {activeTab === 'serviceOrder' && (
                  <>
                    <Text style={styles.dataTitle}>Service Order #{item.scheduleId || item.id || '-'}</Text>
                    <Text style={styles.dataField}>Date: {item.createdAt ? new Date(item.createdAt).toLocaleString() : '-'}</Text>
                    <Text style={styles.dataField}>Status: {item.status || '-'}</Text>
                    <Text style={styles.dataField}>Project: {item.project || '-'}</Text>
                  </>
                )}
              </View>
            ))
          )}
        </View>
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  header: {
    width: '100%',
    justifyContent: 'flex-end',
    paddingBottom: width * 0.04,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.05,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeftIcon: {
    paddingVertical: width * 0.02,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.055,
    fontWeight: '700',
    flex: 1,
    marginLeft: width * 0.03,
    textAlign: 'center',
  },
  equipmentCard: {
    backgroundColor: '#F5F7FA',
    borderRadius: 8,
    padding: 16,
    marginBottom: 18,
    borderWidth: 1,
    borderColor: '#E0E0E0',
  },
  equipmentName: {
    color: '#015185',
    fontWeight: '700',
    fontSize: 18,
    marginBottom: 6,
  },
  equipmentField: {
    color: '#2267A2',
    fontSize: 15,
    marginBottom: 2,
  },
  tabsRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 12,
    marginTop: 8,
  },
  tabButton: {
    flex: 1,
    paddingVertical: 10,
    backgroundColor: '#E9E9E9',
    borderRadius: 6,
    marginHorizontal: 4,
    alignItems: 'center',
  },
  tabButtonActive: {
    backgroundColor: '#015185',
  },
  tabText: {
    color: '#015185',
    fontWeight: '700',
    fontSize: 15,
  },
  tabTextActive: {
    color: '#fff',
  },
  tabContent: {
    minHeight: 120,
    backgroundColor: '#F5F7FA',
    borderRadius: 8,
    padding: 18,
    marginTop: 8,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    alignItems: 'center',
    justifyContent: 'center',
  },
  tabContentText: {
    color: '#2267A2',
    fontSize: 16,
    textAlign: 'center',
  },
  dataCard: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 12,
    marginBottom: 10,
    borderWidth: 1,
    borderColor: '#E0E0E0',
    width: '100%',
  },
  dataTitle: {
    color: '#015185',
    fontWeight: '700',
    fontSize: 15,
    marginBottom: 4,
  },
  dataField: {
    color: '#2267A2',
    fontSize: 14,
    marginBottom: 2,
  },
});

export default TrackEquipmentStatus;