import React, { useState, useRef, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  Modal,
  Alert,
  ScrollView, // Added for modal content
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation, useFocusEffect } from '@react-navigation/native';
import auth from '@react-native-firebase/auth';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { getFilteredTeamMembers } from '../MainHome/Utils/teamMemberUtils';
import JobCardAPI from '../ConfigMongo/JobCardMongoConfig.js';

// Import tab components
import AllWotab from './AllWotab';
import AssignedWotab from './AssignedWotab';
import CompleteWotab from './CompleteWotab';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.48;

const WorkOrdertab = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [activeTab, setActiveTab] = useState('All');
  const [userModalVisible, setUserModalVisible] = useState(false); // New state for user details modal
  const [userDetails, setUserDetails] = useState(null); // Store user details
  const [userCountries, setUserCountries] = useState([]); // For filtering
  const [userProjects, setUserProjects] = useState([]); // For filtering
  const [jobcardPermissions, setJobcardPermissions] = useState([]); // For permission check
  const [canCreateJobCard, setCanCreateJobCard] = useState(false); // For button enable/disable
  const [userUid, setUserUid] = useState(null); // Add this state
  const [refreshKey, setRefreshKey] = useState(0); // Add refreshKey for AllWotab

  // Animation setup for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Fetch user data and permissions on mount
  useEffect(() => {
    const fetchUserData = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) return;
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        const userData = snapshot.val();
        if (!userData) return;
        setUserCountries(Array.isArray(userData.countries) ? userData.countries : []);
        setUserProjects(Array.isArray(userData.projects) ? userData.projects : []);
        const permissions = JobCardAPI.getUserJobCardPermissions(userData);
        setJobcardPermissions(permissions);
        setCanCreateJobCard(JobCardAPI.canCreateJobCard(permissions));
        setUserUid(userUid); // Save userUid in state for AllWotab
      } catch (err) {
        // Optionally handle error
      }
    };
    fetchUserData();
  }, []);

  // Refresh data when screen is focused (increment refreshKey)
  useFocusEffect(
    React.useCallback(() => {
      setRefreshKey((prev) => prev + 1);
    }, [])
  );

  const handleSearch = (text) => {
    setSearchQuery(text);
  };

  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    setTimeout(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }, 2000);
  };

  const handleTabPress = (tabName) => {
    setActiveTab(tabName);
  };

  const handleCreateJobCard = () => {
    if (navigation && navigation.navigate) {
      navigation.navigate('CreateWorkOrder');
    } else {
      Alert.alert('Navigation error', 'Cannot open Create Job Card screen.');
    }
  };

  // Handler for lock icon click - show permissions
  const handleLockIconPress = async () => {
    try {
      // Get user UID from AsyncStorage
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        return;
      }

      // Fetch user data from Firebase
      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const snapshot = await userRef.once('value');
      const userData = snapshot.val();

      if (!userData) {
        Alert.alert('Error', 'User data not found in database.');
        return;
      }

      // Extract jobcard module permissions
      const jobcardPermissions = JobCardAPI.getUserJobCardPermissions(userData);

      // Prepare user details for modal with focus on permissions
      const details = {
        name: userData.name || 'N/A',
        email: userData.email || 'N/A',
        employeeNumber: userData.employeeNumber || 'N/A',
        countries: Array.isArray(userData.countries)
          ? userData.countries.join(', ')
          : 'None',
        projects: Array.isArray(userData.projects)
          ? userData.projects.join(', ')
          : 'None',
        jobcardPermissions: Array.isArray(jobcardPermissions)
          ? jobcardPermissions.join(', ')
          : 'None',
      };

      setUserDetails(details);
      setUserModalVisible(true); // Show modal
    } catch (err) {
      Alert.alert('Error', err.message || 'Failed to fetch user details.');
    }
  };

  const renderTabContent = () => {
    switch (activeTab) {
      case 'All':
        return (
          <AllWotab
            refreshKey={refreshKey}
            searchQuery={searchQuery}
            onRefresh={handleRefresh}
            userCountries={userCountries}
            userProjects={userProjects}
            jobcardPermissions={jobcardPermissions}
            userUid={userUid}
          />
        );
      case 'Assigned':
        return <AssignedWotab />;
      case 'Completed':
        return <CompleteWotab />;
      default:
        return (
          <AllWotab
            searchQuery={searchQuery}
            onRefresh={handleRefresh}
            userCountries={userCountries}
            userProjects={userProjects}
          />
        );
    }
  };

  return (
    <View style={styles.container}>
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Job Card</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleRefresh} style={styles.refreshButton}>
              <Image
                source={require('../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity style={styles.lockButton} onPress={handleLockIconPress}>
              <MaterialIcons 
                name="lock" 
                size={width * 0.06} 
                color="#fff" 
              />
            </TouchableOpacity>
          </View>
        </View>

        {/* Search Bar */}
        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton}>
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>

        {/* Tabs */}
        <View style={styles.tabContainer}>
          <TouchableOpacity
            style={[styles.tabButton, activeTab === 'All' && styles.activeTab]}
            onPress={() => handleTabPress('All')}
          >
            <Text style={[styles.tabText, activeTab === 'All' && styles.activeTabText]}>All</Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.tabButton, activeTab === 'Assigned' && styles.activeTab]}
            onPress={() => handleTabPress('Assigned')}
          >
            <Text style={[styles.tabText, activeTab === 'Assigned' && styles.activeTabText]}>
              Assigned
            </Text>
          </TouchableOpacity>
          <TouchableOpacity
            style={[styles.tabButton, activeTab === 'Completed' && styles.activeTab]}
            onPress={() => handleTabPress('Completed')}
          >
            <Text style={[styles.tabText, activeTab === 'Completed' && styles.activeTabText]}>
              Completed
            </Text>
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Main Content */}
      <View style={styles.mainContent}>{renderTabContent()}</View>

      {/* Bottom Button */}
      <View style={styles.buttonContainer}>
        <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
          <TouchableOpacity
            style={[styles.startButton, !canCreateJobCard && { opacity: 0.5 }]}
            onPress={handleCreateJobCard}
            disabled={!canCreateJobCard}
          >
            <Ionicons name="add" size={width * 0.06} color="#fff" />
            <Text style={styles.buttonText}>Create Job Card</Text>
          </TouchableOpacity>
        </LinearGradient>
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* User Details Modal */}
      <Modal
        visible={userModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setUserModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.userModalContent}>
            <Text style={styles.modalTitle}>User Information</Text>
            {userDetails ? (
              <ScrollView style={styles.modalScroll}>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Name:</Text>
                  <Text style={styles.userInfoValue}>{userDetails.name}</Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Email:</Text>
                  <Text style={styles.userInfoValue}>{userDetails.email}</Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Countries:</Text>
                  <Text style={styles.userInfoValue}>
                    {userDetails.countries && userDetails.countries !== 'None' 
                      ? userDetails.countries 
                      : 'N/A'}
                  </Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Projects:</Text>
                  <Text style={styles.userInfoValue}>
                    {userDetails.projects && userDetails.projects !== 'None' 
                      ? userDetails.projects 
                      : 'N/A'}
                  </Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Job Card Permissions:</Text>
                  <Text style={styles.userInfoValue}>
                    {userDetails.jobcardPermissions && userDetails.jobcardPermissions !== 'None' 
                      ? userDetails.jobcardPermissions 
                      : 'None'}
                  </Text>
                </View>
                <View style={styles.userInfoItem}>
                  <Text style={styles.userInfoLabel}>Employee Number:</Text>
                  <Text style={styles.userInfoValue}>{userDetails.employeeNumber || 'N/A'}</Text>
                </View>
              </ScrollView>
            ) : (
              <Text style={styles.noDataText}>No user data available.</Text>
            )}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setUserModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  lockButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  tabContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
  },
  tabButton: {
    flex: 1,
    alignItems: 'center',
    paddingVertical: width * 0.03,
    marginTop: width * 0.02,
  },
  activeTab: {
    borderBottomWidth: 2,
    borderBottomColor: '#fff',
  },
  tabText: {
    color: '#ccc',
    fontSize: width * 0.04,
    fontWeight: '600',
  },
  activeTabText: {
    color: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  userModalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    width: '80%',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#333',
    marginBottom: width * 0.025,
    textAlign: 'center',
  },
  modalScroll: {
    maxHeight: width * 0.8,
    width: '100%',
  },
  detailRow: {
    flexDirection: 'row',
    marginBottom: 10,
    flexWrap: 'wrap',
  },
  detailLabel: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#015185',
    width: '40%',
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#333',
    width: '60%',
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#888',
    marginBottom: 15,
  },
  modalCloseButton: {
    marginTop: width * 0.025,
    alignItems: 'center',
  },
  modalCloseText: {
    fontSize: width * 0.04,
    color: '#0078D4',
    fontWeight: '600',
  },
  permissionSection: {
    marginVertical: 10,
    paddingVertical: 8,
    borderTopWidth: 1,
    borderTopColor: '#e0e0e0',
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 8,
  },
  permissionRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 6,
    paddingVertical: 4,
  },
  permissionLabel: {
    fontSize: width * 0.04,
    fontWeight: '500',
    color: '#333',
    flex: 1,
  },
  permissionValue: {
    fontSize: width * 0.04,
    fontWeight: '600',
    flex: 1,
    textAlign: 'right',
  },
  userInfoItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.015,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  userInfoLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
  },
  userInfoValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
    width: '60%',
  },
});

export default WorkOrdertab;