// MongoDB API Utility Functions
// This file contains all the API calls to interact with MongoDB backend

// API Base URLs
const MONGODB_API_BASE = 'https://api.titandrillingzm.com:6003';
const EQUIPMENT_API_BASE = 'https://api.titandrillingzm.com:6001';
const DOCUMENTS_API_BASE = 'https://api.titandrillingzm.com:6002';

// Generic API call function
const apiCall = async (url, options = {}) => {
  try {
    const response = await fetch(url, {
      headers: {
        'Content-Type': 'application/json',
        ...options.headers,
      },
      ...options,
    });

    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }

    return await response.json();
  } catch (error) {
    console.error('API call error:', error);
    throw error;
  }
};

// User Management API
export const UserAPI = {
  // Get all users
  getAllUsers: async () => {
    return await apiCall(`${MONGODB_API_BASE}/users/all`);
  },

  // Get user by ID
  getUserById: async (userId) => {
    return await apiCall(`${MONGODB_API_BASE}/users/${userId}`);
  },

  // Update user
  updateUser: async (userId, userData) => {
    return await apiCall(`${MONGODB_API_BASE}/users/${userId}`, {
      method: 'PUT',
      body: JSON.stringify(userData),
    });
  },

  // Create user
  createUser: async (userData) => {
    return await apiCall(`${MONGODB_API_BASE}/users/upload`, {
      method: 'POST',
      body: JSON.stringify(userData),
    });
  },

  // Bulk create users
  bulkCreateUsers: async (users) => {
    return await apiCall(`${MONGODB_API_BASE}/users/bulk-upload`, {
      method: 'POST',
      body: JSON.stringify({ users }),
    });
  },

  // Delete user
  deleteUser: async (userId) => {
    return await apiCall(`${MONGODB_API_BASE}/users/${userId}`, {
      method: 'DELETE',
    });
  },

  // Clean up user data
  cleanupUserData: async () => {
    return await apiCall(`${MONGODB_API_BASE}/users/cleanup`, {
      method: 'POST',
    });
  },

  // Get users overview
  getUsersOverview: async () => {
    return await apiCall(`${MONGODB_API_BASE}/users`);
  },
};

// Equipment Management API
export const EquipmentAPI = {
  // Get all equipment
  getAllEquipment: async () => {
    return await apiCall(`${EQUIPMENT_API_BASE}/equipment`);
  },

  // Get equipment by ID
  getEquipmentById: async (equipmentId) => {
    return await apiCall(`${EQUIPMENT_API_BASE}/equipment/${equipmentId}`);
  },

  // Create equipment
  createEquipment: async (equipmentData) => {
    return await apiCall(`${EQUIPMENT_API_BASE}/equipment`, {
      method: 'POST',
      body: JSON.stringify(equipmentData),
    });
  },

  // Update equipment
  updateEquipment: async (equipmentId, equipmentData) => {
    return await apiCall(`${EQUIPMENT_API_BASE}/equipment/${equipmentId}`, {
      method: 'PUT',
      body: JSON.stringify(equipmentData),
    });
  },

  // Delete equipment
  deleteEquipment: async (equipmentId) => {
    return await apiCall(`${EQUIPMENT_API_BASE}/equipment/${equipmentId}`, {
      method: 'DELETE',
    });
  },
};

// Documents Management API
export const DocumentsAPI = {
  // Get documents overview
  getDocumentsOverview: async () => {
    return await apiCall(`${DOCUMENTS_API_BASE}/documents`);
  },

  // Get user documents
  getUserDocuments: async (userId) => {
    return await apiCall(`${DOCUMENTS_API_BASE}/Documents/${userId}`);
  },

  // Get documents by subfolder
  getDocumentsBySubfolder: async (userId, documentType, subfolderName) => {
    return await apiCall(`${DOCUMENTS_API_BASE}/Documents/${userId}/${documentType}/${subfolderName}`);
  },

  // Upload document
  uploadDocument: async (formData) => {
    try {
      const response = await fetch(`${DOCUMENTS_API_BASE}/Documents/upload`, {
        method: 'POST',
        body: formData,
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      return await response.json();
    } catch (error) {
      console.error('Document upload error:', error);
      throw error;
    }
  },

  // Delete document
  deleteDocument: async (userId, documentType, subfolderName, documentId) => {
    return await apiCall(`${DOCUMENTS_API_BASE}/Documents/${userId}/${documentType}/${subfolderName}/${documentId}`, {
      method: 'DELETE',
    });
  },

  // Serve file
  serveFile: async (documentId) => {
    return await apiCall(`${DOCUMENTS_API_BASE}/documents/${documentId}`);
  },
};

// Authentication API
export const AuthAPI = {
  // Login user
  loginUser: async (emailOrEmployeeId, password) => {
    try {
      // Get all users
      const usersResponse = await UserAPI.getAllUsers();
      if (!usersResponse.success || !usersResponse.data) {
        throw new Error('Failed to fetch users data');
      }

      const users = usersResponse.data;
      
      // Find user by email or employee number
      const user = users.find(u => 
        u.email?.toLowerCase() === emailOrEmployeeId.toLowerCase() || 
        u.employeeNumber === emailOrEmployeeId
      );

      if (!user) {
        throw new Error('User not found');
      }

      // Verify password
      let isPasswordValid = false;
      if (user.hashedPassword) {
        // Verify with bcrypt
        const bcrypt = require('react-native-bcrypt');
        isPasswordValid = await bcrypt.compare(password, user.hashedPassword);
      } else if (user.password) {
        // Fallback to plain text comparison
        isPasswordValid = user.password === password;
      } else {
        // Default password fallback
        isPasswordValid = user.employeeNumber === password || password === 'password';
      }

      if (!isPasswordValid) {
        throw new Error('Invalid password');
      }

      return {
        success: true,
        user: user,
      };
    } catch (error) {
      console.error('Login error:', error);
      throw error;
    }
  },

  // Update last login
  updateLastLogin: async (userId) => {
    try {
      return await UserAPI.updateUser(userId, {
        loginStatus: true,
        lastLogin: new Date().toISOString(),
      });
    } catch (error) {
      console.error('Update last login error:', error);
      throw error;
    }
  },

  // Logout user
  logoutUser: async (userId) => {
    try {
      return await UserAPI.updateUser(userId, {
        loginStatus: false,
      });
    } catch (error) {
      console.error('Logout error:', error);
      throw error;
    }
  },
};

// Utility functions
export const Utils = {
  // Check if user has permission for module
  hasPermission: (userModules, moduleKey) => {
    return userModules.includes(moduleKey) || userModules.includes('All');
  },

  // Get user's enabled modules
  getEnabledModules: (userModules) => {
    return userModules.filter(module => module !== 'All');
  },

  // Format date
  formatDate: (date) => {
    if (!date) return 'N/A';
    const d = new Date(date);
    return d.toLocaleDateString('en-GB', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  },

  // Validate email
  isValidEmail: (email) => {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
  },

  // Generate random ID
  generateId: () => {
    return `user_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  },
};

// Export all APIs
export default {
  UserAPI,
  EquipmentAPI,
  DocumentsAPI,
  AuthAPI,
  Utils,
};
