import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  Animated,
  Modal,
  Image,
  Dimensions,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import JobCardAPI from '../ConfigMongo/JobCardMongoConfig.js';
import OfflineWorkOrderHelper from './OfflineWorkOrderHelper';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';

const { width } = Dimensions.get('window');

const DROPDOWN_OPTIONS = [
  {
    label: 'Edit',
    value: 'edit',
    icon: <MaterialIcons name="edit" size={22} color="#2196f3" />,
  },
  {
    label: 'View PDF',
    value: 'view-pdf',
    icon: <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" />,
  },
  {
    label: 'Delete',
    value: 'delete',
    icon: <MaterialIcons name="delete" size={22} color="#e74c3c" />,
  },
];

const AllWotab = ({ refreshKey, searchQuery, onRefresh, userCountries = [], userProjects = [], jobcardPermissions = [], userUid }) => {
  const navigation = useNavigation();
  const [workOrders, setWorkOrders] = useState([]);
  const [filteredWorkOrders, setFilteredWorkOrders] = useState([]);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const [isDataLoaded, setIsDataLoaded] = useState(false);
  const [shouldRenderReports, setShouldRenderReports] = useState(false);
  const [reportsReady, setReportsReady] = useState(false);

  // Animation for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotationLoopRef = useRef(null);
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const startRotation = useCallback(() => {
    if (rotationLoopRef.current) {
      return;
    }
    rotateAnim.setValue(0);
    rotationLoopRef.current = Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    );
    rotationLoopRef.current.start();
  }, [rotateAnim]);

  const stopRotation = useCallback(() => {
    if (rotationLoopRef.current) {
      rotationLoopRef.current.stop();
      rotationLoopRef.current = null;
    }
    rotateAnim.setValue(0);
  }, [rotateAnim]);

  // Animation for dropdown
  const fadeAnim = useRef(new Animated.Value(0)).current;

  // Show toast notifications
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  const loadWorkOrders = useCallback(async ({ showLoader = false } = {}) => {
    if (showLoader) {
      setSyncModalVisible(true);
      startRotation();
    }

    try {
      console.log('📊 Loading job cards from MongoDB...');
      console.log('📊 API Base URL:', 'https://api.titandrillingzm.com:6005');
      
      let workOrderList = [];
      
      // Try to get all job cards from MongoDB
      try {
        const response = await JobCardAPI.getAllJobCards({ limit: 1000 });
        
        console.log('📊 API Response:', JSON.stringify(response, null, 2));
        
        if (response.success && response.data && Array.isArray(response.data)) {
          console.log('✅ Job cards loaded from MongoDB:', response.data.length);
          
          workOrderList = response.data.map(wo => ({
            id: wo._id,
            ...wo,
            adminUid: wo.adminUid
          }));
          
          // Cache for offline access
          await AsyncStorage.setItem('globalWorkOrders', JSON.stringify(workOrderList));
          console.log('✅ Cached', workOrderList.length, 'work orders to AsyncStorage');
        } else {
          throw new Error('No data returned from MongoDB');
        }
      } catch (mongoError) {
        console.log('⚠️ MongoDB load failed, loading from AsyncStorage cache...');
        console.log('⚠️ Error:', mongoError.message);
        
        // Load from cache if MongoDB fails
        const cachedWorkOrdersRaw = await AsyncStorage.getItem('globalWorkOrders');
        if (cachedWorkOrdersRaw) {
          workOrderList = JSON.parse(cachedWorkOrdersRaw);
          console.log('✅ Loaded', workOrderList.length, 'work orders from cache');
        } else {
          console.log('⚠️ No cached work orders found in AsyncStorage');
          workOrderList = [];
        }
      }
      
      // Load offline work orders
      const offlineWorkOrders = await OfflineWorkOrderHelper.getOfflineWorkOrders();
      console.log('📴 Offline work orders found:', offlineWorkOrders.length);
      
      if (offlineWorkOrders.length > 0) {
        console.log('📋 First offline work order details:');
        console.log('   ID:', offlineWorkOrders[0]._id);
        console.log('   Request #:', offlineWorkOrders[0].requestNumber);
        console.log('   PDF Local Path:', offlineWorkOrders[0].pdfLocalPath || 'Not set');
        console.log('   Title:', offlineWorkOrders[0].title);
      }
      
      // Format offline work orders to match display structure - PRESERVE pdfLocalPath
      const formattedOfflineWorkOrders = offlineWorkOrders.map(wo => ({
        ...wo,
        id: wo._id,
        offlineSyncPending: true, // Mark as pending sync
        requestNumber: wo.requestNumber || 'OFFLINE',
        pdfLocalPath: wo.pdfLocalPath || '', // CRITICAL: Preserve PDF path
      }));
      
      // Combine online and offline work orders
      workOrderList = [...workOrderList, ...formattedOfflineWorkOrders];
      
      // Sort by creation date (newest first)
      workOrderList.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

      // Filter by onlyMineView permission
      let filtered = workOrderList;
      if (Array.isArray(jobcardPermissions) && jobcardPermissions.includes('onlyMineView') && userUid) {
        filtered = workOrderList.filter((wo) => wo.adminUid === userUid);
      }

      // Filter out invalid/empty job cards
      filtered = filtered.filter(wo => {
        const title = (wo.title || '').trim().toLowerCase();
        const isUntitled = !title || title.includes('untitled work order');
        const equipmentName = (wo.equipment?.equipmentName || '').trim().toLowerCase();
        const isNoEquipment = !wo.equipment || equipmentName === 'no equipment' || !equipmentName;
        const type = (wo.workOrderType || '').trim().toLowerCase();
        const isTypeNA = !type || type === 'n/a';
        
        return !(isUntitled || isNoEquipment || isTypeNA);
      });

      setWorkOrders(filtered);
      handleSearch(searchQuery, filtered);
      console.log('✅ Total job cards (online + offline):', filtered.length);
      
      setIsDataLoaded(true);
      setShouldRenderReports(true);
      setReportsReady(true);
    } catch (error) {
      console.error('❌ Error loading job cards from MongoDB:', error);
      showToast('error', 'Error', 'Failed to load job cards from database.');
      setIsDataLoaded(true);
      setShouldRenderReports(false);
      setReportsReady(false);
    } finally {
      if (showLoader) {
        setSyncModalVisible(false);
        stopRotation();
      }
    }
  }, [jobcardPermissions, searchQuery, startRotation, stopRotation, userUid]);

  useEffect(() => {
    loadWorkOrders({ showLoader: true });
  }, [loadWorkOrders]);

  useEffect(() => {
    if (typeof refreshKey !== 'undefined') {
      loadWorkOrders({ showLoader: true });
    }
  }, [refreshKey, loadWorkOrders]);

  // Refresh when refreshKey changes (from parent)
  useEffect(() => {
    if (typeof refreshKey !== 'undefined' && refreshKey > 0) {
      // Silent reload without showing sync modal
      loadWorkOrders();
    }
  }, [refreshKey]);

  // Handle search
  const handleSearch = (query, workOrdersList = workOrders) => {
    if (query) {
      const filtered = workOrdersList.filter(
        (wo) =>
          (wo.title || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.equipment?.equipmentName || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.priority || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.assignTo?.some((user) => user.name.toLowerCase().includes(query.toLowerCase())) || false) ||
          (wo.workOrderType || '').toLowerCase().includes(query.toLowerCase())
      );
      setFilteredWorkOrders(filtered);
    } else {
      setFilteredWorkOrders(workOrdersList);
    }
  };

  useEffect(() => {
    handleSearch(searchQuery);
  }, [searchQuery, workOrders]);

  // Handle refresh
  const handleRefresh = async () => {
    setIsRefreshing(true);
    // Don't show sync modal on manual refresh
    await loadWorkOrders();
    setIsRefreshing(false);
    if (onRefresh) onRefresh();
  };

  // Delete work order from MongoDB
  const deleteWorkOrder = async (workOrderId, adminUid) => {
    try {
      console.log('🗑️ Deleting job card from MongoDB:', workOrderId);
      
      const response = await JobCardAPI.deleteJobCard(adminUid, workOrderId);
      
      if (response.success) {
        const updatedWorkOrders = workOrders.filter((wo) => wo.id !== workOrderId);
        setWorkOrders(updatedWorkOrders);
        handleSearch(searchQuery, updatedWorkOrders);
        setDropdownVisible(null);
        showToast('success', 'Success', 'Job Card deleted successfully!');
        console.log('✅ Job card deleted successfully');
      } else {
        throw new Error(response.error || 'Failed to delete job card');
      }
    } catch (error) {
      console.error('❌ Error deleting job card from MongoDB:', error);
      showToast('error', 'Error', 'Failed to delete job card.');
    }
  };

  // Confirm before deleting
  const confirmDeleteWorkOrder = (workOrderId, adminUid) => {
    Alert.alert(
      'Delete Work Order',
      'Are you sure you want to delete this work order? This action cannot be undone.',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'Delete', style: 'destructive', onPress: () => deleteWorkOrder(workOrderId, adminUid) },
      ],
      { cancelable: true }
    );
  };

  // View PDF - Smart loading from local or MongoDB
  const handleViewPDF = async (workOrderId) => {
    try {
      console.log('═══════════════════════════════════════');
      console.log('📄 OPENING PDF FOR JOB CARD');
      console.log('═══════════════════════════════════════');
      console.log('🆔 Work Order ID:', workOrderId);
      
      // Find the work order to get its data
      const workOrder = workOrders.find(wo => wo.id === workOrderId);
      const requestNumber = workOrder?.requestNumber;
      const pdfLocalPath = workOrder?.pdfLocalPath;
      
      console.log('📋 Work Order Data:');
      console.log('   Request #:', requestNumber);
      console.log('   PDF Local Path:', pdfLocalPath || 'Not set');
      console.log('   Offline Created:', workOrder?.offlineCreated);
      console.log('   Sync Pending:', workOrder?.offlineSyncPending);
      
      // Check network status
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      console.log('🌐 Network:', isOnline ? 'Online' : 'Offline');
      
      let pdfUrl = null;
      let isLocalFile = false;
      
      // PRIORITY 1: Check if PDF path is in work order data (for offline-created)
      if (pdfLocalPath) {
        console.log('🔍 Strategy 1: Checking pdfLocalPath from work order data');
        const fileExists = await RNFS.exists(pdfLocalPath);
        console.log('   File exists:', fileExists);
        
        if (fileExists) {
          pdfUrl = `file://${pdfLocalPath}`;
          isLocalFile = true;
          console.log('✅ Using PDF from work order data:', pdfUrl);
        }
      }
      
      // PRIORITY 2: Try local PDF with request number
      if (!pdfUrl && requestNumber) {
        console.log('🔍 Strategy 2: Checking AsyncStorage with request number');
        const pdfKey = `workorder_pdf_local_${requestNumber}`;
        console.log('   Key:', pdfKey);
        const localPath = await AsyncStorage.getItem(pdfKey);
        console.log('   Path:', localPath || 'Not found');
        
        if (localPath) {
          const fileExists = await RNFS.exists(localPath);
          console.log('   File exists:', fileExists);
          
          if (fileExists) {
            pdfUrl = `file://${localPath}`;
            isLocalFile = true;
            console.log('✅ Using PDF from AsyncStorage (request number)');
          }
        }
      }
      
      // PRIORITY 3: Try with work order ID
      if (!pdfUrl) {
        console.log('🔍 Strategy 3: Checking AsyncStorage with work order ID');
        const pdfKey = `workorder_pdf_local_${workOrderId}`;
        console.log('   Key:', pdfKey);
        const localPath = await AsyncStorage.getItem(pdfKey);
        console.log('   Path:', localPath || 'Not found');
        
        if (localPath) {
          const fileExists = await RNFS.exists(localPath);
          console.log('   File exists:', fileExists);
          
          if (fileExists) {
            pdfUrl = `file://${localPath}`;
            isLocalFile = true;
            console.log('✅ Using PDF from AsyncStorage (ID)');
          }
        }
      }
      
      // PRIORITY 4: If online, try MongoDB
      if (!pdfUrl && isOnline) {
        console.log('🔍 Strategy 4: Trying MongoDB');
        try {
          const response = await JobCardAPI.getJobCard(workOrderId);
          
          if (response.success && response.data?.pdfDownloadUrl) {
            pdfUrl = response.data.pdfDownloadUrl;
            console.log('✅ Using MongoDB PDF:', pdfUrl);
          } else {
            console.log('   MongoDB response success:', response.success);
            console.log('   Has PDF URL:', !!response.data?.pdfDownloadUrl);
          }
        } catch (mongoError) {
          console.error('❌ MongoDB fetch failed:', mongoError.message);
        }
      }
      
      // DEBUG: If still no PDF, show all stored keys
      if (!pdfUrl) {
        console.log('🔍 PDF NOT FOUND - Debugging AsyncStorage keys...');
        const allKeys = await AsyncStorage.getAllKeys();
        const pdfKeys = allKeys.filter(k => k.includes('pdf') || k.includes('workorder'));
        console.log('📦 Found', pdfKeys.length, 'potential PDF keys:');
        pdfKeys.forEach(key => console.log('   -', key));
        
        console.error('❌ No PDF found in local storage or MongoDB');
        showToast('error', 'Error', 'PDF not available. Try creating the job card again or sync if offline.');
        return;
      }
      
      // Navigate to PDF viewer
      console.log('✅ Opening PDF viewer');
      console.log('   URL:', pdfUrl);
      console.log('   Is Local:', isLocalFile);
      console.log('═══════════════════════════════════════');
      
      setDropdownVisible(null);
      const adminUid = await AsyncStorage.getItem('userUid');
      
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        userId: adminUid,
        isLocalFile,
        onGoBack: () => {
          if (onRefresh) onRefresh();
          navigation.goBack();
        },
      });
      
    } catch (error) {
      console.error('❌ Error opening PDF:', error);
      console.error('❌ Error stack:', error.stack);
      showToast('error', 'Error', 'Could not open the PDF. Please try again.');
    }
  };

  // Get priority color
  const getPriorityColor = (priority) => {
    switch (priority) {
      case 'Low':
        return '#2ecc71';
      case 'Medium':
        return '#f1c40f';
      case 'High':
        return '#e67e22';
      case 'Emergency':
        return '#e74c3c';
      default:
        return '#666';
    }
  };

  const getStatusColor = (status) => {
    const normalized = (status || '').toString().trim().toLowerCase();
    switch (normalized) {
      case 'hold':
      case 'on hold':
        return '#ffd205';
      case 'completed':
      case 'closed':
        return '#2ecc71';
      case 'in progress':
      case 'in-process':
      case 'processing':
        return '#3498db';
      case 'service scheduled':
      case 'scheduled':
        return '#00B894';
      case 'cancelled':
      case 'canceled':
        return '#e74c3c';
      case 'pending':
        return '#f1c40f';
      default:
        return '#545d6b';
    }
  };

  const getStatusTextColor = (status) => {
    const normalized = (status || '').toString().trim().toLowerCase();
    if (['hold', 'on hold', 'pending', 'service scheduled', 'scheduled'].includes(normalized)) {
      return '#0055A5';
    }
    return '#fff';
  };

const getDisplayTitle = (item) => {
  const requestNumber = (item?.requestNumber || '').toString().trim();
  if (requestNumber) {
    return requestNumber;
  }

  const rawTitle = (item?.title || '').toString().trim();
  if (!rawTitle) {
    return 'Untitled Work Order';
  }

  if (rawTitle.includes(' - ')) {
    return rawTitle.split(' - ')[0].trim();
  }

  return rawTitle;
};

  // Format date
  const formatDate = (date) => {
    if (!date) return 'N/A';
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  // Handle dropdown actions
  const handleDropdownAction = async (action, workOrder) => {
    switch (action) {
      case 'edit':
        setDropdownVisible(null);
        navigation.navigate('UpdateCreateWorkOrder', { workOrder });
        break;
      case 'view-pdf':
        handleViewPDF(workOrder.id);
        break;
      case 'delete':
        confirmDeleteWorkOrder(workOrder.id, workOrder.adminUid);
        break;
      default:
        break;
    }
  };

  // Toggle dropdown with animation
  const toggleDropdown = (id) => {
    if (dropdownVisible === id) {
      Animated.timing(fadeAnim, {
        toValue: 0,
        duration: 200,
        useNativeDriver: true,
      }).start(() => setDropdownVisible(null));
    } else {
      setDropdownVisible(id);
      Animated.timing(fadeAnim, {
        toValue: 1,
        duration: 200,
        useNativeDriver: true,
      }).start();
    }
  };

  // Render work order item
  const renderWorkOrder = ({ item, index }) => {
    const title = (item.title || '').trim().toLowerCase();
    const equipmentName = (item.equipment?.equipmentName || '').trim().toLowerCase();
    const type = (item.workOrderType || '').trim().toLowerCase();
    if (!title || title.includes('untitled work order') || !equipmentName || equipmentName === 'no equipment' || !type || type === 'n/a') {
      return null;
    }
    const dropdownPosition = index > workOrders.length - 4 ? 'top' : 'auto';
  const displayTitle = getDisplayTitle(item);
    
    return (
      <View style={styles.workOrderItem}>
        <View style={styles.workOrderHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
          <Text style={styles.workOrderTitle}>{displayTitle}</Text>
            
            {/* Offline Sync Pending Badge */}
            {item.offlineSyncPending && (
              <View style={styles.offlineBadge}>
                <MaterialIcons name="cloud-upload" size={12} color="#fff" />
                <Text style={styles.offlineBadgeText}>SYNC</Text>
              </View>
            )}
          </View>
          <TouchableOpacity style={[styles.statusButton, { backgroundColor: getPriorityColor(item.priority) }]}> 
            <Text style={styles.statusButtonText}>{item.priority || 'N/A'}</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.workOrderDetails}>
          {item.status && (
            <View style={[styles.statusBadgeRow, { backgroundColor: getStatusColor(item.status) }]}>
              <Text style={[styles.statusBadgeText, { color: getStatusTextColor(item.status) }]}>
                {item.status}
              </Text>
            </View>
          )}
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Equipment</Text>
            <Text style={styles.detailValue}>{item.equipment?.equipmentName || 'No Equipment'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Type</Text>
            <Text style={styles.detailValue}>{item.workOrderType || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Start Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.startDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Due Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.dueDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Assigned To</Text>
            <Text style={styles.detailValue}>{item.assignTo?.map((user) => user.name).join(', ') || 'Unassigned'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={[styles.detailLabel, { color: '#666', marginRight: 10 }]}>Date</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.detailValue}>{formatDate(item.createdAt)}</Text>
              <Dropdown
                style={styles.dropdown}
                data={DROPDOWN_OPTIONS}
                labelField="label"
                valueField="value"
                value={null}
                onChange={(option) => handleDropdownAction(option.value, item)}
                renderRightIcon={() => (
                  <MaterialIcons name="more-vert" size={22} color="#333" />
                )}
                renderItem={(option) => (
                  <View style={styles.dropdownItem}>
                    {option.icon}
                    <Text
                      style={[
                        styles.dropdownText,
                        { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                      ]}
                    >
                      {option.label}
                    </Text>
                  </View>
                )}
                placeholder=""
                showsVerticalScrollIndicator={false}
                dropdownPosition={dropdownPosition}
                containerStyle={styles.dropdownContainer}
                itemTextStyle={styles.dropdownItemText}
                selectedTextStyle={{ display: 'none' }}
              />
            </View>
          </View>
          <View style={styles.detailItem}>
            <TouchableOpacity
              style={styles.trackButton}
              onPress={() => navigation.navigate('WorkerDashboardhome', {
                workOrderId: item.id,
                adminUid: item.adminUid,
                onGoBack: () => navigation.navigate('WorkOrdertab'),
              })}
            >
              <Text style={styles.trackButtonText}>Track</Text>
            </TouchableOpacity>
            {item.serviceScheduleId ? (
              <View style={styles.serviceScheduledBadge}>
                <Text style={styles.serviceScheduledText}>Service scheduled</Text>
              </View>
            ) : null}
          </View>
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <View style={styles.mainContent}>
        {isDataLoaded && reportsReady && workOrders.length >= 0 ? (
          <FlatList
            data={filteredWorkOrders}
            renderItem={renderWorkOrder}
            keyExtractor={(item) => item.id}
            contentContainerStyle={styles.workOrderList}
            style={styles.flatList}
            ListEmptyComponent={
              <Text style={styles.emptyText}>
                {searchQuery ? 'No matching work orders found' : 'No work orders available'}
              </Text>
            }
            refreshing={isRefreshing}
            onRefresh={handleRefresh}
          />
        ) : (
          <View style={{ flex: 1, justifyContent: 'center', alignItems: 'center' }}>
            <Text style={styles.emptyText}>Loading reports...</Text>
          </View>
        )}
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    // paddingHorizontal: width * 0.04,
  },
  flatList: {
    flex: 1,
  },
  workOrderList: {
    paddingTop: width * 0.012,
    paddingBottom: width * 0.24,
  },
  workOrderItem: {
    marginVertical: width * 0.012,
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
  },
  workOrderHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingTop: width * 0.025,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: width * 0.05,
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  workOrderTitle: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  statusButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    minWidth: width * 0.08,
    alignItems: 'center',
    justifyContent: 'center',
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  workOrderDetails: {
    paddingHorizontal: width * 0.05,
  },
  detailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.012,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  detailLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: width * 0.008,
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
    backgroundColor: '#fff',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  emptyText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  trackButton: {
    backgroundColor: '#015185',
    paddingVertical: 6,
    paddingHorizontal: 16,
    borderRadius: 6,
    marginTop: 8,
    alignSelf: 'flex-start',
  },
  trackButtonText: {
    color: '#fff',
    fontWeight: 'bold',
  },
  serviceScheduledBadge: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    marginLeft: 12,
    alignSelf: 'center',
  },
  serviceScheduledText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 14,
  },
  statusBadgeRow: {
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    alignSelf: 'flex-start',
    marginBottom: 8,
    marginTop: 8,
  },
  statusBadgeText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 14,
  },
  offlineBadge: {
    marginLeft: width * 0.02,
    paddingHorizontal: width * 0.025,
    paddingVertical: width * 0.012,
    borderRadius: width * 0.015,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#ff9800',
  },
  offlineBadgeText: {
    fontSize: width * 0.028,
    color: '#fff',
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
});

export default AllWotab;