import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  Animated,
  Modal,
  Image,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation } from '@react-navigation/native';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import JobCardAPI from '../ConfigMongo/JobCardMongoConfig.js';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';

const { width } = Dimensions.get('window');

const DROPDOWN_OPTIONS = [
  {
    label: 'Edit',
    value: 'edit',
    icon: <MaterialIcons name="edit" size={22} color="#2196f3" />,
  },
  {
    label: 'View PDF',
    value: 'view-pdf',
    icon: <MaterialIcons name="picture-as-pdf" size={22} color="#2196f3" />,
  },
  {
    label: 'Delete',
    value: 'delete',
    icon: <MaterialIcons name="delete" size={22} color="#e74c3c" />,
  },
];

const CompleteWotab = ({ searchQuery, onRefresh }) => {
  const navigation = useNavigation();
  const [workOrders, setWorkOrders] = useState([]);
  const [filteredWorkOrders, setFilteredWorkOrders] = useState([]);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [dropdownVisible, setDropdownVisible] = useState(null);

  // Animation for sync icon
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Animation for dropdown
  const fadeAnim = useRef(new Animated.Value(0)).current;

  // Show toast notifications
  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Load work orders from MongoDB
  const loadWorkOrders = async () => {
    try {
      console.log('📊 Loading completed job cards from MongoDB...');
      console.log('📊 API Base URL:', 'https://api.titandrillingzm.com:6005');
      
      // Get all job cards from MongoDB
      const response = await JobCardAPI.getAllJobCards({ limit: 1000 });
      
      console.log('📊 Complete API Response:', JSON.stringify(response, null, 2));
      
      if (response.success && response.data) {
        console.log('✅ Job cards loaded from MongoDB:', response.data.length);
        
        let workOrderList = response.data.map(wo => ({
          id: wo._id,
          ...wo,
          adminUid: wo.adminUid
        }));
        
        // Filter for completed job cards only
        const completedList = workOrderList.filter((wo) => {
          const normalizedStatus = (wo.status || '').toString().trim().toLowerCase();
          return ['complete', 'completed', 'closed', 'resolved', 'finished'].includes(normalizedStatus);
        });
        
        // Sort by creation date (newest first)
        completedList.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        
        console.log('✅ Completed job cards:', completedList.length);
        setWorkOrders(completedList);
        handleSearch(searchQuery, completedList);
      } else {
        console.log('⚠️ No job cards found in MongoDB');
        setWorkOrders([]);
        handleSearch(searchQuery, []);
      }
    } catch (error) {
      console.error('❌ Error loading completed job cards from MongoDB:', error);
      showToast('error', 'Error', 'Failed to load completed job cards.');
      setWorkOrders([]);
    }
  };

  useEffect(() => {
    loadWorkOrders();
  }, []);

  // Handle search
  const handleSearch = (query, workOrdersList = workOrders) => {
    if (query) {
      const filtered = workOrdersList.filter(
        (wo) =>
          (wo.title || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.equipment?.equipmentName || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.priority || '').toLowerCase().includes(query.toLowerCase()) ||
          (wo.assignTo?.some((user) => user.name.toLowerCase().includes(query.toLowerCase())) || false) ||
          (wo.workOrderType || '').toLowerCase().includes(query.toLowerCase())
      );
      setFilteredWorkOrders(filtered);
    } else {
      setFilteredWorkOrders(workOrdersList);
    }
  };

  useEffect(() => {
    handleSearch(searchQuery);
  }, [searchQuery]);

  // Handle refresh
  const handleRefresh = () => {
    setIsRefreshing(true);
    setSyncModalVisible(true);

    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    loadWorkOrders();

    setTimeout(() => {
      setIsRefreshing(false);
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      if (onRefresh) onRefresh();
    }, 2000);
  };

  // Delete work order from MongoDB
  const deleteWorkOrder = async (workOrderId) => {
    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated. Please log in again.');
        navigation.navigate('AdminLoginScreen');
        return;
      }
      
      console.log('🗑️ Deleting job card from MongoDB:', workOrderId);
      
      const response = await JobCardAPI.deleteJobCard(adminUid, workOrderId);
      
      if (response.success) {
        const updatedWorkOrders = workOrders.filter((wo) => wo.id !== workOrderId);
        setWorkOrders(updatedWorkOrders);
        handleSearch(searchQuery, updatedWorkOrders);
        setDropdownVisible(null);
        showToast('success', 'Success', 'Job Card deleted successfully!');
        console.log('✅ Job card deleted successfully');
      } else {
        throw new Error(response.error || 'Failed to delete job card');
      }
    } catch (error) {
      console.error('❌ Error deleting job card from MongoDB:', error);
      showToast('error', 'Error', 'Failed to delete job card.');
    }
  };

  // View PDF - Smart loading from local or MongoDB
  const handleViewPDF = async (workOrderId) => {
    try {
      console.log('📄 Opening PDF for job card:', workOrderId);
      
      // Find the work order to get its request number
      const workOrder = workOrders.find(wo => wo.id === workOrderId);
      const requestNumber = workOrder?.requestNumber;
      
      console.log('📄 Request number:', requestNumber);
      
      // Check network status
      const netInfo = await NetInfo.fetch();
      const isOnline = netInfo.isConnected && netInfo.isInternetReachable !== false;
      
      console.log('📄 Network status:', isOnline ? 'Online' : 'Offline');
      
      let pdfUrl = null;
      let isLocalFile = false;
      
      // PRIORITY 1: Try local PDF first (works offline and online)
      if (requestNumber) {
        const pdfKey = `workorder_pdf_local_${requestNumber}`;
        const localPdfPath = await AsyncStorage.getItem(pdfKey);
        
        console.log('📄 Checking local PDF with key:', pdfKey);
        console.log('📄 Local PDF path:', localPdfPath);
        
        if (localPdfPath) {
          const fileExists = await RNFS.exists(localPdfPath);
          console.log('📄 Local file exists:', fileExists);
          
          if (fileExists) {
            pdfUrl = `file://${localPdfPath}`;
            isLocalFile = true;
            console.log('✅ Using local PDF:', pdfUrl);
          }
        }
      }
      
      // PRIORITY 2: If no local PDF and online, try MongoDB
      if (!pdfUrl && isOnline) {
        console.log('🌐 No local PDF, trying MongoDB...');
        try {
          const response = await JobCardAPI.getJobCard(workOrderId);
          
          if (response.success && response.data?.pdfDownloadUrl) {
            pdfUrl = response.data.pdfDownloadUrl;
            console.log('✅ Using MongoDB PDF:', pdfUrl);
          }
        } catch (mongoError) {
          console.warn('⚠️ MongoDB PDF fetch failed:', mongoError.message);
        }
      }
      
      // If still no PDF URL found
      if (!pdfUrl) {
        console.error('❌ No PDF found in local storage or MongoDB');
        showToast('error', 'Error', 'PDF not available. Please recreate the job card.');
        return;
      }
      
      // Navigate to PDF viewer
      setDropdownVisible(null);
      const adminUid = await AsyncStorage.getItem('userUid');
      
      navigation.navigate('PdfviewPage', {
        pdfUrl,
        userId: adminUid,
        isLocalFile,
        onGoBack: () => {
          if (onRefresh) onRefresh();
          navigation.goBack();
        },
      });
      
      console.log('✅ Navigated to PDF viewer');
      
    } catch (error) {
      console.error('❌ Error opening PDF:', error);
      showToast('error', 'Error', 'Could not open the PDF. Please try again.');
    }
  };

  // Get priority color
  const getPriorityColor = (priority) => {
    switch (priority) {
      case 'Low':
        return '#2ecc71';
      case 'Medium':
        return '#f1c40f';
      case 'High':
        return '#e67e22';
      case 'Emergency':
        return '#e74c3c';
      default:
        return '#666';
    }
  };

  const getStatusColor = (status) => {
    const normalized = (status || '').toString().trim().toLowerCase();
    switch (normalized) {
      case 'hold':
      case 'on hold':
        return '#ffd205';
      case 'completed':
      case 'complete':
      case 'closed':
        return '#2ecc71';
      case 'in progress':
      case 'in-progress':
      case 'processing':
        return '#3498db';
      case 'service scheduled':
      case 'scheduled':
        return '#00B894';
      case 'cancelled':
      case 'canceled':
        return '#e74c3c';
      case 'pending':
        return '#f1c40f';
      default:
        return '#545d6b';
    }
  };

  const getStatusTextColor = (status) => {
    const normalized = (status || '').toString().trim().toLowerCase();
    if (['hold', 'on hold', 'pending', 'service scheduled', 'scheduled'].includes(normalized)) {
      return '#0055A5';
    }
    return '#fff';
  };

const getDisplayTitle = (item) => {
  const requestNumber = (item?.requestNumber || '').toString().trim();
  if (requestNumber) {
    return requestNumber;
  }

  const rawTitle = (item?.title || '').toString().trim();
  if (!rawTitle) {
    return 'Untitled Work Order';
  }

  if (rawTitle.includes(' - ')) {
    return rawTitle.split(' - ')[0].trim();
  }

  return rawTitle;
};

  // Format date
  const formatDate = (date) => {
    if (!date) return 'N/A';
    return new Date(date).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  // Handle dropdown actions
  const handleDropdownAction = async (action, workOrder) => {
    switch (action) {
      case 'edit':
        setDropdownVisible(null);
        navigation.navigate('UpdateCreateWorkOrder', { workOrder });
        break;
      case 'view-pdf':
        handleViewPDF(workOrder.id);
        break;
      case 'delete':
        deleteWorkOrder(workOrder.id);
        break;
      default:
        break;
    }
  };

  // Toggle dropdown with animation
  const toggleDropdown = (id) => {
    if (dropdownVisible === id) {
      Animated.timing(fadeAnim, {
        toValue: 0,
        duration: 200,
        useNativeDriver: true,
      }).start(() => setDropdownVisible(null));
    } else {
      setDropdownVisible(id);
      Animated.timing(fadeAnim, {
        toValue: 1,
        duration: 200,
        useNativeDriver: true,
      }).start();
    }
  };

  // Render work order item
  const renderWorkOrder = ({ item, index }) => {
    const dropdownPosition = index > workOrders.length - 4 ? 'top' : 'auto';
  const displayTitle = getDisplayTitle(item);
    
    return (
      <View style={styles.workOrderItem}>
        <View style={styles.workOrderHeader}>
          <View style={{ flexDirection: 'row', alignItems: 'center' }}>
          <Text style={styles.workOrderTitle}>{displayTitle}</Text>
          </View>
          <TouchableOpacity style={[styles.statusButton, { backgroundColor: getPriorityColor(item.priority) }]}>
            <Text style={styles.statusButtonText}>{item.priority || 'N/A'}</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.workOrderDetails}>
          {/* Status row before Track row */}
          {item.status && (
            <View style={[styles.statusBadgeRow, { backgroundColor: getStatusColor(item.status) }]}>
              <Text style={[styles.statusBadgeText, { color: getStatusTextColor(item.status) }]}>
                {item.status}
              </Text>
            </View>
          )}
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Equipment</Text>
            <Text style={styles.detailValue}>{item.equipment?.equipmentName || 'No Equipment'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Type</Text>
            <Text style={styles.detailValue}>{item.workOrderType || 'N/A'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Start Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.startDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Due Date</Text>
            <Text style={styles.detailValue}>{formatDate(item.dueDate)}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={styles.detailLabel}>Assigned To</Text>
            <Text style={styles.detailValue}>{item.assignTo?.map((user) => user.name).join(', ') || 'Unassigned'}</Text>
          </View>
          <View style={styles.detailItem}>
            <Text style={[styles.detailLabel, { color: '#666', marginRight: 10 }]}>Date</Text>
            <View style={styles.dateContainer}>
              <Text style={styles.detailValue}>{formatDate(item.createdAt)}</Text>
              <Dropdown
                style={styles.dropdown}
                data={DROPDOWN_OPTIONS}
                labelField="label"
                valueField="value"
                value={null}
                onChange={(option) => handleDropdownAction(option.value, item)}
                renderRightIcon={() => (
                  <MaterialIcons name="more-vert" size={22} color="#333" />
                )}
                renderItem={(option) => (
                  <View style={styles.dropdownItem}>
                    {option.icon}
                    <Text
                      style={[
                        styles.dropdownText,
                        { color: option.value === 'delete' ? '#e74c3c' : '#015185' },
                      ]}
                    >
                      {option.label}
                    </Text>
                  </View>
                )}
                placeholder=""
                showsVerticalScrollIndicator={false}
                dropdownPosition={dropdownPosition}
                containerStyle={styles.dropdownContainer}
                itemTextStyle={styles.dropdownItemText}
                selectedTextStyle={{ display: 'none' }}
              />
            </View>
          </View>
          <View style={styles.detailItem}>
            <TouchableOpacity
              style={styles.trackButton}
              onPress={() => navigation.navigate('WorkerDashboardhome', {
                workOrderId: item.id,
                adminUid: item.adminUid || item.adminUid, // fallback if field is named differently
                onGoBack: () => navigation.navigate('WorkOrdertab'),
              })}
            >
              <Text style={styles.trackButtonText}>Track</Text>
            </TouchableOpacity>
            {item.serviceScheduleId ? (
              <View style={styles.serviceScheduledBadge}>
                <Text style={styles.serviceScheduledText}>Service scheduled</Text>
              </View>
            ) : null}
          </View>
        </View>
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <View style={styles.mainContent}>
        <FlatList
          data={filteredWorkOrders}
          renderItem={renderWorkOrder}
          keyExtractor={(item) => item.id}
          contentContainerStyle={styles.workOrderList}
          style={styles.flatList}
          ListEmptyComponent={
            <Text style={styles.emptyText}>
              {searchQuery ? 'No matching work orders found' : 'No work orders available'}
            </Text>
          }
          refreshing={isRefreshing}
          onRefresh={handleRefresh}
        />
      </View>

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Cloud Data Sync in Progress</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
  },
  flatList: {
    flex: 1,
  },
  workOrderList: {
    paddingTop: width * 0.012,
    paddingBottom: width * 0.24,
  },
  workOrderItem: {
    marginVertical: width * 0.012,
    backgroundColor: '#F2F2F2',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#eee',
  },
  workOrderHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingTop: width * 0.025,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: width * 0.05,
    borderTopLeftRadius: width * 0.012,
    borderTopRightRadius: width * 0.012,
  },
  workOrderTitle: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  statusButton: {
    paddingVertical: width * 0.01,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.015,
    minWidth: width * 0.08,
    alignItems: 'center',
    justifyContent: 'center',
  },
  statusButtonText: {
    color: '#fff',
    fontSize: width * 0.03,
    fontWeight: '600',
  },
  workOrderDetails: {
    paddingHorizontal: width * 0.05,
  },
  detailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.012,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  detailLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: width * 0.008,
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  dateContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  dropdown: {
    width: 40,
    backgroundColor: 'transparent',
    borderWidth: 0,
  },
  dropdownContainer: {
    width: 220,
    borderRadius: 3,
    paddingVertical: 4,
    paddingHorizontal: 0,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    elevation: 5,
    padding: 10,
    left: '40%',
    backgroundColor: '#fff',
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 11,
    paddingHorizontal: 11,
  },
  dropdownText: {
    marginLeft: 12,
    fontSize: 16,
  },
  dropdownItemText: {
    fontSize: 16,
  },
  emptyText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 24,
    height: 24,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  trackButton: {
    backgroundColor: '#015185',
    paddingVertical: 6,
    paddingHorizontal: 16,
    borderRadius: 6,
    marginTop: 8,
    alignSelf: 'flex-start',
  },
  trackButtonText: {
    color: '#fff',
    fontWeight: 'bold',
  },
  serviceScheduledBadge: {
    backgroundColor: '#015185',
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    marginLeft: 12,
    alignSelf: 'center',
  },
  serviceScheduledText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 14,
  },
  statusBadgeRow: {
    borderRadius: 6,
    paddingVertical: 6,
    paddingHorizontal: 14,
    alignSelf: 'flex-start',
    marginBottom: 8,
    marginTop: 8,
  },
  statusBadgeText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 14,
  },
});

export default CompleteWotab