<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\AccountDeletionRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class AccountDeletionController extends Controller
{
    /**
     * Show the account deletion request form
     */
    public function showForm()
    {
        return view('account-deletion.request');
    }

    /**
     * Handle account deletion request submission
     */
    public function submitRequest(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|max:255',
            'phone_number' => 'required|string|max:20',
            'nrc_id' => 'required|string|max:50',
            'reason' => 'nullable|string|max:1000'
        ], [
            'email.required' => 'Email address is required.',
            'email.email' => 'Please enter a valid email address.',
            'phone_number.required' => 'Phone number is required.',
            'nrc_id.required' => 'NRC ID is required.',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Check if there's already a pending request for this email
        $existingRequest = AccountDeletionRequest::where('email', $request->email)
            ->where('status', 'pending')
            ->first();

        if ($existingRequest) {
            return redirect()->back()
                ->with('error', 'You already have a pending account deletion request. Please wait for it to be processed.')
                ->withInput();
        }

        // Create new deletion request
        AccountDeletionRequest::create([
            'email' => $request->email,
            'phone_number' => $request->phone_number,
            'nrc_id' => $request->nrc_id,
            'reason' => $request->reason,
            'status' => 'pending'
        ]);

        return redirect()->back()
            ->with('success', 'Your account deletion request has been submitted successfully. We will process it within 7 business days.');
    }

    /**
     * Show admin panel for managing deletion requests
     */
    public function adminIndex()
    {
        $requests = AccountDeletionRequest::with('processedBy')
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('account-deletion.admin.index', compact('requests'));
    }

    /**
     * Show details of a specific deletion request
     */
    public function adminShow($id)
    {
        $request = AccountDeletionRequest::with('processedBy')->findOrFail($id);
        return view('account-deletion.admin.show', compact('request'));
    }

    /**
     * Approve a deletion request
     */
    public function adminApprove(Request $request, $id)
    {
        $deletionRequest = AccountDeletionRequest::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'admin_notes' => 'nullable|string|max:1000'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator);
        }

        $deletionRequest->update([
            'status' => 'approved',
            'admin_notes' => $request->admin_notes,
            'processed_at' => now(),
            'processed_by' => Auth::id()
        ]);

        return redirect()->route('account-deletion.admin.index')
            ->with('success', 'Account deletion request has been approved.');
    }

    /**
     * Reject a deletion request
     */
    public function adminReject(Request $request, $id)
    {
        $deletionRequest = AccountDeletionRequest::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'admin_notes' => 'required|string|max:1000'
        ], [
            'admin_notes.required' => 'Please provide a reason for rejection.'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator);
        }

        $deletionRequest->update([
            'status' => 'rejected',
            'admin_notes' => $request->admin_notes,
            'processed_at' => now(),
            'processed_by' => Auth::id()
        ]);

        return redirect()->route('account-deletion.admin.index')
            ->with('success', 'Account deletion request has been rejected.');
    }
}
