<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class DvirController extends Controller
{
    private $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
    
    /**
     * Save DVIR inspection report to Firebase
     */
    public function saveDvirReport(Request $request): JsonResponse
    {
        try {
            Log::info('DVIR report creation request received', [
                'request_data' => $request->all(),
                'headers' => $request->headers->all()
            ]);

            // Validate the request data
            $request->validate([
                'reportData' => 'required|array',
                'firebasePath' => 'required|string',
            ]);

            $reportData = $request->input('reportData');
            $firebasePath = $request->input('firebasePath');
            
            // Use the ID from the report data (generated in frontend)
            $timestamp = $reportData['id'] ?? round(microtime(true) * 1000);
            
            // Debug: Log the data being saved
            Log::info('Saving DVIR data to Firebase:', [
                'report_data' => $reportData,
                'firebase_path' => $firebasePath,
                'timestamp' => $timestamp
            ]);
            
            // Save to Firebase using the provided path and ID
            $response = Http::put(
                $this->firebaseUrl . $firebasePath . '/' . $timestamp . '.json',
                $reportData
            );
            
            if ($response->successful()) {
                Log::info('DVIR report saved successfully', [
                    'dvirNumber' => $reportData['dvirNumber'] ?? 'N/A',
                    'firebaseKey' => $timestamp,
                    'timestamp' => now()
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'DVIR inspection report saved successfully',
                    'data' => [
                        'firebaseKey' => (string)$timestamp,
                        'dvirNumber' => $reportData['dvirNumber'] ?? 'N/A',
                        'timestamp' => $reportData['createdAt'] ?? now()->toISOString(),
                        'pdfUrl' => '', // Will be generated after PDF creation
                        'signatureUrl' => '' // Will be generated after signature upload
                    ]
                ]);
            } else {
                Log::error('Failed to save DVIR report to Firebase', [
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'dvirNumber' => $reportData['dvirNumber'] ?? 'N/A'
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to save DVIR report to database. Status: ' . $response->status()
                ], 500);
            }
            
        } catch (\Exception $e) {
            Log::error('Error saving DVIR report', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'requestData' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error saving DVIR report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Transform Step 2 data to match Firebase structure
     */
    private function transformStep2Data($step2Data)
    {
        $items = [];
        
        // Requirements section
        $requirements = ['req1', 'req2', 'req3', 'req4', 'req5', 'req6', 'req7'];
        foreach ($requirements as $req) {
            $items[] = [
                'itemId' => $req,
                'label' => $this->getRequirementLabel($req),
                'note' => $step2Data['inspectionNotes'][$req] ?? '',
                'section' => 'Requirements',
                'status' => $step2Data['inspectionSelections'][$req] ?? ''
            ];
        }
        
        // Hazards section
        $hazards = ['haz1', 'haz2', 'haz3', 'haz4', 'haz5', 'haz6', 'haz7'];
        foreach ($hazards as $haz) {
            $items[] = [
                'itemId' => $haz,
                'label' => $this->getHazardLabel($haz),
                'note' => $step2Data['inspectionNotes'][$haz] ?? '',
                'section' => 'Hazards – Are you aware of:',
                'status' => $step2Data['inspectionSelections'][$haz] ?? ''
            ];
        }
        
        return $items;
    }

    /**
     * Get requirement labels
     */
    private function getRequirementLabel($reqId)
    {
        $labels = [
            'req1' => 'I hold a valid driver\'s licence',
            'req2' => 'I will obey speed limits and road signs',
            'req3' => 'I will not overload the vehicle and ensure the load is secure',
            'req4' => 'I have completed the vehicle pre check list and satisfied all is order for my trip',
            'req5' => 'I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)',
            'req6' => 'I have read and understand the Titan driving procedure, National driving rules and cliental rules',
            'req7' => 'I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)'
        ];
        
        return $labels[$reqId] ?? '';
    }

    /**
     * Get hazard labels
     */
    private function getHazardLabel($hazId)
    {
        $labels = [
            'haz1' => 'Pedestrians crossing the road, school children and drunkards',
            'haz2' => 'Motor bikes and taxis stopping suddenly or pulling in to the road suddenly',
            'haz3' => 'Oncoming vehicles overtaking on blind rises or corners',
            'haz4' => 'Heavy trucks or plant taking up width of the road',
            'haz5' => 'Riots or unrest',
            'haz6' => 'Gravel/Sandy roads can be slippery – Loss of control',
            'haz7' => 'Tyre blowouts or punctures – Loss of control'
        ];
        
        return $labels[$hazId] ?? '';
    }

    /**
     * Generate DVIR PDF
     */
    public function generateDvirPdf(Request $request)
    {
        try {
            $request->validate([
                'requestNumber' => 'required|string',
                'step1' => 'required|array',
                'step2' => 'required|array',
                'step3' => 'required|array',
            ]);

            $dvirData = $request->all();
            
            // Generate PDF content (you can customize this based on your needs)
            $pdfContent = $this->generateDvirPdfContent($dvirData);
            
            // For now, return success response
            // You can implement actual PDF generation later
            return response()->json([
                'success' => true,
                'message' => 'DVIR PDF generated successfully',
                'data' => [
                    'requestNumber' => $dvirData['requestNumber'],
                    'pdfContent' => $pdfContent
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error generating DVIR PDF: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error generating DVIR PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate DVIR PDF content
     */
    private function generateDvirPdfContent($dvirData)
    {
        // This is a placeholder for PDF generation
        // You can implement actual PDF generation logic here
        return "DVIR Report: " . $dvirData['requestNumber'];
    }

    /**
     * Get DVIR reports from Firebase
     */
    public function getDvirReports()
    {
        try {
            $response = Http::get($this->firebaseUrl . 'GlobalDvirReports/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/inspectionReports.json');
            
            if ($response->successful()) {
                $dvirData = $response->json() ?? [];
                return response()->json([
                    'success' => true,
                    'data' => $dvirData
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch DVIR reports'
                ], 500);
            }
            
        } catch (\Exception $e) {
            Log::error('Error fetching DVIR reports: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error fetching DVIR reports: ' . $e->getMessage()
            ], 500);
        }
    }
} 