<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class HrApplicationController extends Controller
{
    protected function resolveHrAdminUid(): string
    {
        return config('services.hr_module.admin_uid', 'cahj3jJqe5Z16pQdLGSWrNgm84o1');
    }

    /**
     * Get all applications by type from MongoDB
     */
    public function getApplicationsByType($applicationType)
    {
        try {
            $adminUid = $this->resolveHrAdminUid();
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'https://api.titandrillingzm.com:6013');
            
            $client = Http::timeout(30)
                ->acceptJson()
                ->withoutVerifying();
            
            $url = rtrim($baseUrl, '/') . "/hr-applications/{$applicationType}/{$adminUid}";
            
            $response = $client->get($url);
            
            if (!$response->successful()) {
                Log::error("Failed to fetch {$applicationType} applications", [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'data' => [],
                    'count' => 0
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error("Error fetching {$applicationType} applications: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'data' => [],
                'count' => 0,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Create new HR application in MongoDB
     */
    public function store(Request $request)
    {
        try {
            $adminUid = $this->resolveHrAdminUid();
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'https://api.titandrillingzm.com:6013');
            
            // Prepare application data
            $applicationData = $request->all();
            $applicationData['adminUid'] = $adminUid;
            $applicationData['createdAt'] = now()->toISOString();
            
            Log::info('Creating HR application', [
                'type' => $applicationData['applicationType'] ?? 'unknown',
                'adminUid' => $adminUid,
                'base_url' => $baseUrl
            ]);
            
            // Create HTTP client with SSL verification disabled
            $client = Http::timeout(30)
                ->acceptJson()
                ->withoutVerifying();
            
            $url = rtrim($baseUrl, '/') . '/hr-applications';
            
            try {
                $response = $client->post($url, $applicationData);
            } catch (\Illuminate\Http\Client\ConnectionException $connectionException) {
                // Try HTTP fallback if HTTPS fails
                $httpBaseUrl = preg_replace('#^https://#i', 'http://', $baseUrl);
                $httpUrl = rtrim($httpBaseUrl, '/') . '/hr-applications';
                
                Log::warning('HTTPS connection failed, trying HTTP fallback', [
                    'original_url' => $url,
                    'fallback_url' => $httpUrl,
                    'error' => $connectionException->getMessage()
                ]);
                
                $response = Http::timeout(30)
                    ->acceptJson()
                    ->withoutVerifying()
                    ->post($httpUrl, $applicationData);
            }
            
            if (!$response->successful()) {
                $errorBody = $response->body();
                Log::error('Failed to create HR application', [
                    'status' => $response->status(),
                    'status_text' => $response->statusText(),
                    'body' => $errorBody,
                    'url' => $url
                ]);
                
                $errorMessage = 'Failed to create application';
                try {
                    $errorData = json_decode($errorBody, true);
                    if (isset($errorData['error'])) {
                        $errorMessage = $errorData['error'];
                    } elseif (isset($errorData['message'])) {
                        $errorMessage = $errorData['message'];
                    }
                } catch (\Exception $e) {
                    // Keep default error message
                }
                
                return response()->json([
                    'success' => false,
                    'error' => $errorMessage,
                    'details' => $errorBody
                ], $response->status() ?: 500);
            }
            
            $result = $response->json();
            Log::info('HR application created successfully', [
                'applicationNumber' => $result['data']['applicationNumber'] ?? 'N/A'
            ]);
            
            return response()->json($result);
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Connection error creating HR application', [
                'message' => $e->getMessage(),
                'base_url' => $baseUrl ?? 'not set'
            ]);
            return response()->json([
                'success' => false,
                'error' => 'Cannot connect to HR application service. Please check your network connection.',
                'details' => $e->getMessage()
            ], 500);
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('Connection error creating HR application', [
                'message' => $e->getMessage(),
                'base_url' => $baseUrl ?? 'not set'
            ]);
            return response()->json([
                'success' => false,
                'error' => 'Cannot connect to HR application service. Please check if the service is running on port 6013.',
                'message' => 'Connection failed. Please try again later.'
            ], 500);
        } catch (\Exception $e) {
            Log::error('Error creating HR application', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'class' => get_class($e),
                'trace' => substr($e->getTraceAsString(), 0, 1000)
            ]);
            
            // Provide user-friendly error message
            $errorMessage = 'Failed to create application.';
            if (str_contains($e->getMessage(), 'Target class')) {
                $errorMessage = 'Application service error. Please contact support.';
            } elseif (str_contains($e->getMessage(), 'Connection')) {
                $errorMessage = 'Cannot connect to application service. Please check your network connection.';
            } elseif (str_contains($e->getMessage(), 'timeout')) {
                $errorMessage = 'Request timed out. Please try again.';
            }
            
            return response()->json([
                'success' => false,
                'error' => $errorMessage,
                'message' => $errorMessage
            ], 500);
        }
    }
    
    /**
     * Update application status (approve/reject)
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'https://api.titandrillingzm.com:6013');
            
            $statusData = [
                'status' => $request->status,
                'comments' => $request->comments,
                'approverUid' => auth()->id() ?? $this->resolveHrAdminUid(),
                'approverName' => $request->approverName,
                'approverRole' => $request->approverRole,
                'reason' => $request->reason,
                'timestamp' => now()->toISOString()
            ];
            
            Log::info('Updating HR application status', [
                'id' => $id,
                'status' => $request->status,
                'approverRole' => $request->approverRole
            ]);
            
            $client = Http::timeout(30)
                ->acceptJson()
                ->withoutVerifying();
            
            $url = rtrim($baseUrl, '/') . "/hr-applications/{$id}/status";
            $response = $client->put($url, $statusData);
            
            if (!$response->successful()) {
                Log::error('Failed to update application status', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'error' => 'Failed to update status'
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error('Error updating application status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Delete HR application
     */
    public function destroy($id)
    {
        try {
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'https://api.titandrillingzm.com:6013');
            
            Log::info('Deleting HR application', ['id' => $id]);
            
            $client = Http::timeout(30)
                ->acceptJson()
                ->withoutVerifying();
            
            $url = rtrim($baseUrl, '/') . "/hr-applications/{$id}";
            $response = $client->delete($url);
            
            if (!$response->successful()) {
                Log::error('Failed to delete application', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'error' => 'Failed to delete application'
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error('Error deleting application: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get application counts by type
     */
    public function getCounts()
    {
        try {
            $adminUid = $this->resolveHrAdminUid();
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'https://api.titandrillingzm.com:6013');
            
            $client = Http::timeout(30)
                ->acceptJson()
                ->withoutVerifying();
            
            $url = rtrim($baseUrl, '/') . "/hr-applications/counts/{$adminUid}";
            $response = $client->get($url);
            
            if (!$response->successful()) {
                Log::error('Failed to fetch HR application counts', [
                    'status' => $response->status()
                ]);
                return response()->json([
                    'success' => false,
                    'data' => []
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error('Error fetching counts: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
