<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;

class InspectionController extends Controller
{
    private $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
    
    public function jobCard(Request $request)
    {
        $inspectionId = $request->get('inspectionId');
        $requestNumber = $request->get('requestNumber');
        
        // Get logged-in user's MongoDB UID from session
        $adminUid = $request->session()->get('mongo_user_uid', null);
        
        // If inspection data is provided, use JobCardInsReport view (for creating from inspection)
        if ($inspectionId) {
            return view('inspection.JobCardInsReport', [
                'inspectionId' => $inspectionId,
                'requestNumber' => $requestNumber,
                'adminUid' => $adminUid
            ]);
        }
        
        // Otherwise use the standard job card view
        return view('inspection.job_card', [
            'adminUid' => $adminUid
        ]);
    }

    public function jobCardReport()
    {
        return view('inspection.JobCardInsReport');
    }

    public function jobCardIndex()
    {
        return view('inspection.job_card_index');
    }

    public function correctiveActionSteps(Request $request)
    {
        return view('inspection.InsCorrectiveAction.StepsCorrectiveAction', [
            'inspectionId' => $request->get('inspectionId'),
            'requestNumber' => $request->get('requestNumber'),
            'mode' => $request->get('mode')
        ]);
    }

    protected function inspectionHandlerHttp()
    {
        return Http::withOptions([
            'verify' => config('services.inspection_handler.verify_ssl', true),
            'timeout' => 45,
        ]);
    }

    protected function inspectionHandlerUrl(string $path): string
    {
        return rtrim(config('services.inspection_handler.base_url'), '/') . $path;
    }

    public function proxyInspectionReport(string $reportId): JsonResponse
    {
        try {
            $response = $this->inspectionHandlerHttp()->get(
                $this->inspectionHandlerUrl('/inspections/report/' . urlencode($reportId))
            );
            return response()->json($response->json(), $response->status());
        } catch (ConnectionException $e) {
            Log::error('Inspection proxy failed', ['id' => $reportId, 'error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Unable to fetch inspection'], 500);
        }
    }

    public function proxyGetCorrectiveAction(string $reportId): JsonResponse
    {
        try {
            $response = $this->inspectionHandlerHttp()->get(
                $this->inspectionHandlerUrl('/corrective-actions/' . urlencode($reportId))
            );
            return response()->json($response->json(), $response->status());
        } catch (ConnectionException $e) {
            Log::error('Corrective action fetch failed', ['id' => $reportId, 'error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Unable to fetch corrective action'], 500);
        }
    }

    public function proxySaveCorrectiveAction(Request $request): JsonResponse
    {
        $payload = $request->only(['reportId', 'userId', 'correctiveActionData']);
        try {
            $response = $this->inspectionHandlerHttp()->post(
                $this->inspectionHandlerUrl('/corrective-actions'),
                $payload
            );
            return response()->json($response->json(), $response->status());
        } catch (ConnectionException $e) {
            Log::error('Corrective action save failed', [
                'reportId' => $payload['reportId'] ?? null,
                'error' => $e->getMessage()
            ]);
            return response()->json(['success' => false, 'message' => 'Unable to save corrective action'], 500);
        }
    }

    public function proxyUploadCorrectivePdf(Request $request, string $reportId): JsonResponse
    {
        $request->validate([
            'pdf' => 'required|file|mimes:pdf|max:10240',
            'userId' => 'nullable|string'
        ]);

        try {
            $http = $this->inspectionHandlerHttp();
            $http = $http->attach('pdf', fopen($request->file('pdf')->getPathname(), 'r'), $request->file('pdf')->getClientOriginalName());
            if ($request->filled('userId')) {
                $http = $http->attach('userId', $request->input('userId'));
            }
            if ($request->filled('fileName')) {
                $http = $http->attach('fileName', $request->input('fileName'));
            }
            $response = $http->post(
                $this->inspectionHandlerUrl('/corrective-actions/' . urlencode($reportId) . '/upload-pdf')
            );
            return response()->json($response->json(), $response->status());
        } catch (\Throwable $e) {
            Log::error('Corrective PDF upload failed', ['id' => $reportId, 'error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Unable to upload PDF'], 500);
        }
    }

    public function proxyUploadCorrectiveSignature(Request $request, string $reportId): JsonResponse
    {
        $request->validate([
            'signature' => 'required|file|mimes:png,jpg,jpeg|max:4096',
            'signatureType' => 'required|string',
            'userId' => 'nullable|string'
        ]);

        try {
            $http = $this->inspectionHandlerHttp();
            $http = $http->attach('signature', fopen($request->file('signature')->getPathname(), 'r'), $request->file('signature')->getClientOriginalName());
            $http = $http->attach('signatureType', $request->input('signatureType'));
            if ($request->filled('userId')) {
                $http = $http->attach('userId', $request->input('userId'));
            }
            $response = $http->post(
                $this->inspectionHandlerUrl('/corrective-actions/' . urlencode($reportId) . '/upload-signature')
            );
            return response()->json($response->json(), $response->status());
        } catch (\Throwable $e) {
            Log::error('Corrective signature upload failed', ['id' => $reportId, 'error' => $e->getMessage()]);
            return response()->json(['success' => false, 'message' => 'Unable to upload signature'], 500);
        }
    }

    public function proxyGenerateCorrectivePdf(Request $request, string $reportId): JsonResponse
    {
        $payload = [
            'userId' => $request->input('userId', auth()->user()->email ?? 'unknown-admin')
        ];

        try {
            $response = $this->inspectionHandlerHttp()->post(
                $this->inspectionHandlerUrl('/corrective-actions/' . urlencode($reportId) . '/generate-pdf'),
                $payload
            );

            return response()->json($response->json(), $response->status());
        } catch (ConnectionException $e) {
            Log::error('Corrective PDF generation failed', [
                'reportId' => $reportId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Unable to generate corrective action PDF'
            ], 500);
        }
    }

    public function getJobCards(Request $request)
    {
        try {
            Log::info('Fetching job cards from Firebase');

            // Get pagination parameters
            $perPage = $request->get('per_page', 10);
            $page = $request->get('page', 1);

            // For now, we'll use mock data since we don't have actual job card data structure
            // In a real implementation, you would fetch from Firebase or database
            $mockJobCards = [
                [
                    'jobCardNumber' => 'ZAM-KOB-JC-1002',
                    'title' => 'Engine Maintenance - Light Vehicle',
                    'priority' => 'Medium',
                    'status' => 'In Progress',
                    'equipmentNumber' => 'ZAM-KOB-001',
                    'equipmentName' => 'Toyota Hilux',
                    'project' => 'Kobold',
                    'startDate' => '2025-01-15',
                    'dueDate' => '2025-01-20',
                    'assignedTo' => 'John Smith',
                    'createdAt' => '2025-01-10'
                ],
                [
                    'jobCardNumber' => 'ZAM-KOB-JC-1003',
                    'title' => 'Brake System Inspection',
                    'priority' => 'High',
                    'status' => 'Initiated',
                    'equipmentNumber' => 'ZAM-KOB-002',
                    'equipmentName' => 'Caterpillar 320D',
                    'project' => 'Kobold',
                    'startDate' => '2025-01-16',
                    'dueDate' => '2025-01-18',
                    'assignedTo' => 'Mike Johnson',
                    'createdAt' => '2025-01-12'
                ],
                [
                    'jobCardNumber' => 'ZAM-KOB-JC-1004',
                    'title' => 'Drilling Equipment Calibration',
                    'priority' => 'Low',
                    'status' => 'On Hold',
                    'equipmentNumber' => 'ZAM-KOB-003',
                    'equipmentName' => 'Atlas Copco ROC L8',
                    'project' => 'Kobold',
                    'startDate' => '2025-01-18',
                    'dueDate' => '2025-01-25',
                    'assignedTo' => 'Sarah Wilson',
                    'createdAt' => '2025-01-14'
                ]
            ];

            // Apply pagination
            $totalJobCards = count($mockJobCards);
            $totalPages = ceil($totalJobCards / $perPage);
            $offset = ($page - 1) * $perPage;
            $paginatedJobCards = array_slice($mockJobCards, $offset, $perPage);

            return response()->json([
                'success' => true,
                'message' => 'Job cards retrieved successfully',
                'jobCards' => $paginatedJobCards,
                'pagination' => [
                    'current_page' => (int) $page,
                    'per_page' => (int) $perPage,
                    'total' => $totalJobCards,
                    'total_pages' => $totalPages,
                    'from' => $offset + 1,
                    'to' => min($offset + $perPage, $totalJobCards),
                    'has_more_pages' => $page < $totalPages
                ]
            ]);

        } catch (\Exception $e) {
            Log::error('Error fetching job cards: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error fetching job cards: ' . $e->getMessage()
            ], 500);
        }
    }

    public function index()
    {
        $inspectionData = $this->fetchInspectionData();
        return view('inspection.inspection', compact('inspectionData'));
    }

    public function inspectionReport()
    {
        // Just return the view - data will be loaded via JavaScript from MongoDB API
        return view('inspection.inspection');
    }
    
    /**
     * API endpoint to fetch inspection reports from MongoDB
     */
    public function getAllInspectionReports(Request $request)
    {
        try {
            $page = $request->get('page', 1);
            $limit = $request->get('limit', 10000); // Get all for now
            
            $mongoUrl = env('MONGODB_INSPECTION_GET_ALL_URL', 'https://api.titandrillingzm.com:6004/inspections/all');
            
            Log::info('Fetching inspections from MongoDB', [
                'url' => $mongoUrl,
                'page' => $page,
                'limit' => $limit
            ]);
            
            $response = Http::timeout(30)->get($mongoUrl, [
                'page' => $page,
                'limit' => $limit
            ]);
            
            if ($response->successful()) {
                $data = $response->json();
                $allInspections = $data['data'] ?? [];
                
                Log::info('Inspections fetched successfully from MongoDB', [
                    'total' => count($allInspections)
                ]);
                
                return response()->json([
                    'success' => true,
                    'data' => $allInspections,
                    'pagination' => $data['pagination'] ?? [],
                    'total' => count($allInspections)
                ]);
            }
            
            Log::error('Failed to fetch inspections from MongoDB', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch inspection reports from MongoDB'
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('Exception in getAllInspectionReports', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * API endpoint to fetch inspection reports with linked Job Cards and Corrective Actions
     * Matches React Native functionality from Inspection_Report_Home.js
     */
    public function getAllInspectionsWithRelations(Request $request)
    {
        try {
            Log::info('🔗 Fetching inspections with job cards and corrective actions...');
            
            // Step 1: Fetch all inspections from MongoDB
            $inspectionsUrl = 'https://api.titandrillingzm.com:6004/inspections/all';
            $inspectionsResponse = Http::timeout(30)->get($inspectionsUrl, [
                'page' => 1,
                'limit' => 10000
            ]);
            
            if (!$inspectionsResponse->successful()) {
                throw new \Exception('Failed to fetch inspections from MongoDB');
            }
            
            $inspections = $inspectionsResponse->json()['data'] ?? [];
            Log::info('✅ Fetched ' . count($inspections) . ' inspections');
            
            // Step 2: Fetch all job cards from MongoDB
            $jobCardsUrl = 'https://api.titandrillingzm.com:6005/jobcards/all';
            $jobCardsResponse = Http::timeout(30)->get($jobCardsUrl, [
                'page' => 1,
                'limit' => 10000
            ]);
            
            $jobCards = [];
            $jobCardsMap = [];
            
            if ($jobCardsResponse->successful()) {
                $jobCardsData = $jobCardsResponse->json()['data'] ?? [];
                Log::info('✅ Fetched ' . count($jobCardsData) . ' job cards');
                
                // Create a map of inspection ID to job card (like React Native)
                // Match by both inspection ID AND adminUid to ensure correct linking
                foreach ($jobCardsData as $jobCard) {
                    // Check multiple possible linkage fields (prioritize in order)
                    // CRITICAL: serviceScheduleId is the PRIMARY field for linking (matches React Native)
                    $inspectionId = null;
                    $jobCardAdminUid = $jobCard['adminUid'] ?? null;
                    
                    // Priority 1: serviceScheduleId (PRIMARY - this is set when creating job card from inspection)
                    if (isset($jobCard['serviceScheduleId']) && !empty($jobCard['serviceScheduleId'])) {
                        $inspectionId = $jobCard['serviceScheduleId'];
                    }
                    
                    // Priority 2: Direct linkInspection field (matches React Native logic)
                    if (!$inspectionId && isset($jobCard['linkInspection'])) {
                        $inspectionId = $jobCard['linkInspection']['id'] ?? 
                                      $jobCard['linkInspection']['_id'] ?? 
                                      $jobCard['linkInspection']['inspectionId'] ?? null;
                    }
                    
                    // Priority 3: Direct inspectionId field
                    if (!$inspectionId && isset($jobCard['inspectionId'])) {
                        $inspectionId = $jobCard['inspectionId'];
                    }
                    
                    // Priority 4: Check if title or description contains inspection request number
                    if (!$inspectionId) {
                        $title = $jobCard['title'] ?? '';
                        $description = $jobCard['description'] ?? '';
                        $combined = $title . ' ' . $description;
                        
                        // Look for inspection request number pattern (e.g., ZAM-KAN-IR-001)
                        if (preg_match('/([A-Z]{3}-[A-Z]{3}-IR-\d{3,4})/', $combined, $matches)) {
                            // Store the request number as a potential match
                            // We'll link it in the next step
                            $jobCard['_inspectionRequestNumber'] = $matches[1];
                        }
                    }
                    
                    // Create composite key: inspectionId + adminUid (matches React Native behavior)
                    // This ensures job cards are matched to inspections from the same admin
                    if ($inspectionId) {
                        // Use inspection ID as primary key, but store adminUid for verification
                        $jobCard['_matchedAdminUid'] = $jobCardAdminUid;
                        $jobCardsMap[$inspectionId] = $jobCard;
                    }
                }
                
                Log::info('📊 Job cards mapped to ' . count($jobCardsMap) . ' inspections');
            } else {
                Log::warning('⚠️ Failed to fetch job cards, continuing without them');
            }
            
            // Step 3: Create reverse map for job cards with inspection request numbers
            $requestNumberToJobCard = [];
            foreach ($jobCardsData ?? [] as $jobCard) {
                if (isset($jobCard['_inspectionRequestNumber'])) {
                    $requestNumberToJobCard[$jobCard['_inspectionRequestNumber']] = $jobCard;
                }
            }
            
            // Step 4: Enhance inspections with job card and corrective action data
            $enhancedInspections = [];
            
            foreach ($inspections as $inspection) {
                $inspectionId = $inspection['_id'] ?? $inspection['id'] ?? null;
                $inspectionAdminUid = $inspection['adminUID'] ?? $inspection['userId'] ?? $inspection['adminUid'] ?? null;
                $requestNumber = $inspection['requestNumber'] ?? $inspection['step1']['requestNumber'] ?? null;
                
                // Try to find job card by ID first, then by request number
                // Match by both inspection ID AND adminUid (like React Native)
                $linkedJobCard = null;
                
                if ($inspectionId && isset($jobCardsMap[$inspectionId])) {
                    $potentialJobCard = $jobCardsMap[$inspectionId];
                    $jobCardAdminUid = $potentialJobCard['adminUid'] ?? $potentialJobCard['_matchedAdminUid'] ?? null;
                    
                    // Match by inspection ID AND adminUid (if both are available)
                    // If adminUid matches or if inspection doesn't have adminUid, use the job card
                    if (!$inspectionAdminUid || !$jobCardAdminUid || $inspectionAdminUid === $jobCardAdminUid) {
                        $linkedJobCard = $potentialJobCard;
                    }
                }
                
                // If not found by ID, try matching by request number (fallback)
                if (!$linkedJobCard && $requestNumber && isset($requestNumberToJobCard[$requestNumber])) {
                    $potentialJobCard = $requestNumberToJobCard[$requestNumber];
                    $jobCardAdminUid = $potentialJobCard['adminUid'] ?? null;
                    
                    // Also verify adminUid match for request number matching
                    if (!$inspectionAdminUid || !$jobCardAdminUid || $inspectionAdminUid === $jobCardAdminUid) {
                        $linkedJobCard = $potentialJobCard;
                    }
                }
                if ($linkedJobCard) {
                    $inspection['linkedJobCard'] = [
                        'id' => $linkedJobCard['_id'] ?? $linkedJobCard['id'],
                        'requestNumber' => $linkedJobCard['requestNumber'] ?? $linkedJobCard['title'],
                        'status' => $linkedJobCard['status'] ?? 'Hold',
                        'title' => $linkedJobCard['title'] ?? 'N/A',
                        'pdfDownloadUrl' => $linkedJobCard['pdfDownloadUrl'] ?? null,
                        'createdAt' => $linkedJobCard['createdAt'] ?? null
                    ];
                }
                
                // Add corrective action status from existing data
                $hasCorrectiveAction = isset($inspection['correctiveActionData']) && 
                                      !empty($inspection['correctiveActionData']);
                
                $inspection['hasCorrectiveAction'] = $hasCorrectiveAction;
                
                if ($hasCorrectiveAction) {
                    $inspection['correctiveAction'] = [
                        'exists' => true,
                        'pdfDownloadUrl' => $inspection['correctiveActionData']['pdfDownloadUrl'] ?? null,
                        'mechanicName' => $inspection['correctiveActionData']['mechanicName'] ?? null,
                        'operatorName' => $inspection['correctiveActionData']['operatorName'] ?? null,
                        'isSatisfactory' => $inspection['correctiveActionData']['isSatisfactory'] ?? false
                    ];
                }
                
                $enhancedInspections[] = $inspection;
            }
            
            Log::info('✅ Enhanced ' . count($enhancedInspections) . ' inspections with relations');
            
            return response()->json([
                'success' => true,
                'data' => $enhancedInspections,
                'total' => count($enhancedInspections),
                'jobCardsLinked' => count($jobCardsMap)
            ]);
            
        } catch (\Exception $e) {
            Log::error('❌ Error in getAllInspectionsWithRelations: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Test MongoDB connection
     */
    public function testMongoConnection()
    {
        try {
            $mongoUrl = 'https://api.titandrillingzm.com:6004/inspectionreports';
            
            Log::info('🧪 Testing MongoDB connection', ['url' => $mongoUrl]);
            
            $response = Http::timeout(10)->get($mongoUrl);
            
            if ($response->successful()) {
                $data = $response->json();
                return response()->json([
                    'success' => true,
                    'message' => 'MongoDB server is accessible',
                    'data' => $data,
                    'status' => $response->status()
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'MongoDB server responded with error',
                'status' => $response->status(),
                'body' => $response->body()
            ], $response->status());
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot connect to MongoDB server',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Create new inspection report in MongoDB
     */
    public function createInspectionReport(Request $request)
    {
        try {
            // Get data from request body
            $userId = $request->input('userId');
            $reportData = $request->input('reportData');
            
            // Fallback to getting all data if structure is different
            if (!$reportData) {
                $reportData = $request->all();
                $userId = $reportData['userId'] ?? auth()->user()->email ?? 'unknown';
            }
            
            // IMPORTANT: Ensure _id is always set (never undefined)
            if (!isset($reportData['_id']) || empty($reportData['_id'])) {
                $reportData['_id'] = $reportData['requestNumber'] ?? 'report_' . time();
                Log::info('📝 Generated missing _id', ['_id' => $reportData['_id']]);
            }
            
            // Ensure requestNumber is set
            if (!isset($reportData['requestNumber']) || empty($reportData['requestNumber'])) {
                $reportData['requestNumber'] = $reportData['_id'];
                Log::info('📝 Generated missing requestNumber', ['requestNumber' => $reportData['requestNumber']]);
            }
            
            Log::info('📝 Creating inspection report in MongoDB', [
                '_id' => $reportData['_id'],
                'userId' => $userId,
                'requestNumber' => $reportData['requestNumber'],
                'hasStep1' => isset($reportData['step1']),
                'hasStep2' => isset($reportData['step2']),
                'hasStep3' => isset($reportData['step3'])
            ]);
            
            // Send to MongoDB API
            $mongoUrl = 'https://api.titandrillingzm.com:6004/inspections';
            
            $response = Http::timeout(30)->post($mongoUrl, [
                'userId' => $userId,
                'reportData' => $reportData
            ]);
            
            if ($response->successful()) {
                $result = $response->json();
                
                // Safely extract report ID from response
                $savedReportId = $result['data']['_id'] ?? 
                                $result['_id'] ?? 
                                $reportData['_id'] ?? 
                                'unknown';
                
                Log::info('✅ Inspection report created successfully', [
                    'reportId' => $savedReportId,
                    'responseKeys' => array_keys($result)
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Inspection report created successfully',
                    'data' => $result['data'] ?? $result,
                    'reportId' => $savedReportId
                ]);
            }
            
            $errorBody = $response->body();
            $errorData = json_decode($errorBody, true);
            
            Log::error('❌ Failed to create inspection in MongoDB', [
                'status' => $response->status(),
                'body' => $errorBody,
                'decoded_error' => $errorData,
                'mongodb_url' => $mongoUrl
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create inspection report',
                'error' => $errorData['error'] ?? $errorData['message'] ?? $errorBody,
                'mongodb_status' => $response->status(),
                'debug_info' => [
                    'url' => $mongoUrl,
                    'has_userId' => !empty($userId),
                    'has_reportData' => !empty($reportData),
                    'reportId' => $reportData['_id'] ?? 'not_set'
                ]
            ], 500);
            
        } catch (\Exception $e) {
            Log::error('❌ Exception in createInspectionReport', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error creating report: ' . $e->getMessage(),
                'error' => config('app.debug') ? [
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ] : null
            ], 500);
        }
    }

    public function inspection2()
    {
        return view('inspection.inspection2');
    }

    public function doInspection()
    {
        return view('inspection.do_inspection');
    }

    public function store(Request $request)
    {
        try {
            // Add your validation and storage logic here
            Log::info('Inspection store request received', $request->all());
            
            // TODO: Implement actual storage logic
            return redirect()->route('inspection.index')->with('success', 'Inspection data stored successfully');
        } catch (\Exception $e) {
            Log::error('Error storing inspection data: ' . $e->getMessage());
            return redirect()->route('inspection.index')->with('error', 'Failed to store inspection data');
        }
    }

    public function incidentReport()
    {
        return view('inspection.incident_report');
    }

    public function hseReports()
    {
        return view('inspection.HSEmodule.HSEreportsALLc');
    }

    protected function getHseInspectionBaseUrl(): string
    {
        return rtrim(config('services.hse_inspections.base_url', ''), '/');
    }

    protected function getHseCorrectiveBaseUrl(): string
    {
        return rtrim(config('services.hse_corrective_actions.base_url', ''), '/');
    }

    protected function buildHseUrl(string $baseUrl, string $endpoint): string
    {
        return rtrim($baseUrl, '/') . '/' . ltrim($endpoint, '/');
    }

    protected function shouldRetryOverHttp(string $baseUrl, \Throwable $exception): bool
    {
        if (!str_starts_with(strtolower($baseUrl), 'https://')) {
            return false;
        }

        $message = strtolower($exception->getMessage());

        // Always retry on connection errors - might be SSL or network issue
        if ($exception instanceof \Illuminate\Http\Client\ConnectionException) {
            return true;
        }

        return str_contains($message, 'ssl3_get_record')
            || str_contains($message, 'wrong version number')
            || str_contains($message, 'unknown protocol')
            || str_contains($message, 'sslv3 alert handshake failure')
            || str_contains($message, 'connection')
            || str_contains($message, 'timeout')
            || str_contains($message, 'failed to connect');
    }

    protected function callHseService(string $service, string $method, string $endpoint, array $options = [])
    {
        $baseUrl = $service === 'inspections'
            ? $this->getHseInspectionBaseUrl()
            : $this->getHseCorrectiveBaseUrl();

        if (empty($baseUrl)) {
            throw new \RuntimeException("HSE {$service} service URL is not configured.");
        }

        $client = $this->hseHttpClient($service);
        $url = $this->buildHseUrl($baseUrl, $endpoint);
        
        // For GET requests, query parameters should be passed separately
        $httpMethod = strtolower($method);
        $queryParams = [];
        $bodyData = [];
        
        if ($httpMethod === 'get') {
            // For GET requests, all options are query parameters
            $queryParams = $options;
        } else {
            // For POST/PUT/DELETE, options go in the request body
            $bodyData = $options;
        }

        try {
            Log::debug("Calling HSE service", [
                'service' => $service,
                'method' => $httpMethod,
                'url' => $url,
                'query_params' => $queryParams,
                'has_body' => !empty($bodyData),
            ]);
            
            if ($httpMethod === 'get') {
                return $client->get($url, $queryParams);
            } else {
                return $client->{strtolower($method)}($url, $bodyData);
            }
        } catch (\Throwable $exception) {
            // Always try HTTP fallback for connection exceptions or HTTPS URLs
            $shouldRetry = $exception instanceof \Illuminate\Http\Client\ConnectionException 
                || $this->shouldRetryOverHttp($baseUrl, $exception);
            
            if ($shouldRetry && str_starts_with(strtolower($baseUrl), 'https://')) {
                $fallbackBase = preg_replace('#^https://#i', 'http://', $baseUrl);
                Log::warning("Retrying HSE {$service} request over HTTP", [
                    'endpoint' => $endpoint,
                    'original_url' => $baseUrl,
                    'fallback_url' => $fallbackBase,
                    'exception' => $exception->getMessage(),
                    'exception_class' => get_class($exception),
                ]);

                try {
                    $fallbackUrl = $this->buildHseUrl($fallbackBase, $endpoint);
                    $fallbackClient = $this->hseHttpClient($service);
                    
                    if ($httpMethod === 'get') {
                        $fallbackResponse = $fallbackClient->get($fallbackUrl, $queryParams);
                        if ($fallbackResponse->successful()) {
                            Log::info("HTTP fallback succeeded for HSE {$service}");
                            return $fallbackResponse;
                        }
                    } else {
                        $fallbackResponse = $fallbackClient->{strtolower($method)}($fallbackUrl, $bodyData);
                        if ($fallbackResponse->successful()) {
                            Log::info("HTTP fallback succeeded for HSE {$service}");
                            return $fallbackResponse;
                        }
                    }
                } catch (\Throwable $fallbackException) {
                    Log::error("HTTP fallback also failed for HSE {$service}", [
                        'exception' => $fallbackException->getMessage(),
                    ]);
                }
            }

            Log::error("HSE service call failed", [
                'service' => $service,
                'method' => $httpMethod,
                'url' => $url,
                'exception' => $exception->getMessage(),
                'exception_class' => get_class($exception),
            ]);

            throw $exception;
        }
    }

    protected function hseHttpClient(string $service = 'inspections', bool $useHttp = false)
    {
        $config = $service === 'inspections'
            ? config('services.hse_inspections', [])
            : config('services.hse_corrective_actions', []);

        // Always disable SSL verification for production to avoid certificate issues
        $client = Http::timeout(30)
            ->acceptJson()
            ->withoutVerifying(); // Always disable SSL verification

        return $client;
    }

    public function getHseReportsData(Request $request): JsonResponse
    {
        $baseUrl = $this->getHseInspectionBaseUrl();
        if (empty($baseUrl)) {
            Log::error('HSE Inspection service URL is not configured');
            return response()->json([
                'success' => false,
                'data' => [],
                'message' => 'HSE Inspection service URL is not configured.',
            ], 500);
        }

        $page = max(1, (int) $request->get('page', 1));
        $limit = max(1, min((int) $request->get('limit', 500), 1000)); // Cap at 1000

        // Try multiple ports as fallback (6014 for HTTPS, 5014 for HTTP)
        $baseUrlsToTry = [
            $baseUrl, // Primary configured URL (port 6014)
            preg_replace('#:6014#', ':5014', $baseUrl), // Fallback to port 5014
            preg_replace('#https://#', 'http://', preg_replace('#:6014#', ':5014', $baseUrl)), // HTTP on 5014
        ];
        
        $lastException = null;
        $lastError = null;

        foreach ($baseUrlsToTry as $tryUrl) {
            try {
                Log::info('Attempting to fetch HSE inspections', [
                    'base_url' => $tryUrl,
                    'page' => $page,
                    'limit' => $limit,
                ]);

                // Temporarily override base URL for this attempt
                $originalBaseUrl = $baseUrl;
                
                // Build the full URL manually for this attempt
                $fullUrl = rtrim($tryUrl, '/') . '/hse-inspections/all';
                $client = $this->hseHttpClient('inspections');
                
                $response = $client->get($fullUrl, [
                    'page' => $page,
                    'limit' => $limit,
                ]);

                if ($response->successful()) {
                    $payload = $response->json();
                    Log::info('Successfully fetched HSE inspections', [
                        'url_used' => $tryUrl,
                        'count' => count($payload['data'] ?? []),
                    ]);

                    return response()->json([
                        'success' => $payload['success'] ?? true,
                        'data' => $payload['data'] ?? [],
                        'pagination' => $payload['pagination'] ?? [],
                        'message' => $payload['message'] ?? 'HSE inspections fetched successfully.',
                    ]);
                } else {
                    $lastError = [
                        'url' => $tryUrl,
                        'status' => $response->status(),
                        'status_text' => $response->statusText(),
                        'body' => substr($response->body(), 0, 500),
                    ];
                    Log::warning('HSE service returned error status', $lastError);
                }
            } catch (\Illuminate\Http\Client\ConnectionException $exception) {
                $lastException = $exception;
                Log::warning('Connection failed for HSE inspections URL', [
                    'url' => $tryUrl,
                    'error' => $exception->getMessage(),
                ]);
                continue; // Try next URL
            } catch (\Exception $exception) {
                $lastException = $exception;
                Log::warning('Exception fetching HSE inspections from URL', [
                    'url' => $tryUrl,
                    'error' => $exception->getMessage(),
                ]);
                continue; // Try next URL
            }
        }

        // All URLs failed - return error with details
        $errorMessage = 'Cannot connect to HSE inspection service.';
        if ($lastException) {
            $errorMessage .= ' Error: ' . $lastException->getMessage();
        } elseif ($lastError) {
            $errorMessage .= ' Service returned status ' . ($lastError['status'] ?? 'unknown');
        }
        
        $errorMessage .= ' Tried URLs: ' . implode(', ', array_slice($baseUrlsToTry, 0, 3));

        Log::error('All HSE inspection URLs failed', [
            'tried_urls' => $baseUrlsToTry,
            'last_exception' => $lastException ? $lastException->getMessage() : null,
            'last_error' => $lastError,
        ]);

        return response()->json([
            'success' => false,
            'data' => [],
            'message' => $errorMessage,
            'error' => $lastException ? $lastException->getMessage() : ($lastError['status_text'] ?? 'Unknown error'),
        ], 500);
    }

    public function getHseCorrectiveAction(string $reportId): JsonResponse
    {
        $baseUrl = $this->getHseCorrectiveBaseUrl();
        if (empty($baseUrl)) {
            return response()->json([
                'success' => false,
                'data' => null,
                'message' => 'HSE Corrective Action service URL is not configured.',
            ], 500);
        }

        try {
            $response = $this->callHseService(
                'corrective',
                'get',
                '/hse-corrective-actions/' . urlencode($reportId)
            );

            if (!$response->successful()) {
                Log::warning('Failed to fetch corrective action', [
                    'reportId' => $reportId,
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);

                return response()->json([
                    'success' => false,
                    'data' => null,
                    'message' => 'Corrective action not found.',
                ], $response->status());
            }

            return response()->json($response->json());
        } catch (\Exception $exception) {
            Log::error('Exception fetching corrective action', [
                'reportId' => $reportId,
                'message' => $exception->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'data' => null,
                'message' => 'Unexpected error fetching corrective action.',
            ], 500);
        }
    }

    public function deleteHseReport(Request $request, string $userId, string $reportId): JsonResponse
    {
        $baseUrl = $this->getHseInspectionBaseUrl();
        if (empty($baseUrl)) {
            return response()->json([
                'success' => false,
                'message' => 'HSE Inspection service URL is not configured.',
            ], 500);
        }

        try {
            $response = $this->callHseService(
                'inspections',
                'delete',
                '/hse-inspections/' . urlencode($userId) . '/' . urlencode($reportId)
            );

            if (!$response->successful()) {
                Log::warning('Failed to delete HSE inspection', [
                    'userId' => $userId,
                    'reportId' => $reportId,
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete HSE inspection.',
                ], $response->status());
            }

            return response()->json($response->json());
        } catch (\Exception $exception) {
            Log::error('Exception deleting HSE inspection', [
                'userId' => $userId,
                'reportId' => $reportId,
                'message' => $exception->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Unexpected error deleting HSE inspection.',
            ], 500);
        }
    }

    public function backToIncidentReport()
    {
        return view('inspection.back_to_incident_report');
    }

    public function incidentChecklist()
    {
        return view('inspection.incident_checklist');
    }

    public function reminder()
    {
        return view('inspection.reminder');
    }

    public function operatorDailyReporting()
    {
        $operatorReports = $this->fetchOperatorDailyReports();
        return view('inspection.operator_daily_reporting', compact('operatorReports'));
    }

    /**
     * Fetch operator daily reports from MongoDB
     */
    public function fetchOperatorDailyReports()
    {
        try {
            Log::info('Fetching operator daily reports from MongoDB...');
            
            // Fetch ALL daily reports from MongoDB API
            $mongoUrl = 'https://api.titandrillingzm.com:6008/daily-reports';
            
            $response = Http::timeout(10)->get($mongoUrl);
            
            if (!$response->successful()) {
                Log::warning('Failed to fetch operator daily reports from MongoDB', [
                    'status' => $response->status(),
                    'url' => $mongoUrl
                ]);
                return [];
            }
            
            $result = $response->json();
            
            if (!isset($result['success']) || !$result['success'] || !isset($result['data'])) {
                Log::warning('Invalid response format from MongoDB API');
                return [];
            }
            
            $operatorData = $result['data'];
            
            if (empty($operatorData)) {
                Log::info('No operator daily reports found in MongoDB');
                return [];
            }
            
            Log::info('Fetched ' . count($operatorData) . ' daily reports from MongoDB');
            
            // Process the data to match our expected format
            $processedData = [];
            $groupedStatuses = [];
            $dayKeyMap = [
                1 => 'mon',
                2 => 'tue',
                3 => 'wed',
                4 => 'thu',
                5 => 'fri',
                6 => 'sat',
                7 => 'sun'
            ];
            
            foreach ($operatorData as $report) {
                $pdfDownloadUrl = $report['pdfDownloadUrl'] ?? ($report['pdfUrl'] ?? null);
                
                if (empty($pdfDownloadUrl) || !is_string($pdfDownloadUrl)) {
                    continue;
                }
                
                // Extract equipment/vehicle information
                $equipmentName = 'N/A';
                if (isset($report['selectedEquipment'])) {
                    $equipmentName = $report['selectedEquipment']['equipmentName'] ?? 
                                    $report['selectedEquipment']['vehicleNumber'] ?? 
                                    $report['selectedEquipment']['name'] ?? 'N/A';
                }
                
                // Parse report date (start date or created date)
                $rawStartDate = $report['step1']['startDate'] ?? null;
                $reportDate = null;
                
                if ($rawStartDate) {
                    $reportDate = $this->parseDateSafely($rawStartDate);
                }
                
                if (!$reportDate && !empty($report['createdAt'])) {
                    $reportDate = $this->parseDateSafely($report['createdAt']);
                }
                
                if (!$reportDate) {
                    $reportDate = Carbon::now();
                }
                
                $startDate = $reportDate->format('d/m/Y');
                
                // Determine submission status
                $hasStep2 = isset($report['step2']) && !empty($report['step2']);
                $hasStep3 = isset($report['step3']) && !empty($report['step3']);
                $submitted = $hasStep3;
                
                // Checklist metadata
                $checklistDetails = $report['step2']['checklistDetails'] ?? 
                                    $report['checklistDetails'] ?? 
                                    $report['checklist'] ?? [];
                
                $checklistTitle = is_array($checklistDetails)
                    ? ($checklistDetails['formTitle'] ?? $checklistDetails['title'] ?? 'N/A')
                    : 'N/A';
                
                $checklistKey = is_array($checklistDetails)
                    ? ($checklistDetails['checklistId'] ?? $checklistDetails['id'] ?? $checklistTitle)
                    : $checklistTitle;
                
                $equipmentId = $report['selectedEquipment']['id'] ?? 
                               $report['selectedEquipment']['equipmentId'] ?? 
                               $report['selectedEquipment']['uniqueId'] ?? 
                               $report['selectedEquipment']['vehicleNumber'] ?? 
                               $report['requestNumber'] ?? 'UNKNOWN_EQUIPMENT';
                
                $weekStart = $reportDate->copy()->startOfWeek(Carbon::MONDAY)->toDateString();
                $dayKey = $dayKeyMap[$reportDate->dayOfWeekIso ?? 1] ?? 'mon';
                
                $groupKey = implode('|', [$equipmentId, $checklistKey, $weekStart]);
                
                if (!isset($groupedStatuses[$groupKey])) {
                    $groupedStatuses[$groupKey] = [
                        'mon' => 'pending',
                        'tue' => 'pending',
                        'wed' => 'pending',
                        'thu' => 'pending',
                        'fri' => 'pending',
                        'sat' => 'pending',
                        'sun' => 'pending'
                    ];
                }
                
                if ($submitted) {
                    $groupedStatuses[$groupKey][$dayKey] = 'completed';
                }
                
                $processedData[] = [
                    'id' => $report['_id'] ?? $report['id'] ?? uniqid(),
                    'reportNumber' => $report['requestNumber'] ?? 'N/A',
                    'startDate' => $startDate,
                    'vehicleNumber' => $equipmentName,
                    'vehicleName' => $equipmentName,
                    'mon' => 'pending',
                    'tue' => 'pending',
                    'wed' => 'pending',
                    'thu' => 'pending',
                    'fri' => 'pending',
                    'sat' => 'pending',
                    'sun' => 'pending',
                    'submitted' => $submitted,
                    'created_at' => $report['createdAt'] ?? date('Y-m-d H:i:s'),
                    'updated_at' => $report['uploadTimestamp'] ?? date('Y-m-d H:i:s'),
                    'pdfUrl' => $pdfDownloadUrl,
                    'checklistTitle' => $checklistTitle,
                    'checklistDetails' => $checklistDetails,
                    'equipmentId' => $equipmentId,
                    'weekStart' => $weekStart,
                    'dayOfWeek' => $dayKey,
                    'groupKey' => $groupKey,
                    'reportDateIso' => $reportDate->toDateString()
                ];
            }
            
            // Apply grouped weekly statuses to each record
            foreach ($processedData as &$entry) {
                $groupKey = $entry['groupKey'] ?? null;
                if ($groupKey && isset($groupedStatuses[$groupKey])) {
                    foreach ($groupedStatuses[$groupKey] as $day => $status) {
                        $entry[$day] = $status;
                    }
                }
                
                unset($entry['groupKey']);
            }
            unset($entry);
            
            Log::info('Processed operator daily reports count: ' . count($processedData));
            return $processedData;
            
        } catch (\Exception $e) {
            Log::error('Error fetching operator daily reports from MongoDB: ' . $e->getMessage(), [
                'exception' => $e->getTraceAsString()
            ]);
            return [];
        }
    }

    /**
     * Safely parse a date string into a Carbon instance.
     */
    private function parseDateSafely($dateValue): ?Carbon
    {
        if (empty($dateValue)) {
            return null;
        }

        $formats = [
            Carbon::ISO8601,
            'Y-m-d\TH:i:s.u\Z',
            'Y-m-d\TH:i:s\Z',
            'Y-m-d\TH:i:sP',
            'Y-m-d H:i:s',
            'd/m/Y',
            'Y-m-d'
        ];

        foreach ($formats as $format) {
            try {
                return Carbon::createFromFormat($format, $dateValue);
            } catch (\Exception $e) {
                // continue trying other formats
            }
        }

        try {
            return Carbon::parse($dateValue);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * API endpoint to get operator daily reports data
     */
    public function getOperatorDailyReports()
    {
        try {
            $operatorReports = $this->fetchOperatorDailyReports();
            
            return response()->json([
                'success' => true,
                'data' => $operatorReports,
                'count' => count($operatorReports)
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error in getOperatorDailyReports API: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error fetching operator daily reports: ' . $e->getMessage(),
                'data' => []
            ], 500);
        }
    }

    public function testInspectionData()
    {
        try {
            $response = Http::get('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json');
            
            if ($response->successful()) {
                $inspectionData = $response->json() ?? [];
                return response()->json([
                    'success' => true,
                    'status' => $response->status(),
                    'total_records' => is_array($inspectionData) ? count($inspectionData) : 'Not array',
                    'sample_data' => is_array($inspectionData) && !empty($inspectionData) ? array_slice($inspectionData, 0, 3, true) : [],
                    'message' => 'Inspection data retrieved successfully'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'status' => $response->status(),
                    'message' => 'Failed to fetch inspection data'
                ], 500);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Error fetching inspection data'
            ], 500);
        }
    }

    public function getIncidentReports(Request $request)
    {
        try {
            Log::info('Fetching incident reports from Firebase');

            // Get pagination parameters
            $perPage = $request->get('per_page', 10);
            $page = $request->get('page', 1);

            // Try multiple Firebase paths to find the data
            $firebaseUrls = [
                'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/incidentReports.json',
                'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/.json'
            ];

            Log::info('Trying to fetch from Firebase URLs', ['urls' => $firebaseUrls]);

            $response = null;
            $incidentData = null;
            $usedUrl = null;

            foreach ($firebaseUrls as $url) {
                try {
                    Log::info('Attempting to fetch from URL', ['url' => $url]);
                    $response = Http::get($url);

                    if ($response->successful()) {
                        $data = $response->json();
                        Log::info('Response from URL', [
                            'url' => $url,
                            'status' => $response->status(),
                            'data_type' => gettype($data),
                            'is_array' => is_array($data),
                            'has_incidentReports' => is_array($data) && isset($data['incidentReports']),
                            'count' => is_array($data) ? count($data) : 'N/A'
                        ]);

                        // Check if this URL has the data we're looking for
                        if (is_array($data)) {
                            if (isset($data['incidentReports']) && is_array($data['incidentReports'])) {
                                $incidentData = $data['incidentReports'];
                                $usedUrl = $url . ' (incidentReports key found)';
                                break;
                            } elseif (count($data) > 0 && is_array(reset($data))) {
                                $incidentData = $data;
                                $usedUrl = $url . ' (direct array found)';
                                break;
                            }
                        }
                    } else {
                        Log::warning('Failed to fetch from URL', [
                            'url' => $url,
                            'status' => $response->status(),
                            'body' => $response->body()
                        ]);
                    }
                } catch (\Exception $e) {
                    Log::error('Error fetching from URL', [
                        'url' => $url,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            if (!$incidentData) {
                Log::error('No incident data found from any URL', [
                    'attempted_urls' => $firebaseUrls
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'No incident data found in Firebase database',
                    'debug' => [
                        'attempted_urls' => $firebaseUrls,
                        'responses' => $firebaseUrls
                    ]
                ], 404);
            }

            Log::info('Successfully found incident data', [
                'used_url' => $usedUrl,
                'data_count' => count($incidentData)
            ]);

            Log::info('Firebase response received', [
                'status' => $response->status(),
                'body_length' => strlen($response->body())
            ]);

            if ($response->successful()) {
                $incidentData = $response->json() ?? [];

                Log::info('Raw Firebase data received', [
                    'data_type' => gettype($incidentData),
                    'is_array' => is_array($incidentData),
                    'count' => is_array($incidentData) ? count($incidentData) : 'N/A',
                    'sample_keys' => is_array($incidentData) ? array_keys(array_slice($incidentData, 0, 3, true)) : 'N/A'
                ]);

                // Transform the data to extract incident information
                $incidents = [];

                if (is_array($incidentData)) {
                    foreach ($incidentData as $incidentId => $incident) {
                        if (is_array($incident)) {
                            // Extract the incident information based on the structure
                            $incidentInfo = [
                                'incidentNumber' => $incidentId, // Use the Firebase key as incident number
                                'incidentDate' => $incident['incidentDate'] ?? null,
                                'vehicleNumber' => $incident['vehicleNumber'] ?? null,
                                'vehicleName' => $incident['vehicleName'] ?? null,
                                'country' => $incident['country'] ?? null,
                                'location' => $incident['incidentArea'] ?? $incident['location'] ?? null,
                                'reportedBy' => $incident['reportedBy'] ?? null,
                                'pdfDownloadUrl' => $incident['pdfDownloadUrl'] ?? null,
                                'status' => $incident['status'] ?? 'completed',
                                'description' => $incident['incidentCause'] ?? null,
                                'severity' => $incident['severity'] ?? null,
                                'actions' => $incident['immediateCorrectiveAction'] ?? null,
                                'createdAt' => $incident['createdAt'] ?? null,
                                'updatedAt' => $incident['updatedAt'] ?? null,
                                'incidentCategory' => $incident['incidentCategory'] ?? null,
                                'incidentArea' => $incident['incidentArea'] ?? null,
                                'involvesEquipment' => $incident['involvesEquipment'] ?? false,
                                'additionalComment' => $incident['additionalComment'] ?? null,
                                'equipmentDamage' => $incident['equipmentDamage'] ?? null,
                                'incidentCause' => $incident['incidentCause'] ?? null,
                                'immediateCorrectiveAction' => $incident['immediateCorrectiveAction'] ?? null,
                                'selectedIncidents' => $incident['incidentTypes'] ?? [], // Map incidentTypes to selectedIncidents
                                'project' => $incident['project'] ?? null,
                                'otherIncidentNote' => $incident['otherIncidentNote'] ?? null,
                                'gpsAddress' => $incident['gpsAddress'] ?? null,
                                'coordinates' => $incident['coordinates'] ?? null,
                                'adminUid' => $incident['adminUid'] ?? null
                            ];

                            $incidents[] = $incidentInfo;
                        }
                    }
                }

                Log::info('Processed incident reports', [
                    'total_incidents' => count($incidents),
                    'sample_incident' => !empty($incidents) ? $incidents[0] : null
                ]);

                // Apply pagination
                $totalIncidents = count($incidents);
                $totalPages = ceil($totalIncidents / $perPage);
                $offset = ($page - 1) * $perPage;
                $paginatedIncidents = array_slice($incidents, $offset, $perPage);

                return response()->json([
                    'success' => true,
                    'message' => 'Incident reports retrieved successfully',
                    'incidents' => $paginatedIncidents,
                    'pagination' => [
                        'current_page' => (int) $page,
                        'per_page' => (int) $perPage,
                        'total' => $totalIncidents,
                        'total_pages' => $totalPages,
                        'from' => $offset + 1,
                        'to' => min($offset + $perPage, $totalIncidents),
                        'has_more_pages' => $page < $totalPages
                    ]
                ]);
            } else {
                Log::error('Failed to fetch incident reports from Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch incident reports',
                    'status' => $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error fetching incident reports: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error fetching incident reports: ' . $e->getMessage()
            ], 500);
        }
    }

    public function debugFirebaseData()
    {
        try {
            Log::info('Debug Firebase data request');

            // Try multiple Firebase paths to find the data
            $firebaseUrls = [
                'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/incidentReports.json',
                'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/.json'
            ];

            $results = [];

            foreach ($firebaseUrls as $url) {
                try {
                    $response = Http::get($url);

                    if ($response->successful()) {
                        $data = $response->json() ?? [];

                        $results[] = [
                            'url' => $url,
                            'status' => $response->status(),
                            'data_type' => gettype($data),
                            'is_array' => is_array($data),
                            'has_incidentReports' => is_array($data) && isset($data['incidentReports']),
                            'count' => is_array($data) ? count($data) : 'N/A',
                            'sample_keys' => is_array($data) ? array_keys(array_slice($data, 0, 3, true)) : 'N/A',
                            'sample_data' => is_array($data) ? json_encode(array_slice($data, 0, 1, true)) : 'N/A'
                        ];
                    } else {
                        $results[] = [
                            'url' => $url,
                            'status' => $response->status(),
                            'error' => 'Failed to fetch',
                            'body' => substr($response->body(), 0, 500)
                        ];
                    }
                } catch (\Exception $e) {
                    $results[] = [
                        'url' => $url,
                        'error' => $e->getMessage()
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Firebase debug completed',
                'results' => $results,
                'total_attempts' => count($results)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error in debug: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ]);
        }
    }

    public function storeIncident(Request $request)
    {
        try {
            Log::info('Incident creation received', [
                'request_data' => $request->all(),
                'files' => $request->allFiles()
            ]);

            // Validate the request data
            $request->validate([
                'incidentNumber' => 'required|string|max:255',
                'incidentDate' => 'required|string',
                'location' => 'required|string|max:255',
                'incidentType' => 'required|string|max:100',
                'incidentArea' => 'nullable|string|max:255',
                'severity' => 'required|string|max:50',
                'vehicleNumber' => 'required|string|max:255',
                'vehicleName' => 'required|string|max:255',
                'vehicleCategory' => 'nullable|string|max:255',
                'vinNumber' => 'nullable|string|max:255',
                'operatorName' => 'nullable|string|max:255',
                'weatherCondition' => 'required|string|max:100',
                'vehicleActivity' => 'required|string|max:100',
                'incidentCause' => 'nullable|string',
                'equipmentDamage' => 'nullable|string',
                'additionalComment' => 'nullable|string',
                'reportedBy' => 'nullable|string|max:255',
            ]);

            // Handle image uploads dynamically
            $imageUrls = [];
            $allFiles = $request->allFiles();
            
            foreach ($allFiles as $key => $file) {
                if (strpos($key, 'incidentImage') === 0 && $file) {
                    // Validate file type
                    if (!$file->isValid() || !$file->getMimeType() || !str_starts_with($file->getMimeType(), 'image/')) {
                        continue;
                    }
                    
                    // Validate file size (2MB limit)
                    if ($file->getSize() > 2 * 1024 * 1024) {
                        continue;
                    }
                    
                    try {
                        $imageName = time() . '_' . $key . '.' . $file->getClientOriginalExtension();
                        $imagePath = $file->storeAs('incident_images', $imageName, 'public');
                        $imageUrls[$key] = asset('storage/' . $imagePath);
                    } catch (\Exception $e) {
                        Log::warning("Failed to upload image {$key}: " . $e->getMessage());
                        continue;
                    }
                }
            }

            // Prepare the incident data
            $incidentData = [
                'incidentNumber' => $request->input('incidentNumber'),
                'incidentDate' => $request->input('incidentDate'),
                'location' => $request->input('location'),
                'incidentType' => $request->input('incidentType'),
                'incidentArea' => $request->input('incidentArea'),
                'severity' => $request->input('severity'),
                'vehicleNumber' => $request->input('vehicleNumber'),
                'vehicleName' => $request->input('vehicleName'),
                'vehicleCategory' => $request->input('vehicleCategory'),
                'vinNumber' => $request->input('vinNumber'),
                'operatorName' => $request->input('operatorName'),
                'weatherCondition' => $request->input('weatherCondition'),
                'vehicleActivity' => $request->input('vehicleActivity'),
                'incidentCause' => $request->input('incidentCause'),
                'equipmentDamage' => $request->input('equipmentDamage'),
                'additionalComment' => $request->input('additionalComment'),
                'reportedBy' => $request->input('reportedBy'),
                'status' => 'Open',
                'createdAt' => now()->toISOString(),
                'updatedAt' => now()->toISOString()
            ];

            if (!empty($imageUrls)) {
                $incidentData['images'] = $imageUrls;
            }

            Log::info('Processed incident data', [
                'incidentData' => $incidentData,
                'imageUrls' => $imageUrls
            ]);

            // Store the data in Firebase with the incident number as key
            $response = Http::put('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//incidentReports/' . $request->input('incidentNumber') . '.json', $incidentData);
            
            Log::info('Firebase response received', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            
            if ($response->successful()) {
                Log::info('Incident created successfully', ['incidentNumber' => $request->input('incidentNumber')]);
                
                // Generate PDF for the incident
                try {
                    $pdfUrl = $this->generateIncidentPdf($incidentData);
                    $incidentData['pdfDownloadUrl'] = $pdfUrl;
                    
                    // Update Firebase with PDF URL
                    Http::put('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//incidentReports/' . $request->input('incidentNumber') . '.json', $incidentData);
                    
                    Log::info('PDF generated and stored successfully', ['pdfUrl' => $pdfUrl]);
                } catch (\Exception $e) {
                    Log::error('Failed to generate PDF: ' . $e->getMessage());
                    // Continue without PDF - incident is still created
                }
                
                return response()->json([
                    'success' => true,
                    'message' => 'Incident report created successfully',
                    'incidentNumber' => $request->input('incidentNumber'),
                    'data' => $incidentData
                ]);
            } else {
                Log::error('Failed to create incident in Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create incident report',
                    'status' => $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error creating incident: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error creating incident report: ' . $e->getMessage()
            ], 500);
        }
    }

    public function createIncident()
    {
        return view('inspection.create_incident_report');
    }

    public function viewInspectionPdf($id)
    {
        try {
            Log::info('Fetching inspection PDF', ['id' => $id]);
            
            // Fetch inspection data from Firebase using the specified path
            $response = Http::get("https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/{$id}/step3/pdfDownloadUrl.json");
            
            if ($response->successful()) {
                $pdfUrl = $response->json();
                
                if ($pdfUrl) {
                    Log::info('Inspection PDF found', ['id' => $id, 'pdfUrl' => $pdfUrl]);
                    
                    // Redirect to the PDF URL in a new tab
                    return redirect()->away($pdfUrl);
                } else {
                    Log::warning('No PDF found for inspection', ['id' => $id]);
                    return response()->json([
                        'success' => false,
                        'message' => 'No PDF report available for this inspection'
                    ], 404);
                }
            } else {
                Log::error('Failed to fetch inspection from Firebase', ['id' => $id, 'status' => $response->status()]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch inspection data'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error viewing inspection PDF: ' . $e->getMessage(), ['id' => $id]);
            return response()->json([
                'success' => false,
                'message' => 'Error viewing inspection PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    public function viewDvirPdf($id)
    {
        try {
            Log::info('Fetching DVIR PDF', ['id' => $id]);
            
            // Fetch DVIR data from Firebase using the specified path
            $response = Http::get("https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//GlobalDvirReports/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/inspectionReports/{$id}/step3/pdfDownloadUrl.json");
            
            if ($response->successful()) {
                $pdfUrl = $response->json();
                
                if ($pdfUrl) {
                    Log::info('DVIR PDF found', ['id' => $id, 'pdfUrl' => $pdfUrl]);
                    
                    // Redirect to the PDF URL in a new tab
                    return redirect()->away($pdfUrl);
                } else {
                    Log::warning('No PDF found for DVIR', ['id' => $id]);
                    return response()->json([
                        'success' => false,
                        'message' => 'No PDF report available for this DVIR'
                    ], 404);
                }
            } else {
                Log::error('Failed to fetch DVIR from Firebase', ['id' => $id, 'status' => $response->status()]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch DVIR data'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error viewing DVIR PDF: ' . $e->getMessage(), ['id' => $id]);
            return response()->json([
                'success' => false,
                'message' => 'Error viewing DVIR PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    public function viewIncidentPdf($id)
    {
        try {
            Log::info('Fetching incident PDF', ['id' => $id]);
            
            // First, try to fetch the incident data from Firebase
            $response = Http::get("https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//incidentReports/{$id}.json");
            
            if ($response->successful()) {
                $incidentData = $response->json();
                
                if ($incidentData) {
                    // Check if PDF already exists
                    if (isset($incidentData['pdfDownloadUrl']) && $incidentData['pdfDownloadUrl']) {
                        Log::info('Incident PDF found', ['id' => $id, 'pdfUrl' => $incidentData['pdfDownloadUrl']]);
                        return redirect()->away($incidentData['pdfDownloadUrl']);
                    } else {
                        // Generate PDF on-demand
                        Log::info('Generating PDF on-demand for incident', ['id' => $id]);
                        $pdfUrl = $this->generateIncidentPdf($incidentData);
                        
                        // Update Firebase with the new PDF URL
                        $incidentData['pdfDownloadUrl'] = $pdfUrl;
                        Http::put("https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//incidentReports/{$id}.json", $incidentData);
                        
                        return redirect()->away($pdfUrl);
                    }
                } else {
                    Log::warning('No incident data found', ['id' => $id]);
                    return response()->json([
                        'success' => false,
                        'message' => 'No incident data found'
                    ], 404);
                }
            } else {
                Log::error('Failed to fetch incident from Firebase', ['id' => $id, 'status' => $response->status()]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch incident data'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error viewing incident PDF: ' . $e->getMessage(), ['id' => $id]);
            return response()->json([
                'success' => false,
                'message' => 'Error viewing incident PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    private function generateIncidentPdf($incidentData)
    {
        try {
            // Prepare data for PDF template
            $step1 = [
                'vehicleName' => $incidentData['vehicleName'] ?? '-',
                'vehicleId' => $incidentData['vehicleNumber'] ?? '-',
                'incidentDate' => $incidentData['incidentDate'] ?? '-',
                'incidentDate2' => $incidentData['operatorName'] ?? '-',
                'project' => $incidentData['location'] ?? '-',
                'incidentArea' => $incidentData['incidentArea'] ?? '-',
                'coordinates' => '-',
                'gpsAddress' => '-'
            ];
            
            $step2 = [
                'incidentCause' => $incidentData['incidentCause'] ?? '-',
                'equipmentDamage' => $incidentData['equipmentDamage'] ?? '-',
                'additionalComment' => $incidentData['additionalComment'] ?? '-'
            ];
            
            // Generate incident types HTML
            $incidentTypesHtml = $this->generateIncidentTypesHtml($incidentData['incidentType']);
            
            // Generate severity HTML
            $severityHtml = $this->generateSeverityHtml($incidentData['severity']);
            
            // Generate images HTML
            $imagesHtml = $this->generateImagesHtml($incidentData['images'] ?? []);
            
            // Get user name (you can modify this based on your authentication)
            $userName = $incidentData['reportedBy'] ?? 'System User';
            
            // Generate the HTML content
            $html = $this->generateIncidentPdfHtml($step1, $step2, $incidentTypesHtml, $severityHtml, $imagesHtml, $userName, $incidentData);
            
            // Generate PDF
            $pdf = Pdf::loadHTML($html);
            $pdf->setPaper('A4', 'portrait');
            
            // Generate unique filename
            $filename = 'incident_' . $incidentData['incidentNumber'] . '_' . time() . '.pdf';
            
            // Store PDF in storage
            $pdfPath = 'incident_pdfs/' . $filename;
            Storage::disk('public')->put($pdfPath, $pdf->output());
            
            // Return the public URL
            return asset('storage/' . $pdfPath);
            
        } catch (\Exception $e) {
            Log::error('Error generating incident PDF: ' . $e->getMessage());
            throw $e;
        }
    }
    
    private function generateIncidentPdfHtml($step1, $step2, $incidentTypesHtml, $severityHtml, $imagesHtml, $userName, $incidentData)
    {
        // Helper functions
        $dashIfNA = function($value) {
            return $value && $value !== '-' ? $value : '-';
        };
        
        $formatDate = function($dateString) {
            if (!$dateString || $dateString === '-') return '-';
            try {
                $date = new \DateTime($dateString);
                return $date->format('M d, Y');
            } catch (\Exception $e) {
                return $dateString;
            }
        };
        
        $formatTime = function($dateString) {
            if (!$dateString || $dateString === '-') return '-';
            try {
                $date = new \DateTime($dateString);
                return $date->format('H:i A');
            } catch (\Exception $e) {
                return $dateString;
            }
        };
        
        return '
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Incident Report</title>
          <style>
            body { font-family: Arial, sans-serif; margin: 0; padding: 0; }
            .container { max-width: 900px; margin: 0 auto; padding: 12px; }
            .header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px; }
            .header h1 { font-size: 22px; font-weight: bold; margin: 0; }
            .logo { font-size: 22px; font-weight: bold; color: #fff;  border-radius: 50%; width: 150px; height: 48px; display: flex; align-items: center; justify-content: center; }
            .section-title { font-weight: bold; font-size: 14px; margin: 10px 0 4px; background-color: #e6e6e6; padding: 4px; border: 1px solid #ccc; }
            table { width: 100%; border-collapse: collapse; margin-bottom: 8px; }
            th, td { border: 1px solid #ccc; padding: 6px; text-align: left; font-size: 13px; }
            th { background-color: #f2f2f2; font-weight: bold; }
            .checkbox-table td { border: none; padding: 2px 6px; font-size: 13px; }
            .checkbox-table input[type="checkbox"] { margin-right: 4px; }
            .text-section { border: 1px solid #ccc; padding: 8px; min-height: 32px; font-size: 13px; margin-bottom: 8px; }
            .footer { text-align: center; margin-top: 10px; font-size: 11px; color: #000; border-top: 1px solid #000; padding-top: 6px; }
            .reported-by { text-align: right; font-size: 13px; margin-top: 8px; }
            .reported-by span { border-bottom: 1px solid #000; padding-bottom: 2px; }
            .logo img { width: 100%; height: 100%; object-fit: contain; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1>Incident Report</h1>
              <div class="logo">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo"/></div>
            </div>
            <table>
              <tr>
                <th colspan="2">Vehicle Details</th>
                <th colspan="2">Incident Details</th>
              </tr>
              <tr>
                <td>Vehicle Category</td>
                <td>' . $dashIfNA($step1['vehicleName']) . '</td>
                <td>Incident Number</td>
                <td>' . $dashIfNA($incidentData['incidentNumber'] ?? '-') . '</td>
              </tr>
              <tr>
                <td>Vehicle Name</td>
                <td>' . $dashIfNA($step1['vehicleName']) . '</td>
                <td>Incident Date</td>
                <td>' . $dashIfNA($formatDate($step1['incidentDate'])) . '</td>
              </tr>
              <tr>
                <td>Vehicle #</td>
                <td>' . $dashIfNA($step1['vehicleId']) . '</td>
                <td>Incident Time</td>
                <td>' . $dashIfNA($formatTime($step1['incidentDate'])) . '</td>
              </tr>
              <tr>
                <td>VIN #</td>
                <td>' . $dashIfNA($incidentData['vinNumber'] ?? '-') . '</td>
                <td>Location</td>
                <td>' . $dashIfNA($step1['project']) . '</td>
              </tr>
              <tr>
                <td>Operator</td>
                <td>' . $dashIfNA($step1['incidentDate2']) . '</td>
                <td>Incident Area</td>
                <td>' . $dashIfNA($step1['incidentArea']) . '</td>
              </tr>
              <tr>
                <td>GPS Coordinates</td>
                <td colspan="3">' . $dashIfNA($step1['coordinates']) . '</td>
              </tr>
              <tr>
                <td>GPS Address</td>
                <td colspan="3">' . $dashIfNA($step1['gpsAddress']) . '</td>
              </tr>
            </table>
            <div class="section-title">Types of Incident</div>
            ' . $incidentTypesHtml . '
            <div class="section-title">Incident Severity Rating</div>
            ' . $severityHtml . '
            <div class="section-title">How incident occurred and what were the causes?</div>
            <div class="text-section">' . $step2['incidentCause'] . '</div>
            <div class="section-title">What broke on the equipment, list all damages?</div>
            <div class="text-section">' . $step2['equipmentDamage'] . '</div>
            <div class="section-title">Additional Comment</div>
            <div class="text-section">' . $step2['additionalComment'] . '</div>
            <div class="section-title">Incident Images</div>
            ' . $imagesHtml . '
            <div class="reported-by">
              <span>' . $userName . '</span><br>
              Reported By
            </div>
            <div class="footer">
              Prepared By: Titan Drilling            </div>
          </div>
        </body>
        </html>';
    }
    
    private function generateIncidentTypesHtml($incidentType)
    {
        $allTypes = [
            'Brake Failure', 'Chemical Spill', 'Collision', 'Drowned in Water',
            'Electrical Failure', 'Engine Failure', 'Equipment Failure', 'Explosion/fire',
            'Fall Attachments', 'Fall from elevation', 'Leakage in Vehicle',
            'Loss of Stability', 'Oil Spill', 'Person hit by Vehicle', 'Steering Failure',
            'Stuck', 'Swamping', 'Tire Burst', 'Turned Over', 'Other'
        ];
        
        $html = '<table class="checkbox-table">';
        foreach ($allTypes as $type) {
            $checked = ($type === $incidentType) ? 'checked' : '';
            $html .= '<tr><td><input type="checkbox" ' . $checked . ' disabled> ' . $type . '</td></tr>';
        }
        $html .= '</table>';
        
        return $html;
    }
    
    private function generateSeverityHtml($severity)
    {
        $allSeverities = ['High', 'Medium', 'Low'];
        
        $html = '<table class="checkbox-table">';
        foreach ($allSeverities as $sev) {
            $checked = ($sev === $severity) ? 'checked' : '';
            $html .= '<tr><td><input type="checkbox" ' . $checked . ' disabled> ' . $sev . '</td></tr>';
        }
        $html .= '</table>';
        
        return $html;
    }
    
    private function generateImagesHtml($images)
    {
        if (empty($images)) {
            return '<div class="text-section">No images uploaded</div>';
        }
        
        $html = '<div style="display: flex; flex-wrap: wrap; gap: 10px; margin-bottom: 10px;">';
        foreach ($images as $imageUrl) {
            $html .= '<div style="width: 150px; height: 150px; border: 1px solid #ccc; display: flex; align-items: center; justify-content: center; background-color: #f8f9fa;">
                        <img src="' . $imageUrl . '" style="max-width: 100%; max-height: 100%; object-fit: contain;" alt="Incident Image">
                      </div>';
        }
        $html .= '</div>';
        
        return $html;
    }
    
    public function downloadIncidentPdf($id)
    {
        try {
            Log::info('Downloading incident PDF', ['id' => $id]);
            
            // Fetch incident data from Firebase
            $response = Http::get("https://titan-drilling-1f8e9-default-rtdb.firebaseio.com//incidentReports/{$id}.json");
            
            if ($response->successful()) {
                $incidentData = $response->json();
                
                if ($incidentData) {
                    // Generate PDF
                    $pdf = $this->generateIncidentPdfContent($incidentData);
                    
                    // Return PDF as download
                    return response($pdf)
                        ->header('Content-Type', 'application/pdf')
                        ->header('Content-Disposition', 'attachment; filename="incident_' . $id . '.pdf"');
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'No incident data found'
                    ], 404);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch incident data'
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error downloading incident PDF: ' . $e->getMessage(), ['id' => $id]);
            return response()->json([
                'success' => false,
                'message' => 'Error downloading incident PDF: ' . $e->getMessage()
            ], 500);
        }
    }
    
    private function generateIncidentPdfContent($incidentData)
    {
        try {
            // Prepare data for PDF template
            $step1 = [
                'vehicleName' => $incidentData['vehicleName'] ?? '-',
                'vehicleId' => $incidentData['vehicleNumber'] ?? '-',
                'incidentDate' => $incidentData['incidentDate'] ?? '-',
                'incidentDate2' => $incidentData['operatorName'] ?? '-',
                'project' => $incidentData['location'] ?? '-',
                'incidentArea' => $incidentData['incidentArea'] ?? '-',
                'coordinates' => '-',
                'gpsAddress' => '-'
            ];
            
            $step2 = [
                'incidentCause' => $incidentData['incidentCause'] ?? '-',
                'equipmentDamage' => $incidentData['equipmentDamage'] ?? '-',
                'additionalComment' => $incidentData['additionalComment'] ?? '-'
            ];
            
            // Generate incident types HTML
            $incidentTypesHtml = $this->generateIncidentTypesHtml($incidentData['incidentType']);
            
            // Generate severity HTML
            $severityHtml = $this->generateSeverityHtml($incidentData['severity']);
            
            // Generate images HTML
            $imagesHtml = $this->generateImagesHtml($incidentData['images'] ?? []);
            
            // Get user name
            $userName = $incidentData['reportedBy'] ?? 'System User';
            
            // Generate the HTML content
            $html = $this->generateIncidentPdfHtml($step1, $step2, $incidentTypesHtml, $severityHtml, $imagesHtml, $userName, $incidentData);
            
            // Generate PDF
            $pdf = Pdf::loadHTML($html);
            $pdf->setPaper('A4', 'portrait');
            
            return $pdf->output();
            
        } catch (\Exception $e) {
            Log::error('Error generating incident PDF content: ' . $e->getMessage());
            throw $e;
        }
    }

    private function fetchInspectionData()
    {
        try {
            Log::info('Starting to fetch inspection data from Firebase...');
            
            $inspectionResponse = Http::get($this->firebaseUrl . 'GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json');
            
            Log::info('Firebase response status: ' . $inspectionResponse->status());
            Log::info('Firebase response headers: ' . json_encode($inspectionResponse->headers()));
            
            if (!$inspectionResponse->successful()) {
                Log::error('Firebase request failed with status: ' . $inspectionResponse->status());
                Log::error('Firebase response body: ' . $inspectionResponse->body());
                return [];
            }
            
            $inspectionData = $inspectionResponse->json() ?? [];
            
            Log::info('Raw inspection data type: ' . gettype($inspectionData));
            Log::info('Raw inspection data count: ' . (is_array($inspectionData) ? count($inspectionData) : 'Not array'));
            
            if (!is_array($inspectionData)) {
                Log::error('Inspection data is not an array: ' . json_encode($inspectionData));
                return [];
            }

            if (empty($inspectionData)) {
                Log::warning('Inspection data is empty');
                return [];
            }

            $processedData = [];
            $processedCount = 0;
            $errorCount = 0;
            
            foreach ($inspectionData as $reportId => $report) {
                try {
                    if (is_array($report)) {
                        $processedReport = [
                            'id' => $reportId,
                            'requestNumber' => $this->extractRequestNumber($report),
                            'createdAt' => $this->extractCreatedAt($report),
                            'equipmentName' => $this->extractEquipmentName($report),
                            'faultyItems' => $this->extractFaultyItems($report),
                            'equipmentStatus' => $report['equipmentStatus'] ?? 'N/A',
                            'location' => $this->extractLocation($report),
                            'inspector' => $this->extractInspector($report),
                            'fault' => $this->extractFaults($report),
                            'correctiveAction' => $this->extractCorrectiveAction($report),
                            'linkedWo' => $this->extractLinkedWo($report)
                        ];
                        
                        $processedData[] = $processedReport;
                        $processedCount++;
                        
                        // Log first few reports for debugging
                        if ($processedCount <= 3) {
                            Log::info('Processed report ' . $processedCount . ': ' . json_encode($processedReport));
                        }
                    } else {
                        Log::warning('Report is not an array: ' . json_encode($report));
                        $errorCount++;
                    }
                } catch (\Exception $e) {
                    Log::error('Error processing report ' . $reportId . ': ' . $e->getMessage());
                    $errorCount++;
                }
            }

            // Sort by creation date (newest first)
            usort($processedData, function($a, $b) {
                try {
                    $dateA = strtotime($a['createdAt']);
                    $dateB = strtotime($b['createdAt']);
                    return $dateB - $dateA;
                } catch (\Exception $e) {
                    return 0;
                }
            });

            Log::info('Successfully processed ' . $processedCount . ' inspection reports');
            if ($errorCount > 0) {
                Log::warning('Encountered ' . $errorCount . ' errors while processing reports');
            }
            
            return $processedData;

        } catch (\Exception $e) {
            Log::error('Error fetching inspection data: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            return [];
        }
    }

    private function extractRequestNumber($report)
    {
        // Use the specific path: step1/requestNumber
        if (isset($report['step1']['requestNumber'])) {
            return $report['step1']['requestNumber'];
        }
        // Fallback to root level
        if (isset($report['requestNumber'])) {
            return $report['requestNumber'];
        }
        return 'N/A';
    }

    private function extractCreatedAt($report)
    {
        // Use the specific path: step1/createdAt
        if (isset($report['step1']['createdAt'])) {
            return $report['step1']['createdAt'];
        }
        // Fallback to root level
        if (isset($report['createdAt'])) {
            return $report['createdAt'];
        }
        return 'N/A';
    }

    private function extractEquipmentName($report)
    {
        try {
            // Use the specific path: step1/selectedEquipment
            if (isset($report['step1']['selectedEquipment'])) {
                $selectedEquipment = $report['step1']['selectedEquipment'];
                
                // Log the structure for debugging
                Log::info('SelectedEquipment structure: ' . json_encode($selectedEquipment));
                
                // Handle different data types
                if (is_string($selectedEquipment)) {
                    return $selectedEquipment;
                } elseif (is_array($selectedEquipment)) {
                    // Try to extract name from the array
                    if (isset($selectedEquipment['name'])) {
                        return $selectedEquipment['name'];
                    } elseif (isset($selectedEquipment['equipmentName'])) {
                        return $selectedEquipment['equipmentName'];
                    } elseif (isset($selectedEquipment['id'])) {
                        return $selectedEquipment['id'];
                    } else {
                        // If it's an array but no clear name field, log and return first value
                        $firstValue = reset($selectedEquipment);
                        if (is_string($firstValue)) {
                            return $firstValue;
                        }
                        Log::warning('SelectedEquipment array has no string values: ' . json_encode($selectedEquipment));
                        return 'Equipment Array';
                    }
                } elseif (is_object($selectedEquipment)) {
                    // Handle object case
                    if (method_exists($selectedEquipment, '__toString')) {
                        return (string) $selectedEquipment;
                    } elseif (isset($selectedEquipment->name)) {
                        return $selectedEquipment->name;
                    } else {
                        Log::warning('SelectedEquipment object has no string conversion: ' . json_encode($selectedEquipment));
                        return 'Equipment Object';
                    }
                } else {
                    Log::warning('SelectedEquipment has unexpected type: ' . gettype($selectedEquipment));
                    return 'Unknown Equipment';
                }
            }
            
            // Fallback to root level
            if (isset($report['equipmentName'])) {
                return $report['equipmentName'];
            }
            
            return 'N/A';
            
        } catch (\Exception $e) {
            Log::error('Error extracting equipment name: ' . $e->getMessage());
            return 'Error Loading Equipment';
        }
    }

    private function extractFaultyItems($report)
    {
        // Use the specific path: step3/faultyItems (for DVIR reports)
        if (isset($report['step3']['faultyItems'])) {
            return $report['step3']['faultyItems'];
        }
        // Fallback to step2/faultyItems (for other inspection types)
        if (isset($report['step2']['faultyItems'])) {
            return $report['step2']['faultyItems'];
        }
        // Fallback to root level
        if (isset($report['faultyItems'])) {
            return $report['faultyItems'];
        }
        return 0;
    }

    private function extractLocation($report)
    {
        // Use the specific path: step1/country
        if (isset($report['step1']['country'])) {
            return $report['step1']['country'];
        }
        // Fallback to other location fields
        if (isset($report['step1']['location'])) {
            return $report['step1']['location'];
        }
        if (isset($report['step1']['site'])) {
            return $report['step1']['site'];
        }
        if (isset($report['location'])) {
            return $report['location'];
        }
        return 'N/A';
    }

    private function extractInspector($report)
    {
        // Try to extract inspector from various possible fields
        if (isset($report['step1']['inspector'])) {
            return $report['step1']['inspector'];
        }
        if (isset($report['step1']['inspectorName'])) {
            return $report['step1']['inspectorName'];
        }
        if (isset($report['inspector'])) {
            return $report['inspector'];
        }
        if (isset($report['createdBy'])) {
            return $report['createdBy'];
        }
        return 'N/A';
    }

    private function extractFaults($report)
    {
        $faults = [];
        
        // Check step2 for faultyItems (primary source)
        if (isset($report['step2']['faultyItems']) && is_array($report['step2']['faultyItems'])) {
            foreach ($report['step2']['faultyItems'] as $fault) {
                if (isset($fault['description'])) {
                    $faults[] = $fault['description'];
                } elseif (is_string($fault)) {
                    $faults[] = $fault;
                }
            }
        }
        
        // Check step1 for faults (fallback)
        if (isset($report['step1']['faults']) && is_array($report['step1']['faults'])) {
            foreach ($report['step1']['faults'] as $fault) {
                if (isset($fault['description'])) {
                    $faults[] = $fault['description'];
                } elseif (is_string($fault)) {
                    $faults[] = $fault;
                }
            }
        }
        
        // Check step3 for faults (fallback)
        if (isset($report['step3']['faults']) && is_array($report['step3']['faults'])) {
            foreach ($report['step3']['faults'] as $fault) {
                if (isset($fault['description'])) {
                    $faults[] = $fault['description'];
                } elseif (is_string($fault)) {
                    $faults[] = $fault;
                }
            }
        }
        
        return !empty($faults) ? implode(', ', array_slice($faults, 0, 3)) : 'No faults found';
    }

    private function extractCorrectiveAction($report)
    {
        $actions = [];
        
        // Check step1 for corrective actions
        if (isset($report['step1']['correctiveActions']) && is_array($report['step1']['correctiveActions'])) {
            foreach ($report['step1']['correctiveActions'] as $action) {
                if (isset($action['action'])) {
                    $actions[] = $action['action'];
                }
            }
        }
        
        // Check step2 for corrective actions
        if (isset($report['step2']['correctiveActions']) && is_array($report['step2']['correctiveActions'])) {
            foreach ($report['step2']['correctiveActions'] as $action) {
                if (isset($action['action'])) {
                    $actions[] = $action['action'];
                }
            }
        }
        
        // Check step3 for corrective actions
        if (isset($report['step3']['correctiveActions']) && is_array($report['step3']['correctiveActions'])) {
            foreach ($report['step3']['correctiveActions'] as $action) {
                if (isset($action['action'])) {
                    $actions[] = $action['action'];
                }
            }
        }
        
        return !empty($actions) ? implode(', ', array_slice($actions, 0, 3)) : 'No actions specified';
    }

    private function extractLinkedWo($report)
    {
        // Try to extract linked work order from various possible fields
        if (isset($report['linkedWorkOrder'])) {
            return $report['linkedWorkOrder'];
        }
        if (isset($report['workOrderId'])) {
            return $report['workOrderId'];
        }
        if (isset($report['step3']['workOrder'])) {
            return $report['step3']['workOrder'];
        }
        return 'N/A';
    }

    public function storeInspection(Request $request)
    {
        try {
            Log::info('Inspection creation request received', [
                'request_data' => $request->all(),
                'headers' => $request->headers->all()
            ]);

            // Validate the request data
            $request->validate([
                'reportNumber' => 'required|string|max:255',
                'location' => 'required|string|max:255',
                'inspectorName' => 'required|string|max:255',
                'inspectionChecklist' => 'required|string|max:255',
                'vehicleNumber' => 'required|string|max:255',
                'vehicleName' => 'required|string|max:255',
                'vinNumber' => 'required|string|max:255',
                'meterReading' => 'required|string|max:255',
                'meterUnit' => 'required|string|max:50',
                'overallCondition' => 'nullable|string|max:100',
                'vehicleSafeToUse' => 'nullable|string|max:10',
                'maintenanceRequired' => 'nullable|string|max:10',
                'vehicleStatus' => 'nullable|string|max:100',
                'additionalNote' => 'nullable|string',
                'inspectionItems' => 'nullable|array'
            ]);

            // Get the next available index
            $nextIndex = $this->getNextInspectionIndex();
            
            // Prepare the inspection data structure
            $inspectionData = [
                'step1' => [
                    'requestNumber' => $request->input('reportNumber'),
                    'location' => $request->input('location'),
                    'inspectorName' => $request->input('inspectorName'),
                    'inspectionChecklist' => $request->input('inspectionChecklist'),
                    'vehicleNumber' => $request->input('vehicleNumber'),
                    'vehicleName' => $request->input('vehicleName'),
                    'vinNumber' => $request->input('vinNumber'),
                    'meterReading' => $request->input('meterReading'),
                    'meterUnit' => $request->input('meterUnit'),
                    'createdAt' => now()->toISOString(),
                    'updatedAt' => now()->toISOString()
                ],
                'step2' => [
                    'inspectionItems' => $request->input('inspectionItems', []),
                    'completedAt' => now()->toISOString()
                ],
                'step3' => [
                    'overallCondition' => $request->input('overallCondition'),
                    'vehicleSafeToUse' => $request->input('vehicleSafeToUse'),
                    'maintenanceRequired' => $request->input('maintenanceRequired'),
                    'vehicleStatus' => $request->input('vehicleStatus'),
                    'additionalNote' => $request->input('additionalNote'),
                    'completedAt' => now()->toISOString()
                ],
                'equipmentName' => $request->input('vehicleName'),
                'equipmentStatus' => $request->input('vehicleStatus', 'Active'),
                'status' => 'Completed',
                'createdAt' => now()->toISOString(),
                'updatedAt' => now()->toISOString()
            ];

            Log::info('Processed inspection data', [
                'nextIndex' => $nextIndex,
                'inspectionData' => $inspectionData
            ]);

            // Store the data in Firebase with auto-incrementing index
            $response = Http::put($this->firebaseUrl . 'GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/' . $nextIndex . '.json', $inspectionData);
            
            Log::info('Firebase response received', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            
            if ($response->successful()) {
                Log::info('Inspection created successfully', ['index' => $nextIndex]);
                return response()->json([
                    'success' => true,
                    'message' => 'Inspection report created successfully',
                    'index' => $nextIndex,
                    'data' => $inspectionData
                ]);
            } else {
                Log::error('Failed to create inspection in Firebase', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create inspection report',
                    'status' => $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error creating inspection: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error creating inspection report: ' . $e->getMessage()
            ], 500);
        }
    }

    private function getNextInspectionIndex()
    {
        try {
            // Get current inspection data to find the next available index
            $response = Http::get($this->firebaseUrl . 'GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json');
            
            if ($response->successful()) {
                $inspectionData = $response->json() ?? [];
                
                if (is_array($inspectionData) && !empty($inspectionData)) {
                    // Find the highest numeric index
                    $maxIndex = 0;
                    foreach (array_keys($inspectionData) as $key) {
                        if (is_numeric($key) && (int)$key > $maxIndex) {
                            $maxIndex = (int)$key;
                        }
                    }
                    return $maxIndex + 1;
                }
            }
            
            // If no existing data or error, start with index 1
            return 1;
        } catch (\Exception $e) {
            Log::error('Error getting next inspection index: ' . $e->getMessage());
            // Fallback to timestamp-based index
            return (int)(time() / 1000);
        }
    }

    /**
     * Fetch equipment data from Firebase
     */
    public function fetchEquipmentFromFirebase(): JsonResponse
    {
        try {
            // Firebase configuration - you may want to move these to .env file
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
            $equipmentPath = '/EquipmentsDataJSON/GlobalEquipmentJsonData.json';
            
            // Make request to Firebase
            $response = Http::get($firebaseUrl . $equipmentPath);
            
            if ($response->successful()) {
                $equipmentData = $response->json();
                
                // Process the data to ensure it's in the expected format
                $processedEquipment = [];
                
                if (is_array($equipmentData)) {
                    foreach ($equipmentData as $index => $item) {
                        if (is_array($item) || is_object($item)) {
                            $item = (array) $item;
                            
                            // Add firebaseKey for reference
                            $item['firebaseKey'] = $index;
                            
                            // Ensure required fields exist
                            $processedEquipment[] = [
                                'id' => $item['id'] ?? $item['firebaseKey'] ?? $index,
                                'firebaseKey' => $index,
                                'equipmentName' => $item['equipmentName'] ?? $item['vehicleNumber'] ?? 'Unknown Equipment',
                                'vehicleNumber' => $item['vehicleNumber'] ?? $item['id'] ?? $index,
                                'model' => $item['model'] ?? 'N/A',
                                'mainCategory' => $item['mainCategory'] ?? $item['equipmentCategory'] ?? 'N/A',
                                'equipmentCategory' => $item['equipmentCategory'] ?? $item['mainCategory'] ?? 'N/A',
                                'status' => $item['status'] ?? 'Available',
                                'meterReading' => $item['meterReading'] ?? 0,
                                'meterUnit' => $item['meterUnit'] ?? 'KM',
                                'createdAt' => $item['createdAt'] ?? now()->toISOString(),
                                'updatedAt' => $item['updatedAt'] ?? now()->toISOString(),
                            ];
                        }
                    }
                }
                
                return response()->json([
                    'success' => true,
                    'equipment' => $processedEquipment,
                    'count' => count($processedEquipment)
                ]);
            } else {
                Log::error('Firebase API error', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch equipment data from Firebase. Status: ' . $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error fetching equipment from Firebase', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error fetching equipment data: ' . $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Fetch inspection details by Firebase key for task creation
     */
    public function fetchInspectionDetailsByKey(Request $request): JsonResponse
    {
        try {
            $firebaseKey = $request->input('firebase_key');

            if (!$firebaseKey) {
                return response()->json([
                    'success' => false,
                    'message' => 'Firebase key is required'
                ], 400);
            }

            // Firebase configuration
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
            $inspectionPath = "GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1/{$firebaseKey}.json";

            // Make request to Firebase
            $response = Http::get($firebaseUrl . $inspectionPath);

            if ($response->successful()) {
                $inspectionData = $response->json();

                if (!$inspectionData) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Inspection not found'
                    ], 404);
                }

                // Extract faulty items from step2 with enhanced detection
                $faultyItems = [];
                $allItems = [];

                Log::info("=== EXTRACTING FAULTY ITEMS ===");
                Log::info("Extracting faulty items for firebase key: ${firebaseKey}");
                Log::info("Inspection data structure", [
                    'has_step2' => isset($inspectionData['step2']),
                    'step2_keys' => isset($inspectionData['step2']) ? array_keys($inspectionData['step2']) : 'No step2',
                    'full_structure' => array_keys($inspectionData)
                ]);

                // Enhanced faulty items detection with multiple structures
                if (isset($inspectionData['step2'])) {
                    Log::info("Step2 structure keys", ['keys' => array_keys($inspectionData['step2'])]);

                    // Check new structure: selections + sections (from inspection2.blade.php)
                    if (isset($inspectionData['step2']['selections']) && isset($inspectionData['step2']['sections'])) {
                        Log::info("Processing selections + sections structure", [
                            'selections_count' => count($inspectionData['step2']['selections']),
                            'sections_count' => count($inspectionData['step2']['sections'])
                        ]);

                        $selections = $inspectionData['step2']['selections'];
                        $sections = $inspectionData['step2']['sections'];
                        $notes = $inspectionData['step2']['notes'] ?? [];
                        $numericInputs = $inspectionData['step2']['numericInputs'] ?? [];

                        foreach ($sections as $section) {
                            $sectionName = $section['name'] ?? $section['text'] ?? 'General';
                            if (isset($section['subItems']) && is_array($section['subItems'])) {
                                foreach ($section['subItems'] as $item) {
                                    $itemKey = $item['key'] ?? $item['id'] ?? '';
                                    $itemName = $item['name'] ?? $item['label'] ?? $item['title'] ?? $item['itemName'] ?? $itemKey;
                                    $status = $selections[$itemKey] ?? '';
                                    $note = $notes[$itemKey] ?? '';
                                    $numericInput = $numericInputs[$itemKey] ?? '';

                                    Log::info("Processing item from sections", [
                                        'item_key' => $itemKey,
                                        'item_name' => $itemName,
                                        'status' => $status,
                                        'section' => $sectionName,
                                        'has_numeric_input' => !empty($numericInput)
                                    ]);

                                    // Check if this item is faulty with STRICT detection
                                    $isFaulty = false;
                                    $faultReason = '';

                                    if (!empty($status)) {
                                        $statusLower = strtolower(trim($status));

                                        // ONLY consider these as faulty (strict detection)
                                        if ($statusLower === 'reject' || $statusLower === 'no' ||
                                            $statusLower === 'fail' || $statusLower === 'failed' ||
                                            $statusLower === 'faulty' || $statusLower === 'defective' ||
                                            $statusLower === 'not ok' || $statusLower === 'needs repair' ||
                                            $statusLower === 'requires attention' || $statusLower === 'bad' ||
                                            $statusLower === 'poor' || $statusLower === 'replace') {
                                            $isFaulty = true;
                                            $faultReason = $status;
                                        }
                                        // N/A is NOT faulty unless there's a specific issue
                                        elseif ($statusLower === 'n/a' || $statusLower === 'na') {
                                            // Only consider N/A as faulty if there's a note indicating an issue
                                            if (!empty($note) && stripos($note, 'fault') !== false) {
                                                $isFaulty = true;
                                                $faultReason = 'N/A with fault note';
                                            }
                                        }
                                        // OK is explicitly NOT faulty
                                        elseif ($statusLower === 'ok' || $statusLower === 'pass' || $statusLower === 'passed') {
                                            $isFaulty = false;
                                        }
                                    } else {
                                        // No status provided - only consider faulty if there's a specific issue noted
                                        if (!empty($note) && (stripos($note, 'fault') !== false || stripos($note, 'issue') !== false)) {
                                            $isFaulty = true;
                                            $faultReason = 'No status but issue noted';
                                        }
                                    }

                                    if ($isFaulty) {
                                        $faultyItems[] = [
                                            'id' => $itemKey,
                                            'name' => $itemName,
                                            'status' => $status,
                                            'note' => $note,
                                            'section' => $sectionName,
                                            'faultReason' => $faultReason,
                                            'numericInput' => $numericInput,
                                            'category' => $item['category'] ?? $sectionName,
                                            'description' => $item['description'] ?? '',
                                            'instruction' => $item['instruction'] ?? '',
                                            'sortOrder' => $item['sortOrder'] ?? 0,
                                            'isInspectionRequired' => $item['isInspectionRequired'] ?? false,
                                            'inputType' => $item['inputType'] ?? 'Ok/Reject/N/A'
                                        ];

                                        Log::info("✅ FOUND FAULTY ITEM from sections", [
                                            'id' => $itemKey,
                                            'name' => $itemName,
                                            'status' => $status,
                                            'section' => $sectionName,
                                            'faultReason' => $faultReason
                                        ]);
                                    }
                                }
                            }
                        }
                    }
                    // Check old structure: inspectionItems
                    elseif (isset($inspectionData['step2']['inspectionItems'])) {
                        $items = $inspectionData['step2']['inspectionItems'];
                        Log::info("Processing inspectionItems structure", ['count' => count($items)]);

                        foreach ($items as $itemId => $item) {
                            Log::info("Processing inspection item", [
                                'item_id' => $itemId,
                                'item_keys' => array_keys($item),
                                'has_status' => isset($item['status']),
                                'status' => $item['status'] ?? 'No status'
                            ]);

                            $status = $item['status'] ?? '';
                            $label = $item['label'] ?? $item['itemName'] ?? $item['name'] ?? $itemId;

                            // Enhanced status checking with multiple variations
                            $isFaulty = false;
                            $statusLower = strtolower($status);

                            Log::info("Checking status", [
                                'status' => $status,
                                'statusLower' => $statusLower,
                                'isFaulty' => $isFaulty
                            ]);

                            if ($statusLower === 'no' || $statusLower === 'faulty' ||
                                $statusLower === 'fail' || $statusLower === 'failed' ||
                                $statusLower === 'not ok' || $statusLower === 'defective' ||
                                $statusLower === 'needs repair' || $statusLower === 'requires attention' ||
                                $statusLower === 'n/a' || $statusLower === 'na' ||
                                $statusLower === 'bad' || $statusLower === 'poor' ||
                                $statusLower === 'replace' || $statusLower === 'repair') {
                                $isFaulty = true;
                            }

                            Log::info("Fault check result", [
                                'status' => $status,
                                'isFaulty' => $isFaulty
                            ]);

                            if ($isFaulty) {
                                $faultyItems[] = [
                                    'id' => $itemId,
                                    'name' => $label,
                                    'status' => $status,
                                    'note' => $item['note'] ?? $item['description'] ?? $item['comment'] ?? '',
                                    'section' => $item['section'] ?? $item['category'] ?? 'General'
                                ];

                                Log::info("✅ FOUND FAULTY ITEM", [
                                    'item_id' => $itemId,
                                    'name' => $label,
                                    'status' => $status,
                                    'section' => $item['section'] ?? 'General'
                                ]);
                            } else {
                                Log::info("❌ Not faulty", [
                                    'item_id' => $itemId,
                                    'status' => $status
                                ]);
                            }
                        }
                    }

                    // Check items structure (alternative)
                    if (isset($inspectionData['step2']['items'])) {
                        $items = $inspectionData['step2']['items'];
                        Log::info("Processing items structure", ['count' => count($items)]);

                        foreach ($items as $itemId => $item) {
                            Log::info("Processing items item", [
                                'item_id' => $itemId,
                                'item_keys' => array_keys($item),
                                'has_status' => isset($item['status']),
                                'status' => $item['status'] ?? 'No status'
                            ]);

                            $status = $item['status'] ?? '';
                            $label = $item['label'] ?? $item['itemName'] ?? $item['name'] ?? $itemId;

                            $isFaulty = false;
                            $statusLower = strtolower($status);

                            if ($statusLower === 'no' || $statusLower === 'faulty' ||
                                $statusLower === 'fail' || $statusLower === 'failed' ||
                                $statusLower === 'not ok' || $statusLower === 'defective' ||
                                $statusLower === 'needs repair' || $statusLower === 'requires attention' ||
                                $statusLower === 'n/a' || $statusLower === 'na' ||
                                $statusLower === 'bad' || $statusLower === 'poor' ||
                                $statusLower === 'replace' || $statusLower === 'repair') {
                                $isFaulty = true;
                            }

                            if ($isFaulty) {
                                $faultyItems[] = [
                                    'id' => $itemId,
                                    'name' => $label,
                                    'status' => $status,
                                    'note' => $item['note'] ?? $item['description'] ?? $item['comment'] ?? '',
                                    'section' => $item['section'] ?? $item['category'] ?? 'General'
                                ];

                                Log::info("✅ FOUND FAULTY ITEM in alternative structure", [
                                    'item_id' => $itemId,
                                    'name' => $label,
                                    'status' => $status,
                                    'section' => $item['section'] ?? 'General'
                                ]);
                            } else {
                                Log::info("❌ Not faulty in alternative structure", [
                                    'item_id' => $itemId,
                                    'status' => $status
                                ]);
                            }
                        }
                    }
                }

                // Fallback: Check root level structures
                if (empty($faultyItems)) {
                    Log::info("No faulty items found in step2, checking root level structures");

                    // Check root level items
                    if (isset($inspectionData['items']) && is_array($inspectionData['items'])) {
                        $items = $inspectionData['items'];
                        Log::info("Processing root items structure", ['count' => count($items)]);

                        foreach ($items as $itemId => $item) {
                            if (isset($item['status'])) {
                                $status = $item['status'];
                                $label = $item['label'] ?? $item['itemName'] ?? $item['name'] ?? $itemId;

                                $isFaulty = false;
                                $statusLower = strtolower($status);

                                if ($statusLower === 'no' || $statusLower === 'faulty' ||
                                    $statusLower === 'fail' || $statusLower === 'failed' ||
                                    $statusLower === 'not ok' || $statusLower === 'defective' ||
                                    $statusLower === 'needs repair' || $statusLower === 'requires attention' ||
                                    $statusLower === 'n/a' || $statusLower === 'na' ||
                                    $statusLower === 'bad' || $statusLower === 'poor') {
                                    $isFaulty = true;
                                }

                                if ($isFaulty) {
                                    $faultyItems[] = [
                                        'id' => $itemId,
                                        'name' => $label,
                                        'status' => $status,
                                        'note' => $item['note'] ?? $item['description'] ?? $item['comment'] ?? '',
                                        'section' => $item['section'] ?? $item['category'] ?? 'General'
                                    ];

                                    Log::info("Found faulty item in root structure", [
                                        'item_id' => $itemId,
                                        'name' => $label,
                                        'status' => $status
                                    ]);
                                }
                            }
                        }
                    }
                }

                // Additional fallback: Check for any structure that might contain faulty items
                if (empty($faultyItems)) {
                    Log::info("=== ADDITIONAL FALLBACK CHECKS ===");

                    // Search through the entire inspection data for any faulty indicators
                    $searchInStructure = function($data, $path = '') use (&$faultyItems, &$searchInStructure) {
                        if (is_array($data)) {
                            foreach ($data as $key => $value) {
                                $currentPath = $path ? $path . '.' . $key : $key;

                                // Check if this key contains status information
                                if (is_array($value) && isset($value['status'])) {
                                    $status = $value['status'] ?? '';
                                    $label = $value['label'] ?? $value['itemName'] ?? $value['name'] ?? $key;
                                    $statusLower = strtolower($status);

                                    // Check for faulty status
                                    if ($statusLower === 'no' || $statusLower === 'faulty' ||
                                        $statusLower === 'fail' || $statusLower === 'failed' ||
                                        $statusLower === 'not ok' || $statusLower === 'defective' ||
                                        $statusLower === 'needs repair' || $statusLower === 'requires attention' ||
                                        $statusLower === 'n/a' || $statusLower === 'na' ||
                                        $statusLower === 'bad' || $statusLower === 'poor' ||
                                        $statusLower === 'replace' || $statusLower === 'repair') {

                                        $faultyItems[] = [
                                            'id' => $key,
                                            'name' => $label,
                                            'status' => $status,
                                            'note' => $value['note'] ?? $value['description'] ?? $value['comment'] ?? '',
                                            'section' => $value['section'] ?? $value['category'] ?? 'General'
                                        ];

                                        Log::info("✅ FOUND FAULTY ITEM in fallback search", [
                                            'path' => $currentPath,
                                            'item_id' => $key,
                                            'name' => $label,
                                            'status' => $status
                                        ]);
                                    }
                                } else {
                                    // Recursively search
                                    $searchInStructure($value, $currentPath);
                                }
                            }
                        }
                    };

                    $searchInStructure($inspectionData);

                    Log::info("Fallback search completed", [
                        'found_additional_faulty' => count($faultyItems)
                    ]);
                }

                Log::info("=== FINAL FAULTY ITEMS RESULT ===");
                Log::info("Final faulty items result", [
                    'found_count' => count($faultyItems),
                    'faulty_items' => $faultyItems
                ]);

                return response()->json([
                    'success' => true,
                    'inspection' => [
                        'firebaseKey' => $firebaseKey,
                        'requestNumber' => $inspectionData['requestNumber'] ?? 'Unknown',
                        'equipment' => [
                            'name' => $inspectionData['step1']['equipment']['equipmentName'] ?? 'N/A',
                            'number' => $inspectionData['step1']['equipment']['vehicleNumber'] ?? 'N/A',
                            'model' => $inspectionData['step1']['equipment']['model'] ?? 'N/A'
                        ],
                        'createdAt' => $inspectionData['createdAt'] ?? now()->toISOString(),
                        'updatedAt' => $inspectionData['updatedAt'] ?? now()->toISOString(),
                        'country' => $inspectionData['step1']['country'] ?? 'N/A',
                        'project' => $inspectionData['step1']['project'] ?? 'N/A',
                        'inspector' => $inspectionData['step1']['inspector'] ?? 'N/A'
                    ],
                    'faultyItems' => $faultyItems,
                    'faultyItemsCount' => count($faultyItems)
                ]);
            } else {
                Log::error('Firebase API error for inspection details', [
                    'firebaseKey' => $firebaseKey,
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch inspection details from Firebase. Status: ' . $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error fetching inspection details from Firebase', [
                'firebaseKey' => $request->input('firebase_key'),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error fetching inspection details: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch all inspection reports from Firebase for linking to job cards
     */
    public function fetchInspectionReportsForLinking(): JsonResponse
    {
        try {
            Log::info('Fetching inspection reports for linking from Firebase');

            // Firebase configuration
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
            $inspectionPath = 'GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json';

            // Make request to Firebase
            $response = Http::get($firebaseUrl . $inspectionPath);

            if ($response->successful()) {
                $inspectionData = $response->json();

                Log::info('Raw Firebase inspection data structure', [
                    'data_type' => gettype($inspectionData),
                    'is_array' => is_array($inspectionData),
                    'sample_keys' => is_array($inspectionData) ? array_keys(array_slice($inspectionData, 0, 1, true)) : 'Not array',
                    'sample_data' => is_array($inspectionData) ? array_slice($inspectionData, 0, 1, true) : 'Not array'
                ]);

                // Process the data to extract inspection reports
                $processedInspections = [];

                if (is_array($inspectionData)) {
                    foreach ($inspectionData as $index => $item) {
                        if (is_array($item) || is_object($item)) {
                            $item = (array) $item;

                            // Extract request number using the same method as other functions
                            $requestNumber = $this->extractRequestNumber($item);

                            // Count actual faulty items (more accurate than just checking faultyItemsCount)
                            $actualFaultyItemsCount = 0;
                            $faultyItemsDetails = [];

                            // Check if this inspection has any faulty items by examining multiple possible structures
                            if (isset($item['step2'])) {
                                Log::info("Processing inspection ${index} for faulty items", [
                                    'step2_keys' => array_keys($item['step2']),
                                    'has_selections' => isset($item['step2']['selections']),
                                    'has_sections' => isset($item['step2']['sections']),
                                    'has_inspectionItems' => isset($item['step2']['inspectionItems']),
                                    'has_items' => isset($item['step2']['items']),
                                    'has_faultyItemsCount' => isset($item['step2']['faultyItemsCount']),
                                    'requestNumber' => $requestNumber
                                ]);

                                // Check new structure: selections + sections (from inspection2.blade.php)
                                if (isset($item['step2']['selections']) && isset($item['step2']['sections'])) {
                                    Log::info("Processing selections + sections structure for inspection ${index}");

                                    $selections = $item['step2']['selections'];
                                    $sections = $item['step2']['sections'];
                                    $notes = $item['step2']['notes'] ?? [];
                                    $numericInputs = $item['step2']['numericInputs'] ?? [];

                                    foreach ($sections as $section) {
                                        $sectionName = $section['name'] ?? $section['text'] ?? 'General';
                                        if (isset($section['subItems']) && is_array($section['subItems'])) {
                                            foreach ($section['subItems'] as $subItem) {
                                                $itemKey = $subItem['key'] ?? $subItem['id'] ?? '';
                                                $itemName = $subItem['name'] ?? $subItem['label'] ?? $subItem['title'] ?? $subItem['itemName'] ?? $itemKey;
                                                $status = $selections[$itemKey] ?? '';
                                                $note = $notes[$itemKey] ?? '';
                                                $numericInput = $numericInputs[$itemKey] ?? '';

                                                // Check for faulty status with STRICT detection
                                                $isFaulty = false;
                                                if (!empty($status)) {
                                                    $statusLower = strtolower(trim($status));
                                                    if ($statusLower === 'reject' || $statusLower === 'no' ||
                                                        $statusLower === 'fail' || $statusLower === 'failed' ||
                                                        $statusLower === 'faulty' || $statusLower === 'defective' ||
                                                        $statusLower === 'not ok' || $statusLower === 'needs repair' ||
                                                        $statusLower === 'requires attention' || $statusLower === 'bad' ||
                                                        $statusLower === 'poor' || $statusLower === 'replace') {
                                                        $isFaulty = true;
                                                    }
                                                    // N/A is NOT faulty unless there's a specific issue
                                                    elseif ($statusLower === 'n/a' || $statusLower === 'na') {
                                                        // Only consider N/A as faulty if there's a note indicating an issue
                                                        if (!empty($note) && stripos($note, 'fault') !== false) {
                                                            $isFaulty = true;
                                                        }
                                                    }
                                                    // OK is explicitly NOT faulty
                                                    elseif ($statusLower === 'ok' || $statusLower === 'pass' || $statusLower === 'passed') {
                                                        $isFaulty = false;
                                                    }
                                                } else {
                                                    // No status provided - only consider faulty if there's a specific issue noted
                                                    if (!empty($note) && (stripos($note, 'fault') !== false || stripos($note, 'issue') !== false)) {
                                                        $isFaulty = true;
                                                    }
                                                }

                                                if ($isFaulty) {
                                                    $actualFaultyItemsCount++;
                                                    $faultyItemsDetails[] = [
                                                        'id' => $itemKey,
                                                        'name' => $itemName,
                                                        'status' => $status,
                                                        'note' => $note,
                                                        'section' => $sectionName,
                                                        'numericInput' => $numericInput,
                                                        'category' => $subItem['category'] ?? $sectionName
                                                    ];

                                                    Log::info("Found faulty item in sections structure", [
                                                        'item_id' => $itemKey,
                                                        'status' => $status,
                                                        'name' => $itemName,
                                                        'section' => $sectionName
                                                    ]);
                                                }
                                            }
                                        }
                                    }
                                }
                                // Debug the structure of inspectionItems if it exists
                                elseif (isset($item['step2']['inspectionItems'])) {
                                    $sampleItem = reset($item['step2']['inspectionItems']);
                                    Log::info("Sample inspectionItems structure", [
                                        'sample_keys' => array_keys($sampleItem),
                                        'sample_status' => $sampleItem['status'] ?? 'No status field',
                                        'total_items' => count($item['step2']['inspectionItems'])
                                    ]);

                                // Primary structure: step2.inspectionItems
                                if (isset($item['step2']['inspectionItems'])) {
                                    $items = $item['step2']['inspectionItems'];
                                    Log::info("Counting faulty items in inspectionItems for inspection ${index}", [
                                        'total_items' => count($items)
                                    ]);

                                    foreach ($items as $itemId => $inspectionItem) {
                                        $status = $inspectionItem['status'] ?? '';
                                        $label = $inspectionItem['label'] ?? $inspectionItem['itemName'] ?? $inspectionItem['name'] ?? $itemId;
                                        $note = $inspectionItem['note'] ?? $inspectionItem['description'] ?? $inspectionItem['comment'] ?? '';

                                        // Check for faulty status with STRICT detection
                                        $isFaulty = false;
                                        $faultReason = '';

                                        if (!empty($status)) {
                                            $statusLower = strtolower(trim($status));
                                            if ($statusLower === 'reject' || $statusLower === 'no' ||
                                                $statusLower === 'fail' || $statusLower === 'failed' ||
                                                $statusLower === 'faulty' || $statusLower === 'defective' ||
                                                $statusLower === 'not ok' || $statusLower === 'needs repair' ||
                                                $statusLower === 'requires attention' || $statusLower === 'bad' ||
                                                $statusLower === 'poor' || $statusLower === 'replace') {
                                                $isFaulty = true;
                                                $faultReason = $status;
                                            }
                                            // N/A is NOT faulty unless there's a specific issue
                                            elseif ($statusLower === 'n/a' || $statusLower === 'na') {
                                                // Only consider N/A as faulty if there's a note indicating an issue
                                                if (!empty($note) && stripos($note, 'fault') !== false) {
                                                    $isFaulty = true;
                                                    $faultReason = 'N/A with fault note';
                                                }
                                            }
                                            // OK is explicitly NOT faulty
                                            elseif ($statusLower === 'ok' || $statusLower === 'pass' || $statusLower === 'passed') {
                                                $isFaulty = false;
                                            }
                                        } else {
                                            // No status provided - only consider faulty if there's a specific issue noted
                                            if (!empty($note) && (stripos($note, 'fault') !== false || stripos($note, 'issue') !== false)) {
                                                $isFaulty = true;
                                                $faultReason = 'No status but issue noted';
                                            }
                                        }

                                        if ($isFaulty) {
                                            $actualFaultyItemsCount++;
                                            $faultyItemsDetails[] = [
                                                'id' => $itemId,
                                                'name' => $label,
                                                'status' => $status,
                                                'note' => $note,
                                                'section' => $inspectionItem['section'] ?? $inspectionItem['category'] ?? 'General',
                                                'faultReason' => $faultReason
                                            ];

                                            Log::info("Found faulty item in inspectionItems", [
                                                'item_id' => $itemId,
                                                'status' => $status,
                                                'label' => $label,
                                                'section' => $inspectionItem['section'] ?? 'General',
                                                'faultReason' => $faultReason
                                            ]);
                                        }
                                    }
                                }
                                // Alternative structure: step2.items
                                elseif (isset($item['step2']['items'])) {
                                    $items = $item['step2']['items'];
                                    Log::info("Counting faulty items in items (alternative structure) for inspection ${index}", [
                                        'total_items' => count($items)
                                    ]);

                                    foreach ($items as $itemId => $inspectionItem) {
                                        $status = $inspectionItem['status'] ?? '';
                                        $label = $inspectionItem['label'] ?? $inspectionItem['itemName'] ?? $inspectionItem['name'] ?? $itemId;
                                        $note = $inspectionItem['note'] ?? $inspectionItem['description'] ?? $inspectionItem['comment'] ?? '';

                                        // Check for faulty status with STRICT detection
                                        $isFaulty = false;
                                        $faultReason = '';

                                        if (!empty($status)) {
                                            $statusLower = strtolower(trim($status));
                                            if ($statusLower === 'reject' || $statusLower === 'no' ||
                                                $statusLower === 'fail' || $statusLower === 'failed' ||
                                                $statusLower === 'faulty' || $statusLower === 'defective' ||
                                                $statusLower === 'not ok' || $statusLower === 'needs repair' ||
                                                $statusLower === 'requires attention' || $statusLower === 'bad' ||
                                                $statusLower === 'poor' || $statusLower === 'replace') {
                                                $isFaulty = true;
                                                $faultReason = $status;
                                            }
                                            // N/A is NOT faulty unless there's a specific issue
                                            elseif ($statusLower === 'n/a' || $statusLower === 'na') {
                                                // Only consider N/A as faulty if there's a note indicating an issue
                                                if (!empty($note) && stripos($note, 'fault') !== false) {
                                                    $isFaulty = true;
                                                    $faultReason = 'N/A with fault note';
                                                }
                                            }
                                            // OK is explicitly NOT faulty
                                            elseif ($statusLower === 'ok' || $statusLower === 'pass' || $statusLower === 'passed') {
                                                $isFaulty = false;
                                            }
                                        } else {
                                            // No status provided - only consider faulty if there's a specific issue noted
                                            if (!empty($note) && (stripos($note, 'fault') !== false || stripos($note, 'issue') !== false)) {
                                                $isFaulty = true;
                                                $faultReason = 'No status but issue noted';
                                            }
                                        }

                                        if ($isFaulty) {
                                            $actualFaultyItemsCount++;
                                            $faultyItemsDetails[] = [
                                                'id' => $itemId,
                                                'name' => $label,
                                                'status' => $status,
                                                'note' => $note,
                                                'section' => $inspectionItem['section'] ?? $inspectionItem['category'] ?? 'General',
                                                'faultReason' => $faultReason
                                            ];

                                            Log::info("Found faulty item in items (alt structure)", [
                                                'item_id' => $itemId,
                                                'status' => $status,
                                                'label' => $label,
                                                'faultReason' => $faultReason
                                            ]);
                                        }
                                    }
                                }
                                // Check root level items as well
                                elseif (isset($item['items'])) {
                                    $items = $item['items'];
                                    Log::info("Counting faulty items in root items for inspection ${index}", [
                                        'total_items' => count($items)
                                    ]);

                                    foreach ($items as $itemId => $inspectionItem) {
                                        $status = $inspectionItem['status'] ?? '';
                                        $label = $inspectionItem['label'] ?? $inspectionItem['itemName'] ?? $inspectionItem['name'] ?? $itemId;
                                        $note = $inspectionItem['note'] ?? $inspectionItem['description'] ?? $inspectionItem['comment'] ?? '';

                                        $isFaulty = false;
                                        $faultReason = '';

                                        if (!empty($status)) {
                                            $statusLower = strtolower(trim($status));
                                            if ($statusLower === 'reject' || $statusLower === 'no' ||
                                                $statusLower === 'fail' || $statusLower === 'failed' ||
                                                $statusLower === 'faulty' || $statusLower === 'defective' ||
                                                $statusLower === 'not ok' || $statusLower === 'needs repair' ||
                                                $statusLower === 'requires attention' || $statusLower === 'bad' ||
                                                $statusLower === 'poor' || $statusLower === 'replace') {
                                                $isFaulty = true;
                                                $faultReason = $status;
                                            }
                                            // N/A is NOT faulty unless there's a specific issue
                                            elseif ($statusLower === 'n/a' || $statusLower === 'na') {
                                                // Only consider N/A as faulty if there's a note indicating an issue
                                                if (!empty($note) && stripos($note, 'fault') !== false) {
                                                    $isFaulty = true;
                                                    $faultReason = 'N/A with fault note';
                                                }
                                            }
                                            // OK is explicitly NOT faulty
                                            elseif ($statusLower === 'ok' || $statusLower === 'pass' || $statusLower === 'passed') {
                                                $isFaulty = false;
                                            }
                                        } else {
                                            // No status provided - only consider faulty if there's a specific issue noted
                                            if (!empty($note) && (stripos($note, 'fault') !== false || stripos($note, 'issue') !== false)) {
                                                $isFaulty = true;
                                                $faultReason = 'No status but issue noted';
                                            }
                                        }

                                        if ($isFaulty) {
                                            $actualFaultyItemsCount++;
                                            $faultyItemsDetails[] = [
                                                'id' => $itemId,
                                                'name' => $label,
                                                'status' => $status,
                                                'note' => $note,
                                                'section' => $inspectionItem['section'] ?? $inspectionItem['category'] ?? 'General',
                                                'faultReason' => $faultReason
                                            ];
                                        }
                                    }
                            } else {
                                Log::info("No inspectionItems, items, or root items found for inspection ${index}", [
                                    'step2_keys' => array_keys($item['step2']),
                                    'root_keys' => isset($item) ? array_keys($item) : 'No item data'
                                ]);

                                // Check if there's any other structure that might contain faulty items
                                $allKeys = isset($item) ? array_keys($item) : [];
                                Log::info("Looking for alternative faulty item structures in inspection ${index}", [
                                    'all_keys' => $allKeys,
                                    'requestNumber' => $requestNumber
                                ]);

                                // Check for any key that might contain inspection data
                                foreach ($allKeys as $key) {
                                    if (strpos(strtolower($key), 'fault') !== false ||
                                        strpos(strtolower($key), 'inspect') !== false ||
                                        strpos(strtolower($key), 'item') !== false) {
                                        Log::info("Found potentially relevant key: ${key}", [
                                            'key_type' => gettype($item[$key]),
                                            'key_content' => is_array($item[$key]) ? array_keys($item[$key]) : $item[$key]
                                        ]);
                                    }
                                }
                            }
                            } else {
                                Log::info("No step2 found for inspection ${index}", [
                                    'item_keys' => isset($item) ? array_keys($item) : 'No item data',
                                    'requestNumber' => $requestNumber
                                ]);

                                // Log the entire structure for debugging
                                if (isset($item)) {
                                    Log::info("Full inspection structure for debugging", [
                                        'item_keys' => array_keys($item),
                                        'item_type' => gettype($item),
                                        'sample_data' => json_encode(array_slice((array)$item, 0, 3, true))
                                    ]);
                                }
                            }

                            // Enhanced fallback: Check for any field that might indicate faulty items
                            if ($actualFaultyItemsCount === 0) {
                                $potentialFaultyFields = [
                                    'hasFaultyItems', 'faultyItems', 'faultyItemsCount',
                                    'hasIssues', 'issues', 'issuesCount', 'problems',
                                    'problemCount', 'defects', 'defectCount'
                                ];

                                foreach ($potentialFaultyFields as $field) {
                                    if (isset($item[$field])) {
                                        Log::info("Found potential faulty items field: ${field}", [
                                            'field_value' => $item[$field],
                                            'field_type' => gettype($item[$field])
                                        ]);

                                        if (is_numeric($item[$field]) && $item[$field] > 0) {
                                            $actualFaultyItemsCount = intval($item[$field]);
                                            Log::info("Using ${field} as faulty items count: ${actualFaultyItemsCount}");
                                            break;
                                        } elseif (is_bool($item[$field]) && $item[$field] === true) {
                                            $actualFaultyItemsCount = 1; // Assume at least 1 faulty item
                                            Log::info("Using ${field} boolean flag, assuming 1 faulty item");
                                            break;
                                        } elseif (is_array($item[$field]) && count($item[$field]) > 0) {
                                            $actualFaultyItemsCount = count($item[$field]);
                                            Log::info("Using ${field} array count: ${actualFaultyItemsCount}");
                                            break;
                                        }
                                    }
                                }

                                // Also check nested structures
                                if (isset($item['step2']) && is_array($item['step2'])) {
                                    foreach ($item['step2'] as $step2Key => $step2Value) {
                                        if (strpos(strtolower($step2Key), 'fault') !== false && is_numeric($step2Value) && $step2Value > 0) {
                                            $actualFaultyItemsCount = intval($step2Value);
                                            Log::info("Found faulty count in step2.${step2Key}: ${actualFaultyItemsCount}");
                                            break;
                                        }
                                    }
                                }
                            }

                            Log::info("Final faulty items count for inspection ${index}", [
                                'actualFaultyItemsCount' => $actualFaultyItemsCount,
                                'faultyItemsDetails' => $faultyItemsDetails,
                                'hasFaultyItems' => $actualFaultyItemsCount > 0,
                                'requestNumber' => $requestNumber,
                                'total_fields_checked' => isset($item) ? count(array_keys($item)) : 0
                            ]);

                            // Extract relevant information for linking
                            $processedInspections[] = [
                                'id' => $requestNumber ?? $index,
                                'firebaseKey' => $index,
                                'requestNumber' => $requestNumber ?? 'Unknown',
                                'inspectionNumber' => $requestNumber ?? 'Unknown',
                                'equipment' => [
                                    'name' => $item['step1']['equipment']['equipmentName'] ?? $item['step1']['selectedEquipment']['equipmentName'] ?? $item['step1']['equipment']['name'] ?? 'N/A',
                                    'number' => $item['step1']['equipment']['vehicleNumber'] ?? $item['step1']['selectedEquipment']['vehicleNumber'] ?? $item['step1']['equipment']['number'] ?? 'N/A',
                                    'model' => $item['step1']['equipment']['model'] ?? $item['step1']['selectedEquipment']['model'] ?? $item['step1']['equipment']['model'] ?? 'N/A',
                                    'vinNumber' => $item['step1']['equipment']['vinNumber'] ?? $item['step1']['selectedEquipment']['vinNumber'] ?? 'N/A'
                                ],
                                'createdAt' => $item['createdAt'] ?? $item['step1']['createdAt'] ?? now()->toISOString(),
                                'updatedAt' => $item['updatedAt'] ?? $item['step1']['updatedAt'] ?? now()->toISOString(),
                                'country' => $item['step1']['country'] ?? 'N/A',
                                'project' => $item['step1']['project'] ?? 'N/A',
                                'inspector' => $item['step1']['inspector'] ?? $item['step1']['inspectorName'] ?? 'N/A',
                                'hasFaultyItems' => $actualFaultyItemsCount > 0,
                                'faultyItemsCount' => $actualFaultyItemsCount,
                                // Add additional fields for auto-population
                                'equipmentDisplay' => ($item['step1']['equipment']['equipmentName'] ?? $item['step1']['selectedEquipment']['equipmentName'] ?? $item['step1']['equipment']['name'] ?? 'N/A') . ' - ' . ($item['step1']['equipment']['model'] ?? $item['step1']['selectedEquipment']['model'] ?? $item['step1']['equipment']['model'] ?? 'N/A') . ' (' . ($item['step1']['equipment']['vehicleNumber'] ?? $item['step1']['selectedEquipment']['vehicleNumber'] ?? $item['step1']['equipment']['number'] ?? 'N/A') . ')'
                            ];
                        }
                    }
                }

                // Sort by created date (newest first)
                usort($processedInspections, function($a, $b) {
                    return strtotime($b['createdAt']) - strtotime($a['createdAt']);
                });

                Log::info('Successfully fetched inspection reports for linking', ['count' => count($processedInspections)]);

                return response()->json([
                    'success' => true,
                    'inspections' => $processedInspections,
                    'count' => count($processedInspections)
                ]);
            } else {
                Log::error('Firebase API error for inspection reports', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch inspection reports from Firebase. Status: ' . $response->status()
                ], 500);
            }
        }
        } catch (\Exception $e) {
            Log::error('Error fetching inspection reports from Firebase', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error fetching inspection reports: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Fetch checklist data from Firebase
     */
    public function fetchChecklistsFromFirebase(): JsonResponse
    {
        try {
            // Firebase configuration - you may want to move these to .env file
            $firebaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
            $checklistPath = '/AdminChecklistJson/GlobalChecklistJson/Checklistfile/documents.json';
            
            // Make request to Firebase
            $response = Http::get($firebaseUrl . $checklistPath);
            
            if ($response->successful()) {
                $checklistData = $response->json();
                
                // Process the data to ensure it's in the expected format
                $processedChecklists = [];
                
                if (is_array($checklistData)) {
                    foreach ($checklistData as $index => $item) {
                        if (is_array($item) || is_object($item)) {
                            $item = (array) $item;
                            
                            // Process checklist structure
                            $checklistItems = [];
                            if (isset($item['checklist']) && is_array($item['checklist'])) {
                                foreach ($item['checklist'] as $sectionKey => $sectionData) {
                                    $sectionItems = [];
                                    
                                    if (is_array($sectionData)) {
                                        if (isset($sectionData['items']) && is_array($sectionData['items'])) {
                                            // Section has items array
                                            foreach ($sectionData['items'] as $itemIndex => $checklistItem) {
                                                $sectionItems[] = [
                                                    'itemName' => is_string($checklistItem) ? $checklistItem : ($checklistItem['itemName'] ?? 'Unnamed Item'),
                                                    'inputType' => 'Ok/Reject/N/A',
                                                    'description' => '',
                                                    'instruction' => '',
                                                    'sortOrder' => $itemIndex + 1,
                                                    'isInspectionRequired' => false,
                                                ];
                                            }
                                        } else {
                                            // Section is directly an array of items
                                            foreach ($sectionData as $itemIndex => $checklistItem) {
                                                $sectionItems[] = [
                                                    'itemName' => is_string($checklistItem) ? $checklistItem : ($checklistItem['itemName'] ?? 'Unnamed Item'),
                                                    'inputType' => 'Ok/Reject/N/A',
                                                    'description' => '',
                                                    'instruction' => '',
                                                    'sortOrder' => $itemIndex + 1,
                                                    'isInspectionRequired' => false,
                                                ];
                                            }
                                        }
                                    }
                                    
                                    if (!empty($sectionItems)) {
                                        $checklistItems[] = [
                                            'text' => $sectionKey,
                                            'subItems' => $sectionItems
                                        ];
                                    }
                                }
                            }
                            
                            $processedChecklists[] = [
                                'id' => $item['document_id'] ?? $index,
                                'document_id' => $item['document_id'] ?? $index,
                                'title' => $item['document_name'] ?? 'Untitled Checklist',
                                'document_name' => $item['document_name'] ?? 'Untitled Checklist',
                                'description' => $item['description'] ?? '',
                                'items' => $checklistItems,
                                'status' => 'Available',
                                'version' => $item['version'] ?? '',
                                'version_date' => $item['version_date'] ?? '',
                                'approved_by' => $item['approved_by'] ?? '',
                                'Header_title' => $item['Header_title'] ?? '',
                                'header_inputs' => $item['header_inputs'] ?? [],
                                'footer_inputs' => $item['footer_inputs'] ?? [],
                                'checklist' => $item['checklist'] ?? [],
                                'createdAt' => $item['createdAt'] ?? now()->toISOString(),
                                'updatedAt' => $item['updatedAt'] ?? now()->toISOString(),
                            ];
                        }
                    }
                }
                
                return response()->json([
                    'success' => true,
                    'checklists' => $processedChecklists,
                    'count' => count($processedChecklists)
                ]);
            } else {
                Log::error('Firebase API error', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to fetch checklist data from Firebase. Status: ' . $response->status()
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error fetching checklists from Firebase', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error fetching checklist data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Save inspection report to Firebase database
     */
    public function saveInspectionReport(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'requestNumber' => 'required|string',
                'step1' => 'required|array',
                'step2' => 'required|array',
                'step3' => 'required|array',
            ]);

            $inspectionData = $request->all();
            
            // Add timestamp and status
            $inspectionData['createdAt'] = now()->toISOString();
            $inspectionData['updatedAt'] = now()->toISOString();
            $inspectionData['status'] = 'completed';
            
            // Firebase path for GlobalInspectionReport
            $firebasePath = 'GlobalInspectionReport/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1';
            
            // Generate unique key for the inspection report
            $uniqueKey = 'inspection_' . time() . '_' . uniqid();
            
            // Save to Firebase
            $response = Http::put(
                $this->firebaseUrl . $firebasePath . '/' . $uniqueKey . '.json',
                $inspectionData
            );
            
            if ($response->successful()) {
                Log::info('Inspection report saved successfully', [
                    'requestNumber' => $inspectionData['requestNumber'],
                    'firebaseKey' => $uniqueKey,
                    'timestamp' => now()
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Inspection report saved successfully',
                    'data' => [
                        'firebaseKey' => $uniqueKey,
                        'requestNumber' => $inspectionData['requestNumber'],
                        'timestamp' => $inspectionData['createdAt']
                    ]
                ]);
            } else {
                Log::error('Failed to save inspection report to Firebase', [
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'requestNumber' => $inspectionData['requestNumber']
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to save inspection report to database. Status: ' . $response->status()
                ], 500);
            }
            
        } catch (\Exception $e) {
            Log::error('Error saving inspection report', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'requestData' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error saving inspection report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate inspection PDF (matches React Native PDF generation)
     * Accepts HTML content from frontend for PDF generation
     */
    public function generateInspectionPdf(Request $request)
    {
        try {
            $inputData = $request->all();
            
            Log::info('📄 Generating inspection PDF', [
                'hasHtml' => isset($inputData['html']),
                'hasReportData' => isset($inputData['reportData']),
                'requestNumber' => $inputData['requestNumber'] ?? 'N/A'
            ]);
            
            // Check if HTML content is provided from frontend
            if (isset($inputData['html']) && !empty($inputData['html'])) {
                // Use HTML content from frontend (generated by blade's generatePDFHTML function)
                $pdfContent = $inputData['html'];
                Log::info('✅ Using HTML content from frontend', [
                    'htmlLength' => strlen($pdfContent),
                    'requestNumber' => $inputData['requestNumber'] ?? 'N/A'
                ]);
            } else {
                // Fallback: Generate HTML on backend
                $inspectionData = $inputData['reportData'] ?? $inputData;
                
                if (!isset($inspectionData['step1']) || !isset($inspectionData['step2']) || !isset($inspectionData['step3'])) {
                    throw new \Exception('Missing required inspection data (html content or step1, step2, step3)');
                }
                
                $pdfContent = $this->generateInspectionPdfContent($inspectionData);
                Log::info('⚠️ Using backend-generated HTML (fallback)');
            }
            
            // Generate PDF using DomPDF
            $pdf = PDF::loadHTML($pdfContent);
            $pdf->setPaper('A4', 'portrait');
            
            $requestNumber = $inputData['requestNumber'] ?? $inputData['reportData']['requestNumber'] ?? 'Inspection_Report';
            
            Log::info('✅ PDF generated successfully', ['requestNumber' => $requestNumber]);
            
            // Return PDF as downloadable response (for JavaScript to receive as Blob)
            return response($pdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="Inspection_Report_' . $requestNumber . '.pdf"');
            
        } catch (\Exception $e) {
            Log::error('❌ Error generating inspection PDF', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error generating PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate corrective action PDF using DomPDF
     * Accepts HTML content from frontend for PDF generation
     */
    public function generateCorrectiveActionPdf(Request $request)
    {
        try {
            $inputData = $request->all();
            
            Log::info('📄 Generating corrective action PDF', [
                'hasHtml' => isset($inputData['html']),
                'htmlLength' => isset($inputData['html']) ? strlen($inputData['html']) : 0,
                'inspectionId' => $inputData['inspectionId'] ?? 'N/A',
                'requestNumber' => $inputData['requestNumber'] ?? 'N/A',
                'allKeys' => array_keys($inputData)
            ]);
            
            // Check if HTML content is provided from frontend
            if (!isset($inputData['html']) || empty($inputData['html'])) {
                Log::error('❌ HTML content missing in request', [
                    'inputKeys' => array_keys($inputData),
                    'inputData' => array_map(function($value) {
                        return is_string($value) ? substr($value, 0, 100) : gettype($value);
                    }, $inputData)
                ]);
                throw new \Exception('HTML content is required for PDF generation');
            }
            
            $htmlContent = $inputData['html'];
            Log::info('✅ Using HTML content from frontend', [
                'htmlLength' => strlen($htmlContent),
                'requestNumber' => $inputData['requestNumber'] ?? 'N/A'
            ]);
            
            // Replace logo URL with absolute file path for reliable rendering
            $logoPath = public_path('images/titan-drlling_blue-removebg-preview.png');
            if (file_exists($logoPath)) {
                // Convert logo to base64 for embedding
                $logoData = file_get_contents($logoPath);
                $logoBase64 = 'data:image/png;base64,' . base64_encode($logoData);
                // Replace any logo URLs in the HTML with base64
                $htmlContent = preg_replace(
                    '/<img[^>]+src=["\']([^"\']*titan[^"\']*\.(png|jpg|jpeg))["\'][^>]*>/i',
                    '<img src="' . $logoBase64 . '" alt="Titan Africa Drilling Logo" style="width: 140px; height: 50px; object-fit: contain; display: block;"/>',
                    $htmlContent
                );
            }
            
            // Generate PDF using DomPDF with remote image support
            $pdf = PDF::loadHTML($htmlContent);
            $pdf->setPaper('A4', 'portrait');
            
            // Enable remote image loading
            $dompdf = $pdf->getDomPDF();
            $options = $dompdf->getOptions();
            $options->setIsRemoteEnabled(true);
            $options->setIsHtml5ParserEnabled(true);
            $dompdf->setOptions($options);
            
            $requestNumber = $inputData['requestNumber'] ?? 'Corrective_Action_Report';
            
            Log::info('✅ Corrective action PDF generated successfully', ['requestNumber' => $requestNumber]);
            
            // Return PDF as downloadable response (for JavaScript to receive as Blob)
            return response($pdf->output())
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'attachment; filename="Corrective_Action_' . $requestNumber . '.pdf"');
            
        } catch (\Exception $e) {
            Log::error('❌ Error generating corrective action PDF', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error generating PDF: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate PDF content for inspection report
     */
    private function generateInspectionPdfContent(array $inspectionData): string
    {
        $step1 = $inspectionData['step1'];
        $step2 = $inspectionData['step2'];
        $step3 = $inspectionData['step3'];
        
        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="utf-8">
            <title>Inspection Report - ' . $inspectionData['requestNumber'] . '</title>
            <style>
                body { font-family: Arial, sans-serif; font-size: 12px; line-height: 1.4; }
                .header { text-align: center; margin-bottom: 30px; border-bottom: 2px solid #333; padding-bottom: 20px; }
                .section { margin-bottom: 25px; }
                .section-title { background-color: #f0f0f0; padding: 10px; font-weight: bold; font-size: 14px; }
                .row { display: flex; margin-bottom: 8px; }
                .label { font-weight: bold; width: 200px; }
                .value { flex: 1; }
                .table { width: 100%; border-collapse: collapse; margin-top: 10px; }
                .table th, .table td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                .table th { background-color: #f5f5f5; font-weight: bold; }
                .signature-section { margin-top: 20px; }
                .signature-box { border: 1px solid #ddd; height: 60px; margin-top: 10px; }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>INSPECTION REPORT</h1>
                <h2>Request Number: ' . $inspectionData['requestNumber'] . '</h2>
                <p>Generated on: ' . date('F j, Y \a\t g:i A') . '</p>
            </div>
            
            <div class="section">
                <div class="section-title">Step 1: Inspection Information</div>
                <div class="row">
                    <div class="label">Country:</div>
                    <div class="value">' . ($step1['country'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Project:</div>
                    <div class="value">' . ($step1['project'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Inspector:</div>
                    <div class="value">' . ($step1['inspector'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Equipment:</div>
                    <div class="value">' . ($step1['equipment']['vehicleName'] ?? 'N/A') . ' - ' . ($step1['equipment']['vinNumber'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Meter Reading:</div>
                    <div class="value">' . ($step1['equipment']['meterReading'] ?? 'N/A') . ' ' . ($step1['equipment']['meterUnit'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">GPS Coordinates:</div>
                    <div class="value">' . ($step1['gpsLocation']['coordinates'] ?? 'N/A') . '</div>
                </div>
            </div>
            
            <div class="section">
                <div class="section-title">Step 2: Inspection Results</div>
                <div class="row">
                    <div class="label">Checklist:</div>
                    <div class="value">' . ($step2['checklist']['document_name'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Faulty Items Count:</div>
                    <div class="value">' . ($step2['faultyItemsCount'] ?? 0) . '</div>
                </div>
            </div>
            
            <div class="section">
                <div class="section-title">Step 3: Inspection Summary</div>
                <div class="row">
                    <div class="label">Overall Condition:</div>
                    <div class="value">' . ($step3['overallCondition'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Equipment Status:</div>
                    <div class="value">' . ($step3['equipmentStatus'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Priority:</div>
                    <div class="value">' . ($step3['priority'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Safe to Use:</div>
                    <div class="value">' . ($step3['safeToUse'] ?? 'N/A') . '</div>
                </div>
                <div class="row">
                    <div class="label">Additional Notes:</div>
                    <div class="value">' . ($step3['additionalNote'] ?? 'N/A') . '</div>
                </div>
            </div>
            
            <div class="section">
                <div class="section-title">Signatures</div>';
        
        if (isset($step3['signatureFields']) && is_array($step3['signatureFields'])) {
            foreach ($step3['signatureFields'] as $signatureType) {
                $signatureLabel = ucfirst(str_replace('_', ' ', $signatureType));
                $signatureName = $step3['signatureValues'][$signatureType . '_name'] ?? 'N/A';
                
                $html .= '
                <div class="row">
                    <div class="label">' . $signatureLabel . ' Name:</div>
                    <div class="value">' . $signatureName . '</div>
                </div>
                <div class="row">
                    <div class="label">' . $signatureLabel . ' Signature:</div>
                    <div class="value">
                        <div class="signature-box"></div>
                    </div>
                </div>';
            }
        }
        
        $html .= '
            </div>
        </body>
        </html>';

        return $html;
    }
    
    /**
     * Delete inspection report from MongoDB (Admin can delete any report)
     */
    public function deleteInspectionReport($inspectionId)
    {
        try {
            Log::info('🗑️ Deleting inspection from MongoDB', [
                'inspectionId' => $inspectionId,
                'user' => auth()->user()->email ?? 'unknown'
            ]);
            
            // Use delete-by-id endpoint (allows admin to delete any report)
            $mongoUrl = "https://api.titandrillingzm.com:6004/inspections/delete-by-id/{$inspectionId}";
            
            Log::info('📤 Sending delete request to MongoDB', [
                'url' => $mongoUrl,
                'method' => 'DELETE',
                'inspectionId' => $inspectionId
            ]);
            
            $response = Http::timeout(30)->delete($mongoUrl);
            
            $responseBody = $response->body();
            $responseData = json_decode($responseBody, true);
            
            Log::info('📊 MongoDB delete response', [
                'status' => $response->status(),
                'successful' => $response->successful(),
                'body' => $responseBody,
                'decoded' => $responseData
            ]);
            
            // Check if deletion was successful
            if ($response->successful() && isset($responseData['success']) && $responseData['success'] === true) {
                Log::info('✅ Inspection deleted successfully from MongoDB', [
                    'inspectionId' => $inspectionId,
                    'deletedRequestNumber' => $responseData['data']['deletedRequestNumber'] ?? 'N/A'
                ]);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Inspection report deleted successfully from MongoDB',
                    'data' => $responseData['data'] ?? null
                ]);
            }
            
            // If MongoDB returned an error or 404
            $errorMessage = $responseData['error'] ?? $responseData['message'] ?? 'Unknown error from MongoDB';
            
            Log::error('❌ Failed to delete inspection from MongoDB', [
                'status' => $response->status(),
                'body' => $responseBody,
                'decoded_error' => $responseData,
                'inspectionId' => $inspectionId,
                'errorMessage' => $errorMessage
            ]);
            
            return response()->json([
                'success' => false,
                'message' => $errorMessage,
                'error' => $errorMessage,
                'debug' => [
                    'inspectionId' => $inspectionId,
                    'mongoUrl' => $mongoUrl,
                    'httpStatus' => $response->status(),
                    'mongoResponse' => $responseData
                ]
            ], 400);
            
        } catch (\Exception $e) {
            Log::error('❌ Exception deleting inspection report', [
                'error' => $e->getMessage(),
                'inspectionId' => $inspectionId,
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Server error while deleting inspection',
                'error' => $e->getMessage(),
                'debug' => [
                    'inspectionId' => $inspectionId,
                    'exception' => get_class($e)
                ]
            ], 500);
        }
    }
    
    /**
     * Proxy endpoint to view PDF inline (with proper headers for viewing, not downloading)
     */
    public function viewPdfProxy(Request $request)
    {
        try {
            // Get the PDF path from query parameter
            $pdfPath = $request->query('path');
            
            if (!$pdfPath) {
                return response()->json([
                    'success' => false,
                    'message' => 'PDF path is required'
                ], 400);
            }
            
            // Construct full MongoDB URL
            $mongoBaseUrl = 'https://api.titandrillingzm.com:6004';
            $fullUrl = $pdfPath;
            
            // Add base URL if path is relative
            if (!str_starts_with($pdfPath, 'http')) {
                $fullUrl = $mongoBaseUrl . $pdfPath;
            }
            
            Log::info('📄 Fetching PDF from MongoDB for inline viewing', ['url' => $fullUrl]);
            
            // Fetch PDF from MongoDB server
            $response = Http::timeout(60)->get($fullUrl);
            
            if ($response->successful()) {
                Log::info('✅ PDF fetched successfully, serving with inline headers');
                
                // Return PDF with INLINE viewing headers (not attachment)
                return response($response->body())
                    ->header('Content-Type', 'application/pdf')
                    ->header('Content-Disposition', 'inline; filename="inspection_report.pdf"')
                    ->header('Cache-Control', 'public, max-age=3600')
                    ->header('X-Content-Type-Options', 'nosniff')
                    ->header('Accept-Ranges', 'bytes');
            }
            
            Log::error('❌ Failed to fetch PDF from MongoDB', [
                'status' => $response->status(),
                'url' => $fullUrl
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'PDF not found or unavailable'
            ], 404);
            
        } catch (\Exception $e) {
            Log::error('❌ Exception viewing PDF', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to load PDF',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}

