// Global variables
let selectedEquipment = null;
let selectedUsers = [];
let selectedInspection = null;
let selectedWorkOrder = null;
let currentCategory = 'Task';
let items = {
    Task: [],
    Parts: [],
    Labor: [],
    'Additional Cost': [],
    Remarks: []
};

// Project data based on country
const projectData = {
    'Zambia': [
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' }
    ],
    'DRC': [
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' }
    ]
};

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initializeForm();
    setupEventListeners();
    updateRequestNumber();
});

// Initialize form
function initializeForm() {
    // Set up country change handler
    const countrySelect = document.getElementById('country');
    const projectSelect = document.getElementById('project');
    
    countrySelect.addEventListener('change', function() {
        updateProjectOptions();
        updateRequestNumber();
    });
    
    projectSelect.addEventListener('change', function() {
        updateRequestNumber();
    });
    
    // Set up priority radio buttons
    setupPriorityButtons();
    
    // Set up collapsible sections
    setupCollapsibleSections();
}

// Setup event listeners
function setupEventListeners() {
    // Equipment search
    const equipmentSearch = document.getElementById('equipmentSearch');
    if (equipmentSearch) {
        equipmentSearch.addEventListener('input', filterEquipment);
    }
    
    // User search
    const userSearch = document.getElementById('userSearch');
    if (userSearch) {
        userSearch.addEventListener('input', filterUsers);
    }
}

// Update project options based on country
function updateProjectOptions() {
    const country = document.getElementById('country').value;
    const projectSelect = document.getElementById('project');
    
    // Clear existing options
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    
    if (country && projectData[country]) {
        projectData[country].forEach(project => {
            const option = document.createElement('option');
            option.value = project.value;
            option.textContent = project.label;
            projectSelect.appendChild(option);
        });
    }
}

// Update request number
function updateRequestNumber() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const requestNumberElement = document.getElementById('requestNumber');
    
    if (country && project) {
        // Generate request number based on country and project
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
        const projectCode = project.slice(0, 3).toUpperCase();
        const timestamp = Date.now().toString().slice(-4);
        const requestNumber = `${countryCode}-${projectCode}-JC-${timestamp}`;
        requestNumberElement.textContent = requestNumber;
    } else {
        requestNumberElement.textContent = 'Select Country & Project';
    }
}

// Setup priority buttons
function setupPriorityButtons() {
    const priorityButtons = document.querySelectorAll('.priority-button');
    priorityButtons.forEach(button => {
        button.addEventListener('click', function() {
            const radio = this.previousElementSibling;
            radio.checked = true;
            
            // Update visual state
            priorityButtons.forEach(btn => btn.classList.remove('selected'));
            this.classList.add('selected');
        });
    });
}

// Setup collapsible sections
function setupCollapsibleSections() {
    const sections = document.querySelectorAll('.collapsible-section');
    sections.forEach(section => {
        const header = section.querySelector('.section-header');
        const content = section.querySelector('.section-content');
        const icon = section.querySelector('.section-header i');
        
        header.addEventListener('click', function() {
            const isCollapsed = content.style.display === 'none';
            content.style.display = isCollapsed ? 'block' : 'none';
            icon.classList.toggle('fa-chevron-down', isCollapsed);
            icon.classList.toggle('fa-chevron-up', !isCollapsed);
        });
    });
}

// Open equipment modal
function openEquipmentModal() {
    const modal = new bootstrap.Modal(document.getElementById('equipmentModal'));
    loadEquipment();
    modal.show();
}

// Load equipment data
async function loadEquipment() {
    try {
        const response = await fetch('/api/equipment');
        const equipment = await response.json();
        displayEquipment(equipment);
    } catch (error) {
        console.error('Error loading equipment:', error);
        displayEquipment([]);
    }
}

// Display equipment in modal
function displayEquipment(equipmentList) {
    const container = document.getElementById('equipmentList');
    container.innerHTML = '';
    
    equipmentList.forEach(equipment => {
        const item = document.createElement('div');
        item.className = 'list-item';
        item.innerHTML = `
            <div class="list-item-title">${equipment.equipmentName || equipment.name}</div>
            <div class="list-item-details">
                Equipment #: ${equipment.equipmentNumber || 'N/A'}<br>
                Category: ${equipment.mainCategory || 'N/A'}
            </div>
        `;
        
        item.addEventListener('click', function() {
            selectEquipment(equipment);
        });
        
        container.appendChild(item);
    });
}

// Filter equipment
function filterEquipment() {
    const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
    const items = document.querySelectorAll('#equipmentList .list-item');
    
    items.forEach(item => {
        const title = item.querySelector('.list-item-title').textContent.toLowerCase();
        const details = item.querySelector('.list-item-details').textContent.toLowerCase();
        
        if (title.includes(searchTerm) || details.includes(searchTerm)) {
            item.style.display = 'block';
        } else {
            item.style.display = 'none';
        }
    });
}

// Select equipment
function selectEquipment(equipment) {
    selectedEquipment = equipment;
    
    // Update display
    const display = document.getElementById('selectedEquipmentDisplay');
    display.innerHTML = `
        <div>
            <div style="font-weight: bold; color: #015185;">${equipment.equipmentName || equipment.name}</div>
            <div style="font-size: 12px; color: #666;">${equipment.equipmentNumber || 'N/A'}</div>
        </div>
    `;
    
    // Set hidden input
    document.getElementById('selectedEquipmentId').value = equipment.id;
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('equipmentModal'));
    modal.hide();
}

// Open assign to modal
function openAssignToModal() {
    const modal = new bootstrap.Modal(document.getElementById('assignToModal'));
    loadUsers();
    modal.show();
}

// Load users data
async function loadUsers() {
    try {
        const response = await fetch('/api/users');
        const users = await response.json();
        displayUsers(users);
    } catch (error) {
        console.error('Error loading users:', error);
        displayUsers([]);
    }
}

// Display users in modal
function displayUsers(usersList) {
    const container = document.getElementById('usersList');
    container.innerHTML = '';
    
    usersList.forEach(user => {
        const item = document.createElement('div');
        item.className = 'list-item';
        item.innerHTML = `
            <div class="list-item-title">${user.name || user.fullName}</div>
            <div class="list-item-details">
                Employee #: ${user.employeeNumber || 'N/A'}<br>
                Role: ${user.role || 'N/A'}
            </div>
        `;
        
        item.addEventListener('click', function() {
            toggleUserSelection(user, item);
        });
        
        container.appendChild(item);
    });
}

// Filter users
function filterUsers() {
    const searchTerm = document.getElementById('userSearch').value.toLowerCase();
    const items = document.querySelectorAll('#usersList .list-item');
    
    items.forEach(item => {
        const title = item.querySelector('.list-item-title').textContent.toLowerCase();
        const details = item.querySelector('.list-item-details').textContent.toLowerCase();
        
        if (title.includes(searchTerm) || details.includes(searchTerm)) {
            item.style.display = 'block';
        } else {
            item.style.display = 'none';
        }
    });
}

// Toggle user selection
function toggleUserSelection(user, element) {
    const index = selectedUsers.findIndex(u => u.id === user.id);
    
    if (index > -1) {
        selectedUsers.splice(index, 1);
        element.classList.remove('selected');
    } else {
        selectedUsers.push(user);
        element.classList.add('selected');
    }
}

// Save user selection
function saveUserSelection() {
    updateSelectedUsersDisplay();
    
    // Update assign to text
    const assignToText = document.getElementById('assignToText');
    assignToText.textContent = selectedUsers.length > 0 ? `${selectedUsers.length} Selected` : 'Select';
    
    // Set hidden input
    document.getElementById('assignToInput').value = JSON.stringify(selectedUsers);
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('assignToModal'));
    modal.hide();
}

// Update selected users display
function updateSelectedUsersDisplay() {
    const container = document.getElementById('selectedUsersContainer');
    
    if (selectedUsers.length === 0) {
        container.style.display = 'none';
        return;
    }
    
    container.style.display = 'block';
    container.innerHTML = selectedUsers.map(user => `
        <div class="selected-user-item">
            <div>
                <div class="selected-user-name">${user.name || user.fullName}</div>
                <div class="selected-user-details">Emp#: ${user.employeeNumber || 'N/A'} | Role: ${user.role || 'N/A'}</div>
            </div>
            <button type="button" class="remove-user-btn" onclick="removeUser(${user.id})">
                <i class="fas fa-times"></i>
            </button>
        </div>
    `).join('');
}

// Remove user from selection
function removeUser(userId) {
    selectedUsers = selectedUsers.filter(u => u.id !== userId);
    updateSelectedUsersDisplay();
    
    // Update assign to text
    const assignToText = document.getElementById('assignToText');
    assignToText.textContent = selectedUsers.length > 0 ? `${selectedUsers.length} Selected` : 'Select';
    
    // Set hidden input
    document.getElementById('assignToInput').value = JSON.stringify(selectedUsers);
}

// Open inspection modal
function openInspectionModal() {
    const modal = new bootstrap.Modal(document.getElementById('inspectionModal'));
    loadInspections();
    modal.show();
}

// Load inspections data
async function loadInspections() {
    try {
        const response = await fetch('/api/inspections');
        const inspections = await response.json();
        displayInspections(inspections);
    } catch (error) {
        console.error('Error loading inspections:', error);
        displayInspections([]);
    }
}

// Display inspections in modal
function displayInspections(inspectionsList) {
    const container = document.getElementById('inspectionsList');
    container.innerHTML = '';
    
    inspectionsList.forEach(inspection => {
        const item = document.createElement('div');
        item.className = 'list-item';
        item.innerHTML = `
            <div class="list-item-title">Request #: ${inspection.requestNumber || 'N/A'}</div>
            <div class="list-item-details">
                Equipment: ${inspection.equipment?.equipmentName || 'N/A'}<br>
                Date: ${inspection.date ? new Date(inspection.date).toLocaleDateString() : 'N/A'}<br>
                Faulty Items: ${inspection.faultyItems || 0}
            </div>
        `;
        
        item.addEventListener('click', function() {
            selectInspection(inspection);
        });
        
        container.appendChild(item);
    });
}

// Select inspection
function selectInspection(inspection) {
    selectedInspection = inspection;
    
    // Update display
    const inspectionText = document.getElementById('inspectionText');
    inspectionText.textContent = inspection.requestNumber || 'Selected';
    
    // Set hidden input
    document.getElementById('linkInspectionId').value = inspection.id;
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('inspectionModal'));
    modal.hide();
}

// Open work order modal
function openWorkOrderModal() {
    const modal = new bootstrap.Modal(document.getElementById('workOrderModal'));
    loadWorkOrders();
    modal.show();
}

// Load work orders data
async function loadWorkOrders() {
    try {
        const response = await fetch('/api/work-orders');
        const workOrders = await response.json();
        displayWorkOrders(workOrders);
    } catch (error) {
        console.error('Error loading work orders:', error);
        displayWorkOrders([]);
    }
}

// Display work orders in modal
function displayWorkOrders(workOrdersList) {
    const container = document.getElementById('workOrdersList');
    container.innerHTML = '';
    
    workOrdersList.forEach(workOrder => {
        const item = document.createElement('div');
        item.className = 'list-item';
        item.innerHTML = `
            <div class="list-item-title">${workOrder.title}</div>
            <div class="list-item-details">
                Job Card #: ${workOrder.requestNumber || 'N/A'}<br>
                Equipment: ${workOrder.equipment?.equipmentName || 'N/A'}
            </div>
        `;
        
        item.addEventListener('click', function() {
            selectWorkOrder(workOrder);
        });
        
        container.appendChild(item);
    });
}

// Select work order
function selectWorkOrder(workOrder) {
    selectedWorkOrder = workOrder;
    
    // Update display
    const workOrderText = document.getElementById('workOrderText');
    workOrderText.textContent = workOrder.title || 'Selected';
    
    // Set hidden input
    document.getElementById('linkedWorkOrderId').value = workOrder.id;
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('workOrderModal'));
    modal.hide();
}

// Switch tab
function switchTab(category) {
    // Update tab buttons
    document.querySelectorAll('.tab-button').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-category="${category}"]`).classList.add('active');
    
    // Update tab content
    document.querySelectorAll('.tab-content').forEach(content => {
        content.style.display = 'none';
    });
    document.getElementById(`${category}-content`).style.display = 'block';
    
    currentCategory = category;
    updateItemsDisplay();
}

// Open add item modal
function openAddItemModal(category) {
    currentCategory = category;
    const modal = new bootstrap.Modal(document.getElementById('addItemModal'));
    const modalTitle = document.getElementById('addItemModalTitle');
    const modalBody = document.getElementById('addItemModalBody');
    
    modalTitle.textContent = `Add ${category}`;
    modalBody.innerHTML = getAddItemForm(category);
    
    modal.show();
}

// Get add item form based on category
function getAddItemForm(category) {
    switch (category) {
        case 'Task':
            return `
                <div class="form-group">
                    <label class="form-label">Task Description</label>
                    <input type="text" class="form-control" id="itemText" placeholder="Enter task description">
                </div>
                <div class="form-group">
                    <label class="form-label">Type</label>
                    <select class="form-control" id="itemType">
                        <option value="Repair">Repair</option>
                        <option value="Replace">Replace</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Note</label>
                    <textarea class="form-control" id="itemNote" rows="3" placeholder="Enter note"></textarea>
                </div>
            `;
        case 'Parts':
            return `
                <div class="form-group">
                    <label class="form-label">Item Number</label>
                    <input type="text" class="form-control" id="itemNumber" placeholder="Enter item number">
                </div>
                <div class="form-group">
                    <label class="form-label">Part Number</label>
                    <input type="text" class="form-control" id="partNumber" placeholder="Enter part number">
                </div>
                <div class="form-group">
                    <label class="form-label">Item Name</label>
                    <input type="text" class="form-control" id="itemName" placeholder="Enter item name">
                </div>
                <div class="form-group">
                    <label class="form-label">Quantity</label>
                    <input type="number" class="form-control" id="quantity" placeholder="Enter quantity">
                </div>
                <div class="form-group">
                    <label class="form-label">Unit</label>
                    <select class="form-control" id="unit">
                        <option value="Piece">Piece</option>
                        <option value="Unit">Unit</option>
                        <option value="Box">Box</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Item Cost</label>
                    <input type="number" class="form-control" id="itemCost" placeholder="Enter cost" step="0.01">
                </div>
                <div class="form-group">
                    <label class="form-label">Site</label>
                    <input type="text" class="form-control" id="site" placeholder="Enter site">
                </div>
                <div class="form-group">
                    <label class="form-label">Note</label>
                    <textarea class="form-control" id="itemNote" rows="3" placeholder="Enter note"></textarea>
                </div>
            `;
        case 'Labor':
            return `
                <div class="form-group">
                    <label class="form-label">Worker Name</label>
                    <input type="text" class="form-control" id="worker" placeholder="Enter worker name">
                </div>
                <div class="form-group">
                    <label class="form-label">Labor Code</label>
                    <input type="text" class="form-control" id="code" placeholder="Enter labor code">
                </div>
                <div class="form-group">
                    <label class="form-label">Rate</label>
                    <input type="number" class="form-control" id="rate" placeholder="Enter rate" step="0.01">
                </div>
                <div class="form-group">
                    <label class="form-label">Hours</label>
                    <input type="number" class="form-control" id="hours" placeholder="Enter hours">
                </div>
                <div class="form-group">
                    <label class="form-label">Minutes</label>
                    <input type="number" class="form-control" id="minutes" placeholder="Enter minutes">
                </div>
                <div class="form-group">
                    <label class="form-label">Note</label>
                    <textarea class="form-control" id="itemNote" rows="3" placeholder="Enter note"></textarea>
                </div>
            `;
        case 'Additional Cost':
            return `
                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea class="form-control" id="description" rows="4" placeholder="Enter description"></textarea>
                </div>
                <div class="form-group">
                    <label class="form-label">Amount</label>
                    <input type="number" class="form-control" id="amount" placeholder="Enter amount" step="0.01">
                </div>
            `;
        case 'Remarks':
            return `
                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea class="form-control" id="description" rows="4" placeholder="Enter remarks"></textarea>
                </div>
            `;
        default:
            return '';
    }
}

// Save item
function saveItem() {
    const item = getItemData();
    if (!item) return;
    
    items[currentCategory].push(item);
    updateItemsDisplay();
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('addItemModal'));
    modal.hide();
}

// Get item data from form
function getItemData() {
    const baseItem = {
        id: Date.now().toString(),
        category: currentCategory,
        status: 'Initiated'
    };
    
    switch (currentCategory) {
        case 'Task':
            const text = document.getElementById('itemText').value;
            const type = document.getElementById('itemType').value;
            const note = document.getElementById('itemNote').value;
            
            if (!text.trim()) {
                alert('Task description is required');
                return null;
            }
            
            return {
                ...baseItem,
                text,
                type,
                note
            };
        case 'Parts':
            const itemNumber = document.getElementById('itemNumber').value;
            const partNumber = document.getElementById('partNumber').value;
            const itemName = document.getElementById('itemName').value;
            const quantity = document.getElementById('quantity').value;
            const unit = document.getElementById('unit').value;
            const itemCost = document.getElementById('itemCost').value;
            const site = document.getElementById('site').value;
            const note = document.getElementById('itemNote').value;
            
            if (!itemNumber || !partNumber || !itemName || !quantity || !itemCost) {
                alert('All required fields must be filled');
                return null;
            }
            
            return {
                ...baseItem,
                itemNumber,
                partNumber,
                itemName,
                quantity,
                unit,
                itemCost,
                site,
                note
            };
        case 'Labor':
            const worker = document.getElementById('worker').value;
            const code = document.getElementById('code').value;
            const rate = document.getElementById('rate').value;
            const hours = document.getElementById('hours').value;
            const minutes = document.getElementById('minutes').value;
            const note = document.getElementById('itemNote').value;
            
            if (!worker || !code || !rate || !hours || !minutes) {
                alert('All required fields must be filled');
                return null;
            }
            
            return {
                ...baseItem,
                worker,
                code,
                rate,
                hours,
                minutes,
                note
            };
        case 'Additional Cost':
            const description = document.getElementById('description').value;
            const amount = document.getElementById('amount').value;
            
            if (!description || !amount) {
                alert('All fields must be filled');
                return null;
            }
            
            return {
                ...baseItem,
                description,
                amount
            };
        case 'Remarks':
            const remarksDescription = document.getElementById('description').value;
            
            if (!remarksDescription.trim()) {
                alert('Description is required');
                return null;
            }
            
            return {
                ...baseItem,
                description: remarksDescription
            };
        default:
            return null;
    }
}

// Update items display
function updateItemsDisplay() {
    const container = document.getElementById(`${currentCategory}ItemsList`);
    if (!container) return;
    
    container.innerHTML = '';
    
    items[currentCategory].forEach(item => {
        const itemElement = createItemElement(item);
        container.appendChild(itemElement);
    });
}

// Create item element
function createItemElement(item) {
    const div = document.createElement('div');
    div.className = 'item-card';
    
    let content = '';
    let details = '';
    
    switch (item.category) {
        case 'Task':
            content = item.text;
            details = `Type: ${item.type} | Note: ${item.note || 'N/A'}`;
            break;
        case 'Parts':
            content = item.itemName;
            details = `Qty: ${item.quantity} ${item.unit} | Cost: ${item.itemCost}`;
            break;
        case 'Labor':
            content = item.worker;
            details = `Hours: ${item.hours}h ${item.minutes}m | Rate: ${item.rate}`;
            break;
        case 'Additional Cost':
            content = item.description;
            details = `Amount: ${item.amount}`;
            break;
        case 'Remarks':
            content = item.description;
            details = '';
            break;
    }
    
    div.innerHTML = `
        <div class="item-content">
            <div class="item-title">${content}</div>
            <div class="item-details">${details}</div>
            <span class="item-status status-${item.status.toLowerCase().replace(' ', '-')}">${item.status}</span>
        </div>
        <div class="item-actions">
            <button type="button" class="action-btn" onclick="editItem('${item.id}')" title="Edit">
                <i class="fas fa-edit"></i>
            </button>
            <button type="button" class="action-btn" onclick="deleteItem('${item.id}')" title="Delete">
                <i class="fas fa-trash"></i>
            </button>
        </div>
    `;
    
    return div;
}

// Edit item
function editItem(itemId) {
    const category = Object.keys(items).find(cat => 
        items[cat].some(item => item.id === itemId)
    );
    
    if (!category) return;
    
    const item = items[category].find(item => item.id === itemId);
    if (!item) return;
    
    currentCategory = category;
    openAddItemModal(category);
    
    // Populate form with item data
    setTimeout(() => {
        populateItemForm(item);
    }, 100);
}

// Populate item form
function populateItemForm(item) {
    switch (item.category) {
        case 'Task':
            document.getElementById('itemText').value = item.text || '';
            document.getElementById('itemType').value = item.type || '';
            document.getElementById('itemNote').value = item.note || '';
            break;
        case 'Parts':
            document.getElementById('itemNumber').value = item.itemNumber || '';
            document.getElementById('partNumber').value = item.partNumber || '';
            document.getElementById('itemName').value = item.itemName || '';
            document.getElementById('quantity').value = item.quantity || '';
            document.getElementById('unit').value = item.unit || '';
            document.getElementById('itemCost').value = item.itemCost || '';
            document.getElementById('site').value = item.site || '';
            document.getElementById('itemNote').value = item.note || '';
            break;
        case 'Labor':
            document.getElementById('worker').value = item.worker || '';
            document.getElementById('code').value = item.code || '';
            document.getElementById('rate').value = item.rate || '';
            document.getElementById('hours').value = item.hours || '';
            document.getElementById('minutes').value = item.minutes || '';
            document.getElementById('itemNote').value = item.note || '';
            break;
        case 'Additional Cost':
            document.getElementById('description').value = item.description || '';
            document.getElementById('amount').value = item.amount || '';
            break;
        case 'Remarks':
            document.getElementById('description').value = item.description || '';
            break;
    }
}

// Delete item
function deleteItem(itemId) {
    if (confirm('Are you sure you want to delete this item?')) {
        Object.keys(items).forEach(category => {
            items[category] = items[category].filter(item => item.id !== itemId);
        });
        updateItemsDisplay();
    }
}

// Toggle section
function toggleSection(sectionId) {
    const content = document.getElementById(`${sectionId}Content`);
    const icon = document.getElementById(`${sectionId}Icon`);
    
    if (content.style.display === 'none') {
        content.style.display = 'block';
        icon.classList.remove('fa-chevron-down');
        icon.classList.add('fa-chevron-up');
    } else {
        content.style.display = 'none';
        icon.classList.remove('fa-chevron-up');
        icon.classList.add('fa-chevron-down');
    }
}

// Save work order
async function saveWorkOrder() {
    // Validate required fields
    if (!validateForm()) {
        return;
    }
    
    // Show PDF loading modal
    const pdfModal = new bootstrap.Modal(document.getElementById('pdfModal'));
    pdfModal.show();
    
    try {
        // Prepare form data
        const formData = new FormData(document.getElementById('workOrderForm'));
        
        // Add items data
        formData.append('items', JSON.stringify(items));
        
        // Submit form
        const response = await fetch('/maintenance/work-orders', {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });
        
        if (response.ok) {
            const result = await response.json();
            
            // Hide PDF modal
            pdfModal.hide();
            
            // Show success modal
            const successModal = new bootstrap.Modal(document.getElementById('successModal'));
            successModal.show();
            
            // Store work order ID for PDF viewing
            window.workOrderId = result.workOrderId;
        } else {
            throw new Error('Failed to create work order');
        }
    } catch (error) {
        console.error('Error saving work order:', error);
        pdfModal.hide();
        alert('Error creating work order: ' + error.message);
    }
}

// Validate form
function validateForm() {
    const title = document.getElementById('workOrderTitle').value.trim();
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const equipment = document.getElementById('selectedEquipmentId').value;
    
    if (!title) {
        alert('Job Card Title is required');
        return false;
    }
    
    if (!country) {
        alert('Country is required');
        return false;
    }
    
    if (!project) {
        alert('Project is required');
        return false;
    }
    
    if (!equipment) {
        alert('Equipment is required');
        return false;
    }
    
    return true;
}

// View PDF
function viewPDF() {
    if (window.workOrderId) {
        window.open(`/maintenance/work-orders/${window.workOrderId}/pdf`, '_blank');
    }
}
