@extends('welcome')

@section('title', 'Inspection KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="inspection-kpi-container">
    <!-- Loading Overlay - Matching Workorder Design -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon" aria-hidden="true"></div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Inspection Data...</div>
        </div>
    </div>
    
    <!-- Header - Matching Workorder Design -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Inspection KPI Dashboard</h1>
                    </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn-header">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                        </button>
                <button type="button" class="create-btn" id="refresh-data-btn-header">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                        </button>
                    </div>
                </div>
            </div>
    
    <div class="inspection-kpi-content">
        <div class="content-box">
            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="datewise">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                            <line x1="16" y1="2" x2="16" y2="6"/>
                            <line x1="8" y1="2" x2="8" y2="6"/>
                            <line x1="3" y1="10" x2="21" y2="10"/>
                        </svg>
                        Datewise
                    </button>
                    <button class="tab-btn" data-tab="inspection-class">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M9 12l2 2 4-4"/>
                            <path d="M21 12c-1 0-3-1-3-3s2-3 3-3 3 1 3 3-2 3-3 3"/>
                            <path d="M3 12c1 0 3-1 3-3s-2-3-3-3-3 1-3 3 2 3 3 3"/>
                            <path d="M12 3c0 1-1 3-3 3s-3-2-3-3 1-3 3-3 3 2 3 3"/>
                            <path d="M12 21c0-1 1-3 3-3s3 2 3 3-1 3-3 3-3-2-3-3"/>
                        </svg>
                        Inspection Classification
                    </button>
                    <button class="tab-btn" data-tab="by-inspector">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M16 21v-2a4 4 0 0 0-4-4H6a4 4 0 0 0-4 4v2"/>
                            <circle cx="9" cy="7" r="4"/>
                            <path d="M22 21v-2a4 4 0 0 0-3-3.87"/>
                            <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                        </svg>
                        By Inspector
                    </button>
                    {{-- <button class="tab-btn" data-tab="by-vehicle">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                            <path d="M14 11a5 5 0 0 0-10 0"/>
                        </svg>
                        By Vehicle
                    </button>
                    <button class="tab-btn" data-tab="vehicle-category">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M3 3h18v18H3zM9 9h6v6H9z"/>
                            <path d="M9 1v6"/>
                            <path d="M15 1v6"/>
                            <path d="M9 17v6"/>
                            <path d="M15 17v6"/>
                            <path d="M1 9h6"/>
                            <path d="M17 9h6"/>
                            <path d="M1 15h6"/>
                            <path d="M17 15h6"/>
                        </svg>
                        Vehicle Category
                    </button> --}}
                    <button class="tab-btn" data-tab="by-checklist">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M9 12l2 2 4-4"/>
                            <path d="M21 12c-1 0-3-1-3-3s2-3 3-3 3 1 3 3-2 3-3 3"/>
                            <path d="M3 12c1 0 3-1 3-3s-2-3-3-3-3 1-3 3 2 3 3 3"/>
                            <path d="M12 3c0 1-1 3-3 3s-3-2-3-3 1-3 3-3 3 2 3 3"/>
                            <path d="M12 21c0-1 1-3 3-3s3 2 3 3-1 3-3 3-3-2-3-3"/>
                        </svg>
                        By Checklist
                    </button>
                    <button class="tab-btn" data-tab="equipment">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                            <path d="M3 10h18"/>
                            <path d="M10 4v6"/>
                        </svg>
                        Equipment
                    </button>
                </div>
                <div class="tab-content">
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <!-- KPI Card Template -->
                            <div class="kpi-card" data-kpi="total-inspection">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M9 12l2 2 4-4"/>
                                        <path d="M21 12c-1 0-3-1-3-3s2-3 3-3 3 1 3 3-2 3-3 3"/>
                                        <path d="M3 12c1 0 3-1 3-3s-2-3-3-3-3 1-3 3 2 3 3 3"/>
                                        <path d="M12 3c0 1-1 3-3 3s-3-2-3-3 1-3 3-3 3 2 3 3"/>
                                        <path d="M12 21c0-1 1-3 3-3s3 2 3 3-1 3-3 3-3-2-3-3"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Inspection</h3>
                                    <div class="kpi-value">02</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="good-inspection">
                                <div class="kpi-icon success">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M20 6L9 17l-5-5"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Good Inspection</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="faulty-inspection">
                                <div class="kpi-icon warning">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"/>
                                        <line x1="12" y1="9" x2="12" y2="13"/>
                                        <line x1="12" y1="17" x2="12.01" y2="17"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Faulty Inspection</h3>
                                    <div class="kpi-value">02</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                        </div>
                        <div class="overview-bottom-section">
                            <div class="additional-kpi-cards">
                                <div class="kpi-card">
                                    <div class="kpi-icon info">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                            <polyline points="14,2 14,8 20,8"/>
                                            <line x1="16" y1="13" x2="8" y2="13"/>
                                            <line x1="16" y1="17" x2="8" y2="17"/>
                                            <polyline points="10,9 9,9 8,9"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Total Linked Workorder</h3>
                                        <div class="kpi-value">02</div>
                                        <div class="kpi-label">Count</div>
                                    </div>
                                </div>
                                
                                <div class="kpi-card">
                                    <div class="kpi-icon secondary">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                            <polyline points="14,2 14,8 20,8"/>
                                            <line x1="16" y1="13" x2="8" y2="13"/>
                                            <line x1="16" y1="17" x2="8" y2="17"/>
                                            <polyline points="10,9 9,9 8,9"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Custom Inspection Form</h3>
                                        <div class="kpi-value">00</div>
                                        <div class="kpi-label">Count</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="chart-section">
                                <div class="chart-card">
                                    <div class="chart-header">
                                        <h3>Corrective Action Status</h3>
                                    </div>
                                    <div class="chart-container">
                                        <canvas id="running"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div id="datewise" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Date wise Inspection Status Count</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="datawise"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Date wise Inspection Count</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="datawise-count"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="inspection-class" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Safe to Use</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="safe-use"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspection Count by Maintenance Priority</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="maintenance-priority"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspection Count by Vehicle Status</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="vehicle-status"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspection Count by Maintenance Required</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="maintenance-required"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Status By Overall Condition</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="overall"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="by-inspector" class="tab-pane">
                        <div class="filter-section">
                            <div class="filter-form">
                                <div class="form-group">
                                    <label for="inspector-select">Select Inspector:</label>
                                    <select class="form-select" id="inspector-select" name="inspector">
                                        <option value="">All Inspectors</option>
                                    </select>
                                </div>
                                <button class="btn-go">Go</button>
                            </div>
                        </div>
                        
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspection Count by Inspector</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="Inspector"></canvas>
                                </div>
                            </div>
                        </div>
                        
                        <div class="inspector-table-section" style="margin-top: 24px;">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspector Reports Details</h3>
                                </div>
                                <div class="table-container" style="overflow-x: auto; padding: 16px;">
                                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 16px;">
                                        <div style="display: flex; align-items: center; gap: 12px;">
                                            <label for="inspector-per-page" style="font-size: 14px; font-weight: 500; color: #495057;">Items per page:</label>
                                            <select id="inspector-per-page" style="padding: 6px 12px; border: 1px solid #dee2e6; border-radius: 4px; font-size: 14px; cursor: pointer;">
                                                <option value="5" selected>5</option>
                                                <option value="10">10</option>
                                                <option value="25">25</option>
                                                <option value="50">50</option>
                                                <option value="100">100</option>
                                            </select>
                                        </div>
                                        <div id="inspector-pagination-info" style="font-size: 14px; color: #6c757d; font-weight: 500;">
                                            Showing 0 - 0 of 0
                                        </div>
                                    </div>
                                    <table id="inspector-reports-table" class="data-table" style="width: 100%; border-collapse: collapse;">
                                        <thead>
                                            <tr style="background: #f8f9fa; border-bottom: 2px solid #dee2e6;">
                                                <th style="padding: 12px; text-align: center; font-weight: 600; color: #495057; width: 60px;">S.No</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Inspector Name</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Request Number</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Report ID</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Equipment Used</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Equipment Number</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Category</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Date</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Status</th>
                                            </tr>
                                        </thead>
                                        <tbody id="inspector-reports-table-body">
                                            <tr>
                                                <td colspan="9" style="padding: 24px; text-align: center; color: #6c757d;">
                                                    Loading inspector data...
                                                </td>
                                            </tr>
                                        </tbody>
                                    </table>
                                    <div id="inspector-pagination" style="display: flex; justify-content: center; align-items: center; gap: 8px; margin-top: 16px; padding: 16px;">
                                        <!-- Pagination buttons will be inserted here -->
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    {{-- <div id="by-vehicle" class="tab-pane">
                        <div class="filter-section">
                            <div class="filter-form">
                                <div class="form-group">
                                    <label for="vehicle-select">Select Particular Vehicle:</label>
                                    <select class="form-select" id="vehicle-select" name="vehicle">
                                        <option value="">Select Vehicle</option>
                                        <option value="vehicle1">Vehicle 1</option>
                                        <option value="vehicle2">Vehicle 2</option>
                                        <option value="vehicle3">Vehicle 3</option>
                                        <option value="vehicle4">Vehicle 4</option>
                                    </select>
                                </div>
                                <button class="btn-go">Go</button>
                            </div>
                        </div>
                        
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspection Count by Vehicle</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="by-vehicle"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="vehicle-category" class="tab-pane">
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspection Count by Vehicle Category</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="vehicle-cat"></canvas>
                                </div>
                            </div>
                        </div>
                    </div> --}}
                    
                    <div id="by-checklist" class="tab-pane">
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Inspection Count by Checklist Used</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="by-Checklist"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="equipment" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Equipment Usage Statistics</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="equipment-usage"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Equipment by Category</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="equipment-category"></canvas>
                                </div>
                            </div>
                        </div>
                        
                        <div class="equipment-table-section" style="margin-top: 24px;">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Equipment Usage Details</h3>
                                </div>
                                <div class="table-container" style="overflow-x: auto; padding: 16px;">
                                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 16px;">
                                        <div style="display: flex; align-items: center; gap: 12px;">
                                            <label for="equipment-per-page" style="font-size: 14px; font-weight: 500; color: #495057;">Items per page:</label>
                                            <select id="equipment-per-page" style="padding: 6px 12px; border: 1px solid #dee2e6; border-radius: 4px; font-size: 14px; cursor: pointer;">
                                                <option value="5" selected>5</option>
                                                <option value="10">10</option>
                                                <option value="25">25</option>
                                                <option value="50">50</option>
                                                <option value="100">100</option>
                                            </select>
                                        </div>
                                        <div id="equipment-pagination-info" style="font-size: 14px; color: #6c757d; font-weight: 500;">
                                            Showing 0 - 0 of 0
                                        </div>
                                    </div>
                                    <table id="equipment-table" class="data-table" style="width: 100%; border-collapse: collapse;">
                                        <thead>
                                            <tr style="background: #f8f9fa; border-bottom: 2px solid #dee2e6;">
                                                <th style="padding: 12px; text-align: center; font-weight: 600; color: #495057; width: 60px;">S.No</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Equipment Name</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Equipment Number</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Category</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Model</th>
                                                <th style="padding: 12px; text-align: center; font-weight: 600; color: #495057;">Usage Count</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Last Used</th>
                                                <th style="padding: 12px; text-align: left; font-weight: 600; color: #495057;">Status</th>
                                            </tr>
                                        </thead>
                                        <tbody id="equipment-table-body">
                                            <tr>
                                                <td colspan="8" style="padding: 24px; text-align: center; color: #6c757d;">
                                                    Loading equipment data...
                                                </td>
                                            </tr>
                                        </tbody>
                                    </table>
                                    <div id="equipment-pagination" style="display: flex; justify-content: center; align-items: center; gap: 8px; margin-top: 16px; padding: 16px;">
                                        <!-- Pagination buttons will be inserted here -->
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
    <script>
        function toggleSidebar() {
            const sidebar = document.querySelector('.sidebar');
            sidebar.classList.toggle('collapsed');
        }
    </script>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
<!-- Chart.js Library -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.js"></script>
    
<!-- SheetJS Library for Excel Export -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
<script>
    // MongoDB API Configuration
const MONGODB_INSPECTION_API = 'https://api.titandrillingzm.com:6004';
const adminId = "cahj3jJqe5Z16pQdLGSWrNgm84o1";

// Global variables
let allInspectionData = [];
let chartInstances = {};
let currentDateRange = 7; // Default to last 7 days
let customStartDate = null;
let customEndDate = null;

// Fetch inspection data from MongoDB
async function fetchInspectionData() {
    try {
        // Show loading state
        showLoadingState(true);
        
        console.log('🔄 Fetching inspection data from MongoDB...');
        
        // Fetch all inspections from MongoDB (with pagination to get all)
        const response = await fetch(`${MONGODB_INSPECTION_API}/inspections/all?page=1&limit=10000`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('📥 MongoDB response:', result);
        
        allInspectionData = [];
        
        if (result.success && result.data && Array.isArray(result.data)) {
            // Map MongoDB data structure to expected format
            result.data.forEach(inspection => {
                // Ensure inspection has required steps
                if (inspection && (inspection.step1 || inspection.step2 || inspection.step3)) {
                    allInspectionData.push({
                        id: inspection._id || inspection.id,
                        step1: inspection.step1 || {},
                        step2: inspection.step2 || {},
                        step3: inspection.step3 || {},
                        ...inspection
                    });
                }
            });
        }
        
        console.log('✅ Fetched inspections:', allInspectionData.length);
        
        if (allInspectionData.length === 0) {
            showNotification('No inspection data found', 'info');
        } else {
            showNotification(`Loaded ${allInspectionData.length} inspection(s)`, 'success');
        }
        
        updateDashboard();
        showLoadingState(false);
    } catch (error) {
        console.error('❌ Error fetching inspection data:', error);
        showNotification('Error fetching inspection data: ' + error.message, 'error');
        showLoadingState(false);
    }
}

// Show/hide loading state
function showLoadingState(isLoading) {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = isLoading ? 'flex' : 'none';
    }
}

// Filter data by date range
function filterDataByDateRange(days) {
    // Handle custom date range
    if (days === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate);
            start.setHours(0, 0, 0, 0);
            const end = new Date(customEndDate);
            end.setHours(23, 59, 59, 999);
            
            return allInspectionData.filter(inspection => {
                const createdAt = inspection.step1?.createdAt || inspection.step3?.createdAt || inspection.createdAt;
                if (!createdAt) return false;
                
                const inspectionDate = new Date(createdAt);
                return inspectionDate >= start && inspectionDate <= end;
            });
        }
        // If no custom dates selected, return all data
        return allInspectionData;
    }
    
    const now = new Date();
    const startDate = new Date(now);
    
    if (days === 'today') {
        startDate.setHours(0, 0, 0, 0);
    } else {
        startDate.setDate(now.getDate() - parseInt(days));
        startDate.setHours(0, 0, 0, 0);
    }
    
    return allInspectionData.filter(inspection => {
        const createdAt = inspection.step1?.createdAt || inspection.step3?.createdAt || inspection.createdAt;
        if (!createdAt) return false;
        
        const inspectionDate = new Date(createdAt);
        return inspectionDate >= startDate && inspectionDate <= now;
    });
}

// Categorize inspections as good or faulty
function categorizeInspection(inspection) {
    const step2 = inspection.step2 || {};
    const step3 = inspection.step3 || {};
    
    const faultyItems = step2.faultyItems || step2.faultyItemCount || 0;
    const safeToUse = step3.safeToUse;
    const overallCondition = step3.overallCondition;
    const equipmentStatus = step3.equipmentStatus || step3.status;
    
    // Consider inspection "good" if:
    // - No faulty items AND safe to use
    // - OR equipment status is "Available" or "Operational"
    if (faultyItems === 0 && safeToUse === 'Yes') {
        return 'good';
    }
    if (equipmentStatus === 'Available' || equipmentStatus === 'Operational') {
        return 'good';
    }
    if (safeToUse === 'No' || equipmentStatus === 'Maintenance Required') {
        return 'faulty';
    }
    
    // Default to faulty if unsure
    return faultyItems > 0 ? 'faulty' : 'good';
}

    // Update KPI cards
function updateKPICards(filteredData) {
    const totalInspections = filteredData.length;
    const goodInspections = filteredData.filter(i => categorizeInspection(i) === 'good').length;
    const faultyInspections = filteredData.filter(i => categorizeInspection(i) === 'faulty').length;
    
    // Count linked workorders (inspections with faulty items that require maintenance)
    const linkedWorkorders = filteredData.filter(i => {
        const step3 = i.step3 || {};
        return step3.equipmentStatus === 'Maintenance Required' || step3.status === 'Maintenance Required';
    }).length;
    
    // Count custom inspection forms
    const customForms = filteredData.filter(i => {
        const checklistId = i.step1?.selectedChecklistId || i.step1?.checklistId;
        return checklistId && (checklistId.includes('CUSTOM') || checklistId.includes('custom'));
    }).length;
    
    // Update card values
    document.querySelector('[data-kpi="total-inspection"] .kpi-value').textContent = String(totalInspections).padStart(2, '0');
    document.querySelector('[data-kpi="good-inspection"] .kpi-value').textContent = String(goodInspections).padStart(2, '0');
    document.querySelector('[data-kpi="faulty-inspection"] .kpi-value').textContent = String(faultyInspections).padStart(2, '0');
    
    const workorderCards = document.querySelectorAll('.additional-kpi-cards .kpi-card');
    if (workorderCards[0]) {
        workorderCards[0].querySelector('.kpi-value').textContent = String(linkedWorkorders).padStart(2, '0');
    }
    if (workorderCards[1]) {
        workorderCards[1].querySelector('.kpi-value').textContent = String(customForms).padStart(2, '0');
    }
}

// Update all dashboard components
function updateDashboard() {
    const filteredData = filterDataByDateRange(currentDateRange);
    
    updateKPICards(filteredData);
    updateCorrectiveActionChart(filteredData);
    updateDatewiseCharts(filteredData);
    updateInspectionClassificationCharts(filteredData);
    updateInspectorChart(filteredData);
    // updateVehicleChart(filteredData);
    // updateVehicleCategoryChart(filteredData);
    updateChecklistChart(filteredData);
    updateEquipmentChart(filteredData);
    
    // Populate dropdowns with all data
    populateVehicleDropdowns(allInspectionData);
    populateInspectorDropdown(allInspectionData);
}

// Optimized Chart Initialization
(function() {
    'use strict';

    // Chart configuration
    const CHART_CONFIG = {
        defaultOptions: {
            legend: { display: false },
            scales: {
                yAxes: [{ ticks: { beginAtZero: true } }]
            }
        }
    };

    // Chart data configurations (will be replaced with dynamic data)
    const charts = [
        {
            id: "running",
            title: "Corrective Action Status",
        data: {
                labels: ["Taken", "Partially", "Not Taken", "NA"],
            datasets: [{
                    backgroundColor: ["#0055A5", "#0055A1", "#DC0D0E", "#0055A1"],
                    data: [0, 0, 2, 0]
                }]
            }
        },
        {
            id: "datawise",
            title: "Date wise Inspection Status Count",
        data: {
                labels: ["19/02/2025", "20/02/2025", "21/02/2025", "22/02/2025", "23/02/2025", "24/02/2025", "25/02/2025", "26/02/2025"],
            datasets: [{
                    backgroundColor: ["#0055A5", "#0055A1", "#DC0D0E", "#0055A1"],
                    data: [0, 0, 0, 0, 0, 0, 0, 0]
                }]
            }
        },
        {
            id: "datawise-count",
            title: "Date wise Inspection Count",
        data: {
                labels: ["19/02/2025", "20/02/2025", "21/02/2025", "22/02/2025", "23/02/2025", "24/02/2025", "25/02/2025", "26/02/2025"],
            datasets: [{
                    backgroundColor: ["#0055A5", "#0055A1", "#DC0D0E", "#0055A1"],
                    data: [0, 0, 0, 0, 0, 0, 0, 0]
                }]
            }
        },
        {
            id: "safe-use",
            title: "Safe to Use",
        data: {
                labels: ["0", "2", "4"],
            datasets: [{
                    backgroundColor: ["#0055A5", "#5899DA"],
                    data: [0, 1]
                }]
            }
        },
        {
            id: "maintenance-priority",
            title: "Inspection Count by Maintenance Priority",
        data: {
                labels: ["0", "Low", "Medium"],
            datasets: [{
                    backgroundColor: ["#945ECF", "#945ECF", "#945ECF"],
                    data: [0, 1, 1, 2]
                }]
            }
        },
        {
            id: "vehicle-status",
            title: "Inspection Count by Vehicle Status",
        data: {
                labels: ["0", "Available"],
            datasets: [{
                    backgroundColor: ["#0055A5", "#DC0D0E"],
                    data: [0, 1, 2, 3]
                }]
            }
        },
        {
            id: "maintenance-required",
            title: "Inspection Count by Maintenance Required",
        data: {
                labels: ["0", "1"],
            datasets: [{
                    backgroundColor: ["#945ECF", "#29A0AD", "#945ECF"],
                    data: [0, 1, 2, 3]
                }]
            }
        },
        {
            id: "overall",
            title: "Status By Overall Condition",
        data: {
                labels: ["0", "A little wear"],
            datasets: [{
                    backgroundColor: ["#945ECF", "#E8743B", "#945ECF"],
                    data: [0, 1, 2, 3]
                }]
            }
        },
        {
            id: "Inspector",
            title: "Inspection Count by Inspector",
        data: {
                labels: ["0", "User Name"],
            datasets: [{
                    backgroundColor: ["#945ECF", "#4C56DC", "#945ECF"],
                    data: [0, 1, 2, 3]
                }]
            }
        },
        {
            id: "by-vehicle",
            title: "Inspection Count by Vehicle",
        data: {
                labels: ["0", "Test"],
            datasets: [{
                    backgroundColor: ["#945ECF", "#E53839", "#945ECF"],
                    data: [0, 1, 2, 3]
                }]
            }
        },
        {
            id: "vehicle-cat",
            title: "Inspection Count by Vehicle Category",
            data: {
                labels: ["0", "Test"],
            datasets: [{
                    backgroundColor: ["#945ECF", "#D65F5F", "#945ECF"],
                    data: [0, 1, 2, 3]
                }]
            }
        },
        {
            id: "by-Checklist",
            title: "Inspection Count by Checklist Used",
            data: {
                labels: ["0", "120 Days Interval Truck/Tractor Preventive Maintenance"],
                datasets: [{
                    backgroundColor: ["#945ECF", "#B43B96", "#945ECF"],
                    data: [0, 1, 2, 3]
                }]
            }
        }
    ];

    // Initialize all charts
    function initCharts() {
        charts.forEach(chart => {
            const config = {
                type: "bar",
                data: chart.data,
                options: {
                    ...CHART_CONFIG.defaultOptions,
            title: {
                display: true,
                        text: chart.title
                    }
                }
            };
            
            new Chart(chart.id, config);
        });
    }

    // Initialize charts when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initCharts);
    } else {
        initCharts();
    }
})();

// Chart update functions
function updateCorrectiveActionChart(data) {
    const correctiveActions = {
        'Taken': 0,
        'Partially': 0,
        'Not Taken': 0,
        'NA': 0
    };
    
    data.forEach(inspection => {
        const step3 = inspection.step3 || {};
        const additionalNote = step3.additionalNote || '';
        
        // Analyze corrective action based on notes and status
        if (step3.safeToUse === 'Yes') {
            correctiveActions['NA']++;
        } else if (step3.equipmentStatus === 'Maintenance Required') {
            correctiveActions['Not Taken']++;
        } else {
            correctiveActions['Partially']++;
        }
    });
    
    updateChart('running', ['Taken', 'Partially', 'Not Taken', 'NA'], 
                Object.values(correctiveActions),
                ['#0055A5', '#FFA500', '#DC0D0E', '#808080']);
}

function updateDatewiseCharts(data) {
    const dateRange = currentDateRange === 'today' ? 1 : parseInt(currentDateRange) || 7;
    const dates = [];
    const counts = {};
    const statusCounts = {};
    
    // Generate date labels
    for (let i = dateRange - 1; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        const dateStr = date.toLocaleDateString('en-GB');
        dates.push(dateStr);
        counts[dateStr] = 0;
        statusCounts[dateStr] = { good: 0, faulty: 0 };
    }
    
    // Count inspections by date
    data.forEach(inspection => {
        const createdAt = inspection.step1?.createdAt || inspection.step3?.createdAt || inspection.createdAt;
        if (createdAt) {
            const date = new Date(createdAt).toLocaleDateString('en-GB');
            if (counts.hasOwnProperty(date)) {
                counts[date]++;
                const category = categorizeInspection(inspection);
                statusCounts[date][category]++;
            }
        }
    });
    
    // Update count chart
    updateChart('datawise-count', dates, dates.map(d => counts[d]),
                ['#0055A5', '#5899DA', '#DC0D0E', '#945ECF']);
    
    // Update status chart (stacked)
    const ctx = document.getElementById('datawise');
    if (ctx && chartInstances['datawise']) {
        chartInstances['datawise'].destroy();
    }
    
    chartInstances['datawise'] = new Chart('datawise', {
        type: 'bar',
            data: {
            labels: dates,
            datasets: [
                {
                    label: 'Good',
                    backgroundColor: '#10b981',
                    data: dates.map(d => statusCounts[d].good)
                },
                {
                    label: 'Faulty',
                    backgroundColor: '#DC0D0E',
                    data: dates.map(d => statusCounts[d].faulty)
                }
            ]
        },
        options: {
            legend: { display: true },
            scales: {
                xAxes: [{ stacked: true }],
                yAxes: [{ stacked: true, ticks: { beginAtZero: true } }]
            },
            title: {
                display: true,
                text: 'Date wise Inspection Status Count'
            }
        }
    });
}

function updateInspectionClassificationCharts(data) {
    // Safe to Use
    const safeToUse = { 'Yes': 0, 'No': 0 };
    data.forEach(i => {
        const safe = i.step3?.safeToUse || 'No';
        safeToUse[safe]++;
    });
    updateChart('safe-use', ['Yes', 'No'], Object.values(safeToUse), ['#10b981', '#DC0D0E']);
    
    // Maintenance Priority
    const priorities = {};
    data.forEach(i => {
        const priority = i.step3?.priority || 'Low';
        priorities[priority] = (priorities[priority] || 0) + 1;
    });
    updateChart('maintenance-priority', Object.keys(priorities), Object.values(priorities),
                ['#945ECF', '#E8743B', '#DC0D0E']);
    
    // Vehicle Status
    const statuses = {};
    data.forEach(i => {
        const status = i.step3?.equipmentStatus || 'Available';
        statuses[status] = (statuses[status] || 0) + 1;
    });
    updateChart('vehicle-status', Object.keys(statuses), Object.values(statuses),
                ['#0055A5', '#DC0D0E', '#FFA500']);
    
    // Maintenance Required
    const maintenanceReq = { 'Yes': 0, 'No': 0 };
    data.forEach(i => {
        const required = i.step3?.equipmentStatus === 'Maintenance Required' ? 'Yes' : 'No';
        maintenanceReq[required]++;
    });
    updateChart('maintenance-required', Object.keys(maintenanceReq), Object.values(maintenanceReq),
                ['#DC0D0E', '#10b981']);
    
    // Overall Condition
    const conditions = {};
    data.forEach(i => {
        const condition = i.step3?.overallCondition || 'Unknown';
        conditions[condition] = (conditions[condition] || 0) + 1;
    });
    updateChart('overall', Object.keys(conditions), Object.values(conditions),
                ['#10b981', '#FFA500', '#DC0D0E']);
}

function extractInspectorName(step1 = {}) {
    const candidates = [];

    if (!step1 || typeof step1 !== 'object') {
        return 'Unknown Inspector';
    }

    const { inspector, inspectorName, inspectorDetails, operator, performedBy, createdByName, createdBy } = step1;

    if (typeof inspector === 'string' && inspector.trim()) {
        candidates.push(inspector.trim());
    } else if (Array.isArray(inspector) && inspector.length) {
        inspector.filter(Boolean).forEach(entry => {
            if (typeof entry === 'string' && entry.trim()) {
                candidates.push(entry.trim());
            } else if (entry && typeof entry === 'object') {
                const arrayCandidates = [
                    entry.name,
                    entry.fullName,
                    entry.displayName,
                    entry.email
                ];
                candidates.push(...arrayCandidates.filter(Boolean));
            }
        });
    } else if (inspector && typeof inspector === 'object') {
        const objectCandidates = [
            inspector.name,
            inspector.fullName,
            inspector.displayName,
            inspector.firstName && inspector.lastName ? `${inspector.firstName} ${inspector.lastName}`.trim() : null,
            inspector.firstName,
            inspector.lastName,
            inspector.email
        ];
        candidates.push(...objectCandidates.filter(Boolean));
    }

    if (inspectorName) candidates.push(inspectorName);

    if (inspectorDetails && typeof inspectorDetails === 'object') {
        const detailsCandidates = [
            inspectorDetails.name,
            inspectorDetails.fullName,
            inspectorDetails.displayName,
            inspectorDetails.email
        ];
        candidates.push(...detailsCandidates.filter(Boolean));
    }

    if (operator) candidates.push(operator);
    if (performedBy) candidates.push(performedBy);
    if (createdByName) candidates.push(createdByName);
    if (typeof createdBy === 'string') candidates.push(createdBy);

    const validName = candidates.find(name => typeof name === 'string' && name.trim());
    return validName ? validName.trim() : 'Unknown Inspector';
}

function extractVehicleName(step1 = {}) {
    if (!step1 || typeof step1 !== 'object') return 'Unknown';
    const selected = step1.selectedEquipment || step1.equipment || step1.vehicleDetails;
    if (selected) {
        const { equipmentName, name, vehicleNumber, vehicleName } = selected;
        return equipmentName || name || vehicleNumber || vehicleName || 'Unknown';
    }
    return step1.vehicleNumber || step1.vehicleName || step1.vehicle || 'Unknown';
}

function updateInspectorChart(data, selectedInspector = null) {
    let filteredData = data;
    if (selectedInspector) {
        const normalizedSelection = selectedInspector.toLowerCase();
        filteredData = data.filter(i => extractInspectorName(i.step1).toLowerCase() === normalizedSelection);
    }
    
    const inspectors = new Map();
    filteredData.forEach(i => {
        const name = extractInspectorName(i.step1);
        const key = name.toLowerCase();
        const current = inspectors.get(key);
        if (current) {
            current.count += 1;
        } else {
            inspectors.set(key, { label: name, count: 1 });
        }
    });

    const labels = Array.from(inspectors.values()).map(entry => entry.label);
    const counts = Array.from(inspectors.values()).map(entry => entry.count);
    
    updateChart('Inspector', labels, counts,
                ['#4C56DC', '#945ECF', '#E8743B', '#0055A5']);
    
    // Reset pagination to first page when filtering
    inspectorCurrentPage = 1;
    
    // Update inspector reports table
    updateInspectorReportsTable(filteredData);
}

// Inspector pagination state
let inspectorCurrentPage = 1;
let inspectorPerPage = 5;
let inspectorAllData = [];

function updateInspectorReportsTable(data) {
    const tableBody = document.getElementById('inspector-reports-table-body');
    if (!tableBody) return;
    
    // Store all inspector data for pagination
    inspectorAllData = data || [];
    
    if (!data || data.length === 0) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="9" style="padding: 24px; text-align: center; color: #6c757d;">
                    No inspector reports data available
                </td>
            </tr>
        `;
        renderInspectorPagination(0);
        updateInspectorPaginationInfo(0, 0, 0);
        return;
    }
    
    // Sort by date (newest first)
    const sortedData = [...inspectorAllData].sort((a, b) => {
        const dateA = new Date(a.step1?.createdAt || a.createdAt || 0);
        const dateB = new Date(b.step1?.createdAt || b.createdAt || 0);
        return dateB - dateA;
    });
    
    // Calculate pagination
    const totalPages = Math.ceil(sortedData.length / inspectorPerPage);
    const startIndex = (inspectorCurrentPage - 1) * inspectorPerPage;
    const endIndex = Math.min(startIndex + inspectorPerPage, sortedData.length);
    const paginatedData = sortedData.slice(startIndex, endIndex);
    
    // Render table rows with inspector details
    tableBody.innerHTML = paginatedData.map((inspection, index) => {
        const serialNumber = startIndex + index + 1;
        const step1 = inspection.step1 || {};
        const step3 = inspection.step3 || {};
        const equipment = step1.selectedEquipment || step1.equipment || {};
        
        const inspectorName = extractInspectorName(step1);
        const requestNumber = step1.requestNumber || inspection.requestNumber || inspection.id || 'N/A';
        const reportId = inspection._id || inspection.id || 'N/A';
        const equipmentName = equipment.equipmentName || equipment.name || 'N/A';
        const equipmentNumber = equipment.vehicleNumber || equipment.equipmentNumber || equipment.id || 'N/A';
        const category = equipment.mainCategory || equipment.equipmentCategory || equipment.category || 'N/A';
        
        const createdAt = step1.createdAt || inspection.createdAt || new Date();
        const formattedDate = new Date(createdAt).toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric'
        });
        
        // Determine status
        const status = step3.equipmentStatus || step3.safeToUse || 'Unknown';
        const statusColor = status === 'Available' || status === 'Yes' || status === 'Safe'
            ? '#10b981'
            : status === 'Maintenance Required' || status === 'No' || status === 'Unsafe'
            ? '#DC0D0E'
            : '#6c757d';
        
        return `
            <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 12px; text-align: center; color: #495057; font-weight: 500;">${serialNumber}</td>
                <td style="padding: 12px; color: #212529; font-weight: 500;">${escapeHtml(inspectorName)}</td>
                <td style="padding: 12px; color: #0055A5; font-weight: 500;">${escapeHtml(requestNumber)}</td>
                <td style="padding: 12px; color: #495057; font-size: 13px;">${escapeHtml(reportId.substring(0, 20))}${reportId.length > 20 ? '...' : ''}</td>
                <td style="padding: 12px; color: #212529; font-weight: 500;">${escapeHtml(equipmentName)}</td>
                <td style="padding: 12px; color: #495057;">${escapeHtml(equipmentNumber)}</td>
                <td style="padding: 12px; color: #495057;">${escapeHtml(category)}</td>
                <td style="padding: 12px; color: #6c757d; font-size: 13px;">${formattedDate}</td>
                <td style="padding: 12px;">
                    <span style="display: inline-block; padding: 4px 8px; border-radius: 4px; background: ${statusColor}20; color: ${statusColor}; font-size: 12px; font-weight: 500;">
                        ${escapeHtml(status)}
                    </span>
                </td>
            </tr>
        `;
    }).join('');
    
    // Render pagination
    renderInspectorPagination(totalPages);
    updateInspectorPaginationInfo(startIndex + 1, endIndex, sortedData.length);
}

function renderInspectorPagination(totalPages) {
    const paginationContainer = document.getElementById('inspector-pagination');
    if (!paginationContainer) return;
    
    if (totalPages <= 1) {
        paginationContainer.innerHTML = '';
        return;
    }
    
    let paginationHTML = '';
    
    // Previous button
    paginationHTML += `
        <button id="inspector-prev-btn" class="inspector-pagination-btn" 
                style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px; ${inspectorCurrentPage === 1 ? 'opacity: 0.5; cursor: not-allowed;' : ''}"
                ${inspectorCurrentPage === 1 ? 'disabled' : ''}>
            Previous
        </button>
    `;
    
    // Page numbers
    const maxVisiblePages = 5;
    let startPage = Math.max(1, inspectorCurrentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
    
    if (endPage - startPage < maxVisiblePages - 1) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }
    
    if (startPage > 1) {
        paginationHTML += `
            <button class="inspector-pagination-btn inspector-page-btn" data-page="1"
                    style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px;">
                1
            </button>
        `;
        if (startPage > 2) {
            paginationHTML += `<span style="padding: 8px 4px; color: #6c757d;">...</span>`;
        }
    }
    
    for (let i = startPage; i <= endPage; i++) {
        paginationHTML += `
            <button class="inspector-pagination-btn inspector-page-btn ${i === inspectorCurrentPage ? 'active' : ''}" 
                    data-page="${i}"
                    style="padding: 8px 12px; border: 1px solid #dee2e6; background: ${i === inspectorCurrentPage ? '#0055A5' : 'white'}; color: ${i === inspectorCurrentPage ? 'white' : '#495057'}; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: ${i === inspectorCurrentPage ? '600' : '400'};">
                ${i}
            </button>
        `;
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            paginationHTML += `<span style="padding: 8px 4px; color: #6c757d;">...</span>`;
        }
        paginationHTML += `
            <button class="inspector-pagination-btn inspector-page-btn" data-page="${totalPages}"
                    style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px;">
                ${totalPages}
            </button>
        `;
    }
    
    // Next button
    paginationHTML += `
        <button id="inspector-next-btn" class="inspector-pagination-btn"
                style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px; ${inspectorCurrentPage === totalPages ? 'opacity: 0.5; cursor: not-allowed;' : ''}"
                ${inspectorCurrentPage === totalPages ? 'disabled' : ''}>
            Next
        </button>
    `;
    
    paginationContainer.innerHTML = paginationHTML;
    
    // Attach event listeners
    attachInspectorPaginationListeners();
}

function attachInspectorPaginationListeners() {
    // Previous button
    const prevBtn = document.getElementById('inspector-prev-btn');
    if (prevBtn) {
        prevBtn.addEventListener('click', () => {
            if (inspectorCurrentPage > 1) {
                inspectorCurrentPage--;
                updateInspectorReportsTable(inspectorAllData);
            }
        });
    }
    
    // Next button
    const nextBtn = document.getElementById('inspector-next-btn');
    if (nextBtn) {
        nextBtn.addEventListener('click', () => {
            const totalPages = Math.ceil(inspectorAllData.length / inspectorPerPage);
            if (inspectorCurrentPage < totalPages) {
                inspectorCurrentPage++;
                updateInspectorReportsTable(inspectorAllData);
            }
        });
    }
    
    // Page number buttons
    document.querySelectorAll('.inspector-page-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const page = parseInt(btn.getAttribute('data-page'));
            if (page && page !== inspectorCurrentPage) {
                inspectorCurrentPage = page;
                updateInspectorReportsTable(inspectorAllData);
            }
        });
    });
    
    // Items per page selector
    const perPageSelect = document.getElementById('inspector-per-page');
    if (perPageSelect) {
        perPageSelect.addEventListener('change', (e) => {
            inspectorPerPage = parseInt(e.target.value);
            inspectorCurrentPage = 1; // Reset to first page
            updateInspectorReportsTable(inspectorAllData);
        });
    }
}

function updateInspectorPaginationInfo(start, end, total) {
    const infoElement = document.getElementById('inspector-pagination-info');
    if (infoElement) {
        if (total === 0) {
            infoElement.textContent = 'Showing 0 - 0 of 0';
        } else {
            infoElement.textContent = `Showing ${start} - ${end} of ${total}`;
        }
    }
}

function updateVehicleChart(data, selectedVehicle = null) {
    let filteredData = data;
    if (selectedVehicle) {
        filteredData = data.filter(i => {
            const vehicleName = extractVehicleName(i.step1);
            return vehicleName === selectedVehicle;
        });
    }
    
    const vehicles = {};
    filteredData.forEach(i => {
        const vehicle = extractVehicleName(i.step1) || 'Unknown';
        vehicles[vehicle] = (vehicles[vehicle] || 0) + 1;
    });
    
    updateChart('by-vehicle', Object.keys(vehicles), Object.values(vehicles),
                ['#E53839', '#945ECF', '#0055A5', '#FFA500']);
}

function updateVehicleCategoryChart(data) {
    const categories = {};
    data.forEach(i => {
        const equipment = i.step1?.selectedEquipment || i.step1?.equipment || {};
        const category = equipment.equipmentCategory || equipment.category || 'Unknown';
        categories[category] = (categories[category] || 0) + 1;
    });
    
    updateChart('vehicle-cat', Object.keys(categories), Object.values(categories),
                ['#D65F5F', '#945ECF', '#0055A5', '#FFA500']);
}

function updateChecklistChart(data) {
    const checklists = {};
    data.forEach(i => {
        const checklist = i.step1?.checklistTitle || i.step1?.checklist?.title || 'Unknown';
        checklists[checklist] = (checklists[checklist] || 0) + 1;
    });
    
    updateChart('by-Checklist', Object.keys(checklists), Object.values(checklists),
                ['#B43B96', '#945ECF', '#0055A5', '#FFA500']);
}

function updateEquipmentChart(data) {
    // Equipment Usage Statistics (by equipment name)
    const equipmentUsage = new Map();
    
    data.forEach(i => {
        const step1 = i.step1 || {};
        const equipment = step1.selectedEquipment || step1.equipment || {};
        const equipmentName = equipment.equipmentName || equipment.name || 'Unknown';
        const equipmentNumber = equipment.vehicleNumber || equipment.equipmentNumber || equipment.id || 'N/A';
        const category = equipment.mainCategory || equipment.equipmentCategory || equipment.category || 'Unknown';
        const model = equipment.model || 'N/A';
        const status = equipment.status || step1.equipmentStatus || 'Unknown';
        
        const createdAt = step1.createdAt || i.createdAt || new Date();
        const lastUsed = new Date(createdAt);
        
        // Create unique key for equipment
        const equipmentKey = `${equipmentName}_${equipmentNumber}`;
        
        if (equipmentUsage.has(equipmentKey)) {
            const current = equipmentUsage.get(equipmentKey);
            current.count += 1;
            // Update last used date if this inspection is more recent
            if (lastUsed > new Date(current.lastUsed)) {
                current.lastUsed = lastUsed.toISOString();
            }
            // Update status if this is more recent
            if (status && status !== 'Unknown') {
                current.status = status;
            }
        } else {
            equipmentUsage.set(equipmentKey, {
                name: equipmentName,
                number: equipmentNumber,
                category: category,
                model: model,
                status: status,
                count: 1,
                lastUsed: lastUsed.toISOString()
            });
        }
    });
    
    // Sort by last used date (newest first - descending)
    const sortedEquipment = Array.from(equipmentUsage.values()).sort((a, b) => {
        const dateA = new Date(a.lastUsed);
        const dateB = new Date(b.lastUsed);
        return dateB - dateA; // Newest first
    });
    
    // Get top 10 for chart (by usage count for chart)
    const topEquipmentByCount = Array.from(equipmentUsage.values())
        .sort((a, b) => b.count - a.count)
        .slice(0, 10);
    const equipmentLabels = topEquipmentByCount.map(eq => eq.name);
    const equipmentCounts = topEquipmentByCount.map(eq => eq.count);
    
    // Update equipment usage chart
    updateChart('equipment-usage', equipmentLabels.length > 0 ? equipmentLabels : ['No Data'], 
                equipmentCounts.length > 0 ? equipmentCounts : [0],
                ['#0055A5', '#E8743B', '#945ECF', '#10b981', '#FFA500', '#DC0D0E', '#4C56DC', '#B43B96', '#E53839', '#5899DA']);
    
    // Equipment by Category
    const categoryStats = new Map();
    Array.from(equipmentUsage.values()).forEach(eq => {
        const cat = eq.category || 'Unknown';
        if (categoryStats.has(cat)) {
            categoryStats.set(cat, categoryStats.get(cat) + eq.count);
        } else {
            categoryStats.set(cat, eq.count);
        }
    });
    
    const categoryLabels = Array.from(categoryStats.keys());
    const categoryCounts = Array.from(categoryStats.values());
    
    updateChart('equipment-category', categoryLabels.length > 0 ? categoryLabels : ['No Data'],
                categoryCounts.length > 0 ? categoryCounts : [0],
                ['#0055A5', '#945ECF', '#E8743B', '#10b981', '#FFA500', '#DC0D0E']);
    
    // Update equipment table (sorted by last used, newest first)
    updateEquipmentTable(sortedEquipment);
}

// Equipment pagination state
let equipmentCurrentPage = 1;
let equipmentPerPage = 5;
let equipmentAllData = [];

function updateEquipmentTable(equipmentList) {
    const tableBody = document.getElementById('equipment-table-body');
    if (!tableBody) return;
    
    // Store all equipment data for pagination
    equipmentAllData = equipmentList || [];
    
    if (!equipmentList || equipmentList.length === 0) {
        tableBody.innerHTML = `
            <tr>
                <td colspan="8" style="padding: 24px; text-align: center; color: #6c757d;">
                    No equipment data available
                </td>
            </tr>
        `;
        renderEquipmentPagination(0);
        updateEquipmentPaginationInfo(0, 0, 0);
        return;
    }
    
    // Calculate pagination
    const totalPages = Math.ceil(equipmentAllData.length / equipmentPerPage);
    const startIndex = (equipmentCurrentPage - 1) * equipmentPerPage;
    const endIndex = Math.min(startIndex + equipmentPerPage, equipmentAllData.length);
    const paginatedData = equipmentAllData.slice(startIndex, endIndex);
    
    // Render table rows with S.No
    tableBody.innerHTML = paginatedData.map((eq, index) => {
        const serialNumber = startIndex + index + 1;
        const lastUsedDate = new Date(eq.lastUsed);
        const formattedDate = lastUsedDate.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric'
        });
        
        const statusColor = eq.status === 'Available' || eq.status === 'In Use' 
            ? '#10b981' 
            : eq.status === 'Maintenance Required' || eq.status === 'Out of Service'
            ? '#DC0D0E'
            : '#6c757d';
        
        return `
            <tr style="border-bottom: 1px solid #dee2e6;">
                <td style="padding: 12px; text-align: center; color: #495057; font-weight: 500;">${serialNumber}</td>
                <td style="padding: 12px; color: #212529; font-weight: 500;">${escapeHtml(eq.name)}</td>
                <td style="padding: 12px; color: #495057;">${escapeHtml(eq.number)}</td>
                <td style="padding: 12px; color: #495057;">${escapeHtml(eq.category)}</td>
                <td style="padding: 12px; color: #495057;">${escapeHtml(eq.model)}</td>
                <td style="padding: 12px; text-align: center; color: #0055A5; font-weight: 600;">${eq.count}</td>
                <td style="padding: 12px; color: #6c757d; font-size: 13px;">${formattedDate}</td>
                <td style="padding: 12px;">
                    <span style="display: inline-block; padding: 4px 8px; border-radius: 4px; background: ${statusColor}20; color: ${statusColor}; font-size: 12px; font-weight: 500;">
                        ${escapeHtml(eq.status)}
                    </span>
                </td>
            </tr>
        `;
    }).join('');
    
    // Render pagination
    renderEquipmentPagination(totalPages);
    updateEquipmentPaginationInfo(startIndex + 1, endIndex, equipmentAllData.length);
}

function renderEquipmentPagination(totalPages) {
    const paginationContainer = document.getElementById('equipment-pagination');
    if (!paginationContainer) return;
    
    if (totalPages <= 1) {
        paginationContainer.innerHTML = '';
        return;
    }
    
    let paginationHTML = '';
    
    // Previous button
    paginationHTML += `
        <button id="equipment-prev-btn" class="equipment-pagination-btn" 
                style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px; ${equipmentCurrentPage === 1 ? 'opacity: 0.5; cursor: not-allowed;' : ''}"
                ${equipmentCurrentPage === 1 ? 'disabled' : ''}>
            Previous
        </button>
    `;
    
    // Page numbers
    const maxVisiblePages = 5;
    let startPage = Math.max(1, equipmentCurrentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
    
    if (endPage - startPage < maxVisiblePages - 1) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }
    
    if (startPage > 1) {
        paginationHTML += `
            <button class="equipment-pagination-btn equipment-page-btn" data-page="1"
                    style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px;">
                1
            </button>
        `;
        if (startPage > 2) {
            paginationHTML += `<span style="padding: 8px 4px; color: #6c757d;">...</span>`;
        }
    }
    
    for (let i = startPage; i <= endPage; i++) {
        paginationHTML += `
            <button class="equipment-pagination-btn equipment-page-btn ${i === equipmentCurrentPage ? 'active' : ''}" 
                    data-page="${i}"
                    style="padding: 8px 12px; border: 1px solid #dee2e6; background: ${i === equipmentCurrentPage ? '#0055A5' : 'white'}; color: ${i === equipmentCurrentPage ? 'white' : '#495057'}; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: ${i === equipmentCurrentPage ? '600' : '400'};">
                ${i}
            </button>
        `;
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            paginationHTML += `<span style="padding: 8px 4px; color: #6c757d;">...</span>`;
        }
        paginationHTML += `
            <button class="equipment-pagination-btn equipment-page-btn" data-page="${totalPages}"
                    style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px;">
                ${totalPages}
            </button>
        `;
    }
    
    // Next button
    paginationHTML += `
        <button id="equipment-next-btn" class="equipment-pagination-btn"
                style="padding: 8px 12px; border: 1px solid #dee2e6; background: white; border-radius: 4px; cursor: pointer; font-size: 14px; ${equipmentCurrentPage === totalPages ? 'opacity: 0.5; cursor: not-allowed;' : ''}"
                ${equipmentCurrentPage === totalPages ? 'disabled' : ''}>
            Next
        </button>
    `;
    
    paginationContainer.innerHTML = paginationHTML;
    
    // Attach event listeners
    attachEquipmentPaginationListeners();
}

function attachEquipmentPaginationListeners() {
    // Previous button
    const prevBtn = document.getElementById('equipment-prev-btn');
    if (prevBtn) {
        prevBtn.addEventListener('click', () => {
            if (equipmentCurrentPage > 1) {
                equipmentCurrentPage--;
                updateEquipmentTable(equipmentAllData);
            }
        });
    }
    
    // Next button
    const nextBtn = document.getElementById('equipment-next-btn');
    if (nextBtn) {
        nextBtn.addEventListener('click', () => {
            const totalPages = Math.ceil(equipmentAllData.length / equipmentPerPage);
            if (equipmentCurrentPage < totalPages) {
                equipmentCurrentPage++;
                updateEquipmentTable(equipmentAllData);
            }
        });
    }
    
    // Page number buttons
    document.querySelectorAll('.equipment-page-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const page = parseInt(btn.getAttribute('data-page'));
            if (page && page !== equipmentCurrentPage) {
                equipmentCurrentPage = page;
                updateEquipmentTable(equipmentAllData);
            }
        });
    });
    
    // Items per page selector
    const perPageSelect = document.getElementById('equipment-per-page');
    if (perPageSelect) {
        perPageSelect.addEventListener('change', (e) => {
            equipmentPerPage = parseInt(e.target.value);
            equipmentCurrentPage = 1; // Reset to first page
            updateEquipmentTable(equipmentAllData);
        });
    }
}

function updateEquipmentPaginationInfo(start, end, total) {
    const infoElement = document.getElementById('equipment-pagination-info');
    if (infoElement) {
        if (total === 0) {
            infoElement.textContent = 'Showing 0 - 0 of 0';
        } else {
            infoElement.textContent = `Showing ${start} - ${end} of ${total}`;
        }
    }
}

function escapeHtml(text) {
    if (!text) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return String(text).replace(/[&<>"']/g, m => map[m]);
}

// Generic chart update function
function updateChart(chartId, labels, data, colors) {
    const ctx = document.getElementById(chartId);
    if (!ctx) return;
    
    // Destroy existing chart
    if (chartInstances[chartId]) {
        chartInstances[chartId].destroy();
    }
    
    // Handle empty data
    if (!labels || labels.length === 0 || !data || data.length === 0) {
        labels = ['No Data'];
        data = [0];
        colors = ['#e0e0e0'];
    }
    
    // Create new chart
    chartInstances[chartId] = new Chart(chartId, {
        type: 'bar',
            data: {
                labels: labels,
            datasets: [{
                backgroundColor: colors,
                data: data
            }]
        },
        options: {
            legend: { display: false },
            scales: {
                yAxes: [{ ticks: { beginAtZero: true } }]
            },
            title: {
                display: true,
                text: ctx.closest('.chart-card')?.querySelector('h3')?.textContent || ''
            },
            tooltips: {
                enabled: labels[0] !== 'No Data'
            }
        }
    });
}

// Populate vehicle dropdowns
function populateVehicleDropdowns(data) {
    const vehicles = new Set();
    data.forEach(i => {
        const vehicleName = extractVehicleName(i.step1);
        if (vehicleName && vehicleName !== 'Unknown') vehicles.add(vehicleName);
    });
    
    const vehicleDropdown = document.getElementById('vehicle-select');
    if (vehicleDropdown) {
        vehicleDropdown.innerHTML = '<option value="">All Vehicles</option>';
        Array.from(vehicles).sort((a, b) => a.localeCompare(b)).forEach(vehicle => {
            const option = document.createElement('option');
            option.value = vehicle;
            option.textContent = vehicle;
            vehicleDropdown.appendChild(option);
        });
    }
}

function populateInspectorDropdown(data) {
    const inspectors = new Map();
    data.forEach(i => {
        const name = extractInspectorName(i.step1);
        if (!name || name === 'Unknown Inspector') return;
        const key = name.toLowerCase();
        if (!inspectors.has(key)) {
            inspectors.set(key, name);
        }
    });
    
    const inspectorDropdown = document.getElementById('inspector-select');
    if (inspectorDropdown) {
        inspectorDropdown.innerHTML = '<option value="">All Inspectors</option>';
        Array.from(inspectors.values())
            .sort((a, b) => a.localeCompare(b))
            .forEach(name => {
                const option = document.createElement('option');
                option.value = name;
                option.textContent = name;
                inspectorDropdown.appendChild(option);
            });
    }
}

// Export to Excel functionality
async function exportToExcel() {
    try {
        const filteredData = filterDataByDateRange(currentDateRange);
        
        if (filteredData.length === 0) {
            showNotification('No data to export', 'error');
            return;
        }
        
        console.log('📊 Exporting data:', filteredData.length, 'inspections');
        
        // Show loading notification
        showNotification('Preparing Excel export...', 'info');
        
        // Prepare data for Excel
        const excelData = filteredData.map((inspection, index) => {
            const step1 = inspection.step1 || {};
            const step2 = inspection.step2 || {};
            const step3 = inspection.step3 || {};
            const equipment = step1.selectedEquipment || step1.equipment || {};
            
            // Format date properly
            const createdAt = step1.createdAt || step3.createdAt || inspection.createdAt;
            const formattedDate = createdAt ? new Date(createdAt).toLocaleString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }) : 'N/A';
            
            return {
                'S.No': index + 1,
                'Inspection ID': inspection.id || inspection._id || 'N/A',
                'Request Number': step1.requestNumber || inspection.requestNumber || 'N/A',
                'Inspector': step1.inspector || step1.inspectorName || 'N/A',
                'Country': step1.country || 'N/A',
                'Project': step1.project || 'N/A',
                'Equipment Name': equipment.equipmentName || equipment.name || 'N/A',
                'Equipment Number': equipment.equipmentNumber || equipment.number || 'N/A',
                'Equipment Category': equipment.equipmentCategory || equipment.category || 'N/A',
                'Make': equipment.make || 'N/A',
                'Model': equipment.model || 'N/A',
                'Checklist Title': step1.checklistTitle || step1.checklist?.title || 'N/A',
                'Checklist ID': step1.selectedChecklistId || step1.checklistId || 'N/A',
                'Meter Reading': step1.meterReading || step1.meterReadingValue || 'N/A',
                'Created At': formattedDate,
                'Faulty Items': step2.faultyItems || step2.faultyItemCount || 0,
                'Safe To Use': step3.safeToUse || 'N/A',
                'Equipment Status': step3.equipmentStatus || step3.status || 'N/A',
                'Priority': step3.priority || 'N/A',
                'Overall Condition': step3.overallCondition || step3.condition || 'N/A',
                'Additional Notes': step3.additionalNote || step3.notes || step3.note || 'N/A',
                'Inspection Status': categorizeInspection(inspection) === 'good' ? 'Good' : 'Faulty',
                'GPS Address': step1.gpsAddress || step1.location?.address || 'N/A',
                'Coordinates': step1.coordinates || (step1.location ? `${step1.location.latitude},${step1.location.longitude}` : 'N/A') || 'N/A'
            };
        });
        
        // Create workbook and worksheet
        const wb = XLSX.utils.book_new();
        const ws = XLSX.utils.json_to_sheet(excelData);
        
        // Set column widths for better readability
        const colWidths = [
            { wch: 6 },  // S.No
            { wch: 15 }, // Inspection ID
            { wch: 20 }, // Request Number
            { wch: 15 }, // Inspector
            { wch: 12 }, // Country
            { wch: 15 }, // Project
            { wch: 20 }, // Equipment Name
            { wch: 18 }, // Equipment Number
            { wch: 18 }, // Equipment Category
            { wch: 15 }, // Make
            { wch: 15 }, // Model
            { wch: 40 }, // Checklist Title
            { wch: 15 }, // Checklist ID
            { wch: 15 }, // Meter Reading
            { wch: 20 }, // Created At
            { wch: 12 }, // Faulty Items
            { wch: 12 }, // Safe To Use
            { wch: 20 }, // Equipment Status
            { wch: 12 }, // Priority
            { wch: 20 }, // Overall Condition
            { wch: 30 }, // Additional Notes
            { wch: 15 }, // Inspection Status
            { wch: 30 }, // GPS Address
            { wch: 25 }  // Coordinates
        ];
        ws['!cols'] = colWidths;
        
        // Add worksheet to workbook
        XLSX.utils.book_append_sheet(wb, ws, 'Inspection Data');
        
        // Create summary sheet
        const totalInspections = filteredData.length;
        const goodInspections = filteredData.filter(i => categorizeInspection(i) === 'good').length;
        const faultyInspections = filteredData.filter(i => categorizeInspection(i) === 'faulty').length;
        const linkedWorkorders = filteredData.filter(i => {
            const step3 = i.step3 || {};
            return step3.equipmentStatus === 'Maintenance Required';
        }).length;
        const customForms = filteredData.filter(i => {
            const checklistId = i.step1?.selectedChecklistId || i.step1?.checklistId;
            return checklistId && checklistId.includes('CUSTOM');
        }).length;
        
        const summaryData = [
            { 'Metric': 'Total Inspections', 'Value': totalInspections },
            { 'Metric': 'Good Inspections', 'Value': goodInspections },
            { 'Metric': 'Faulty Inspections', 'Value': faultyInspections },
            { 'Metric': 'Linked Workorders', 'Value': linkedWorkorders },
            { 'Metric': 'Custom Forms', 'Value': customForms },
            { 'Metric': 'Date Range', 'Value': getCurrentDateRangeText() },
            { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }) }
        ];
        
        const wsSummary = XLSX.utils.json_to_sheet(summaryData);
        wsSummary['!cols'] = [{ wch: 25 }, { wch: 30 }];
        XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
        
        // Generate filename with current date and time range
        const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
        const timestamp = new Date().toISOString().slice(0, 10);
        const filename = `Inspection_KPI_${dateRangeText}_${timestamp}.xlsx`;
        
        // Export file
        XLSX.writeFile(wb, filename);
        
        showNotification(`✅ Exported ${totalInspections} inspection(s) to Excel successfully!`, 'success');
        console.log('✅ Excel export completed:', filename);
    } catch (error) {
        console.error('❌ Error exporting to Excel:', error);
        showNotification('Error exporting to Excel: ' + error.message, 'error');
    }
}

// Get current date range as text for filename
function getCurrentDateRangeText() {
    if (currentDateRange === 'today') {
        return 'Today';
    } else if (currentDateRange === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            const end = new Date(customEndDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentDateRange}_Days`;
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    fetchInspectionData();
    
    // Add export to Excel button listener (header only)
    const exportBtnHeader = document.getElementById('export-excel-btn-header');
    if (exportBtnHeader) {
        exportBtnHeader.addEventListener('click', () => {
            exportToExcel();
        });
    }
    
    // Add refresh button listener (header only)
    const refreshBtnHeader = document.getElementById('refresh-data-btn-header');
    if (refreshBtnHeader) {
        refreshBtnHeader.addEventListener('click', () => {
            showNotification('Refreshing data...', 'info');
            fetchInspectionData();
        });
    }
    
    // Custom date range functionality
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    // Apply custom dates
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentDateRange = 'custom';
            
            // Update active button
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === 'custom') {
                    btn.classList.add('active');
                }
            });
            
            updateDashboard();
            showNotification(`Showing data from ${new Date(startDate).toLocaleDateString()} to ${new Date(endDate).toLocaleDateString()}`, 'success');
        });
    }
    
    // Clear custom dates
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            customStartDate = null;
            customEndDate = null;
            
            // Switch back to default (7 days)
            currentDateRange = 7;
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            // Hide custom date range picker
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            updateDashboard();
            showNotification('Custom date range cleared', 'info');
        });
    }
    
    // Set max date to today for date inputs
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    // When start date changes, update end date minimum
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                // If end date is before start date, clear it
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
        
        // Set default dates when custom date picker is shown
        const periodButtons = document.querySelectorAll('.period-btn');
        periodButtons.forEach(btn => {
            if (btn.getAttribute('data-period') === 'custom') {
                btn.addEventListener('click', () => {
                    // Set default dates if not already set
                    if (!startDateInput.value || !endDateInput.value) {
                        // Default to last 7 days
                        const end = new Date();
                        const start = new Date();
                        start.setDate(end.getDate() - 7);
                        
                        startDateInput.value = start.toISOString().split('T')[0];
                        endDateInput.value = end.toISOString().split('T')[0];
                    }
                });
            }
        });
    }
});
</script>

<style>
/* CSS Variables for consistent theming - Matching Create New Workorder */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== MAIN CONTAINER ===== */
.inspection-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

/* ===== HEADER STYLES - Matching Workorder Design ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

/* ===== MAIN CONTENT ===== */
.inspection-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin-bottom: 1rem;
    margin: 0;
    overflow: hidden;
}


/* ===== TIME PERIOD SELECTOR ===== */
.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Custom Date Range Picker */
.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

/* ===== MODERN TABS ===== */
.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

.tab-btn.active::after {
    content: none;
}

/* ===== TAB CONTENT ===== */
.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

/* ===== KPI CARDS ===== */
.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon.success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.kpi-icon.warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}

.kpi-icon.info {
    background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
}

.kpi-icon.secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
}

.kpi-content {
    flex: 1;
}

.kpi-content h3 {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-label {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* ===== OVERVIEW BOTTOM SECTION ===== */
.overview-bottom-section {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 1.5rem;
    margin-top: 1.5rem;
}

.additional-kpi-cards {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.chart-section {
    display: flex;
    flex-direction: column;
}

/* ===== CHARTS SECTION ===== */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-container {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-container canvas {
    max-width: 100%;
    max-height: 100%;
}

/* ===== FILTER SECTION ===== */
.filter-section {
    margin-bottom: 2rem;
}

.filter-form {
    display: flex;
    align-items: end;
    gap: 1rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
}

.form-group {
    flex: 1;
    display: flex;
    flex-direction: column;
    margin-bottom: 0;
}

.form-group label {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
    font-size: 14px;
}

.form-select {
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 12px 16px;
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.form-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-go {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    white-space: nowrap;
    height: fit-content;
    min-height: 44px;
}

.btn-go:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 992px) {
    .kpi-cards-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 16px;
    }
    
    .overview-bottom-section {
        grid-template-columns: 1fr;
        gap: 16px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        gap: 10px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .inspection-kpi-container,
    .inspection-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
        margin: 0;
        border-radius: var(--radius-lg);
    }
    
    
    .time-period-selector {
        margin-bottom: 1.5rem;
    }
    
    .period-buttons {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 8px;
    }
    
    .period-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
        text-align: center;
    }
    
    .custom-date-range {
        padding: 1rem;
    }
    
    .date-range-inputs {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .date-input-group {
        width: 100%;
    }
    
    .date-input {
        width: 100%;
        min-width: auto;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        width: 100%;
        justify-content: center;
    }
    
    .tab-navigation {
        flex-direction: column;
        gap: 0;
        border-bottom: 1px solid var(--border-color);
    }
    
    .tab-btn {
        justify-content: flex-start;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
        border-bottom: 1px solid var(--border-color);
    }
    
    .kpi-cards-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .kpi-card {
        padding: 16px;
        flex-direction: column;
        text-align: center;
        gap: 12px;
    }
    
    .kpi-icon {
        width: 48px;
        height: 48px;
    }
    
    .kpi-content h3 {
        font-size: 14px;
    }
    
    .kpi-value {
        font-size: 24px;
    }
    
    .kpi-label {
        font-size: 12px;
    }
    
    .overview-bottom-section {
        grid-template-columns: 1fr;
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .chart-card {
        padding: 16px;
    }
    
    .chart-header h3 {
        font-size: 16px;
    }
    
    .chart-container {
        height: 200px;
    }
    
    .filter-form {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
        padding: 16px;
    }
    
    .form-group label {
        font-size: 13px;
    }
    
    .form-select {
        padding: 12px;
        font-size: 16px;
        min-height: 44px;
    }
    
    .btn-go {
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .header-title {
        font-size: 20px;
    }
    
    .header-right {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
        padding: 10px 12px;
        font-size: 13px;
    }
    
    .content-box {
        padding: 14px;
    }
    
    
    .period-buttons {
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 10px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.875rem;
    }
    
    .date-input {
        padding: 10px;
        font-size: 14px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 10px 14px;
        font-size: 13px;
    }
    
    .tab-btn {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 40px;
    }
    
    .kpi-card {
        padding: 14px;
    }
    
    .kpi-icon {
        width: 44px;
        height: 44px;
    }
    
    .kpi-content h3 {
        font-size: 13px;
    }
    
    .kpi-value {
        font-size: 22px;
    }
    
    .kpi-label {
        font-size: 11px;
    }
    
    .chart-container {
        height: 190px;
    }
    
    .filter-form {
        padding: 14px;
    }
    
    .form-group label {
        font-size: 12px;
    }
    
    .form-select {
        padding: 11px;
        font-size: 15px;
        min-height: 42px;
    }
    
    .btn-go {
        padding: 11px 14px;
        font-size: 13px;
        min-height: 42px;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-title {
        font-size: 18px;
    }
    
    .back-button {
        font-size: 18px;
    }
    
    .header-right {
        width: 100%;
    }
    
    .create-btn {
        font-size: 12px;
        padding: 9px 12px;
    }
    
    .content-box {
        padding: 12px;
    }
    
    
    .period-buttons {
        grid-template-columns: 1fr;
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.75rem;
    }
    
    .date-input {
        padding: 9px;
        font-size: 13px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 9px 12px;
        font-size: 12px;
    }
    
    .tab-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
    }
    
    .kpi-card {
        padding: 12px;
        flex-direction: column;
        text-align: center;
        gap: 8px;
    }
    
    .kpi-icon {
        width: 40px;
        height: 40px;
    }
    
    .kpi-content h3 {
        font-size: 13px;
    }
    
    .kpi-value {
        font-size: 20px;
    }
    
    .kpi-label {
        font-size: 11px;
    }
    
    .chart-card {
        padding: 12px;
    }
    
    .chart-header h3 {
        font-size: 14px;
    }
    
    .chart-container {
        height: 180px;
    }
    
    .filter-form {
        padding: 12px;
    }
    
    .form-group label {
        font-size: 12px;
    }
    
    .form-select {
        padding: 10px;
        font-size: 15px;
        min-height: 40px;
    }
    
    .btn-go {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
    }
}

/* ===== LOADING SPINNER ===== */
@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Loading/Sync Modal - Matching Workorder Design */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 56px;
    height: 56px;
    margin: 0 auto 20px;
    border-radius: 50%;
    border: 6px solid rgba(0, 85, 165, 0.18);
    border-top-color: #0055A5;
    border-right-color: #FFD205;
    animation: spin 0.9s linear infinite;
    box-shadow: 0 10px 24px rgba(0, 85, 165, 0.18);
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* ===== ENHANCED RESPONSIVE DESIGN ===== */
@media (max-width: 992px) {
    .kpi-cards-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 16px;
    }
    
    .overview-bottom-section {
        grid-template-columns: 1fr;
        gap: 16px;
    }
}

/* ===== PRINT STYLES ===== */
@media print {
    .header-gradient,
    .period-buttons,
    .tab-navigation {
        display: none !important;
    }
    
    .inspection-kpi-container {
        background: white;
    }
    
    .kpi-card,
    .chart-card {
        border: 1px solid #000;
        break-inside: avoid;
    }
}
</style>

<script>
// Optimized Inspection KPI Functionality
(function() {
    'use strict';
    
    // Configuration
    const CONFIG = {
        notificationTimeout: 5000,
        selectors: {
            tabButtons: '.tab-btn',
            tabPanes: '.tab-pane',
            periodButtons: '.period-btn',
            kpiCards: '.kpi-card',
            chartContainers: '.chart-container',
            filterForms: '.filter-form'
        }
    };

    // Cached DOM elements
    let elements = {};
    
    // Initialize DOM elements
    function initElements() {
        elements = {
            tabButtons: document.querySelectorAll(CONFIG.selectors.tabButtons),
            tabPanes: document.querySelectorAll(CONFIG.selectors.tabPanes),
            periodButtons: document.querySelectorAll(CONFIG.selectors.periodButtons),
            kpiCards: document.querySelectorAll(CONFIG.selectors.kpiCards),
            chartContainers: document.querySelectorAll(CONFIG.selectors.chartContainers),
            filterForms: document.querySelectorAll(CONFIG.selectors.filterForms)
        };
    }

    // Tab switching functionality
    function initTabSwitching() {
        elements.tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');

            // Remove active class from all tabs and panes
                elements.tabButtons.forEach(btn => btn.classList.remove('active'));
                elements.tabPanes.forEach(pane => pane.classList.remove('active'));

            // Add active class to clicked tab and corresponding pane
            this.classList.add('active');
            const targetPane = document.getElementById(targetTab);
            if (targetPane) {
                targetPane.classList.add('active');
            }
        });
    });
    }

    // Time period selector functionality
    function initPeriodSelector() {
        const customDateRange = document.getElementById('custom-date-range');
        
        elements.periodButtons.forEach(button => {
        button.addEventListener('click', function() {
            // Remove active class from all period buttons
                elements.periodButtons.forEach(btn => btn.classList.remove('active'));
            
            // Add active class to clicked button
            this.classList.add('active');
            
            const period = this.getAttribute('data-period');
            console.log('Period changed to:', period);
            
            // Show/hide custom date range picker
            if (period === 'custom') {
                if (customDateRange) {
                    customDateRange.style.display = 'block';
                }
                currentDateRange = 'custom';
                // Don't update dashboard yet, wait for user to select dates
                return;
            } else {
                // Hide custom date range picker for other periods
                if (customDateRange) {
                    customDateRange.style.display = 'none';
                }
                
                // Update current date range
                if (period === 'today') {
                    currentDateRange = 'today';
                } else if (period === '7days') {
                    currentDateRange = 7;
                } else if (period === '30days') {
                    currentDateRange = 30;
                } else if (period === '90days') {
                    currentDateRange = 90;
                }
                
                // Update dashboard with new date range
                updateDashboard();
            }
        });
    });
    }

    // Filter form functionality
    function initFilterForms() {
        elements.filterForms.forEach(form => {
        const goButton = form.querySelector('.btn-go');
        const selectElement = form.querySelector('.form-select');
        
        if (goButton && selectElement) {
            goButton.addEventListener('click', function() {
                const selectedValue = selectElement.value;
                const filteredData = filterDataByDateRange(currentDateRange);
                
                // Determine which chart to update based on the select element ID
                if (selectElement.id === 'inspector-select') {
                    updateInspectorChart(filteredData, selectedValue || null);
                    if (selectedValue) {
                        showNotification(`Inspector chart filtered: ${selectedValue}`, 'success');
                    } else {
                        showNotification('Showing all inspectors', 'info');
                    }
                } else if (selectElement.id === 'vehicle-select') {
                    updateVehicleChart(filteredData, selectedValue || null);
                    if (selectedValue) {
                        showNotification(`Vehicle chart filtered: ${selectedValue}`, 'success');
                    } else {
                        showNotification('Showing all vehicles', 'info');
                    }
                }
            });
        }
    });
        
        // Populate dropdowns after data is loaded
        if (allInspectionData.length > 0) {
            populateVehicleDropdowns(allInspectionData);
            populateInspectorDropdown(allInspectionData);
        }
    }


    // KPI card hover effects
    function initKpiCardEffects() {
        elements.kpiCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-4px)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
    }

    // Chart container responsive handling
    function initChartContainers() {
        elements.chartContainers.forEach(container => {
        const canvas = container.querySelector('canvas');
        if (canvas) {
            // Ensure canvas is responsive
            canvas.style.maxWidth = '100%';
            canvas.style.height = 'auto';
        }
    });
    }

    // Notification system for user feedback
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close" aria-label="Close notification">&times;</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        
        // Auto remove after timeout
        setTimeout(() => removeNotification(notification), CONFIG.notificationTimeout);
        
        // Close button functionality
        notification.querySelector('.notification-close').addEventListener('click', () => {
            removeNotification(notification);
        });
    }
    
    function removeNotification(notification) {
        if (notification && notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
    }

    // Add notification styles - Matching workorder toast design
    function addNotificationStyles() {
    const notificationStyles = document.createElement('style');
    notificationStyles.textContent = `
        .notification {
            position: fixed;
            top: 20px;
            right: 20px;
                z-index: var(--z-index-notification);
            padding: 16px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                font-family: var(--font-family);
            font-weight: 500;
            max-width: 400px;
            min-width: 300px;
            animation: slideIn 0.3s ease;
            background: white;
        }
        
        .notification-success {
            border-left: 4px solid #28a745;
        }
        
        .notification-error {
            border-left: 4px solid #dc3545;
        }
        
        .notification-info {
            border-left: 4px solid #17a2b8;
        }
        
        .notification-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 1rem;
        }
        
        .notification-content span {
            color: #333;
            font-size: 14px;
        }
        
        .notification-close {
            background: none;
            border: none;
            font-size: 18px;
            color: #999;
            cursor: pointer;
            padding: 0;
            line-height: 1;
            margin-left: 12px;
        }
        
        .notification-close:hover {
            color: #666;
        }
        
        @keyframes slideIn {
            from {
                transform: translateX(100%);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }
    `;
    document.head.appendChild(notificationStyles);
    }

    // Initialize everything
    function init() {
        initElements();
        initTabSwitching();
        initPeriodSelector();
        initFilterForms();
        initKpiCardEffects();
        initChartContainers();
        addNotificationStyles();
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
</script>
@endsection