@extends('welcome')

@section('title', 'Create DVIR Report - Step 1')

@section('content')
<link rel="stylesheet" href="{{ asset('css/Inspection/create_incident_report.blade.css') }}">
<!-- Add html2pdf.js library for PDF generation (same as Work Order) -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>
<div class="create-incident-container" role="main" aria-label="Create DVIR Report">
    <!-- Header Section -->
    <header class="create-incident-header" role="banner">
        <div class="header-content">
            <div class="page-title">
                <h1 style="color: white;">Create DVIR</h1>
                <p style="color: rgba(255, 255, 255, 0.9);">Create a new Driver Vehicle Inspection Report with detailed information and evidence</p>
            </div>
        </div>
    </header>
    
    <nav class="d-flex align-items-center mb-4" aria-label="Navigation">
        <a href="/dvir-dot-inspection" class="btn btn-link text-decoration-none" aria-label="Go back to DVIR report list">
            <i class="fas fa-arrow-left me-2" aria-hidden="true"></i>Back to DVIR Reports
        </a>
    </nav>

    <!-- Step Indicator -->
    <div class="step-indicator-section">
        <div class="step-indicator">
            <div class="step-item active" data-step="1">
                <div class="step-icon">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M9 11H5a2 2 0 0 0-2 2v3c0 1.1.9 2 2 2h4m0-7v7m0-7h10a2 2 0 0 1 2 2v3c0 1.1-.9 2-2 2H9m0-7V9a2 2 0 0 1 2-2h2a2 2 0 0 1 2 2v2"/>
                    </svg>
                </div>
                <div class="step-content">
                    <span class="step-number">1</span>
                    <span class="step-title">Inspection Info</span>
                </div>
            </div>
            <div class="step-connector"></div>
            <div class="step-item" data-step="2">
                <div class="step-icon">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                        <polyline points="14,2 14,8 20,8"/>
                        <line x1="16" y1="13" x2="8" y2="13"/>
                        <line x1="16" y1="17" x2="8" y2="17"/>
                    </svg>
                </div>
                <div class="step-content">
                    <span class="step-number">2</span>
                    <span class="step-title">Do Inspection</span>
                </div>
            </div>
            <div class="step-connector"></div>
            <div class="step-item" data-step="3">
                <div class="step-icon">
                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M9 12l2 2 4-4"/>
                        <path d="M21 12c-1 0-3-1-3-3s2-3 3-3 3 1 3 3-2 3-3 3"/>
                        <path d="M3 12c1 0 3-1 3-3s-2-3-3-3-3 1-3 3 2 3 3 3"/>
                        <path d="M13 12h3"/>
                        <path d="M8 12H5"/>
                    </svg>
                </div>
                <div class="step-content">
                    <span class="step-number">3</span>
                    <span class="step-title">Summary</span>
                </div>
            </div>
        </div>
    </div>

    <!-- Main Form Container -->
    <div class="form-container" id="step1Container">
        <div class="form-card">
            <!-- Request Number Display -->
            <div class="request-number-display">
                <div class="request-number-row">
                    <span class="request-label">Request #</span>
                    <span class="request-value" id="requestNumber">Select country and project first</span>
                </div>
            </div>

            <!-- Country Selection -->
            <div class="form-group">
                <label class="form-label required-field">Country</label>
                <select class="form-control" id="country" onchange="updateProjects()">
                    <option value="">Select Country</option>
                    <option value="Zambia">Zambia</option>
                    <option value="DRC">DRC</option>
                </select>
            </div>

            <!-- Project Selection -->
            <div class="form-group">
                <label class="form-label required-field">Project</label>
                <select class="form-control" id="project" disabled onchange="generateRequestNumber()">
                    <option value="">Select Project</option>
                </select>
            </div>

            <!-- Inspector -->
            <div class="form-group">
                <label class="form-label required-field">Inspector</label>
                <input type="text" class="form-control" id="inspector" placeholder="Enter Inspector Name">
            </div>

            <!-- Equipment Selection -->
            <div class="form-group">
                <label class="form-label required-field">Equipment *</label>
                <div class="input-with-button">
                    <input type="text" class="form-control" id="equipmentDisplay" name="equipmentDisplay" placeholder="Select Equipment" onclick="openEquipmentModal()" readonly required>
                    <button type="button" class="btn-select" onclick="openEquipmentModal()" title="Select Equipment">
                        <i class="fas fa-search"></i>
                    </button>
                    <button type="button" class="btn-clear" onclick="clearEquipmentAutoFill()" title="Clear Equipment" style="margin-left: 5px;">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <input type="hidden" id="vehicleNumber" name="vehicleNumber" required>
            </div>

            <!-- Selected Equipment Details -->
            <div class="equipment-details-box" id="equipmentDetailsBox" style="display: none;">
                <div class="equipment-info-row">
                    <span class="equipment-info-text" id="equipmentInfoText"></span>
                </div>
                
                <!-- Meter Reading Section -->
                <div class="meter-reading-section">
                    <label class="meter-reading-label required-field">Meter Reading</label>
                    <div class="meter-reading-input-row">
                        <input type="number" class="meter-reading-input" id="meterReading" placeholder="Enter meter reading">
                        <span class="meter-unit" id="meterUnit">Miles</span>
                    </div>
                    <div class="meter-reading-error" id="meterReadingError" style="display: none;"></div>
                    <div class="current-meter-row">
                        <span class="current-meter-label">Current Meter Reading: </span>
                        <span class="current-meter-value" id="currentMeterValue">0 Miles</span>
                    </div>
                </div>
            </div>

            <!-- Inspection Type Selection -->
            <div class="inspection-type-section">
                <div class="checkbox-row">
                    <input type="checkbox" id="dvirCheckbox" class="inspection-checkbox" checked>
                    <label for="dvirCheckbox" class="checkbox-label">DVIR - Driver Vehicle Inspection Report</label>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="action-buttons">
            <button class="btn-cancel" onclick="handleBackPress()">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M19 12H5"/>
                    <path d="M12 19l-7-7 7-7"/>
                </svg>
                <span>Cancel</span>
            </button>
            <button class="btn-next" id="nextButton" onclick="handleNextPress()" disabled>
                <span>Next</span>
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M5 12h14"/>
                    <path d="M12 5l7 7-7 7"/>
                </svg>
            </button>
        </div>
    </div>

    <!-- Equipment Selection Modal -->
    <div id="equipmentModal" class="dvir-modal-backdrop" onclick="if(event.target === this) closeEquipmentModal()">
        <div class="dvir-modal-dialog" onclick="event.stopPropagation()">
            <div class="dvir-modal-header-wrapper">
                <div class="dvir-modal-header-icon">
                    <i class="fas fa-hard-hat"></i>
                </div>
                <div class="dvir-modal-header-content">
                    <h3 class="dvir-modal-title">Select Equipment</h3>
                    <p class="dvir-modal-subtitle">Choose from available equipment in your database</p>
                </div>
                <button type="button" class="dvir-modal-close-btn" onclick="closeEquipmentModal()" aria-label="Close modal">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="dvir-modal-body-wrapper">
                <div class="dvir-search-wrapper">
                    <div class="dvir-search-input-group">
                        <i class="fas fa-search dvir-search-icon"></i>
                        <input type="text" id="equipmentSearch" class="dvir-search-input" placeholder="Search by name, number, model, or category..." onkeyup="filterEquipmentBySearch()">
                        <button type="button" class="dvir-search-clear-btn" id="clearSearchBtn" onclick="document.getElementById('equipmentSearch').value=''; filterEquipmentBySearch();" style="display: none;">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <button type="button" class="dvir-search-action-btn" onclick="filterEquipmentBySearch()">
                        <i class="fas fa-filter"></i>
                        <span>Filter</span>
                    </button>
                </div>
                <div class="dvir-equipment-list-container" id="equipmentList">
                    <!-- Equipment items will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <!-- Loading Modal -->
    <div class="dvir-loading-backdrop" id="loadingModal" style="display: none;" onclick="if(event.target === this) hideLoadingModal()">
        <div class="dvir-loading-dialog" onclick="event.stopPropagation()">
            <div class="dvir-loading-spinner-wrapper">
                <div class="dvir-loading-spinner"></div>
                <div class="dvir-loading-pulse"></div>
            </div>
            <div class="dvir-loading-content">
                <h3 class="dvir-loading-title">Please Wait</h3>
                <p class="dvir-loading-message">Processing Equipment Data</p>
                <div class="dvir-loading-progress-bar">
                    <div class="dvir-loading-progress-fill"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Toast Container -->
    <div class="toast-container" id="toastContainer"></div>

    <!-- Step 2: Inspection Checklist -->
    <div class="step-container" id="step2Container" style="display: none;">
        <div class="step-header">
            <h1>Inspection Checklist</h1>
            <p>Complete the inspection checklist items</p>
        </div>

        <div class="step-content inspection-content">
            <div class="instruction-text">
                <h3>How to complete this step</h3>
                <p>Work through each checklist group, capture a note or supporting photo whenever something needs attention.</p>
            </div>

            <div class="inspection-guidelines">
                <div class="guideline-card">
                    <span class="guideline-icon">①</span>
                    <div>
                        <h4>Expand every group</h4>
                        <p>Use the chevron on each card to reveal Requirements or Hazard Awareness items.</p>
                    </div>
                </div>
                <div class="guideline-card">
                    <span class="guideline-icon">②</span>
                    <div>
                        <h4>Log context fast</h4>
                        <p>Add a quick note or photo directly from the item so all evidence stays with the inspection.</p>
                    </div>
                </div>
                <div class="guideline-card">
                    <span class="guideline-icon">③</span>
                    <div>
                        <h4>Use Yes / No actions</h4>
                        <p>Mark each line item as you go. You can always reopen a card to make edits before submitting.</p>
                    </div>
                </div>
            </div>

            <div class="checklist-container">
                <!-- Requirements Section -->
                <div class="checklist-card" id="requirements-card">
                    <div class="card-header" onclick="toggleSection('requirements')">
                        <div class="header-left">
                            <span class="expand-icon" id="requirements-icon">▼</span>
                            <h3 class="section-title">Requirements</h3>
                        </div>
                    </div>
                    <div class="card-content" id="requirements-content" style="display: none;">
                        <div class="checklist-items">
                            <div class="checklist-item" data-item-id="req1">
                                <div class="item-content">
                                    <p class="item-text">I hold a valid driver's licence</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req1', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req1', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req1" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req1', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req1" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req1')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req1" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req1')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req1')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req2">
                                <div class="item-content">
                                    <p class="item-text">I will obey speed limits and road signs</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req2', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req2', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req2" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req2', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req2" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req2')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req2" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req2')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req2')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req3">
                                <div class="item-content">
                                    <p class="item-text">I will not overload the vehicle and ensure the load is secure</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req3', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req3', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req3" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req3', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req3" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req3')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req3" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req3')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req3')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req4">
                                <div class="item-content">
                                    <p class="item-text">I have completed the vehicle pre check list and satisfied all is order for my trip</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req4', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req4', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req4" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req4', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req4" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req4')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req4" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req4')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req4')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req5">
                                <div class="item-content">
                                    <p class="item-text">I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req5', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req5', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req5" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req5', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req5" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req5')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req5" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req5')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req5')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req6">
                                <div class="item-content">
                                    <p class="item-text">I have read and understand the Titan driving procedure, National driving rules and cliental rules</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req6', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req6', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req6" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req6', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req6" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req6')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req6" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req6')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req6')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="req7">
                                <div class="item-content">
                                    <p class="item-text">I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('requirements', 'req7', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('requirements', 'req7', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-req7" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('requirements', 'req7', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-req7" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('req7')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-req7" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('req7')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('req7')" title="Take Photo">📷</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Hazards Section -->
                <div class="checklist-card" id="hazards-card">
                    <div class="card-header" onclick="toggleSection('hazards')">
                        <div class="header-left">
                            <span class="expand-icon" id="hazards-icon">▼</span>
                            <h3 class="section-title">Hazards – Are you aware of:</h3>
                        </div>
                    </div>
                    <div class="card-content" id="hazards-content" style="display: none;">
                        <div class="checklist-items">
                            <div class="checklist-item" data-item-id="haz1">
                                <div class="item-content">
                                    <p class="item-text">Pedestrians crossing the road, school children and drunkards</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz1', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz1', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz1" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz1', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz1" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz1')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz1" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz1')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz1')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz2">
                                <div class="item-content">
                                    <p class="item-text">Motor bikes and taxis stopping suddenly or pulling in to the road suddenly</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz2', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz2', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz2" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz2', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz2" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz2')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz2" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz2')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz2')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz3">
                                <div class="item-content">
                                    <p class="item-text">Oncoming vehicles overtaking on blind rises or corners</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz3', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz3', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz3" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz3', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz3" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz3')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz3" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz3')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz3')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz4">
                                <div class="item-content">
                                    <p class="item-text">Heavy trucks or plant taking up width of the road</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz4', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz4', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz4" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz4', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz4" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz4')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz4" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz4')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz4')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz5">
                                <div class="item-content">
                                    <p class="item-text">Riots or unrest</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz5', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz5', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz5" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz5', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz5" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz5')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz5" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz5')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz5')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz6">
                                <div class="item-content">
                                    <p class="item-text">Gravel/Sandy roads can be slippery – Loss of control</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz6', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz6', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz6" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz6', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz6" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz6')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz6" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz6')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz6')" title="Take Photo">📷</button>
                                </div>
                            </div>

                            <div class="checklist-item" data-item-id="haz7">
                                <div class="item-content">
                                    <p class="item-text">Tyre blowouts or punctures – Loss of control</p>
                                    <div class="status-buttons">
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz7', 'Yes')">Yes</button>
                                        <button class="status-btn" onclick="setStatus('hazards', 'haz7', 'No')">No</button>
                                    </div>
                                    <div class="note-input-container" id="note-haz7" style="display: none;">
                                        <input type="text" class="note-input" placeholder="Add note..." onchange="updateNote('hazards', 'haz7', this.value)">
                                    </div>
                                    <div class="image-preview-container" id="image-preview-haz7" style="display: none;">
                                        <div class="image-preview-header">
                                            <span class="image-preview-label">Photo Preview:</span>
                                            <button class="remove-image-btn" onclick="removeImage('haz7')" title="Remove Photo">×</button>
                                        </div>
                                        <img class="image-preview" id="preview-img-haz7" src="" alt="Photo Preview">
                                    </div>
                                </div>
                                <div class="item-actions">
                                    <button class="action-btn" onclick="toggleNote('haz7')" title="Add Note">📝</button>
                                    <button class="action-btn" onclick="takePhoto('haz7')" title="Take Photo">📷</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 2 Navigation Buttons -->
        <div class="nav-buttons">
            <button class="nav-btn back-btn" onclick="navigateToStep(1)">
                <span>← Back</span>
            </button>
            <button class="nav-btn next-btn" id="step2NextButton" onclick="handleStep2Next()" disabled>
                <span>Next →</span>
            </button>
        </div>
    </div>

    <!-- Step 3: Summary and Report Generation -->
    <div class="step-container" id="step3Container" style="display: none;">
        <div class="step-header">
            <h1>Summary & Report Generation</h1>
            <p>Review and generate your DVIR report</p>
        </div>

        <div class="step-content summary-content">
            <div class="summary-highlight">
                <div class="summary-pill">
                    <span class="pill-label">Faulty Items</span>
                    <span class="pill-value" id="faultyItemsCount">0</span>
                </div>
                <div class="summary-pill pill-muted" id="driverStatusChip">Driver status pending</div>
                <div class="summary-pill pill-outline" id="safeToUseChip">Equipment safety not evaluated</div>
            </div>

            <div class="summary-grid">
                <div class="summary-card">
                    <div class="form-group">
                        <label class="form-label required-field">Overall Condition</label>
                        <div class="select-container">
                            <select class="form-control" id="overallCondition" onchange="validateStep3()">
                                <option value="">Select Overall Condition</option>
                                <option value="Proceed with journey">Proceed with journey</option>
                                <option value="Do not proceed with journey">Do not proceed with journey</option>
                                <option value="Report to Supervisor">Report to Supervisor</option>
                            </select>
                        </div>
                        <div class="error-message" id="overallConditionError" style="display: none;"></div>
                    </div>
                </div>

                <div class="summary-card">
                    <div class="form-group">
                        <label class="form-label required-field">Driver Status</label>
                        <div class="radio-group">
                            <label class="radio-option">
                                <input type="radio" name="equipmentStatus" value="Fit to travel" onchange="handleDriverStatusChange()">
                                <span class="radio-label">Fit to travel</span>
                            </label>
                            <label class="radio-option">
                                <input type="radio" name="equipmentStatus" value="Unfit to travel" onchange="handleDriverStatusChange()">
                                <span class="radio-label">Unfit to travel</span>
                            </label>
                        </div>
                        <div class="error-message" id="equipmentStatusError" style="display: none;"></div>
                    </div>

                    <div class="form-group priority-wrapper" id="priorityGroup" style="display: none;">
                        <label class="form-label">Priority</label>
                        <div class="priority-container">
                            <div class="priority-option" onclick="selectPriority('Low')">
                                <div class="priority-icon">🟢</div>
                                <span>Low</span>
                            </div>
                            <div class="priority-option" onclick="selectPriority('Medium')">
                                <div class="priority-icon">🟡</div>
                                <span>Medium</span>
                            </div>
                            <div class="priority-option" onclick="selectPriority('High')">
                                <div class="priority-icon">🟠</div>
                                <span>High</span>
                            </div>
                            <div class="priority-option" onclick="selectPriority('Emergency')">
                                <div class="priority-icon">🔴</div>
                                <span>Emergency</span>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="summary-card">
                    <div class="form-group">
                        <label class="form-label required-field">Equipment Safe to Use</label>
                        <div class="radio-group">
                            <label class="radio-option">
                                <input type="radio" name="safeToUse" value="Yes" onchange="validateStep3()">
                                <span class="radio-label">Yes</span>
                            </label>
                            <label class="radio-option">
                                <input type="radio" name="safeToUse" value="No" onchange="validateStep3()">
                                <span class="radio-label">No</span>
                            </label>
                        </div>
                        <div class="error-message" id="safeToUseError" style="display: none;"></div>
                    </div>
                </div>

                <div class="summary-card full-width">
                    <div class="form-group">
                        <label class="form-label">Additional Note</label>
                        <div class="input-with-icon">
                            <textarea class="form-control textarea" id="additionalNote" rows="4" placeholder="Enter additional notes..."></textarea>
                            <button class="voice-btn" onclick="handleVoiceInput()" title="Voice Input">
                                <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M12 1a3 3 0 0 0-3 3v8a3 3 0 0 0 6 0V4a3 3 0 0 0-3-3z"/>
                                    <path d="M19 10v2a7 7 0 0 1-14 0v-2"/>
                                    <line x1="12" y1="19" x2="12" y2="23"/>
                                    <line x1="8" y1="23" x2="16" y2="23"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>

                <div class="summary-card full-width">
                    <div class="form-group">
                        <label class="form-label required-field">Inspector Signature</label>
                        <div class="signature-container">
                            <div class="signature-box" id="signatureBox" onclick="handleSignature()">
                                <div class="signature-placeholder" id="signaturePlaceholder">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
                                    </svg>
                                    <span>Click to add signature</span>
                                </div>
                                <div class="signature-display" id="signatureDisplay" style="display: none;">
                                    <img id="signatureImage" src="" alt="Signature">
                                    <button class="change-signature-btn" onclick="handleSignature()">Change Signature</button>
                                </div>
                            </div>
                        </div>
                        <div class="error-message" id="signatureError" style="display: none;"></div>
                    </div>
                </div>

                <div class="summary-card full-width">
                    <div class="acknowledgment-section">
                        <div class="acknowledgment-header">
                            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <polyline points="20,6 9,17 4,12"/>
                            </svg>
                            <h3>Operator's Acknowledgment</h3>
                        </div>
                        <p>I hereby certify that the inspection has been done accurately and completely according to compliance with all applicable state and Federal Motor Carrier Safety Regulations.</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Step 3 Navigation Buttons -->
        <div class="nav-buttons">
            <button class="nav-btn back-btn" onclick="navigateToStep(2)">
                <span>← Back</span>
            </button>
            <button class="nav-btn next-btn" id="step3NextButton" onclick="handleStep3Next()" disabled>
                <span>Generate Report</span>
            </button>
        </div>
    </div>

    <!-- Signature Modal -->
    <div class="dvir-modal-backdrop" id="signatureModal" style="display: none;" onclick="if(event.target === this) closeSignatureModal()">
        <div class="dvir-modal-dialog dvir-signature-dialog" onclick="event.stopPropagation()">
            <div class="dvir-modal-header-wrapper">
                <div class="dvir-modal-header-icon">
                    <i class="fas fa-pen-fancy"></i>
                </div>
                <div class="dvir-modal-header-content">
                    <h3 class="dvir-modal-title">Add Signature</h3>
                    <p class="dvir-modal-subtitle">Sign in the box below using your mouse or touchpad</p>
                </div>
                <button class="dvir-modal-close-btn" onclick="closeSignatureModal()" aria-label="Close modal">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="signature-pad-container">
                <canvas id="signaturePad" width="600" height="200"></canvas>
            </div>
            <div class="signature-actions">
                <button class="btn-clear" onclick="clearSignature()">Clear</button>
                <button class="btn-save" onclick="saveSignature()">Save Signature</button>
            </div>
        </div>
    </div>

    <!-- Success Modal -->
    <div class="dvir-success-backdrop" id="successModal" style="display: none;" onclick="if(event.target === this) closeSuccessModal()">
        <div class="dvir-success-dialog" onclick="event.stopPropagation()">
            <div class="dvir-success-icon-wrapper">
                <div class="dvir-success-icon-circle">
                    <i class="fas fa-check"></i>
                </div>
                <div class="dvir-success-checkmark">
                    <svg viewBox="0 0 52 52">
                        <circle cx="26" cy="26" r="25" fill="none"/>
                        <path fill="none" d="m14.1 27.2l7.1 7.2 16.7-16.8"/>
                    </svg>
                </div>
            </div>
            <div class="dvir-success-content">
                <h3 class="dvir-success-title">Report Generated Successfully!</h3>
                <p class="dvir-success-message">Your DVIR report has been generated and saved to the database.</p>
                <div class="dvir-success-actions">
                    <button class="dvir-btn-primary" onclick="viewPDF()">
                        <i class="fas fa-file-pdf"></i>
                        <span>View PDF</span>
                    </button>
                    <button class="dvir-btn-success" onclick="goToDvirList()">
                        <i class="fas fa-list"></i>
                        <span>Back to DVIR List</span>
                    </button>
                    <button class="dvir-btn-secondary" onclick="closeSuccessModal()">
                        <i class="fas fa-times"></i>
                        <span>Close</span>
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>

// Global variables
let selectedEquipment = null;
let equipmentData = [];
let filteredEquipment = [];

// Step management
let currentStep = 1;
let step2Data = {};
let step3Data = {};
let signatureData = null;
let signatureStatus = false;

// Configuration
const CONFIG = {
    countries: {
        'Zambia': [
            { label: 'Kansanshi', value: 'Kansanshi' },
            { label: 'Kalumbila', value: 'Kalumbila' },
            { label: 'Kimteto', value: 'Kimteto' },
            { label: 'Kobold', value: 'Kobold' },
            { label: 'FQM Exploration', value: 'FQM Exploration' },
            { label: 'Mimosa', value: 'Mimosa' }
        ],
        'DRC': [
            { label: 'Lubumbashi', value: 'Lubumbashi' },
            { label: 'Musompo', value: 'Musompo' },
            { label: 'IME', value: 'IME' },
            { label: 'Kamoa', value: 'Kamoa' }
        ]
    }
};

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    loadEquipmentData();
    setupEventListeners();
    initializeActionButtons();
    
    // Initialize step indicator
    updateStepIndicator(1);
    
    // Restore previous step if available
    const savedStep = localStorage.getItem('dvirCurrentStep');
    if (savedStep && savedStep !== '1') {
        navigateToStep(parseInt(savedStep));
    }
});

// Initialize action buttons as disabled
function initializeActionButtons() {
    const actionButtons = document.querySelectorAll('.action-btn');
    actionButtons.forEach(btn => {
        btn.disabled = true;
        btn.style.opacity = '0.5';
        btn.style.cursor = 'not-allowed';
    });
    console.log('Action buttons initialized as disabled');
}

// Setup event listeners
function setupEventListeners() {
    // Country and Project validation
    document.getElementById('country').addEventListener('change', validateForm);
    document.getElementById('project').addEventListener('change', validateForm);
    
    // Inspector input validation
    document.getElementById('inspector').addEventListener('input', validateForm);
    
    // Meter reading validation - trigger on both input and change
    const meterReadingField = document.getElementById('meterReading');
    meterReadingField.addEventListener('input', function() {
        validateMeterReading();
        validateForm();
    });
    meterReadingField.addEventListener('change', function() {
        validateMeterReading();
        validateForm();
    });
    
    // Equipment search
    const searchInput = document.getElementById('equipmentSearch');
    if (searchInput) {
        searchInput.addEventListener('input', filterEquipmentBySearch);
    }
    
    // DVIR checkbox
    document.getElementById('dvirCheckbox').addEventListener('change', validateForm);
    
    // Step indicator click events
    document.querySelectorAll('.step-item').forEach(item => {
        item.style.cursor = 'pointer';
        item.addEventListener('click', function() {
            const step = parseInt(this.dataset.step);
            if (step && step !== currentStep) {
                navigateToStep(step);
            }
        });
    });
    
    console.log('✅ Event listeners setup completed');
}

// Update projects based on country selection
function updateProjects() {
    const country = document.getElementById('country').value;
    const projectSelect = document.getElementById('project');
    
    // Clear existing options
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    
    if (country && CONFIG.countries[country]) {
        projectSelect.disabled = false;
        CONFIG.countries[country].forEach(project => {
            const option = document.createElement('option');
            option.value = project.value;
            option.textContent = project.label;
            projectSelect.appendChild(option);
        });
    } else {
        projectSelect.disabled = true;
    }
    
    generateRequestNumber();
    validateForm();
}

// Generate request number (preview only - counter updated on PDF generation)
async function generateRequestNumber() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    
    if (!country || !project) {
        document.getElementById('requestNumber').textContent = 'Select country and project first';
        return;
    }
    
    try {
        console.log('Generating DVIR number preview for:', country, project);
        
        // Get current number from MongoDB
        const currentNumber = await getCurrentDvirNumber();
        console.log('Current DVIR number from MongoDB:', currentNumber);
        
        // Generate preview number (will be updated when PDF is generated)
        const previewNumber = currentNumber + 1;
        console.log('Preview DVIR number:', previewNumber);
        
        // Generate the request number
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
        const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
        const paddedNumber = String(previewNumber).padStart(3, '0'); // 10 → 010, 11 → 011, 12 → 012
        const requestNumber = `${countryCode}-${projectCode}-DVIR-${paddedNumber}`;
        
        console.log('Generated DVIR number preview:', requestNumber);
        document.getElementById('requestNumber').textContent = requestNumber;
        
        // Store the request number for later use
        window.currentDvirNumber = requestNumber;
        window.currentDvirCounter = previewNumber;
        
        validateForm();
        
    } catch (error) {
        console.error('Error generating DVIR number preview:', error);
        // Fallback to timestamp-based number
        const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
        const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
        const timestamp = Date.now();
        const paddedTimestamp = String(timestamp).slice(-3).padStart(3, '0');
        const requestNumber = `${countryCode}-${projectCode}-DVIR-${paddedTimestamp}`;
        
        document.getElementById('requestNumber').textContent = requestNumber;
        showToast('warning', 'Warning', 'Using fallback number generation');
        validateForm();
    }
}

// Get current DVIR number from MongoDB
async function getCurrentDvirNumber() {
    try {
        const response = await fetch('https://api.titandrillingzm.com:6006/reports/count');
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const result = await response.json();
        
        console.log('Fetched current DVIR count from MongoDB:', result.count);
        
        // Return the count (next number will be count + 1)
        return result.count || 0;
    } catch (error) {
        console.error('Error fetching current DVIR number from MongoDB:', error);
        // Return 0 as fallback
        return 0;
    }
}

// Load equipment data from MongoDB
async function loadEquipmentData() {
    showLoadingModal();
    
    try {
        console.log('🔄 Loading equipment from MongoDB...');
        
        // Fetch from MongoDB API
        const response = await fetch('https://api.titandrillingzm.com:6001/EquipmentsDataJSON/GlobalEquipmentJsonData');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('✅ MongoDB equipment response:', result);
        
        if (result.success && Array.isArray(result.data)) {
            console.log(`Found ${result.data.length} equipment items from MongoDB`);
            
            // Transform MongoDB data to our format
            equipmentData = result.data.map((equipment, index) => ({
                id: equipment._id || equipment.id || index.toString(),
                equipmentName: equipment.vehicleNumber || equipment.equipmentName || equipment.name || equipment.model || 'Unknown Equipment',
                equipmentNumber: equipment.vehicleNumber || equipment.equipmentNumber || equipment.equipmentId || 'N/A',
                mainCategory: equipment.mainCategory || equipment.category || equipment.equipmentCategory || equipment.vehicleType || 'General',
                model: equipment.model || equipment.equipmentModel || 'N/A',
                serialNumber: equipment.vin || equipment.serialNumber || equipment.vinNumber || 'N/A',
                meterReading: equipment.odometerReading || equipment.meterReading || equipment.currentMeter || '0',
                meterUnit: equipment.meterUnit || equipment.unit || 'Hours',
                status: equipment.status || equipment.equipmentStatus || 'Available',
                location: equipment.location || equipment.site || equipment.country || 'N/A',
                manufacturer: equipment.make || equipment.manufacturer || 'N/A',
                year: equipment.year || equipment.yearOfManufacture || 'N/A',
                color: equipment.color || 'N/A',
                notes: equipment.comment || equipment.notes || '',
                lastInspection: equipment.lastInspection || equipment.lastInspectionDate || 'N/A',
                nextInspection: equipment.nextInspection || equipment.nextInspectionDate || 'N/A'
            }));
            
            console.log('Transformed equipment data:', equipmentData);
        } else {
            throw new Error('Invalid data format from MongoDB');
        }
        
        filteredEquipment = [...equipmentData];
        renderEquipmentList();
        showToast('success', 'Success', `Loaded ${equipmentData.length} equipment items from MongoDB`);
    } catch (error) {
        console.error('❌ Error loading equipment from MongoDB:', error);
        // Fallback to mock data
        equipmentData = getMockEquipmentData();
        filteredEquipment = [...equipmentData];
        renderEquipmentList();
        showToast('error', 'Error', 'Failed to load equipment from MongoDB. Using offline data.');
    } finally {
        hideLoadingModal();
    }
}


// Mock equipment data for demonstration
function getMockEquipmentData() {
    return [
        {
            id: '1',
            equipmentName: 'Excavator CAT 320D',
            equipmentNumber: 'EQ-001',
            mainCategory: 'Heavy Vehicles TDC',
            model: 'CAT 320D',
            meterReading: 1250,
            meterUnit: 'Hours',
            status: 'Available'
        },
        {
            id: '2',
            equipmentName: 'Pickup Truck Ford F-150',
            equipmentNumber: 'EQ-002',
            mainCategory: 'Light Vehicles TDC',
            model: 'Ford F-150',
            meterReading: 45000,
            meterUnit: 'Miles',
            status: 'Available'
        },
        {
            id: '3',
            equipmentName: 'Bulldozer CAT D6T',
            equipmentNumber: 'EQ-003',
            mainCategory: 'Heavy Vehicles TDC',
            model: 'CAT D6T',
            meterReading: 2100,
            meterUnit: 'Hours',
            status: 'Maintenance Required'
        },
        {
            id: '4',
            equipmentName: 'SUV Toyota Land Cruiser',
            equipmentNumber: 'EQ-004',
            mainCategory: 'Light Vehicles TDC',
            model: 'Toyota Land Cruiser',
            meterReading: 32000,
            meterUnit: 'Miles',
            status: 'Available'
        }
    ];
}

// Open equipment modal
function openEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
        // Force reflow for animation
        modal.offsetHeight;
        requestAnimationFrame(() => {
            if (modal && modal.classList) {
                modal.classList.add('show');
            }
        });
        renderEquipmentList();
    }
}

// Refresh equipment data from MongoDB
async function refreshEquipmentData() {
    showToast('info', 'Refreshing', 'Loading equipment data from MongoDB...');
    await loadEquipmentData();
}

// Close equipment modal
function closeEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    if (modal) {
        if (modal.classList) {
            modal.classList.remove('show');
        }
        setTimeout(() => {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }, 300);
    }
}

// Clear equipment selection
function clearEquipmentAutoFill() {
    selectedEquipment = null;
    document.getElementById('equipmentDisplay').value = '';
    document.getElementById('equipmentDetailsBox').style.display = 'none';
    document.getElementById('meterReading').value = '';
    updateEquipmentSelectionUI();
    validateForm();
    showToast('info', 'Equipment Cleared', 'Equipment selection has been cleared');
}


// Filter equipment by search
function filterEquipmentBySearch() {
    const searchInput = document.getElementById('equipmentSearch');
    if (!searchInput) {
        console.warn('Equipment search input not found');
        return;
    }
    
    const searchQuery = searchInput.value.toLowerCase();
    
    if (!searchQuery) {
        filteredEquipment = [...equipmentData];
    } else {
        filteredEquipment = equipmentData.filter(item => 
            (item.equipmentName || '').toLowerCase().includes(searchQuery) ||
            (item.equipmentNumber || '').toLowerCase().includes(searchQuery) ||
            (item.model || '').toLowerCase().includes(searchQuery) ||
            (item.mainCategory || '').toLowerCase().includes(searchQuery)
        );
    }
    
    renderEquipmentList();
    console.log(`Filtered ${filteredEquipment.length} equipment items`);
}

// Render equipment list
function renderEquipmentList() {
    const equipmentList = document.getElementById('equipmentList');
    equipmentList.innerHTML = '';
    
    if (filteredEquipment.length === 0) {
        equipmentList.innerHTML = '<div class="no-equipment">No equipment found</div>';
        return;
    }
    
    filteredEquipment.forEach(equipment => {
        const equipmentItem = createEquipmentItem(equipment);
        equipmentList.appendChild(equipmentItem);
    });
}

// Create equipment item element
function createEquipmentItem(equipment) {
    const item = document.createElement('div');
    item.className = 'equipment-item';
    
    const statusColor = getStatusColor(equipment.status);
    const isSelected = selectedEquipment && selectedEquipment.id === equipment.id;
    
    item.innerHTML = `
        <div class="equipment-info" onclick="selectEquipment('${equipment.id}')">
            <div class="equipment-header">
                <span class="equipment-name">${equipment.equipmentName || 'Equipment'}</span>
                <span class="equipment-number">${equipment.equipmentNumber || 'N/A'}</span>
            </div>
            <div class="equipment-details">
                <div class="detail-row">
                    <span class="detail-label">Category</span>
                    <span class="detail-value">${equipment.mainCategory || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Model</span>
                    <span class="detail-value">${equipment.model || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Serial Number</span>
                    <span class="detail-value">${equipment.serialNumber || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Meter Reading</span>
                    <span class="detail-value">${equipment.meterReading || '0'} ${equipment.meterUnit || 'Hours'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Location</span>
                    <span class="detail-value">${equipment.location || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <span class="detail-label">Manufacturer</span>
                    <span class="detail-value">${equipment.manufacturer || 'N/A'}</span>
                </div>
                <div class="detail-row no-border">
                    <span class="detail-label">Status</span>
                    <span class="detail-value status" style="color: ${statusColor}">${equipment.status || 'Available'}</span>
                </div>
            </div>
        </div>
        <button class="select-equipment-btn ${isSelected ? 'selected' : ''}" onclick="selectEquipment('${equipment.id}')">
            ${isSelected ? 'Selected' : 'Select'}
        </button>
    `;
    
    return item;
}

// Get status color
function getStatusColor(status) {
    switch (status) {
        case 'Available': return '#2ecc71';
        case 'Maintenance Required': return '#f1c40f';
        case 'In Maintenance': return '#e67e22';
        case 'Breakdown': return '#e74c3c';
        case 'Deploy': return '#3498db';
        default: return '#666';
    }
}

// Select equipment
function selectEquipment(equipmentId) {
    console.log('=== EQUIPMENT SELECTION ===');
    console.log('Equipment ID:', equipmentId);
    
    selectedEquipment = equipmentData.find(item => item.id === equipmentId);
    
    if (selectedEquipment) {
        console.log('✅ Equipment found and selected');
        console.log('');
        
        console.log('📋 EQUIPMENT DETAILS:');
        console.log('  Equipment ID:', selectedEquipment.id);
        console.log('  Equipment Name:', selectedEquipment.equipmentName);
        console.log('  Equipment Number:', selectedEquipment.equipmentNumber);
        console.log('  Main Category:', selectedEquipment.mainCategory);
        console.log('  Model:', selectedEquipment.model);
        console.log('  Serial Number:', selectedEquipment.serialNumber);
        console.log('  Equipment Status:', selectedEquipment.status);
        console.log('  Location:', selectedEquipment.location);
        console.log('  Manufacturer:', selectedEquipment.manufacturer);
        console.log('  Year:', selectedEquipment.year);
        console.log('  Color:', selectedEquipment.color);
        console.log('  Notes:', selectedEquipment.notes);
        console.log('  Last Inspection:', selectedEquipment.lastInspection);
        console.log('  Next Inspection:', selectedEquipment.nextInspection);
        console.log('');
        
        console.log('📊 METER READING DETAILS:');
        console.log('  Current Meter Reading:', selectedEquipment.meterReading);
        console.log('  Meter Unit:', selectedEquipment.meterUnit);
        console.log('');
        
        // Update UI
        document.getElementById('equipmentDisplay').value = selectedEquipment.equipmentName || 'N/A';
        document.getElementById('equipmentDetailsBox').style.display = 'block';
        
        // Update equipment info with more details
        const equipmentInfo = `Equipment: ${selectedEquipment.equipmentName || 'N/A'} | Number: ${selectedEquipment.equipmentNumber || 'N/A'} | Category: ${selectedEquipment.mainCategory || 'N/A'} | Model: ${selectedEquipment.model || 'N/A'}`;
        document.getElementById('equipmentInfoText').textContent = equipmentInfo;
        
        // Update meter reading
        document.getElementById('meterReading').value = selectedEquipment.meterReading || '';
        document.getElementById('meterUnit').textContent = selectedEquipment.meterUnit || 'Hours';
        document.getElementById('currentMeterValue').textContent = `${selectedEquipment.meterReading || '0'} ${selectedEquipment.meterUnit || 'Hours'}`;
        
        console.log('🔄 UI UPDATES:');
        console.log('  Equipment Info Text:', equipmentInfo);
        console.log('  Meter Reading Input Value:', selectedEquipment.meterReading || '');
        console.log('  Meter Unit Display:', selectedEquipment.meterUnit || 'Hours');
        console.log('  Current Meter Value Display:', `${selectedEquipment.meterReading || '0'} ${selectedEquipment.meterUnit || 'Hours'}`);
        console.log('');
        
        // Update all equipment items to show selection state
        updateEquipmentSelectionUI();
        
        // Close modal
        closeEquipmentModal();
        
        // Validate form
        validateForm();
        validateMeterReading();
        
        console.log('✅ Equipment selection completed successfully');
        console.log('=== END EQUIPMENT SELECTION ===');
        
        showToast('success', 'Equipment Selected', `${selectedEquipment.equipmentName} has been selected`);
    } else {
        console.log('❌ Equipment not found with ID:', equipmentId);
        console.log('Available equipment IDs:', equipmentData.map(eq => eq.id));
    }
}

// Update equipment selection UI
function updateEquipmentSelectionUI() {
    // Re-render the equipment list to show updated selection states
    renderEquipmentList();
}

// Validate meter reading
function validateMeterReading() {
    console.log('=== METER READING VALIDATION ===');
    
    const meterReadingInput = document.getElementById('meterReading').value;
    const meterReading = parseFloat(meterReadingInput);
    const errorDiv = document.getElementById('meterReadingError');
    
    console.log('  Meter Reading Input (raw):', meterReadingInput);
    console.log('  Meter Reading Input (parsed):', meterReading);
    console.log('  Selected Equipment:', selectedEquipment ? 'Yes' : 'No');
    
    // Check if meter reading is provided
    if (!meterReadingInput || isNaN(meterReading)) {
        console.log('❌ Meter Reading Validation Failed: No valid meter reading entered');
        errorDiv.textContent = 'Meter reading is required';
        errorDiv.style.display = 'block';
        console.log('=== END METER READING VALIDATION (FAILED) ===');
        return false;
    }
    
    if (selectedEquipment) {
        console.log('  Equipment Current Meter Reading:', selectedEquipment.meterReading);
        console.log('  Equipment Meter Unit:', selectedEquipment.meterUnit);
        console.log('  Equipment Meter Reading (parsed):', parseFloat(selectedEquipment.meterReading));
        
        const currentMeterReading = parseFloat(selectedEquipment.meterReading || 0);
        
        // Only validate if there's a valid current meter reading and new reading is less
        if (currentMeterReading > 0 && meterReading < currentMeterReading) {
            const errorMessage = `Meter reading should not be less than current reading (${currentMeterReading}).`;
            console.log('❌ Meter Reading Validation Failed:', errorMessage);
            errorDiv.textContent = errorMessage;
            errorDiv.style.display = 'block';
            console.log('=== END METER READING VALIDATION (FAILED) ===');
            return false;
        }
    }
    
    console.log('✅ Meter Reading Validation Passed');
    errorDiv.style.display = 'none';
    console.log('=== END METER READING VALIDATION (PASSED) ===');
    return true;
}

// Validate form
function validateForm() {
    console.log('=== FORM VALIDATION ===');
    
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const inspector = document.getElementById('inspector').value.trim();
    const dvirChecked = document.getElementById('dvirCheckbox').checked;
    
    console.log('  Country:', country, country ? '✅' : '❌');
    console.log('  Project:', project, project ? '✅' : '❌');
    console.log('  Inspector:', inspector, inspector ? '✅' : '❌');
    console.log('  Selected Equipment:', selectedEquipment ? '✅' : '❌');
    console.log('  DVIR Checked:', dvirChecked, dvirChecked ? '✅' : '❌');
    
    const meterReadingValid = validateMeterReading();
    console.log('  Meter Reading Valid:', meterReadingValid, meterReadingValid ? '✅' : '❌');
    
    const isValid = country && project && inspector && selectedEquipment && dvirChecked && meterReadingValid;
    
    console.log('  Overall Form Valid:', isValid, isValid ? '✅' : '❌');
    console.log('  Next Button Disabled:', !isValid);
    console.log('=== END FORM VALIDATION ===');
    
    document.getElementById('nextButton').disabled = !isValid;
    
    return isValid;
}

// Handle next button press
function handleNextPress() {
    console.log('=== DVIR STEP 1 - NEXT BUTTON CLICKED ===');
    console.log('Timestamp:', new Date().toISOString());
    console.log('');
    
    if (!validateForm()) {
        console.log('❌ Form validation failed - cannot proceed');
        showToast('error', 'Error', 'Please fill all required fields and ensure meter reading is valid.');
        return;
    }
    
    console.log('✅ Form validation passed - proceeding with data collection');
    console.log('');
    
    // Collect all form data
    const requestNumber = document.getElementById('requestNumber').textContent;
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const inspector = document.getElementById('inspector').value;
    const meterReading = document.getElementById('meterReading').value;
    const meterUnit = document.getElementById('meterUnit').textContent;
    const currentMeterValue = document.getElementById('currentMeterValue').textContent;
    const dvirChecked = document.getElementById('dvirCheckbox').checked;
    
    console.log('📋 REQUEST INFORMATION:');
    console.log('  Request Number:', requestNumber);
    console.log('');
    
    console.log('🌍 LOCATION INFORMATION:');
    console.log('  Country:', country);
    console.log('  Project:', project);
    console.log('');
    
    console.log('👤 INSPECTOR INFORMATION:');
    console.log('  Inspector Name:', inspector);
    console.log('');
    
    console.log('🚛 EQUIPMENT INFORMATION:');
    if (selectedEquipment) {
        console.log('  Equipment ID:', selectedEquipment.id);
        console.log('  Equipment Name:', selectedEquipment.equipmentName);
        console.log('  Equipment Number:', selectedEquipment.equipmentNumber);
        console.log('  Main Category:', selectedEquipment.mainCategory);
        console.log('  Model:', selectedEquipment.model);
        console.log('  Serial Number:', selectedEquipment.serialNumber);
        console.log('  Equipment Status:', selectedEquipment.status);
        console.log('  Location:', selectedEquipment.location);
        console.log('  Manufacturer:', selectedEquipment.manufacturer);
        console.log('  Year:', selectedEquipment.year);
        console.log('  Color:', selectedEquipment.color);
        console.log('  Notes:', selectedEquipment.notes);
        console.log('  Last Inspection:', selectedEquipment.lastInspection);
        console.log('  Next Inspection:', selectedEquipment.nextInspection);
    } else {
        console.log('  ❌ No equipment selected');
    }
    console.log('');
    
    console.log('📊 METER READING INFORMATION:');
    console.log('  Current Meter Reading (from equipment):', selectedEquipment ? selectedEquipment.meterReading : 'N/A');
    console.log('  Meter Unit (from equipment):', selectedEquipment ? selectedEquipment.meterUnit : 'N/A');
    console.log('  New Meter Reading (user input):', meterReading);
    console.log('  Meter Unit (display):', meterUnit);
    console.log('  Current Meter Value (display):', currentMeterValue);
    console.log('  Meter Reading Validation:', validateMeterReading() ? '✅ Valid' : '❌ Invalid');
    console.log('');
    
    console.log('✅ INSPECTION TYPES:');
    console.log('  DVIR Checked:', dvirChecked);
    console.log('');
    
    // Save step 1 data
    const step1Data = {
        requestNumber: requestNumber,
        country: country,
        project: project,
        inspector: inspector,
        selectedEquipment: selectedEquipment,
        meterReading: meterReading,
        meterUnit: meterUnit,
        currentMeterValue: currentMeterValue,
        inspectionTypes: {
            DVIR: dvirChecked
        },
        timestamp: new Date().getTime(),
        dateTime: new Date().toISOString()
    };
    
    console.log('📦 COMPLETE STEP 1 DATA OBJECT:');
    console.log(JSON.stringify(step1Data, null, 2));
    console.log('');
    
    console.log('💾 SAVING TO SESSION STORAGE...');
    // Store in session storage
    sessionStorage.setItem('dvirStep1Data', JSON.stringify(step1Data));
    console.log('✅ Data saved to session storage successfully');
    console.log('');
    
    console.log('🔄 REDIRECTING TO STEP 2...');
    console.log('=== END OF STEP 1 DATA LOGGING ===');
    
    // Navigate to step 2 using the new navigation function
    navigateToStep(2);
}

// Handle back button press
function handleBackPress() {
    if (confirm('Do you want to exit this step? Any unsaved data will be lost.')) {
        window.history.back();
    }
}

// Show loading modal
function showLoadingModal() {
    const modal = document.getElementById('loadingModal');
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
        // Force reflow for animation
        modal.offsetHeight;
        requestAnimationFrame(() => {
            if (modal && modal.classList) {
                modal.classList.add('show');
            }
        });
    }
}

// Hide loading modal
function hideLoadingModal() {
    const modal = document.getElementById('loadingModal');
    if (modal) {
        if (modal.classList) {
            modal.classList.remove('show');
        }
        setTimeout(() => {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }, 300);
    }
}

// Show toast notification
function showToast(type, title, message) {
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.innerHTML = `
        <div class="toast-content">
            <h4>${title}</h4>
            <p>${message}</p>
        </div>
        <button class="toast-close" onclick="this.parentElement.remove()">×</button>
    `;
    
    document.getElementById('toastContainer').appendChild(toast);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (toast.parentElement) {
            toast.remove();
        }
    }, 5000);
}

// ==================== STEP MANAGEMENT FUNCTIONS ====================

// Update step indicator
function updateStepIndicator(activeStep) {
    console.log('Updating step indicator to:', activeStep);
    
    const stepItems = document.querySelectorAll('.step-item');
    const connectors = document.querySelectorAll('.step-connector');
    
    stepItems.forEach((item, index) => {
        item.classList.remove('active', 'completed');
        if (index + 1 === activeStep) {
            item.classList.add('active');
        } else if (index + 1 < activeStep) {
            item.classList.add('completed');
        }
    });
    
    // Update connectors
    connectors.forEach((connector, index) => {
        if (index + 1 < activeStep) {
            connector.style.background = 'var(--primary-color)';
        } else {
            connector.style.background = '#e5e7eb';
        }
    });
}

// Navigate to specific step
function navigateToStep(step) {
    console.log('Navigating to step:', step);
    
    // Validate step before navigation
    if (step === 2 && !validateStep1()) {
        showToast('error', 'Validation Error', 'Please complete Step 1 before proceeding');
        return;
    }
    
    if (step === 3 && !validateStep2()) {
        showToast('error', 'Validation Error', 'Please complete Step 2 before proceeding');
        return;
    }
    
    // Show the appropriate step
    switch(step) {
        case 1:
            showStep1();
            break;
        case 2:
            showStep2();
            break;
        case 3:
            showStep3();
            break;
        default:
            console.error('Invalid step number:', step);
            return;
    }
    
    // Update step indicator
    updateStepIndicator(step);
    
    // Save current step to localStorage
    localStorage.setItem('dvirCurrentStep', step);
}

// Validate Step 1
function validateStep1() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    const inspector = document.getElementById('inspector').value.trim();
    const selectedEquipment = document.getElementById('equipmentDisplay').value.trim() !== '';
    const meterReading = document.getElementById('meterReading').value.trim();
    
    if (!country || !project || !inspector || !selectedEquipment || !meterReading) {
        return false;
    }
    
    return true;
}

// Validate Step 2
function validateStep2() {
    // Check if at least one checklist item has been completed
    const checklistItems = document.querySelectorAll('#checklistContainer .checklist-item');
    let hasCompletedItem = false;
    
    checklistItems.forEach(item => {
        const statusSelect = item.querySelector('.status-select');
        if (statusSelect && statusSelect.value && statusSelect.value !== '') {
            hasCompletedItem = true;
        }
    });
    
    return hasCompletedItem;
}

// Show Step 1
function showStep1() {
    currentStep = 1;
    document.getElementById('step1Container').style.display = 'block';
    document.getElementById('step2Container').style.display = 'none';
    document.getElementById('step3Container').style.display = 'none';
    updateStepIndicator(1);
    console.log('Switched to Step 1');
}

// Show Step 2
function showStep2() {
    currentStep = 2;
    document.getElementById('step1Container').style.display = 'none';
    document.getElementById('step2Container').style.display = 'block';
    document.getElementById('step3Container').style.display = 'none';
    updateStepIndicator(2);
    console.log('Switched to Step 2');
    loadStep2Data();
    validateStep2();
}

// Load existing Step 2 data
function loadStep2Data() {
    // Restore button states
    Object.entries(step2Data).forEach(([key, value]) => {
        if (key.includes('_Status')) {
            const [section, itemId, field] = key.split('_');
            const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
            if (itemContainer) {
                const buttons = itemContainer.querySelectorAll('.status-btn');
                buttons.forEach(btn => {
                    btn.classList.remove('selected');
                    if (btn.textContent.trim() === value.status) {
                        btn.classList.add('selected');
                    }
                });
                
                // Restore note if exists
                if (value.note) {
                    const noteInput = itemContainer.querySelector('.note-input');
                    if (noteInput) {
                        noteInput.value = value.note;
                    }
                }
            }
        }
        
        // Restore photo indicators and previews
        if (key.startsWith('photo_')) {
            const itemId = key.replace('photo_', '');
            updatePhotoIndicator(itemId, true);
            showImagePreview(itemId, value.data);
            console.log(`Restored photo for item ${itemId}:`, value);
        }
    });
    
    console.log('Step 2 data loaded:', step2Data);
}

// Show Step 3
function showStep3() {
    currentStep = 3;
    document.getElementById('step1Container').style.display = 'none';
    document.getElementById('step2Container').style.display = 'none';
    document.getElementById('step3Container').style.display = 'block';
    updateStepIndicator(3);
    console.log('Switched to Step 3');
    loadFaultyItems();
    validateStep3();
}

// ==================== STEP 2 FUNCTIONS ====================

// Toggle section expansion
function toggleSection(sectionName) {
    const content = document.getElementById(`${sectionName}-content`);
    const icon = document.getElementById(`${sectionName}-icon`);
    
    if (content.style.display === 'none') {
        content.style.display = 'block';
        icon.textContent = '▲';
    } else {
        content.style.display = 'none';
        icon.textContent = '▼';
    }
    console.log(`Toggled ${sectionName} section`);
}

// Set status for checklist item
function setStatus(section, itemId, status) {
    const key = `${section}_${itemId}_Status`;
    step2Data[key] = { status: status, note: step2Data[key]?.note || '' };
    
    // Update button states for this specific item
    const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
    if (itemContainer) {
        const buttons = itemContainer.querySelectorAll('.status-btn');
        buttons.forEach(btn => {
            btn.classList.remove('selected');
            if (btn.textContent.trim() === status) {
                btn.classList.add('selected');
            }
        });
        
        // Enable action buttons after selection
        const actionButtons = itemContainer.querySelectorAll('.action-btn');
        actionButtons.forEach(btn => {
            btn.disabled = false;
            btn.style.opacity = '1';
            btn.style.cursor = 'pointer';
        });
    }
    
    console.log(`Set status for ${key}: ${status}`);
    console.log('Current step2Data:', step2Data);
    validateStep2();
}

// Toggle note input for an item
function toggleNote(itemId) {
    const noteContainer = document.getElementById(`note-${itemId}`);
    if (noteContainer) {
        const isVisible = noteContainer.style.display !== 'none';
        noteContainer.style.display = isVisible ? 'none' : 'block';
        
        if (!isVisible) {
            // Focus on the input when showing
            const input = noteContainer.querySelector('.note-input');
            if (input) {
                input.focus();
            }
        }
        
        console.log(`Toggled note for item: ${itemId}, visible: ${!isVisible}`);
    }
}

// Update note for an item
function updateNote(section, itemId, note) {
    const key = `${section}_${itemId}_Status`;
    if (step2Data[key]) {
        step2Data[key].note = note;
    } else {
        step2Data[key] = { status: '', note: note };
    }
    
    console.log(`Updated note for ${key}: ${note}`);
    console.log('Current step2Data:', step2Data);
    validateStep2();
}

// Take photo (enhanced with file input)
function takePhoto(itemId) {
    // Create a file input element
    const fileInput = document.createElement('input');
    fileInput.type = 'file';
    fileInput.accept = 'image/*';
    fileInput.capture = 'environment'; // Use back camera on mobile
    
    fileInput.onchange = function(event) {
        const file = event.target.files[0];
        if (file) {
            // Validate file size (max 5MB)
            if (file.size > 5 * 1024 * 1024) {
                showToast('error', 'Error', 'Image file is too large. Please select an image smaller than 5MB.');
                return;
            }
            
            // Validate file type
            if (!file.type.startsWith('image/')) {
                showToast('error', 'Error', 'Please select a valid image file.');
                return;
            }
            
            // Process the image
            const reader = new FileReader();
            reader.onload = function(e) {
                const imageData = e.target.result;
                
                // Store image data
                const key = `photo_${itemId}`;
                step2Data[key] = {
                    data: imageData,
                    filename: file.name,
                    timestamp: new Date().getTime()
                };
                
                // Show success message
                showToast('success', 'Photo Added', `Photo captured for item ${itemId}`);
                console.log(`Photo captured for item: ${itemId}`, {
                    filename: file.name,
                    size: file.size,
                    type: file.type,
                    key: key,
                    step2Data: step2Data
                });
                
                // Update UI to show photo was taken and display preview
                updatePhotoIndicator(itemId, true);
                showImagePreview(itemId, imageData);
                
                // Show success message with photo count
                const photoCount = Object.keys(step2Data).filter(key => key.startsWith('photo_')).length;
                showToast('success', 'Photo Added', `Photo captured for item ${itemId} (Total photos: ${photoCount})`);
            };
            
            reader.readAsDataURL(file);
        }
    };
    
    // Trigger file input
    fileInput.click();
    console.log(`Photo capture initiated for item: ${itemId}`);
}

// Update photo indicator
function updatePhotoIndicator(itemId, hasPhoto) {
    const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
    if (itemContainer) {
        const photoButton = itemContainer.querySelector('[onclick*="takePhoto"]');
        if (photoButton) {
            if (hasPhoto) {
                photoButton.style.backgroundColor = '#2ecc71';
                photoButton.title = 'Photo Added - Click to change';
            } else {
                photoButton.style.backgroundColor = '';
                photoButton.title = 'Take Photo';
            }
        }
    }
}

// Show image preview
function showImagePreview(itemId, imageData) {
    const previewContainer = document.getElementById(`image-preview-${itemId}`);
    const previewImage = document.getElementById(`preview-img-${itemId}`);
    
    if (previewContainer && previewImage) {
        previewImage.src = imageData;
        previewContainer.style.display = 'block';
        console.log(`Image preview shown for item: ${itemId}`);
    }
}

// Remove image
function removeImage(itemId) {
    const key = `photo_${itemId}`;
    delete step2Data[key];
    
    // Hide preview container
    const previewContainer = document.getElementById(`image-preview-${itemId}`);
    if (previewContainer) {
        previewContainer.style.display = 'none';
    }
    
    // Update photo indicator
    updatePhotoIndicator(itemId, false);
    
    // Show success message
    showToast('info', 'Photo Removed', `Photo removed for item ${itemId}`);
    console.log(`Image removed for item: ${itemId}`);
    
    // Validate step 2
    validateStep2();
}

// Copy item (enhanced functionality)
function copyItem(itemId) {
    // Find the item text
    const itemContainer = document.querySelector(`[data-item-id="${itemId}"]`);
    if (itemContainer) {
        const itemText = itemContainer.querySelector('.item-text').textContent;
        
        // Copy to clipboard
        if (navigator.clipboard) {
            navigator.clipboard.writeText(itemText).then(() => {
                showToast('success', 'Copied', 'Item text copied to clipboard');
                console.log(`Copied item text: ${itemText}`);
            }).catch(err => {
                console.error('Failed to copy text: ', err);
                showToast('error', 'Error', 'Failed to copy text');
            });
        } else {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = itemText;
            document.body.appendChild(textArea);
            textArea.select();
            try {
                document.execCommand('copy');
                showToast('success', 'Copied', 'Item text copied to clipboard');
                console.log(`Copied item text: ${itemText}`);
            } catch (err) {
                console.error('Failed to copy text: ', err);
                showToast('error', 'Error', 'Failed to copy text');
            }
            document.body.removeChild(textArea);
        }
    }
}

// Validate Step 2
function validateStep2() {
    const hasSelections = Object.keys(step2Data).length > 0;
    const nextButton = document.getElementById('step2NextButton');
    
    nextButton.disabled = !hasSelections;
    console.log(`Step 2 validation: ${hasSelections ? 'Valid' : 'Invalid'}`);
    
    return hasSelections;
}

// Handle Step 2 Next
function handleStep2Next() {
    if (!validateStep2()) {
        showToast('error', 'Error', 'Please complete at least one checklist item.');
        return;
    }
    
    console.log('Step 2 data:', step2Data);
    showToast('success', 'Step 2 Complete', 'Checklist data saved successfully');
    navigateToStep(3);
}

// ==================== STEP 3 FUNCTIONS ====================

// Load faulty items count
function loadFaultyItems() {
    let faultyCount = 0;
    Object.entries(step2Data).forEach(([key, value]) => {
        if (key.includes('_Status') && (value.status === 'No' || value.status === 'Fail')) {
            faultyCount++;
        }
    });
    
    document.getElementById('faultyItemsCount').textContent = faultyCount;
    console.log(`Faulty items count: ${faultyCount}`);
}

// Handle driver status change
function handleDriverStatusChange() {
    const status = document.querySelector('input[name="equipmentStatus"]:checked');
    const priorityGroup = document.getElementById('priorityGroup');
    const driverStatusChip = document.getElementById('driverStatusChip');
    
    if (status && status.value === 'Unfit to travel') {
        priorityGroup.style.display = 'block';
    } else {
        priorityGroup.style.display = 'none';
        step3Data.priority = null;
    }

    if (driverStatusChip) {
        if (status) {
            driverStatusChip.textContent = `Driver status: ${status.value}`;
            driverStatusChip.classList.toggle('pill-alert', status.value === 'Unfit to travel');
            driverStatusChip.classList.toggle('pill-muted', false);
        } else {
            driverStatusChip.textContent = 'Driver status pending';
            driverStatusChip.classList.add('pill-muted');
            driverStatusChip.classList.remove('pill-alert');
        }
    }
    
    validateStep3();
    console.log('Driver status changed:', status ? status.value : 'None');
}

// Select priority
function selectPriority(priority) {
    step3Data.priority = priority;
    
    // Update UI
    document.querySelectorAll('.priority-option').forEach(option => {
        option.classList.remove('selected');
        if (option.textContent.includes(priority)) {
            option.classList.add('selected');
        }
    });
    
    console.log('Priority selected:', priority);
    validateStep3();
}

// Handle voice input (placeholder)
function handleVoiceInput() {
    console.log('Voice input triggered');
    showToast('info', 'Voice Input', 'Voice input functionality would be implemented here');
}

// Handle signature
function handleSignature() {
    openSignatureModal();
    console.log('Signature modal opened');
}

// Close signature modal
function closeSignatureModal() {
    const modal = document.getElementById('signatureModal');
    if (modal) {
        if (modal.classList) {
            modal.classList.remove('show');
        }
        setTimeout(() => {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }, 300);
    }
    console.log('Signature modal closed');
}

// Open signature modal
function openSignatureModal() {
    const modal = document.getElementById('signatureModal');
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
        // Force reflow for animation
        modal.offsetHeight;
        requestAnimationFrame(() => {
            if (modal.classList) {
                modal.classList.add('show');
            }
        });
    }
}

// Clear signature
function clearSignature() {
    const canvas = document.getElementById('signaturePad');
    const ctx = canvas.getContext('2d');
    ctx.clearRect(0, 0, canvas.width, canvas.height);
    console.log('Signature cleared');
}

// Save signature
function saveSignature() {
    const canvas = document.getElementById('signaturePad');
    const dataURL = canvas.toDataURL('image/png');
    
    signatureData = dataURL;
    signatureStatus = true;
    
    // Update UI
    document.getElementById('signaturePlaceholder').style.display = 'none';
    document.getElementById('signatureDisplay').style.display = 'block';
    document.getElementById('signatureImage').src = dataURL;
    
    closeSignatureModal();
    validateStep3();
    console.log('Signature saved');
}

// Validate Step 3
function validateStep3() {
    const overallCondition = document.getElementById('overallCondition').value;
    const equipmentStatus = document.querySelector('input[name="equipmentStatus"]:checked');
    const safeToUse = document.querySelector('input[name="safeToUse"]:checked');
    const safeChip = document.getElementById('safeToUseChip');
    
    let isValid = overallCondition && equipmentStatus && safeToUse && signatureStatus;
    
    // Clear previous errors
    document.getElementById('overallConditionError').style.display = 'none';
    document.getElementById('equipmentStatusError').style.display = 'none';
    document.getElementById('safeToUseError').style.display = 'none';
    document.getElementById('signatureError').style.display = 'none';
    
    // Show specific errors
    if (!overallCondition) {
        document.getElementById('overallConditionError').textContent = 'Overall Condition is required';
        document.getElementById('overallConditionError').style.display = 'block';
    }
    
    if (!equipmentStatus) {
        document.getElementById('equipmentStatusError').textContent = 'Driver Status is required';
        document.getElementById('equipmentStatusError').style.display = 'block';
    }
    
    if (!safeToUse) {
        document.getElementById('safeToUseError').textContent = 'Equipment Safe to Use is required';
        document.getElementById('safeToUseError').style.display = 'block';
    }
    
    if (!signatureStatus) {
        document.getElementById('signatureError').textContent = 'Inspector Signature is required';
        document.getElementById('signatureError').style.display = 'block';
    }

    if (safeChip) {
        if (safeToUse) {
            safeChip.textContent = `Equipment safe to use: ${safeToUse.value}`;
            safeChip.classList.remove('pill-outline', 'pill-success', 'pill-danger');
            safeChip.classList.add(safeToUse.value === 'Yes' ? 'pill-success' : 'pill-danger');
        } else {
            safeChip.textContent = 'Equipment safety not evaluated';
            safeChip.classList.remove('pill-success', 'pill-danger');
            safeChip.classList.add('pill-outline');
        }
    }
    
    const nextButton = document.getElementById('step3NextButton');
    nextButton.disabled = !isValid;
    
    console.log(`Step 3 validation: ${isValid ? 'Valid' : 'Invalid'}`);
    return isValid;
}

// Handle Step 3 Next (Generate Report)
function handleStep3Next() {
    if (!validateStep3()) {
        showToast('error', 'Error', 'Please complete all required fields.');
        return;
    }
    
    // Collect Step 3 data - URLs will be generated after actual upload
    const signatureDownloadUrl = signatureData ? null : null; // Will be set after upload
    const pdfDownloadUrl = null; // Will be set after upload
    
    // Use the generated DVIR number from step 1
    const dvirNumber = window.currentDvirNumber || document.getElementById('requestNumber').textContent;
    
    step3Data = {
        dvirNumber: dvirNumber,
        overallCondition: document.getElementById('overallCondition').value,
        equipmentStatus: document.querySelector('input[name="equipmentStatus"]:checked').value,
        priority: step3Data.priority || null,
        safeToUse: document.querySelector('input[name="safeToUse"]:checked').value,
        additionalNote: document.getElementById('additionalNote').value,
        signatureStatus: signatureStatus,
        signatureData: signatureData,
        signatureDownloadUrl: signatureDownloadUrl,
        pdfDownloadUrl: pdfDownloadUrl,
        faultyItems: parseInt(document.getElementById('faultyItemsCount').textContent),
        createdAt: new Date().toISOString(),
        timestamp: Date.now()
    };
    
    console.log('Step 3 data:', step3Data);
    generateReport();
}

// Save report to MongoDB
async function saveReportToMongoDB(reportData) {
    try {
        // For now, we'll use a hardcoded user ID. In a real application, this would come from authentication
        const userId = 'cahj3jJqe5Z16pQdLGSWrNgm84o1'; // This should be dynamic based on logged-in user
        
        console.log('💾 Saving report to MongoDB for user:', userId);
        console.log('Report data to save:', reportData);
        console.log('Report data structure validation:');
        console.log('- Has id:', !!reportData.id);
        console.log('- Has dvirNumber:', !!reportData.dvirNumber);
        console.log('- Has step1:', !!reportData.step1);
        console.log('- Has step2:', !!reportData.step2);
        console.log('- Has step3:', !!reportData.step3);
        console.log('- Step2 has items:', !!(reportData.step2 && reportData.step2.items));
        console.log('- Step2 items count:', reportData.step2 ? reportData.step2.items.length : 0);
        
        // Save to MongoDB via API
        const response = await fetch('https://api.titandrillingzm.com:6006/reports', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                ...reportData,
                userUid: userId,
                reportId: reportData.id
            })
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        
        if (result.success) {
            console.log('✅ Report saved successfully to MongoDB');
            showToast('success', 'Success', 'Report saved to MongoDB successfully');
            return {
                success: true,
                message: 'Report saved successfully',
                data: {
                    reportId: reportData.id,
                    dvirNumber: reportData.dvirNumber,
                    mongoId: result.data._id
                }
            };
        } else {
            throw new Error(result.error || 'Failed to save to MongoDB');
        }
    } catch (error) {
        console.error('❌ Error saving report to MongoDB:', error);
        throw error;
    }
}

// Helper function to convert data URL to blob
function dataURLtoBlob(dataurl) {
    const arr = dataurl.split(',');
    const mime = arr[0].match(/:(.*?);/)[1];
    const bstr = atob(arr[1]);
    let n = bstr.length;
    const u8arr = new Uint8Array(n);
    while (n--) {
        u8arr[n] = bstr.charCodeAt(n);
    }
    return new Blob([u8arr], { type: mime });
}

// Upload PDF to MongoDB server
async function uploadPDFToServer(pdfBlob, userUid, reportId) {
    try {
        console.log('📤 Uploading PDF to MongoDB server...');
        console.log('📤 Upload details:', {
            userUid,
            reportId,
            blobSize: pdfBlob.size,
            blobType: pdfBlob.type
        });
        
        // Create FormData for file upload
        const formData = new FormData();
        formData.append('pdf', pdfBlob, `DVIR_Report_${reportId}.pdf`);
        formData.append('userUid', userUid);
        formData.append('reportId', reportId);
        
        const response = await fetch('https://api.titandrillingzm.com:6006/reports/upload-pdf', {
            method: 'POST',
            body: formData
        });
        
        console.log('📤 Upload response status:', response.status, response.statusText);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ Upload failed with error:', errorText);
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('📤 Upload response data:', result);
        
        if (result.success) {
            console.log('✅ PDF uploaded successfully!');
            console.log('📄 Download URL:', result.downloadUrl);
            console.log('📄 URL starts with http:', result.downloadUrl?.startsWith('http'));
            console.log('📄 URL is full URL:', result.downloadUrl?.startsWith('http://') || result.downloadUrl?.startsWith('https://'));
            return result.downloadUrl;
        } else {
            throw new Error(result.error || 'Failed to upload PDF');
        }
    } catch (error) {
        console.error('❌ Error uploading PDF to server:', error);
        console.error('❌ Error details:', {
            name: error.name,
            message: error.message,
            stack: error.stack
        });
        throw error;
    }
}

// Generate PDF Report
async function generateReport() {
    try {
        showLoadingModal();
        
        // Get Step 1 data
        const step1Data = JSON.parse(sessionStorage.getItem('dvirStep1Data') || '{}');
        
        // Transform step2Data to match the exact React Native format
        const step2Items = [];
        
        // Process requirements
        const requirementsItems = [
            { id: 'req1', label: "I hold a valid driver's licence" },
            { id: 'req2', label: "I will obey speed limits and road signs" },
            { id: 'req3', label: "I will not overload the vehicle and ensure the load is secure" },
            { id: 'req4', label: "I have completed the vehicle pre check list and satisfied all is order for my trip" },
            { id: 'req5', label: "I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)" },
            { id: 'req6', label: "I have read and understand the Titan driving procedure, National driving rules and cliental rules" },
            { id: 'req7', label: "I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)" }
        ];
        
        // Process hazards
        const hazardsItems = [
            { id: 'haz1', label: "Pedestrians crossing the road, school children and drunkards" },
            { id: 'haz2', label: "Motor bikes and taxis stopping suddenly or pulling in to the road suddenly" },
            { id: 'haz3', label: "Oncoming vehicles overtaking on blind rises or corners" },
            { id: 'haz4', label: "Heavy trucks or plant taking up width of the road" },
            { id: 'haz5', label: "Riots or unrest" },
            { id: 'haz6', label: "Gravel/Sandy roads can be slippery – Loss of control" },
            { id: 'haz7', label: "Tyre blowouts or punctures – Loss of control" }
        ];
        
        // Process requirements items
        requirementsItems.forEach(item => {
            const key = `requirements_${item.id}_Status`;
            const data = step2Data[key] || {};
            const photoKey = `photo_${item.id}`;
            const photoData = step2Data[photoKey] || null;
            
            step2Items.push({
                itemId: item.id,
                label: item.label,
                note: data.note || '',
                section: 'Requirements',
                status: data.status || '',
                photo: photoData ? {
                    data: photoData.data,
                    filename: photoData.filename,
                    size: photoData.size,
                    type: photoData.type
                } : null
            });
        });
        
        // Process hazards items
        hazardsItems.forEach(item => {
            const key = `hazards_${item.id}_Status`;
            const data = step2Data[key] || {};
            const photoKey = `photo_${item.id}`;
            const photoData = step2Data[photoKey] || null;
            
            step2Items.push({
                itemId: item.id,
                label: item.label,
                note: data.note || '',
                section: 'Hazards – Are you aware of:',
                status: data.status || '',
                photo: photoData ? {
                    data: photoData.data,
                    filename: photoData.filename,
                    size: photoData.size,
                    type: photoData.type
                } : null
            });
        });
        
        const transformedStep2 = {
            items: step2Items,
            timestamp: Date.now()
        };
        
        // Calculate faulty items count
        let faultyCount = 0;
        step2Items.forEach(item => {
            if (item.status === 'No' || item.status === 'Fail') {
                faultyCount++;
            }
        });
        
        // Update step3Data with faulty count
        step3Data.faultyItems = faultyCount;
        
        // Create report data with exact structure
        const reportData = {
            id: Date.now().toString(),
            dvirNumber: step3Data.dvirNumber,
            step1: {
                ...step1Data,
                requestNumber: step3Data.dvirNumber, // Use the generated DVIR number
                timestamp: step1Data.timestamp || Date.now()
            },
            step2: transformedStep2,
            step3: step3Data,
            createdAt: new Date().toISOString()
        };
        
        console.log('Generating report with data:', reportData);
        console.log('Transformed step2:', transformedStep2);
        console.log('Original step2Data:', step2Data);
        
        // Debug photo associations
        console.log('Photo associations:');
        Object.entries(step2Data).forEach(([key, value]) => {
            if (key.startsWith('photo_')) {
                console.log(`Photo key: ${key}, data:`, value);
            }
        });
        
        // Debug step2 items with photos
        console.log('Step2 items with photos:');
        step2Items.forEach(item => {
            if (item.photo) {
                console.log(`Item ${item.itemId} has photo:`, item.photo);
            }
        });
        
        // Generate PDF HTML
        const pdfHtml = generatePDFHTML(reportData);
        const reportId = reportData.id;
        const userId = 'cahj3jJqe5Z16pQdLGSWrNgm84o1'; // This should be dynamic based on logged-in user
        
        // Upload PDF to MongoDB server
        let pdfDownloadUrl = null;
        try {
            console.log('=== GENERATING REAL PDF FROM HTML ===');
            
            // Create a temporary container for PDF generation (same as Work Order)
            const tempContainer = document.createElement('div');
            tempContainer.id = 'pdf-content-container';
            tempContainer.innerHTML = pdfHtml;
            
            // Style the container - VISIBLE and ON-SCREEN for proper capture
            tempContainer.style.cssText = `
                position: absolute;
                left: 0;
                top: 0;
                width: 794px;
                min-height: 1123px;
                background: white;
                padding: 20px;
                font-family: Arial, Helvetica, sans-serif;
                font-size: 12px;
                color: #000000;
                z-index: 999999;
                visibility: visible;
                display: block;
                box-sizing: border-box;
            `;
            
            document.body.appendChild(tempContainer);
            console.log('✅ Temp container appended to body');
            console.log('   - Container width:', tempContainer.offsetWidth);
            console.log('   - Container height:', tempContainer.offsetHeight);
            console.log('   - Container innerHTML length:', tempContainer.innerHTML.length);
            
            // CRITICAL: Wait longer for content to fully render (especially for images)
            console.log('⏳ Waiting for content to render (3 seconds)...');
            await new Promise(resolve => setTimeout(resolve, 3000));
            
            console.log('✅ Content should be rendered now');
            
            // Get actual content dimensions
            const actualHeight = Math.max(
                tempContainer.scrollHeight,
                tempContainer.offsetHeight,
                tempContainer.clientHeight
            );
            
            console.log('📐 Container measurements:');
            console.log('   - scrollHeight:', tempContainer.scrollHeight);
            console.log('   - offsetHeight:', tempContainer.offsetHeight);
            console.log('   - Using height:', actualHeight);
            
            // Configure html2pdf options - FIXED for width and height (same as Work Order)
            const opt = {
                margin: [10, 10, 10, 10],
                filename: `DVIR_Report_${reportData.dvirNumber}.pdf`,
                image: { type: 'jpeg', quality: 0.95 },
                html2canvas: { 
                    scale: 2,
                    useCORS: true,
                    logging: true,
                    letterRendering: true,
                    allowTaint: false,
                    backgroundColor: '#ffffff',
                    width: 794,
                    height: actualHeight,
                    windowWidth: 794,
                    windowHeight: actualHeight,
                    scrollY: 0,
                    scrollX: 0,
                    x: 0,
                    y: 0,
                    onclone: function(clonedDoc) {
                        console.log('📋 Document cloned for PDF generation');
                        const clonedContainer = clonedDoc.getElementById('pdf-content-container');
                        if (clonedContainer) {
                            clonedContainer.style.width = '794px';
                            clonedContainer.style.maxWidth = '794px';
                            clonedContainer.style.minWidth = '794px';
                            clonedContainer.style.height = 'auto';
                            clonedContainer.style.overflow = 'visible';
                            clonedContainer.style.display = 'block';
                            console.log('   - Cloned container fixed to 794px width');
                        }
                    }
                },
                jsPDF: { 
                    unit: 'mm',
                    format: 'a4',
                    orientation: 'portrait',
                    compress: true
                },
                pagebreak: { mode: ['avoid-all', 'css', 'legacy'], avoid: 'tr' }
            };
            
            console.log('🔄 Converting HTML to PDF with html2pdf...');
            console.log('   - Container dimensions:', tempContainer.offsetWidth, 'x', tempContainer.offsetHeight);
            
            // Generate PDF blob (same as Work Order)
            const pdfBlob = await html2pdf()
                .set(opt)
                .from(tempContainer)
                .toPdf()
                .output('blob');
            
            console.log('✅ PDF blob generated successfully!');
            console.log('   - Size:', pdfBlob.size, 'bytes');
            console.log('   - Type:', pdfBlob.type);
            
            // CRITICAL: Verify PDF blob content is valid (check PDF header) - Same as Work Order
            console.log('🔍 Verifying PDF blob content...');
            const arrayBuffer = await pdfBlob.slice(0, 5).arrayBuffer();
            const headerBytes = new Uint8Array(arrayBuffer);
            const headerString = String.fromCharCode(...headerBytes);
            console.log('   - PDF Header:', headerString, '(should be "%PDF-")');
            
            if (!headerString.startsWith('%PDF-')) {
                console.error('❌ Invalid PDF header detected!');
                console.error('   - Expected: %PDF-');
                console.error('   - Got:', headerString);
                throw new Error('Generated blob is not a valid PDF (invalid header)');
            }
            
            console.log('✅ PDF blob is valid (has correct PDF header)');
            
            // Remove temporary container
            if (document.body.contains(tempContainer)) {
                document.body.removeChild(tempContainer);
                console.log('✅ Temp container removed');
            }
            
            // Verify PDF blob size
            if (!pdfBlob || pdfBlob.size === 0) {
                throw new Error('PDF generation failed: Blob is empty');
            }
            
            if (pdfBlob.size < 5000) {
                console.warn('⚠️ PDF blob is small (', pdfBlob.size, 'bytes), might be incomplete');
                // Don't throw error, just warn - some minimal PDFs can be small
            }
            
            console.log('✅ PDF blob validation complete - ready for upload');
            
            console.log('=== UPLOADING PDF TO MONGODB SERVER ===');
            
            pdfDownloadUrl = await uploadPDFToServer(pdfBlob, userId, reportId);
            console.log('✅ PDF uploaded successfully:', pdfDownloadUrl);
            console.log('✅ PDF URL type:', typeof pdfDownloadUrl);
            console.log('✅ PDF URL is full URL:', pdfDownloadUrl?.startsWith('http://') || pdfDownloadUrl?.startsWith('https://'));
            
            // Update report data with PDF URL
            reportData.step3.pdfDownloadUrl = pdfDownloadUrl;
            reportData.step3.signatureDownloadUrl = signatureData || null;
            
            console.log('=== PDF UPLOAD COMPLETED ===');
            console.log('📄 PDF URL in reportData.step3:', reportData.step3.pdfDownloadUrl);
            console.log('📄 Full step3 object:', JSON.stringify(reportData.step3, null, 2));
            
        } catch (uploadError) {
            console.error('❌ Error generating or uploading PDF:', uploadError);
            console.error('❌ Error stack:', uploadError.stack);
            showToast('error', 'Error', 'Failed to generate PDF: ' + uploadError.message);
            // Continue without PDF
        }
        
        // Save to MongoDB
        try {
            console.log('💾 === PREPARING TO SAVE TO MONGODB ===');
            console.log('💾 Report ID:', reportData.id);
            console.log('💾 DVIR Number:', reportData.dvirNumber);
            console.log('💾 PDF URL in step3:', reportData.step3?.pdfDownloadUrl);
            console.log('💾 PDF URL is full URL:', reportData.step3?.pdfDownloadUrl?.startsWith('http://') || reportData.step3?.pdfDownloadUrl?.startsWith('https://'));
            console.log('💾 Full report data:', JSON.stringify(reportData, null, 2));
            
            const mongoResult = await saveReportToMongoDB(reportData);
            console.log('✅ MongoDB save result:', mongoResult);
            
            // Store report data in session storage for PDF viewing
            sessionStorage.setItem('currentReportData', JSON.stringify(reportData));
            console.log('✅ Report data stored in session storage for PDF viewing');
            
            // Send email notifications
            try {
                console.log('📧 Sending email notifications...');
                const notificationResponse = await fetch('https://api.titandrillingzm.com:6020/api/dvir-notifications/new-dvir', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        dvirData: {
                            _id: reportData.id,
                            dvirNumber: reportData.dvirNumber,
                            requestNumber: reportData.dvirNumber,
                            equipmentName: step1Data.selectedEquipment?.equipmentName || 'N/A',
                            country: step1Data.country,
                            project: step1Data.project,
                            faultyItems: step3Data.faultyItems || 0
                        },
                        createdByName: step1Data.inspector,
                        inspectorEmail: 'inspector@titandrillingzm.com', // Update with actual email if available
                        pdfUrl: reportData.step3.pdfDownloadUrl || null
                    })
                });
                
                if (notificationResponse.ok) {
                    const notificationResult = await notificationResponse.json();
                    console.log('✅ Notifications sent:', notificationResult);
                } else {
                    console.warn('⚠️ Failed to send notifications');
                }
            } catch (notifError) {
                console.error('❌ Error sending notifications:', notifError);
            }
            
            showToast('success', 'Success', 'DVIR report saved to MongoDB successfully');
            
        } catch (error) {
            console.error('❌ Error saving to MongoDB:', error);
            showToast('error', 'Error', 'Failed to save report to MongoDB');
            throw error;
        }
        
        hideLoadingModal();
        showSuccessModal();
        
        console.log('Report generated successfully');
        
    } catch (error) {
        console.error('Error generating report:', error);
        hideLoadingModal();
        showToast('error', 'Error', 'Failed to generate report. Please try again.');
    }
}

// Generate PDF HTML
function generatePDFHTML(reportData) {
    const { step1, step2, step3 } = reportData;
    
    // Build checklist rows - match React Native format exactly
    let checklistRows = '';
    let lastSection = null;
    
    // Process step2 data to match React Native structure
    const step2Items = [];
    
    // Process step2 items (already in the correct format)
    if (step2.items && Array.isArray(step2.items)) {
        step2.items.forEach(item => {
            if (item.status && item.status !== '') {
                step2Items.push({
                    section: item.section,
                    itemId: item.itemId,
                    label: item.label,
                    status: item.status,
                    note: item.note || '',
                    photo: item.photo || null
                });
            }
        });
    }
    
    // Generate table rows
    step2Items.forEach((item, idx) => {
        if (!item.status || item.status === '' || item.status === 'N/A') return;
        
        // Add section header if different from last section
        if (item.section !== lastSection) {
            checklistRows += `
                <tr style="background-color: #e0e0e0;">
                    <td colspan="7"><strong>${item.section}</strong></td>
                </tr>
            `;
            lastSection = item.section;
        }
        
        // Determine class and checkmarks
        const isYes = item.status.toLowerCase() === 'yes' || item.status.toLowerCase() === 'fixed' || item.status.toLowerCase() === 'pass';
        const isNo = item.status.toLowerCase() === 'no' || item.status.toLowerCase() === 'not fixed' || item.status.toLowerCase() === 'fail';
        const isNA = item.status.toLowerCase() === 'na' || item.status.toLowerCase() === 'no need';
        
        // Add photo if exists
        let photoHtml = '';
        if (item.photo && item.photo.data) {
            photoHtml = `<br><div style="margin-top: 5px; padding: 5px; background-color: #f9f9f9; border: 1px solid #ddd; border-radius: 3px;"><strong>📷 Photo:</strong><br><img src="${item.photo.data}" style="max-width: 200px; max-height: 150px; border: 1px solid #ccc; margin-top: 3px;" alt="Photo"></div>`;
            console.log(`Photo found for item ${item.itemId}:`, item.photo);
        }
        
        checklistRows += `
            <tr>
                <td>${item.label}</td>
                <td>A</td>
                <td style="text-align: center;">${isYes ? '✓' : ''}</td>
                <td style="text-align: center;">${isNo ? '✓' : ''}</td>
                <td>${isNA ? 'N/A' : (item.note || '')}${photoHtml}</td>
            </tr>
        `;
    });
    
    if (!checklistRows) {
        checklistRows = `
            <tr>
                <td colspan="5">No inspection items selected</td>
            </tr>
        `;
    }
    
    // Create signature HTML - use base64 data directly if available
    let signatureHtml = '';
    if (step3.signatureData && step3.signatureData.startsWith('data:image')) {
        signatureHtml = `
            <div class="signature-container">
                <h4>Inspector Signature:</h4>
                <img src="${step3.signatureData}" class="signature-image" alt="Signature" style="border: 1px solid #ccc; padding: 5px;" />
            </div>
        `;
    } else {
        signatureHtml = `
            <div class="signature-container">
                <h4>Inspector Signature:</h4>
                <p><strong>Signature:</strong> Not Provided</p>
            </div>
        `;
    }
    
    return `
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8" />
            <meta name="viewport" content="width=device-width, initial-scale=1.0" />
            <title>DVIR Report - ${reportData.dvirNumber}</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { 
                    font-family: Arial, sans-serif; 
                    padding: 20px; 
                    font-size: 12px; 
                    background: white;
                    color: #000;
                    line-height: 1.4;
                    -webkit-print-color-adjust: exact;
                    print-color-adjust: exact;
                }
                .pdf-container {
                    width: 100%;
                    max-width: 210mm;
                    margin: 0 auto;
                    background: white;
                    color: #000;
                }
                .header, .section { 
                    border: 1px solid #000; 
                    padding: 10px; 
                    margin-bottom: 10px;
                    page-break-inside: avoid;
                }
                .header-table, .checklist-table { 
                    width: 100%; 
                    border-collapse: collapse; 
                    margin-bottom: 10px;
                }
                .header-table td, .checklist-table th, .checklist-table td { 
                    border: 1px solid #000; 
                    padding: 5px; 
                    text-align: left;
                    word-wrap: break-word;
                }
                .title { 
                    text-align: center; 
                    font-weight: bold; 
                    font-size: 16px; 
                    margin: 20px 0;
                    page-break-after: avoid;
                }
                .notes { 
                    font-size: 10px; 
                    margin-bottom: 10px;
                    page-break-inside: avoid;
                }
                .checklist-table th { 
                    background-color: #ddd; 
                    font-weight: bold;
                    page-break-after: avoid;
                }
                .checklist-table tr {
                    page-break-inside: avoid;
                }
                .signature-container { 
                    margin-top: 30px; 
                    text-align: center; 
                    border-top: 2px solid #000; 
                    padding-top: 20px;
                    page-break-inside: avoid;
                }
                .signature-image { 
                    width: 200px; 
                    height: 100px; 
                    object-fit: contain;
                    max-width: 100%;
                }
                .summary-section { 
                    margin-top: 20px;
                    page-break-inside: avoid;
                }
                .summary-section p { 
                    margin: 5px 0; 
                }
                .summary-section h3 {
                    margin-bottom: 10px;
                    page-break-after: avoid;
                }
                .signature-container h4 { 
                    margin-bottom: 10px; 
                    color: #015185;
                }
                img {
                    max-width: 100%;
                    height: auto;
                }
                @media print {
                    body { 
                        padding: 0; 
                    }
                    .header, .section {
                        page-break-inside: avoid;
                    }
                }
            </style>
        </head>
        <body>
            <div style="font-family: Arial, Helvetica, sans-serif; padding: 0; margin: 0; font-size: 10px; background: white; color: black; width: 100%; max-width: 794px; height: auto; box-sizing: border-box;">
            <div class="header" style="background: white; padding: 10px; margin-bottom: 10px; border: 1px solid #ccc; box-sizing: border-box;">
                <table class="header-table" style="width: 100%; max-width: 100%; border-collapse: collapse; table-layout: fixed;">
                    <tr>
                        <td rowspan="5" style="border: none; padding: 0; width: 80px; vertical-align: top;">
                            <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" style="width: 75px; height: auto; display: block;">
                        </td>
                        <td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Document Name:</strong> HSE-FOR-8.1.1 DD Prestart Check</td>
                    </tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Document ID:</strong> HSE-028</td></tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Version Date:</strong> 05/08/2021</td></tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Version:</strong> 2</td></tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Approved:</strong> HSM</td></tr>
                </table>
            </div>
            
            <div class="title">DAILY DD RIG PRE-START CHECKLIST</div>
            
            <div class="section">
                <strong>NO:</strong> ${step1.requestNumber || '0001'}<br />
                <strong>DATE:</strong> ${new Date().toLocaleDateString()}<br />
                <strong>RIG:</strong> ${step1.selectedEquipment?.equipmentName || 'N/A'}<br />
                <strong>ODOMETER READING:</strong> ${step1.meterReading || 'N/A'} ${step1.selectedEquipment?.meterUnit || 'Miles'}<br />
                <strong>FUEL LEVEL:</strong> N/A
            </div>
            
            <div class="section notes">
                1. ALL DEFECTS UNDER "A" CLASSIFICATION TO BE ATTENDED IMMEDIATELY<br />
                2. ALL DEFECTS UNDER "B" CLASSIFICATION TO BE ATTENDED WITHIN 48 HOURS<br />
                3. ALL DEFECTS UNDER "C" CLASSIFICATION TO BE ATTENDED AS SOON AS POSSIBLE
            </div>
            
            <table class="checklist-table">
                <tr>
                    <th>Description</th>
                    <th>Class</th>
                    <th>In Order<br/>(✓ Yes)</th>
                    <th>Defective<br/>(✓ No)</th>
                    <th>Comment</th>
                </tr>
                ${checklistRows}
            </table>
            
            <div class="summary-section">
                <h3>Summary</h3>
                <p><strong>Country:</strong> ${step1.country || 'N/A'}</p>
                <p><strong>Project:</strong> ${step1.project || 'N/A'}</p>
                <p><strong>Inspector:</strong> ${step1.inspector || 'N/A'}</p>
                <p><strong>Faulty Items:</strong> ${step3.faultyItems || '0'}</p>
                <p><strong>Overall Condition:</strong> ${step3.overallCondition}</p>
                <p><strong>Driver status:</strong> ${step3.equipmentStatus}</p>
                ${step3.priority ? `<p><strong>Priority:</strong> ${step3.priority}</p>` : ''}
                <p><strong>Safe to Use:</strong> ${step3.safeToUse}</p>
                <p><strong>Additional Note:</strong> ${step3.additionalNote || 'N/A'}</p>
            </div>
            
            ${signatureHtml}
            </div>
        </body>
        </html>
    `;
}

// Get item text by section and item ID
function getItemText(section, itemId) {
    const items = {
        requirements: {
            req1: "I hold a valid driver's licence",
            req2: "I will obey speed limits and road signs",
            req3: "I will not overload the vehicle and ensure the load is secure",
            req4: "I have completed the vehicle pre check list and satisfied all is order for my trip",
            req5: "I am fit for work – (Not under the influence of alcohol, am well rested, am healthy)",
            req6: "I have read and understand the Titan driving procedure, National driving rules and cliental rules",
            req7: "I will ensure every passenger and myself wears a seat belt (No unauthorised passengers)"
        },
        hazards: {
            haz1: "Pedestrians crossing the road, school children and drunkards",
            haz2: "Motor bikes and taxis stopping suddenly or pulling in to the road suddenly",
            haz3: "Oncoming vehicles overtaking on blind rises or corners",
            haz4: "Heavy trucks or plant taking up width of the road",
            haz5: "Riots or unrest",
            haz6: "Gravel/Sandy roads can be slippery – Loss of control",
            haz7: "Tyre blowouts or punctures – Loss of control"
        }
    };
    
    return items[section]?.[itemId] || 'Unknown item';
}

// Show success modal
// Show success modal
function showSuccessModal() {
    const modal = document.getElementById('successModal');
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
        // Force reflow for animation
        modal.offsetHeight;
        requestAnimationFrame(() => {
            if (modal && modal.classList) {
                modal.classList.add('show');
            }
        });
    }
}

// Close success modal
function closeSuccessModal() {
    const modal = document.getElementById('successModal');
    if (modal) {
        if (modal && modal.classList) {
            modal.classList.remove('show');
        }
        setTimeout(() => {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }, 400);
    }
    // Reset form
    resetForm();
}

// Go to DVIR list page
function goToDvirList() {
    window.location.href = '/dvir-dot-inspection';
}

// View PDF from MongoDB server
function viewPDF() {
    console.log('=== VIEW PDF CLICKED ===');
    
    // Get the report data from session storage
    const reportData = JSON.parse(sessionStorage.getItem('currentReportData') || '{}');
    
    console.log('📋 Report data from session storage:', reportData);
    console.log('📋 PDF URL in step3:', reportData.step3?.pdfDownloadUrl);
    
    if (reportData.step3 && reportData.step3.pdfDownloadUrl) {
        let pdfUrl = reportData.step3.pdfDownloadUrl;
        console.log('📄 Original PDF URL:', pdfUrl);
        console.log('📄 URL type:', typeof pdfUrl);
        console.log('📄 URL starts with http:', pdfUrl.startsWith('http'));
        
        // Convert relative URL to absolute URL if needed
        if (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://')) {
            console.warn('⚠️ PDF URL is relative, converting to absolute URL');
            const baseUrl = 'https://api.titandrillingzm.com:6006';
            // Remove leading slash if present
            const relativePath = pdfUrl.startsWith('/') ? pdfUrl : '/' + pdfUrl;
            pdfUrl = baseUrl + relativePath;
            console.log('📄 Converted PDF URL:', pdfUrl);
        }
        
        try {
            console.log('🔗 Opening PDF URL:', pdfUrl);
            
            // Open PDF in new tab
            const pdfWindow = window.open(pdfUrl, '_blank');
            
            // Check if popup was blocked
            if (!pdfWindow || pdfWindow.closed || typeof pdfWindow.closed == 'undefined') {
                console.error('❌ Popup blocked by browser');
                showToast('error', 'Error', 'Popup blocked! Please allow popups for this site and try again.');
                return;
            }
            
            // Optional: Focus the new window
            pdfWindow.focus();
            console.log('✅ PDF opened successfully in new tab');
            showToast('success', 'Success', 'PDF opened in new tab');
            
        } catch (error) {
            console.error('❌ Error opening PDF:', error);
            console.error('❌ Error details:', {
                name: error.name,
                message: error.message,
                stack: error.stack
            });
            showToast('error', 'Error', 'Failed to open PDF: ' + error.message);
        }
    } else {
        console.error('❌ PDF URL not found in report data');
        console.error('📋 Report data structure:', {
            hasStep3: !!reportData.step3,
            hasPdfUrl: !!(reportData.step3 && reportData.step3.pdfDownloadUrl),
            reportDataKeys: Object.keys(reportData),
            step3Keys: reportData.step3 ? Object.keys(reportData.step3) : [],
            fullStep3: reportData.step3
        });
        showToast('error', 'Error', 'PDF URL not available. Please regenerate the report.');
    }
    
    closeSuccessModal();
    console.log('=== END VIEW PDF ===');
}

// Reset form
function resetForm() {
    // Reset all steps
    currentStep = 1;
    step2Data = {};
    step3Data = {};
    signatureData = null;
    signatureStatus = false;
    
    // Reset Step 1
    document.getElementById('country').value = '';
    document.getElementById('project').value = '';
    document.getElementById('inspector').value = '';
    document.getElementById('dvirCheckbox').checked = true;
    selectedEquipment = null;
    document.getElementById('equipmentDetailsBox').style.display = 'none';
    document.getElementById('equipmentDisplay').value = '';
    
    // Reset Step 2
    Object.keys(step2Data).forEach(key => {
        const [section, itemId, field] = key.split('_');
        const buttons = document.querySelectorAll(`[onclick*="setStatus('${section}', '${itemId}'"]`);
        buttons.forEach(btn => btn.classList.remove('selected'));
    });
    
    // Reset Step 3
    document.getElementById('overallCondition').value = '';
    document.querySelectorAll('input[name="equipmentStatus"]').forEach(radio => radio.checked = false);
    document.querySelectorAll('input[name="safeToUse"]').forEach(radio => radio.checked = false);
    document.getElementById('additionalNote').value = '';
    document.getElementById('priorityGroup').style.display = 'none';
    document.getElementById('signaturePlaceholder').style.display = 'block';
    document.getElementById('signatureDisplay').style.display = 'none';
    
    // Show Step 1
    showStep1();
    
    console.log('Form reset successfully');
}

// Initialize signature pad
function initSignaturePad() {
    const canvas = document.getElementById('signaturePad');
    const ctx = canvas.getContext('2d');
    let isDrawing = false;
    
    canvas.addEventListener('mousedown', (e) => {
        isDrawing = true;
        const rect = canvas.getBoundingClientRect();
        ctx.beginPath();
        ctx.moveTo(e.clientX - rect.left, e.clientY - rect.top);
    });
    
    canvas.addEventListener('mousemove', (e) => {
        if (!isDrawing) return;
        const rect = canvas.getBoundingClientRect();
        ctx.lineTo(e.clientX - rect.left, e.clientY - rect.top);
        ctx.stroke();
    });
    
    canvas.addEventListener('mouseup', () => {
        isDrawing = false;
    });
    
    // Touch events for mobile
    canvas.addEventListener('touchstart', (e) => {
        e.preventDefault();
        isDrawing = true;
        const rect = canvas.getBoundingClientRect();
        const touch = e.touches[0];
        ctx.beginPath();
        ctx.moveTo(touch.clientX - rect.left, touch.clientY - rect.top);
    });
    
    canvas.addEventListener('touchmove', (e) => {
        e.preventDefault();
        if (!isDrawing) return;
        const rect = canvas.getBoundingClientRect();
        const touch = e.touches[0];
        ctx.lineTo(touch.clientX - rect.left, touch.clientY - rect.top);
        ctx.stroke();
    });
    
    canvas.addEventListener('touchend', (e) => {
        e.preventDefault();
        isDrawing = false;
    });
    
    // Set up drawing style
    ctx.strokeStyle = '#000';
    ctx.lineWidth = 2;
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';
}

// Initialize signature pad when page loads
document.addEventListener('DOMContentLoaded', function() {
    initSignaturePad();
});
</script>

<style>
/* CSS Variables */
:root {
    --primary-color: #0055a5;
    --primary-hover: #3b82f6;
    --text-primary: #0f172a;
    --text-secondary: #374151;
    --text-muted: #64748b;
    --bg-primary: #f8fafc;
    --bg-white: #ffffff;
    --bg-hover: #f8fafc;
    --border-color: #e2e8f0;
    --border-light: #d1d5db;
    --shadow-sm: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    --shadow-lg: 0 10px 25px rgba(0, 0, 0, 0.15);
    --radius-sm: 4px;
    --radius-md: 8px;
    --radius-lg: 12px;
    --transition-base: all 0.2s ease;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Import Google Fonts */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* Main Container */
.dvir-create-container {
    min-height: 100vh;
    background: var(--bg-primary);
    font-family: var(--font-family);
    width: 100%;
    max-width: none;
    margin: 0;
    padding: 0;
    display: flex;
    flex-direction: column;
}

/* Body improvements for full width */
body {
    margin: 0;
    padding: 0;
    overflow-x: hidden;
}

/* Header Section */
.dvir-header {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin: 24px 24px 0 24px;
    flex-shrink: 0;
}

.header-content {
    padding: 32px;
}

.page-title h1 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.page-title p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

/* Step Navigation Enhancements */
.step-indicator-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin: 24px 24px 24px 24px;
    flex-shrink: 0;
}

.step-indicator {
    display: flex;
    justify-content: center;
    align-items: center;
    max-width: 600px;
    margin: 0 auto;
    padding: 32px;
}

.step-item {
    display: flex;
    flex-direction: column;
    align-items: center;
    position: relative;
    transition: all 0.3s ease;
    cursor: pointer;
}

.step-item:hover {
    transform: translateY(-2px);
}

.step-item:hover .step-icon {
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.step-item.completed .step-icon {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.step-item.completed .step-title {
    color: #10b981;
    font-weight: 600;
}

.step-item.active .step-icon {
    background: linear-gradient(135deg, var(--primary-color) 0%, #003d82 100%);
    color: white;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.step-item.active .step-title {
    color: var(--primary-color);
    font-weight: 600;
}

.step-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: #f3f4f6;
    color: #6b7280;
    display: flex;
    align-items: center;
    justify-content: center;
    margin-bottom: 12px;
    transition: all 0.3s ease;
    border: 2px solid #e5e7eb;
}

.step-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    text-align: center;
}

.step-number {
    font-size: 0.75rem;
    font-weight: 600;
    color: #6b7280;
    margin-bottom: 2px;
}

.step-title {
    font-size: 0.875rem;
    font-weight: 500;
    color: #6b7280;
    transition: all 0.3s ease;
}

.step-connector {
    width: 100px;
    height: 2px;
    background: #e5e7eb;
    margin: 0 16px;
    margin-top: -25px;
    transition: all 0.3s ease;
}

.step-item.completed + .step-connector,
.step-item.active + .step-connector {
    background: var(--primary-color);
}

/* Mobile responsive */
@media (max-width: 768px) {
    .dvir-header {
        margin: 16px 16px 0 16px;
    }
    
    .step-indicator-section {
        margin: 16px 16px 16px 16px;
    }
    
    .form-container {
        padding: 0 16px;
    }
    
    .step-container {
        padding: 16px;
    }
    
    .step-indicator {
        flex-direction: column;
        gap: 24px;
    }
    
    .step-connector {
        width: 2px;
        height: 40px;
        margin: 12px 0;
        margin-top: 0;
    }
    
    .step-item {
        flex-direction: row;
        width: 100%;
        justify-content: flex-start;
        text-align: left;
    }
    
    .step-content {
        align-items: flex-start;
        margin-left: 16px;
    }
    
    .header-content {
        padding: 24px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .form-card {
        padding: 24px;
    }
}

/* Form Container */
.form-container {
    max-width: 100%;
    margin: 0;
    padding: 0 24px;
    width: 100%;
    flex-grow: 1;
}

.form-card {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-md);
    padding: 32px;
    margin-bottom: 24px;
    width: 100%;
    max-width: none;
}

/* Request Number Display */
.request-number-display {
    background: #f0f0f0;
    border-radius: var(--radius-sm);
    padding: 16px;
    margin-bottom: 24px;
}

.request-number-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.request-label {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 500;
}

.request-value {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 600;
}

/* Form Groups */
.form-group {
    margin-bottom: 24px;
}

.form-label {
    display: block;
    font-size: 16px;
    color: var(--primary-color);
    margin-bottom: 8px;
    font-weight: 500;
}

.required-field::after {
    content: ' *';
    color: #e74c3c;
    font-weight: 600;
}

.form-control {
    width: 100%;
    height: 48px;
    border: 1px solid var(--primary-color);
    border-radius: var(--radius-sm);
    padding: 0 16px;
    font-size: 16px;
    color: var(--text-primary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-hover);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.form-control:disabled {
    background: #e0e0e0;
    cursor: not-allowed;
}

/* Equipment Selection */
.equipment-selection-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #f0f0f0;
    padding: 16px;
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
}

.equipment-label {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 600;
}

.equipment-select-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    background: none;
    border: none;
    color: var(--primary-color);
    font-size: 16px;
    cursor: pointer;
    transition: var(--transition-base);
}

.equipment-select-btn:hover {
    color: var(--primary-hover);
}

/* Equipment Details Box */
.equipment-details-box {
    background: #f0f0f0;
    border-radius: var(--radius-sm);
    padding: 20px;
    margin-bottom: 24px;
}

.equipment-info-row {
    margin-bottom: 16px;
}

.equipment-info-text {
    font-size: 16px;
    color: var(--text-primary);
    line-height: 1.5;
}

/* Meter Reading Section */
.meter-reading-section {
    border-top: 1px solid #ddd;
    padding-top: 16px;
}

.meter-reading-label {
    display: block;
    font-size: 16px;
    color: var(--text-primary);
    margin-bottom: 8px;
    font-weight: 600;
}

.meter-reading-input-row {
    display: flex;
    align-items: center;
    background: var(--bg-white);
    border: 1px solid #ccc;
    border-radius: var(--radius-sm);
    padding: 0 16px;
    margin-bottom: 8px;
}

.meter-reading-input {
    flex: 1;
    border: none;
    outline: none;
    font-size: 16px;
    color: var(--text-primary);
    padding: 12px 0;
}

.meter-unit {
    color: var(--text-muted);
    font-size: 14px;
    margin-left: 8px;
}

.meter-reading-error {
    color: #e74c3c;
    font-size: 12px;
    margin-top: 4px;
}

.current-meter-row {
    display: flex;
    align-items: center;
    margin-top: 8px;
}

.current-meter-label {
    font-size: 13px;
    color: var(--text-muted);
}

.current-meter-value {
    font-size: 13px;
    color: var(--text-primary);
    font-weight: 500;
    margin-left: 4px;
}

/* Inspection Type Section */
.inspection-type-section {
    border-top: 1px solid #e0e0e0;
    margin-top: 16px;
    padding-top: 16px;
}

.checkbox-row {
    display: flex;
    align-items: center;
    padding: 8px 0;
}

.inspection-checkbox {
    width: 20px;
    height: 20px;
    margin-right: 12px;
    accent-color: var(--primary-color);
}

.checkbox-label {
    font-size: 16px;
    color: var(--primary-color);
    font-weight: 500;
    cursor: pointer;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    justify-content: space-between;
    padding: 16px 0;
    background: #f0f0f0;
    border-top: 1px solid #ddd;
    position: sticky;
    bottom: 0;
    left: 0;
    right: 0;
    margin: 0 -16px;
    padding: 16px;
}

.btn-cancel {
    display: flex;
    align-items: center;
    gap: 8px;
    background: none;
    border: none;
    color: var(--primary-color);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    padding: 12px 20px;
    transition: var(--transition-base);
}

.btn-cancel:hover {
    color: var(--primary-hover);
}

.btn-next {
    display: flex;
    align-items: center;
    gap: 8px;
    background: var(--primary-color);
    border: none;
    color: var(--bg-white);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    padding: 12px 20px;
    border-radius: var(--radius-sm);
    transition: var(--transition-base);
}

.btn-next:hover:not(:disabled) {
    background: var(--primary-hover);
}

.btn-next:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

/* ============================================
   DVIR MODAL SYSTEM - Modern & Professional
   ============================================ */

/* Equipment Selection Modal Backdrop */
.dvir-modal-backdrop {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, rgba(0, 0, 0, 0.7) 0%, rgba(0, 85, 165, 0.3) 100%);
    display: none;
    justify-content: center;
    align-items: center;
    z-index: 10000;
    backdrop-filter: blur(8px) saturate(180%);
    opacity: 0;
    transition: opacity 0.4s cubic-bezier(0.4, 0, 0.2, 1);
}

.dvir-modal-backdrop.show {
    opacity: 1;
}

/* Modal Dialog Container */
.dvir-modal-dialog {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    margin: 20px auto;
    border-radius: 20px;
    box-shadow: 
        0 25px 50px -12px rgba(0, 0, 0, 0.25),
        0 0 0 1px rgba(255, 255, 255, 0.5),
        0 0 100px rgba(0, 85, 165, 0.1);
    max-width: 950px;
    width: 92%;
    max-height: 88vh;
    display: flex;
    flex-direction: column;
    transform: scale(0.85) translateY(-30px);
    transition: transform 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
    overflow: hidden;
    border: 1px solid rgba(0, 85, 165, 0.1);
}

.dvir-modal-backdrop.show .dvir-modal-dialog {
    transform: scale(1) translateY(0);
}

/* Modal Header */
.dvir-modal-header-wrapper {
    display: flex;
    align-items: center;
    gap: 16px;
    padding: 24px 32px;
    border-bottom: 2px solid rgba(0, 85, 165, 0.1);
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    position: relative;
    overflow: hidden;
}

.dvir-modal-header-wrapper::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #00d4ff, #0055a5, #003f7d);
    animation: shimmer 3s infinite;
}

@keyframes shimmer {
    0%, 100% { transform: translateX(-100%); }
    50% { transform: translateX(100%); }
}

.dvir-modal-header-icon {
    width: 48px;
    height: 48px;
    border-radius: 12px;
    background: rgba(255, 255, 255, 0.2);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 22px;
    backdrop-filter: blur(10px);
}

.dvir-modal-header-content {
    flex: 1;
}

.dvir-modal-title {
    margin: 0;
    font-size: 24px;
    font-weight: 700;
    color: #ffffff;
    letter-spacing: -0.5px;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.dvir-modal-subtitle {
    margin: 4px 0 0 0;
    font-size: 14px;
    font-weight: 400;
    color: rgba(255, 255, 255, 0.85);
}

.dvir-modal-close-btn {
    width: 40px;
    height: 40px;
    border-radius: 10px;
    background: rgba(255, 255, 255, 0.15);
    border: none;
    color: white;
    font-size: 20px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    backdrop-filter: blur(10px);
}

.dvir-modal-close-btn:hover {
    background: rgba(255, 255, 255, 0.25);
    transform: rotate(90deg) scale(1.1);
}

.dvir-modal-close-btn:active {
    transform: rotate(90deg) scale(0.95);
}

/* Modal Body */
.dvir-modal-body-wrapper {
    padding: 28px 32px;
    max-height: calc(88vh - 160px);
    overflow-y: auto;
    flex: 1;
    background: #ffffff;
}

.dvir-modal-body-wrapper::-webkit-scrollbar {
    width: 10px;
}

.dvir-modal-body-wrapper::-webkit-scrollbar-track {
    background: #f1f5f9;
    border-radius: 5px;
}

.dvir-modal-body-wrapper::-webkit-scrollbar-thumb {
    background: linear-gradient(135deg, #0055a5, #003f7d);
    border-radius: 5px;
}

.dvir-modal-body-wrapper::-webkit-scrollbar-thumb:hover {
    background: linear-gradient(135deg, #003f7d, #002a5c);
}

/* Search Wrapper */
.dvir-search-wrapper {
    display: flex;
    gap: 12px;
    margin-bottom: 24px;
    padding: 20px;
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-radius: 14px;
    border: 1px solid rgba(0, 85, 165, 0.1);
}

.dvir-search-input-group {
    flex: 1;
    position: relative;
    display: flex;
    align-items: center;
}

.dvir-search-icon {
    position: absolute;
    left: 16px;
    color: #6b7280;
    font-size: 18px;
    z-index: 1;
}

.dvir-search-input {
    width: 100%;
    padding: 14px 48px 14px 48px;
    border: 2px solid #e2e8f0;
    border-radius: 12px;
    font-size: 15px;
    font-family: inherit;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    background: #ffffff;
    color: #1f2937;
}

.dvir-search-input:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 4px rgba(0, 85, 165, 0.1), 0 4px 12px rgba(0, 85, 165, 0.15);
    transform: translateY(-1px);
}

.dvir-search-clear-btn {
    position: absolute;
    right: 12px;
    width: 28px;
    height: 28px;
    border-radius: 6px;
    background: #f3f4f6;
    border: none;
    color: #6b7280;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 14px;
    transition: all 0.2s;
}

.dvir-search-clear-btn:hover {
    background: #e5e7eb;
    color: #374151;
}

.dvir-search-action-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 14px 24px;
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
    border: none;
    border-radius: 12px;
    font-size: 15px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.dvir-search-action-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(0, 85, 165, 0.4);
}

.dvir-search-action-btn:active {
    transform: translateY(0);
}

/* Equipment List Container */
.dvir-equipment-list-container {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

/* Loading Modal */
.dvir-loading-backdrop {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, rgba(0, 0, 0, 0.75) 0%, rgba(0, 85, 165, 0.5) 100%);
    display: none;
    justify-content: center;
    align-items: center;
    z-index: 10001;
    backdrop-filter: blur(12px) saturate(200%);
    opacity: 0;
    transition: opacity 0.4s cubic-bezier(0.4, 0, 0.2, 1);
}

.dvir-loading-backdrop.show {
    display: flex;
    opacity: 1;
}

.dvir-loading-dialog {
    background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
    padding: 48px 56px;
    border-radius: 20px;
    box-shadow: 
        0 25px 50px -12px rgba(0, 0, 0, 0.3),
        0 0 0 1px rgba(255, 255, 255, 0.5);
    text-align: center;
    min-width: 320px;
    transform: scale(0.8);
    transition: transform 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
    border: 1px solid rgba(0, 85, 165, 0.1);
}

.dvir-loading-backdrop.show .dvir-loading-dialog {
    transform: scale(1);
}

.dvir-loading-spinner-wrapper {
    position: relative;
    width: 64px;
    height: 64px;
    margin: 0 auto 24px;
}

.dvir-loading-spinner {
    width: 64px;
    height: 64px;
    border: 5px solid rgba(0, 85, 165, 0.1);
    border-top: 5px solid #0055a5;
    border-right: 5px solid #003f7d;
    border-radius: 50%;
    animation: dvirSpin 1s linear infinite;
    position: absolute;
    top: 0;
    left: 0;
}

.dvir-loading-pulse {
    width: 64px;
    height: 64px;
    border-radius: 50%;
    background: radial-gradient(circle, rgba(0, 85, 165, 0.2) 0%, transparent 70%);
    position: absolute;
    top: 0;
    left: 0;
    animation: dvirPulse 2s ease-in-out infinite;
}

@keyframes dvirSpin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

@keyframes dvirPulse {
    0%, 100% { transform: scale(1); opacity: 0.7; }
    50% { transform: scale(1.2); opacity: 0.3; }
}

.dvir-loading-content {
    margin-top: 8px;
}

.dvir-loading-title {
    font-size: 20px;
    font-weight: 700;
    color: #0055a5;
    margin: 0 0 8px 0;
    letter-spacing: -0.3px;
}

.dvir-loading-message {
    font-size: 15px;
    color: #6b7280;
    margin: 0 0 20px 0;
}

.dvir-loading-progress-bar {
    width: 100%;
    height: 6px;
    background: #e5e7eb;
    border-radius: 3px;
    overflow: hidden;
    margin-top: 16px;
}

.dvir-loading-progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #0055a5, #00d4ff, #0055a5);
    background-size: 200% 100%;
    border-radius: 3px;
    animation: dvirProgress 2s linear infinite;
}

@keyframes dvirProgress {
    0% { width: 0%; background-position: 0% 0%; }
    50% { width: 75%; background-position: 100% 0%; }
    100% { width: 100%; background-position: 200% 0%; }
}

/* Signature Modal Specific */
.dvir-signature-dialog {
    max-width: 700px;
    width: 90%;
}

/* Success Modal */
.dvir-success-backdrop {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, rgba(0, 0, 0, 0.7) 0%, rgba(34, 197, 94, 0.3) 100%);
    display: none;
    justify-content: center;
    align-items: center;
    z-index: 10001;
    backdrop-filter: blur(8px) saturate(180%);
    opacity: 0;
    transition: opacity 0.4s cubic-bezier(0.4, 0, 0.2, 1);
}

.dvir-success-backdrop.show {
    display: flex;
    opacity: 1;
}

.dvir-success-dialog {
    background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 100%);
    padding: 48px 56px;
    border-radius: 24px;
    box-shadow: 
        0 25px 50px -12px rgba(0, 0, 0, 0.3),
        0 0 0 1px rgba(255, 255, 255, 0.5),
        0 0 100px rgba(34, 197, 94, 0.2);
    text-align: center;
    min-width: 420px;
    max-width: 550px;
    transform: scale(0.8) rotateY(10deg);
    transition: transform 0.5s cubic-bezier(0.34, 1.56, 0.64, 1);
    border: 2px solid rgba(34, 197, 94, 0.2);
    position: relative;
    overflow: hidden;
}

.dvir-success-backdrop.show .dvir-success-dialog {
    transform: scale(1) rotateY(0deg);
}

.dvir-success-dialog::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.4), transparent);
    animation: dvirSuccessShine 3s infinite;
}

@keyframes dvirSuccessShine {
    0% { left: -100%; }
    100% { left: 100%; }
}

.dvir-success-icon-wrapper {
    position: relative;
    width: 120px;
    height: 120px;
    margin: 0 auto 28px;
}

.dvir-success-icon-circle {
    width: 120px;
    height: 120px;
    border-radius: 50%;
    background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    box-shadow: 
        0 10px 40px rgba(34, 197, 94, 0.4),
        inset 0 2px 10px rgba(255, 255, 255, 0.3);
    animation: dvirSuccessBounce 0.6s cubic-bezier(0.68, -0.55, 0.265, 1.55);
}

@keyframes dvirSuccessBounce {
    0% { transform: scale(0); }
    50% { transform: scale(1.1); }
    100% { transform: scale(1); }
}

.dvir-success-icon-circle i {
    font-size: 56px;
    color: white;
    z-index: 1;
}

.dvir-success-checkmark {
    position: absolute;
    top: 0;
    left: 0;
    width: 120px;
    height: 120px;
    animation: dvirCheckmarkDraw 0.8s ease-in-out 0.3s both;
}

.dvir-success-checkmark svg {
    width: 100%;
    height: 100%;
}

.dvir-success-checkmark circle {
    stroke: #22c55e;
    stroke-width: 3;
    fill: none;
    stroke-dasharray: 157;
    stroke-dashoffset: 157;
    animation: dvirCircleDraw 0.6s ease-in-out 0.3s both;
}

.dvir-success-checkmark path {
    stroke: white;
    stroke-width: 4;
    stroke-linecap: round;
    stroke-linejoin: round;
    fill: none;
    stroke-dasharray: 30;
    stroke-dashoffset: 30;
    animation: dvirPathDraw 0.4s ease-in-out 0.9s both;
}

@keyframes dvirCircleDraw {
    to { stroke-dashoffset: 0; }
}

@keyframes dvirPathDraw {
    to { stroke-dashoffset: 0; }
}

.dvir-success-content {
    position: relative;
    z-index: 1;
}

.dvir-success-title {
    font-size: 28px;
    font-weight: 700;
    color: #1f2937;
    margin: 0 0 12px 0;
    letter-spacing: -0.5px;
}

.dvir-success-message {
    font-size: 16px;
    color: #6b7280;
    margin: 0 0 32px 0;
    line-height: 1.6;
}

.dvir-success-actions {
    display: flex;
    gap: 12px;
    justify-content: center;
    flex-wrap: wrap;
}

.dvir-btn-primary,
.dvir-btn-success,
.dvir-btn-secondary {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 14px 28px;
    border-radius: 12px;
    font-size: 15px;
    font-weight: 600;
    cursor: pointer;
    border: none;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
}

.dvir-btn-primary {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
}

.dvir-btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(0, 85, 165, 0.4);
}

.dvir-btn-success {
    background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
    color: white;
}

.dvir-btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(34, 197, 94, 0.4);
}

.dvir-btn-secondary {
    background: linear-gradient(135deg, #f3f4f6 0%, #e5e7eb 100%);
    color: #374151;
    border: 1px solid #d1d5db;
}

.dvir-btn-secondary:hover {
    background: linear-gradient(135deg, #e5e7eb 0%, #d1d5db 100%);
    transform: translateY(-2px);
}

.modal-content {
    background: var(--bg-white);
    border-radius: var(--radius-lg);
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    max-width: 90%;
    max-height: 90%;
    overflow: hidden;
    position: relative;
    transform: scale(0.9);
    transition: transform 0.3s ease;
    animation: modalSlideIn 0.3s ease-out;
}

.modal-overlay.show .modal-content {
    transform: scale(1);
}

.equipment-modal {
    width: 800px;
    max-height: 600px;
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 24px;
    border-bottom: 1px solid var(--border-color);
}

.modal-header-actions {
    display: flex;
    align-items: center;
    gap: 12px;
}

.modal-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.modal-close {
    background: none;
    border: none;
    cursor: pointer;
    padding: 8px;
    border-radius: var(--radius-sm);
    transition: var(--transition-base);
}

.modal-close:hover {
    background: var(--bg-hover);
}

.refresh-equipment-btn {
    display: flex;
    align-items: center;
    justify-content: center;
    width: 48px;
    height: 48px;
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    border-radius: var(--radius-sm);
    cursor: pointer;
    transition: var(--transition-base);
}

.refresh-equipment-btn:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

/* Equipment Search */
.equipment-search-container {
    padding: 16px 24px;
    border-bottom: 1px solid var(--border-color);
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.equipment-search-input {
    width: 100%;
    height: 48px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    padding: 0 16px 0 48px;
    font-size: 16px;
    color: var(--text-primary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.equipment-search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-icon {
    position: absolute;
    left: 16px;
    color: var(--text-muted);
    pointer-events: none;
}

/* Equipment Tabs */
.equipment-tabs {
    display: flex;
    padding: 0 24px;
    border-bottom: 1px solid var(--border-color);
}

.tab-button {
    flex: 1;
    padding: 16px;
    background: none;
    border: none;
    border-bottom: 2px solid transparent;
    font-size: 16px;
    font-weight: 600;
    color: var(--text-muted);
    cursor: pointer;
    transition: var(--transition-base);
}

.tab-button.active {
    color: var(--primary-color);
    border-bottom-color: var(--primary-color);
}

.tab-button:hover {
    color: var(--primary-color);
}

/* Equipment List */
.equipment-list-container {
    max-height: 300px;
    overflow-y: auto;
    padding: 16px 24px;
}

.equipment-list {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.equipment-item {
    display: flex;
    flex-wrap: nowrap;
    gap: 16px;
    background: #F2F2F2;
    border: 1px solid #eee;
    border-radius: var(--radius-sm);
    padding: 16px;
    transition: var(--transition-base);
    align-items: flex-start;
}

.equipment-item:hover {
    box-shadow: var(--shadow-sm);
}

.equipment-info {
    flex: 1 1 260px;
    cursor: pointer;
}

.equipment-header {
    margin-bottom: 12px;
}

.equipment-name {
    font-size: 18px;
    color: var(--primary-color);
    font-weight: 600;
}

.equipment-details {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    padding: 4px 0;
    border-bottom: 1px solid #ddd;
}

.detail-row.no-border {
    border-bottom: none;
}

.detail-label {
    font-size: 14px;
    color: var(--text-muted);
    width: 40%;
}

.detail-value {
    font-size: 14px;
    color: var(--text-primary);
    width: 60%;
    text-align: right;
    font-weight: 500;
}

.detail-value.status {
    font-weight: 600;
}

.select-equipment-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 8px 16px;
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    align-self: flex-start;
    margin-left: auto;
    min-width: 110px;
    display: inline-flex;
    justify-content: center;
    align-items: center;
}

@media (max-width: 720px) {
    .equipment-item {
        flex-wrap: wrap;
    }

    .select-equipment-btn {
        margin-left: 0;
        width: 100%;
        margin-top: 12px;
    }
}

.select-equipment-btn:hover {
    background: var(--primary-hover);
}

.select-equipment-btn.selected {
    background: #2ecc71;
    color: white;
}

.select-equipment-btn.selected:hover {
    background: #27ae60;
}

.no-equipment {
    text-align: center;
    color: var(--text-muted);
    font-size: 16px;
    padding: 40px 0;
}

/* Modal Footer */
.modal-footer {
    display: flex;
    justify-content: flex-end;
    padding: 16px 24px;
    border-top: 1px solid var(--border-color);
}

.btn-close {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-sm);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-close:hover {
    background: var(--primary-hover);
}

/* Loading Modal */
.loading-modal-content {
    background: var(--bg-white);
    padding: 40px;
    border-radius: var(--radius-lg);
    text-align: center;
    box-shadow: var(--shadow-lg);
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid var(--border-color);
    border-top: 4px solid var(--primary-color);
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto 20px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-modal-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0 0 8px 0;
}

.loading-modal-content p {
    font-size: 16px;
    color: var(--text-muted);
    margin: 0;
}

/* Toast Notifications */
.toast-container {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 9999;
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.toast {
    display: flex;
    align-items: center;
    justify-content: space-between;
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    box-shadow: var(--shadow-md);
    padding: 16px;
    min-width: 300px;
    animation: slideIn 0.3s ease;
}

@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

.toast-success {
    border-left: 4px solid #2ecc71;
}

.toast-error {
    border-left: 4px solid #e74c3c;
}

.toast-info {
    border-left: 4px solid #3498db;
}

.toast-content h4 {
    font-size: 14px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0 0 4px 0;
}

.toast-content p {
    font-size: 13px;
    color: var(--text-muted);
    margin: 0;
}

.toast-close {
    background: none;
    border: none;
    font-size: 18px;
    color: var(--text-muted);
    cursor: pointer;
    padding: 4px;
    margin-left: 12px;
}

.toast-close:hover {
    color: var(--text-primary);
}

/* Responsive Design */
@media (max-width: 768px) {
    .form-container {
        padding: 0 8px;
    }
    
    .form-card {
        padding: 20px;
    }
    
    .header-content {
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .equipment-modal {
        width: 95%;
        max-height: 80%;
    }
    
    .equipment-item {
        flex-direction: column;
        gap: 12px;
    }
    
    .select-equipment-btn {
        align-self: stretch;
        text-align: center;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 12px;
    }
    
    .btn-cancel,
    .btn-next {
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .form-card {
        padding: 16px;
    }
    
    .header-content {
        padding: 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .equipment-tabs {
        flex-direction: column;
    }
    
    .tab-button {
        border-bottom: 1px solid var(--border-color);
        border-right: none;
    }
    
    .tab-button.active {
        border-bottom-color: var(--primary-color);
        background: var(--bg-hover);
    }
}

/* ==================== STEP CONTAINER STYLES ==================== */
.step-container {
    min-height: calc(100vh - 200px);
    background: var(--bg-primary);
    font-family: var(--font-family);
    padding: 20px 24px;
    width: 100%;
    max-width: none;
    flex-grow: 1;
}

.step-header {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-sm);
    border-radius: var(--radius-sm);
    margin-bottom: 24px;
    padding: 32px;
    width: 100%;
    max-width: none;
}

.step-header h1 {
    color: var(--text-primary);
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.step-header p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

.step-content {
    max-width: 800px;
    margin: 0 auto;
    padding: 0 16px;
}

/* ==================== STEP 2 STYLES ==================== */
.inspection-content {
    max-width: 1040px;
}

.instruction-text {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    padding: 20px 24px;
    margin-bottom: 24px;
    text-align: left;
    box-shadow: var(--shadow-sm);
}

.instruction-text h3 {
    margin: 0 0 8px 0;
    color: var(--primary-color);
    font-size: 18px;
    font-weight: 600;
}

.instruction-text p {
    color: var(--text-muted);
    font-weight: 500;
    margin: 0;
}

.inspection-guidelines {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 16px;
    margin-bottom: 24px;
}

.guideline-card {
    display: flex;
    gap: 12px;
    align-items: flex-start;
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 16px;
    box-shadow: var(--shadow-sm);
}

.guideline-icon {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    background: rgba(0, 85, 165, 0.12);
    color: var(--primary-color);
    display: inline-flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    font-size: 16px;
}

.guideline-card h4 {
    margin: 0 0 4px 0;
    font-size: 16px;
    color: var(--text-primary);
}

.guideline-card p {
    margin: 0;
    font-size: 14px;
    color: var(--text-muted);
}

.checklist-container {
    display: flex;
    flex-direction: column;
    gap: 24px;
    width: 100%;
}

.checklist-card {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    overflow: hidden;
    box-shadow: var(--shadow-sm);
}

.card-header {
    background: #f0f0f0;
    padding: 16px;
    cursor: pointer;
    transition: var(--transition-base);
    border-bottom: 1px solid var(--border-color);
}

.card-header:hover {
    background: #e9e9e9;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 12px;
}

.expand-icon {
    font-size: 18px;
    color: var(--primary-color);
    transition: var(--transition-base);
}

.section-title {
    font-size: 18px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0;
}

.card-content {
    padding: 16px;
}

.checklist-items {
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.checklist-item {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    padding: 16px;
    background: #f8f9fa;
    border-radius: var(--radius-sm);
    border: 1px solid #e9ecef;
}

.item-content {
    flex: 1;
    margin-right: 16px;
}

.item-text {
    font-size: 14px;
    color: var(--text-primary);
    margin: 0 0 12px 0;
    line-height: 1.4;
}

.status-buttons {
    display: flex;
    gap: 8px;
}

.status-btn {
    padding: 6px 12px;
    border: 1px solid var(--border-color);
    background: var(--bg-white);
    color: var(--text-primary);
    border-radius: var(--radius-sm);
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.status-btn:hover {
    background: var(--bg-hover);
}

.status-btn.selected {
    background: var(--primary-color);
    color: var(--bg-white);
    border-color: var(--primary-color);
}

.item-actions {
    display: flex;
    gap: 8px;
    align-items: center;
}

.action-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 8px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-size: 14px;
    transition: var(--transition-base);
    opacity: 0.5;
    cursor: not-allowed;
}

.action-btn:hover:not(:disabled) {
    background: var(--primary-hover);
}

.action-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.note-input-container {
    margin-top: 12px;
    padding: 8px;
    background: #f8f9fa;
    border-radius: var(--radius-sm);
    border: 1px solid #e9ecef;
}

.note-input {
    width: 100%;
    padding: 8px 12px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 14px;
    color: var(--text-primary);
    background: var(--bg-white);
    transition: var(--transition-base);
}

.note-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 2px rgba(0, 85, 165, 0.1);
}

.note-input::placeholder {
    color: var(--text-muted);
    font-style: italic;
}

/* Image Preview Styles */
.image-preview-container {
    margin-top: 12px;
    padding: 12px;
    background: #f8f9fa;
    border-radius: var(--radius-sm);
    border: 1px solid #e9ecef;
}

.image-preview-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
}

.image-preview-label {
    font-size: 12px;
    font-weight: 600;
    color: var(--primary-color);
}

.remove-image-btn {
    background: #e74c3c;
    color: white;
    border: none;
    border-radius: 50%;
    width: 20px;
    height: 20px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    font-size: 14px;
    font-weight: bold;
    transition: var(--transition-base);
}

.remove-image-btn:hover {
    background: #c0392b;
    transform: scale(1.1);
}

.image-preview {
    max-width: 100%;
    max-height: 200px;
    border-radius: var(--radius-sm);
    border: 1px solid #ddd;
    object-fit: contain;
    display: block;
    margin: 0 auto;
}

/* ==================== STEP 3 STYLES ==================== */
.summary-content {
    max-width: 1040px;
}

.summary-highlight {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 16px;
    margin-bottom: 24px;
}

.summary-pill {
    background: linear-gradient(135deg, rgba(0, 85, 165, 0.08), rgba(0, 85, 165, 0.02));
    border: 1px solid rgba(0, 85, 165, 0.2);
    border-radius: var(--radius-md);
    padding: 16px;
    display: flex;
    flex-direction: column;
    gap: 4px;
    box-shadow: var(--shadow-sm);
}

.pill-label {
    font-size: 14px;
    color: var(--text-muted);
}

.pill-value {
    font-size: 26px;
    font-weight: 700;
    color: var(--primary-color);
}

.summary-pill.pill-muted {
    background: var(--bg-white);
    border-color: var(--border-color);
    color: var(--text-secondary);
    font-weight: 600;
}

.summary-pill.pill-outline {
    background: rgba(46, 204, 113, 0.1);
    border-color: rgba(46, 204, 113, 0.4);
    color: #0f9d58;
    font-weight: 600;
}

.summary-pill.pill-alert {
    background: rgba(231, 76, 60, 0.1);
    border-color: rgba(231, 76, 60, 0.4);
    color: #e74c3c;
}

.summary-pill.pill-success {
    background: rgba(34, 197, 94, 0.12);
    border-color: rgba(34, 197, 94, 0.4);
    color: #15803d;
}

.summary-pill.pill-danger {
    background: rgba(248, 113, 113, 0.12);
    border-color: rgba(248, 113, 113, 0.4);
    color: #b91c1c;
}

.summary-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
    gap: 24px;
}

.summary-card {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 24px;
    box-shadow: var(--shadow-sm);
    position: relative;
    overflow: hidden;
}

.summary-card::before {
    content: '';
    position: absolute;
    inset: 0;
    border-radius: var(--radius-md);
    border-left: 4px solid rgba(0, 85, 165, 0.4);
    pointer-events: none;
    z-index: 0;
}

.summary-card > * {
    position: relative;
    z-index: 1;
}

.summary-card.full-width {
    grid-column: 1 / -1;
}

.summary-card .form-group {
    margin-bottom: 0;
}

.priority-wrapper {
    margin-top: 20px;
}

.select-container {
    position: relative;
}

.radio-group {
    display: flex;
    gap: 16px;
    margin-top: 8px;
}

.radio-option {
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
}

.radio-option input[type="radio"] {
    margin: 0;
    accent-color: var(--primary-color);
}

.radio-label {
    font-size: 14px;
    color: var(--text-primary);
    cursor: pointer;
}

.priority-container {
    display: flex;
    gap: 16px;
    margin-top: 8px;
    flex-wrap: wrap;
}

.priority-option {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 8px;
    padding: 12px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-sm);
    cursor: pointer;
    transition: var(--transition-base);
    min-width: 80px;
}

.priority-option:hover {
    border-color: var(--primary-color);
}

.priority-option.selected {
    border-color: var(--primary-color);
    background: var(--bg-hover);
}

.priority-icon {
    font-size: 24px;
}

.input-with-icon {
    position: relative;
    display: flex;
    align-items: flex-end;
}

.textarea {
    resize: vertical;
    min-height: 100px;
}

.voice-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 12px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    margin-left: 8px;
    transition: var(--transition-base);
}

.voice-btn:hover {
    background: var(--primary-hover);
}

.signature-container {
    margin-top: 16px;
}

.signature-box {
    border: 2px dashed var(--border-color);
    border-radius: var(--radius-sm);
    padding: 20px;
    text-align: center;
    cursor: pointer;
    transition: var(--transition-base);
    min-height: 120px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.signature-box:hover {
    border-color: var(--primary-color);
    background: var(--bg-hover);
}

.signature-placeholder {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 8px;
    color: var(--text-muted);
}

.signature-display {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 12px;
}

.signature-display img {
    max-width: 300px;
    max-height: 150px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
}

.change-signature-btn {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 8px 16px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-size: 12px;
}

.acknowledgment-section {
    background: var(--bg-white);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    padding: 20px;
    margin-top: 24px;
}

.acknowledgment-header {
    display: flex;
    align-items: center;
    gap: 8px;
    margin-bottom: 12px;
}

.acknowledgment-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0;
}

.acknowledgment-section p {
    font-size: 13px;
    color: var(--text-primary);
    line-height: 1.5;
    margin: 0;
}

/* ==================== NAVIGATION BUTTONS ==================== */
.nav-buttons {
    display: flex;
    justify-content: space-between;
    padding: 16px 0;
    background: #f0f0f0;
    border-top: 1px solid #ddd;
    position: sticky;
    bottom: 0;
    left: 0;
    right: 0;
    margin: 0 -16px;
    padding: 16px;
}

.nav-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border-radius: var(--radius-sm);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    border: none;
}

.back-btn {
    background: none;
    color: var(--primary-color);
}

.back-btn:hover {
    color: var(--primary-hover);
}

.next-btn {
    background: var(--primary-color);
    color: var(--bg-white);
}

.next-btn:hover:not(:disabled) {
    background: var(--primary-hover);
}

.next-btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

/* ==================== SIGNATURE MODAL STYLES ==================== */
.signature-modal {
    width: min(640px, 90vw);
    max-height: 85vh;
    padding: 24px;
    display: flex;
    flex-direction: column;
    gap: 16px;
}

.signature-pad-container {
    padding: 24px;
    border: 1px dashed var(--border-color);
    border-radius: var(--radius-md);
    background: #f8fafc;
}

#signaturePad {
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    cursor: crosshair;
}

.signature-actions {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    margin-top: 8px;
}

.btn-clear,
.btn-save {
    min-width: 140px;
    text-align: center;
    font-weight: 600;
    box-shadow: var(--shadow-sm);
}

.btn-clear {
    background: #6c757d;
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
}

.btn-save {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
}

/* ==================== SUCCESS MODAL STYLES ==================== */
.success-modal {
    width: 90%;
    max-width: 400px;
}

.success-content {
    text-align: center;
    padding: 20px;
}

.success-icon {
    font-size: 48px;
    margin-bottom: 16px;
}

.success-content h3 {
    font-size: 20px;
    font-weight: 600;
    color: var(--primary-color);
    margin: 0 0 8px 0;
}

.success-content p {
    font-size: 14px;
    color: var(--text-muted);
    margin: 0 0 20px 0;
}

.success-actions {
    display: flex;
    gap: 12px;
    justify-content: center;
    flex-wrap: wrap;
}

.btn-primary {
    background: var(--primary-color);
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-weight: 500;
}

.btn-success {
    background: #28a745;
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-weight: 500;
    transition: var(--transition-base);
}

.btn-success:hover {
    background: #218838;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(40, 167, 69, 0.3);
}

.btn-secondary {
    background: #6c757d;
    color: var(--bg-white);
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-sm);
    cursor: pointer;
    font-weight: 500;
}

/* Responsive success modal buttons */
@media (max-width: 480px) {
    .success-actions {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-primary,
    .btn-success,
    .btn-secondary {
        width: 100%;
        padding: 12px 20px;
        font-size: 14px;
    }
}

/* ==================== ERROR MESSAGE STYLES ==================== */
.error-message {
    color: #e74c3c;
    font-size: 12px;
    margin-top: 4px;
    display: none;
}

/* ==================== RESPONSIVE DESIGN ==================== */
@media (max-width: 768px) {
    .step-content {
        padding: 0 8px;
    }
    
    .step-header {
        padding: 20px;
    }
    
    .step-header h1 {
        font-size: 24px;
    }
    
    .checklist-item {
        flex-direction: column;
        gap: 12px;
    }
    
    .item-content {
        margin-right: 0;
    }
    
    .item-actions {
        align-self: stretch;
        justify-content: center;
    }
    
    .nav-buttons {
        flex-direction: column;
        gap: 12px;
    }
    
    .nav-btn {
        justify-content: center;
    }
    
    .priority-container {
        justify-content: center;
    }
    
    .radio-group {
        flex-direction: column;
        gap: 8px;
    }
}

@media (max-width: 480px) {
    .step-header {
        padding: 16px;
    }
    
    .step-header h1 {
        font-size: 20px;
    }
    
    .signature-modal {
        width: 95%;
    }
    
    .signature-pad-container {
        padding: 10px;
    }
    
    #signaturePad {
        width: 100%;
        height: 150px;
    }
}

/* ==================== FOOTER STYLES ==================== */
.dvir-footer {
    background: linear-gradient(135deg, #1a365d 0%, #2c5282 100%);
    color: white;
    margin-top: 60px;
    padding: 0;
    width: 100%;
}

.footer-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 60px 24px 40px;
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 60px;
    align-items: start;
}

.footer-left {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.footer-logo {
    display: flex;
    align-items: center;
    gap: 12px;
    margin-bottom: 16px;
}

.footer-logo-img {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    background: white;
    padding: 4px;
}

.footer-company-name {
    font-size: 24px;
    font-weight: 700;
    color: white;
}

.footer-description {
    color: #e2e8f0;
    font-size: 16px;
    line-height: 1.6;
    margin: 0;
    max-width: 300px;
}

.footer-right {
    display: flex;
    justify-content: flex-end;
}

.footer-links {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 40px;
}

.footer-section h4 {
    color: white;
    font-size: 18px;
    font-weight: 600;
    margin: 0 0 20px 0;
    position: relative;
}

.footer-section h4::after {
    content: '';
    position: absolute;
    bottom: -8px;
    left: 0;
    width: 30px;
    height: 2px;
    background: var(--primary-color);
    border-radius: 1px;
}

.footer-section ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.footer-section ul li {
    margin-bottom: 12px;
}

.footer-section ul li a {
    color: #e2e8f0;
    text-decoration: none;
    font-size: 14px;
    transition: all 0.3s ease;
    display: inline-block;
}

.footer-section ul li a:hover {
    color: white;
    transform: translateX(4px);
}

.footer-bottom {
    background: rgba(0, 0, 0, 0.2);
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    padding: 20px 0;
}

.footer-bottom-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 24px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 20px;
}

.footer-bottom p {
    color: #cbd5e0;
    font-size: 14px;
    margin: 0;
}

.footer-social {
    display: flex;
    align-items: center;
    gap: 16px;
}

.footer-social span {
    color: #cbd5e0;
    font-size: 14px;
    margin-right: 8px;
}

.social-link {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 36px;
    height: 36px;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 8px;
    color: #e2e8f0;
    text-decoration: none;
    transition: all 0.3s ease;
}

.social-link:hover {
    background: var(--primary-color);
    color: white;
    transform: translateY(-2px);
}

/* Mobile responsive footer */
@media (max-width: 768px) {
    .footer-content {
        grid-template-columns: 1fr;
        gap: 40px;
        padding: 40px 16px 30px;
    }
    
    .footer-links {
        grid-template-columns: 1fr;
        gap: 30px;
    }
    
    .footer-bottom-content {
        flex-direction: column;
        text-align: center;
        gap: 16px;
    }
    
    .footer-social {
        justify-content: center;
    }
    
    .footer-company-name {
        font-size: 20px;
    }
    
    .footer-description {
        max-width: none;
    }
}

@media (max-width: 480px) {
    .footer-content {
        padding: 30px 16px 20px;
    }
    
    .footer-links {
        gap: 24px;
    }
    
    .footer-section h4 {
        font-size: 16px;
    }
    
    .footer-section ul li a {
        font-size: 13px;
    }
}
</style>


@endsection
