@extends('welcome')

@section('title', 'Create New Request Maintenance - React Style')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.workorder-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Form Groups */
.form-group {
    margin-bottom: 25px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.required {
    color: var(--danger-color);
    font-weight: bold;
}

.form-input {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-input.error {
    border-color: var(--danger-color);
}

.error-text {
    color: var(--danger-color);
    font-size: 12px;
    margin-top: 4px;
}

/* Request Number Display */
.request-number-display {
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 15px 20px;
    margin-bottom: 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.request-number-label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
}

.request-number-value {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary-color);
}

.react-request-number-row {
    display: flex;
    align-items: center;
    justify-content: space-between;
    background-color: #F2F2F2;
    border-radius: 6px;
    padding: 16px;
    margin-bottom: 8px;
}

.react-request-number-label {
    font-size: 15px;
    color: #015185;
    font-weight: 500;
}

.react-request-number-value {
    font-size: 15px;
    color: #025383;
    font-weight: bold;
}

.react-input-field {
    width: 100%;
    border: 1px solid #015185;
    border-radius: 5px;
    padding: 12px 14px;
    font-size: 16px;
    background-color: #fff;
    box-sizing: border-box;
}

.react-input-field:focus {
    outline: none;
    border-color: #015185;
    box-shadow: 0 0 0 2px rgba(1, 81, 133, 0.2);
}

.react-input-field.error {
    border-color: red;
}

.react-error-text {
    color: red;
    font-size: 13px;
    margin-top: 4px;
    margin-left: 2px;
}

.react-priority-container {
    display: flex;
    justify-content: space-around;
    margin-bottom: 8px;
    background-color: #fff;
    border-radius: 5px;
    width: 100%;
}

.react-priority-item {
    display: flex;
    flex-direction: column;
    align-items: center;
}

.react-priority-button {
    padding: 8px;
    border-radius: 5px;
    border: 1px solid #015185;
    width: 85px;
    height: 60px;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    cursor: pointer;
    transition: all 0.2s ease;
}

.react-priority-button:hover {
    background-color: #f0f8ff;
}

.react-priority-button.selected {
    background-color: #015185;
    color: #fff;
}

.react-priority-image {
    width: 24px;
    height: 24px;
    margin-bottom: 3px;
}

.react-priority-text {
    font-size: 10px;
    color: #333;
    text-align: center;
    font-weight: 500;
}

.react-priority-text.selected {
    color: #fff;
    font-weight: 600;
}

.react-select-row {
    display: flex;
    align-items: center;
    justify-content: space-between;
    background-color: #F5F5F5;
    border-radius: 4px;
    padding: 14px 12px;
    margin-bottom: 4px;
    cursor: pointer;
    transition: background-color 0.2s ease;
}

.react-select-row:hover {
    background-color: #e8f4f8;
}

.react-select-label {
    font-size: 15px;
    color: #707173;
    font-weight: 500;
}

.react-select-right {
    display: flex;
    align-items: center;
}

.react-select-text {
    color: #015185;
    font-size: 15px;
    font-weight: 500;
    margin-right: 2px;
}

.react-selected-text {
    font-size: 14px;
    color: #333;
    margin-top: 4px;
    text-align: left;
}

.react-selected-items-container {
    margin-top: 4px;
}

.react-selected-items-row {
    display: flex;
    align-items: center;
    flex-wrap: wrap;
    gap: 8px;
}

.react-selected-item {
    display: flex;
    align-items: center;
    background-color: #e0e0e0;
    border-radius: 20px;
    padding: 4px 8px;
    margin-right: 8px;
    margin-bottom: 4px;
}

.react-selected-item-text {
    font-size: 14px;
    color: #015185;
    margin-right: 8px;
}

.react-close-button {
    padding: 2px;
    cursor: pointer;
}

.react-close-circle {
    width: 16px;
    height: 16px;
    border-radius: 50%;
    background-color: #d32f2f;
    display: flex;
    justify-content: center;
    align-items: center;
    color: #fff;
    font-size: 12px;
}

.react-task-list-container {
    margin-top: 10px;
    max-height: 120px;
    overflow-y: auto;
}

.react-task-card {
    background-color: #fff;
    border-radius: 8px;
    padding: 14px;
    margin-bottom: 10px;
    border: 1px solid #eee;
    position: relative;
}

.react-task-card-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.react-task-card-title {
    font-size: 16px;
    color: #222;
    font-weight: 500;
    flex: 1;
}

.react-task-card-status-pill {
    background-color: #bfc2c5;
    border-radius: 16px;
    padding: 4px 14px;
    margin-left: 8px;
}

.react-task-card-status-text {
    color: #fff;
    font-weight: 500;
    font-size: 14px;
}

.react-task-card-menu-button {
    padding: 4px;
    margin-left: 8px;
    cursor: pointer;
}

.react-task-card-type {
    color: #e6b800;
    font-weight: 500;
    font-size: 15px;
    margin-top: 6px;
    margin-bottom: 2px;
}

.react-add-task-button {
    display: flex;
    align-items: center;
    background-color: #015185;
    padding: 8px 12px;
    border-radius: 8px;
    color: #fff;
    font-size: 14px;
    text-decoration: none;
    cursor: pointer;
    transition: background-color 0.2s ease;
}

.react-add-task-button:hover {
    background-color: #003f7d;
}

.react-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

/* Body scroll lock when modal is open */
body.modal-open {
    overflow: hidden;
}

/* Modal Styles - Matching Create Workorder */
.custom-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    inset: 0;
    width: 100vw;
    min-height: 100vh;
    background-color: rgba(0, 0, 0, 0.5);
    padding: 40px 16px;
    overflow-y: auto;
    align-items: center;
    justify-content: center;
}

.custom-modal.show {
    display: flex;
}

.custom-modal-content {
    background-color: #fefefe;
    margin: auto;
    padding: 0;
    border: 1px solid #888;
    width: 80%;
    max-width: 600px;
    border-radius: 8px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    position: relative;
    transform-origin: center;
    max-height: 80vh;
    overflow-y: auto;
    animation: modalSlideIn 0.3s ease;
}

@media (max-width: 768px) {
    .custom-modal {
        align-items: flex-start;
    }
}

.success-modal-content {
    max-width: 400px;
    text-align: center;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: translateY(-10px) scale(0.96);
    }
    to {
        opacity: 1;
        transform: translateY(0) scale(1);
    }
}

.custom-modal-header {
    background-color: #0055A5;
    color: white;
    padding: 15px 20px;
    border-radius: 8px 8px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 500;
}

.close-btn {
    color: white;
    font-size: 28px;
    font-weight: bold;
    background: none;
    border: none;
    cursor: pointer;
    padding: 0;
    line-height: 1;
}

.close-btn:hover {
    opacity: 0.7;
}

.custom-modal-body {
    padding: 20px;
}

.search-container {
    display: flex;
    gap: 10px;
    margin-bottom: 20px;
}

.search-input {
    flex: 1;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.search-btn {
    padding: 10px 15px;
    background-color: #0055A5;
    color: white;
    border: none;
    border-radius: 4px;
    cursor: pointer;
}

.search-btn:hover {
    background-color: #003f7d;
}

.equipment-list, .user-list, .inspection-list, .jobcard-list {
    max-height: 400px;
    overflow-y: auto;
    border: 1px solid #eee;
    border-radius: 4px;
}

.equipment-item, .user-item, .inspection-item, .jobcard-item {
    padding: 15px;
    border-bottom: 1px solid #eee;
    cursor: pointer;
    transition: background-color 0.2s;
}

.equipment-item:hover, .user-item:hover, .inspection-item:hover, .jobcard-item:hover {
    background-color: #f8f9fa;
}

.equipment-item:last-child, .user-item:last-child, .inspection-item:last-child, .jobcard-item:last-child {
    border-bottom: none;
}

.equipment-name, .user-name, .inspection-name, .jobcard-name {
    font-weight: 600;
    color: #0055A5;
    margin-bottom: 5px;
}

.equipment-details, .user-details, .inspection-details, .jobcard-details {
    font-size: 12px;
    color: #666;
    line-height: 1.4;
}

.modal-footer {
    padding: 15px 20px;
    background-color: #f8f9fa;
    border-top: 1px solid #dee2e6;
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 15px;
}

.modal-footer .btn {
    padding: 10px 20px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    transition: all 0.2s ease;
    text-align: center;
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-footer .btn-primary {
    background-color: #0055A5;
    color: white;
}

.modal-footer .btn-primary:hover {
    background-color: #003f7d;
}

.modal-footer .btn-secondary {
    background-color: #0055A5;
    color: white;
}

.modal-footer .btn-secondary:hover {
    background-color: #003f7d;
}

/* Priority Selection Styles */
.priority-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 15px;
    margin-bottom: 20px;
    background-color: #fff;
    border-radius: 8px;
    padding: 20px;
    border: 1px solid var(--border-light);
}

.priority-item {
    display: flex;
    flex-direction: column;
    align-items: center;
    cursor: pointer;
}

.priority-button {
    padding: 15px;
    border-radius: 8px;
    border: 2px solid var(--border-color);
    width: 100%;
    min-height: 80px;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    cursor: pointer;
    transition: all 0.3s ease;
    background: white;
    position: relative;
}

.priority-button:hover {
    background-color: #f0f8ff;
    border-color: var(--primary-color);
}

.priority-button.selected {
    background-color: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 2px rgba(0, 85, 165, 0.3);
    transform: scale(1.05);
}

.priority-image {
    width: 24px;
    height: 24px;
    margin-bottom: 5px;
}

.priority-text {
    font-size: 11px;
    color: #333;
    text-align: center;
    font-weight: 500;
}

.priority-button.selected .priority-text {
    color: white;
    font-weight: 600;
}

/* Select Row Styles */
.select-row {
    display: flex;
    align-items: center;
    justify-content: space-between;
    background-color: #f8f9fa;
    border: 1px solid var(--border-light);
    border-radius: 8px;
    padding: 15px 20px;
    margin-bottom: 15px;
    cursor: pointer;
    transition: all 0.2s ease;
}

.select-row:hover {
    background-color: #e9ecef;
    border-color: var(--primary-color);
}

.select-label {
    font-size: 14px;
    color: var(--text-secondary);
    font-weight: 500;
}

.select-right {
    display: flex;
    align-items: center;
    gap: 10px;
}

.select-text {
    color: var(--primary-color);
    font-size: 14px;
    font-weight: 500;
}

.select-arrow {
    color: var(--text-secondary);
    font-size: 12px;
}

/* Selected Items Display */
.selected-items-container {
    margin-top: 10px;
}

.selected-items-row {
    display: flex;
    align-items: center;
    flex-wrap: wrap;
    gap: 8px;
}

.selected-item {
    display: flex;
    align-items: center;
    background-color: #e3f2fd;
    border: 1px solid #bbdefb;
    border-radius: 20px;
    padding: 6px 12px;
    margin-right: 8px;
    margin-bottom: 4px;
}

.selected-item-text {
    font-size: 13px;
    color: var(--primary-color);
    margin-right: 8px;
    font-weight: 500;
}

.close-button {
    padding: 2px;
    cursor: pointer;
    color: var(--danger-color);
}

.close-circle {
    width: 16px;
    height: 16px;
    border-radius: 50%;
    background-color: var(--danger-color);
    display: flex;
    justify-content: center;
    align-items: center;
    color: white;
    font-size: 12px;
    font-weight: bold;
}

/* Task List Styles */
.task-list-container {
    margin-top: 15px;
    max-height: 200px;
    overflow-y: auto;
}

.task-card {
    background-color: #fff;
    border: 1px solid var(--border-light);
    border-radius: 8px;
    padding: 15px;
    margin-bottom: 10px;
    position: relative;
    box-shadow: var(--shadow-sm);
}

.task-card-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 8px;
}

.task-card-title {
    font-size: 15px;
    color: var(--text-primary);
    font-weight: 600;
    flex: 1;
}

.task-card-actions {
    display: flex;
    align-items: center;
    gap: 10px;
}

.task-status-dropdown {
    padding: 4px 8px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 12px;
    background: white;
    color: var(--text-primary);
    cursor: pointer;
    transition: var(--transition-base);
}

.task-status-dropdown:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 2px rgba(0, 85, 165, 0.1);
}

.task-delete-btn {
    background: var(--danger-color);
    color: white;
    border: none;
    border-radius: var(--radius-sm);
    padding: 6px 8px;
    cursor: pointer;
    font-size: 12px;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    justify-content: center;
}

.task-delete-btn:hover {
    background: #c82333;
    transform: scale(1.05);
}

.task-delete-btn:active {
    transform: scale(0.95);
}

.task-card-type {
    color: var(--warning-color);
    font-weight: 500;
    font-size: 13px;
    margin-top: 5px;
}

.add-task-button {
    display: flex;
    align-items: center;
    background-color: var(--primary-color);
    padding: 10px 15px;
    border-radius: 8px;
    color: white;
    font-size: 14px;
    text-decoration: none;
    cursor: pointer;
    transition: background-color 0.2s ease;
    border: none;
    gap: 8px;
}

.add-task-button:hover {
    background-color: var(--primary-hover);
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .custom-modal-content {
        width: 95%;
        margin: 10px;
    }
    
    .priority-container {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .priority-button {
        min-height: 70px;
    }
    
    .modal-footer {
        flex-direction: column;
        gap: 10px;
    }
    
    .modal-footer .btn {
        width: 100%;
        max-width: 200px;
    }
}

/* Sync Modal */
.react-sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: rgba(0, 0, 0, 0.5);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 1001;
}

.react-sync-modal-content {
    width: 70%;
    background-color: white;
    border-radius: 10px;
    padding: 20px;
    text-align: center;
}

.react-sync-modal-text {
    font-size: 18px;
    margin-bottom: 10px;
    color: #333;
}

.react-sync-icon {
    width: 30px;
    height: 30px;
    margin-right: 10px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.loading-spinner {
    width: 30px;
    height: 30px;
    border: 3px solid #f3f3f3;
    border-top: 3px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin: 0 auto;
}

.react-sync-icon-div {
    display: flex;
    align-items: center;
    justify-content: center;
}

.react-sync-modal-text-add {
    font-size: 14px;
    color: #333;
}

/* Success Modal */
.react-success-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: rgba(0, 0, 0, 0.5);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 1001;
}

.react-success-modal-content {
    background-color: #fff;
    border-radius: 10px;
    padding: 20px;
    width: 70%;
    text-align: center;
}

.react-success-modal-text {
    font-size: 18px;
    font-weight: bold;
    color: #015185;
    margin-bottom: 10px;
}

.react-success-modal-sub-text {
    font-size: 16px;
    color: #333;
    margin-bottom: 20px;
}

.react-success-modal-button {
    background-color: #015185;
    padding: 8px 20px;
    border-radius: 8px;
    color: #fff;
    font-size: 16px;
    font-weight: 600;
    border: none;
    cursor: pointer;
}

.react-success-modal-button:hover {
    background-color: #003f7d;
}

/* Equipment Display Styles */
.equipment-item {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    border: 1px solid #e9ecef;
    border-radius: 12px;
    padding: 20px;
    margin-bottom: 16px;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    position: relative;
    overflow: hidden;
}

.equipment-item::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #015185, #3481BC);
}

.equipment-item:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(1, 81, 133, 0.2);
    border-color: #015185;
}

.equipment-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
    padding-bottom: 12px;
    border-bottom: 2px solid #e9ecef;
}

.equipment-name {
    font-size: 18px;
    font-weight: 700;
    color: #015185;
    flex: 1;
    text-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
}

.equipment-status {
    padding: 6px 16px;
    border-radius: 25px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.8px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.equipment-status.active {
    background-color: #d4edda;
    color: #155724;
}

.equipment-status.inactive {
    background-color: #f8d7da;
    color: #721c24;
}

.equipment-status.maintenance {
    background-color: #fff3cd;
    color: #856404;
}

.equipment-status.operational {
    background-color: #d1ecf1;
    color: #0c5460;
}

.equipment-details {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 12px;
}

.equipment-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 12px;
    background: rgba(248, 249, 250, 0.5);
    border-radius: 6px;
    border-left: 3px solid #015185;
    transition: all 0.2s ease;
}

.equipment-row:hover {
    background: rgba(1, 81, 133, 0.05);
    transform: translateX(2px);
}

.equipment-label {
    font-size: 14px;
    color: #6c757d;
    font-weight: 600;
    flex: 1;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.equipment-value {
    font-size: 14px;
    color: #495057;
    font-weight: 500;
    text-align: right;
    flex: 1;
    background: rgba(255, 255, 255, 0.8);
    padding: 4px 8px;
    border-radius: 4px;
    border: 1px solid #e9ecef;
}

/* Selected Equipment Card Styles */
.selected-equipment-card {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    border: 2px solid #015185;
    border-radius: 12px;
    padding: 16px;
    margin-top: 8px;
    box-shadow: 0 4px 8px rgba(1, 81, 133, 0.1);
}

.selected-equipment-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 16px;
    padding-bottom: 12px;
    border-bottom: 2px solid #015185;
}

.selected-equipment-name {
    font-size: 18px;
    font-weight: 700;
    color: #015185;
    flex: 1;
}

.selected-equipment-status {
    padding: 6px 16px;
    border-radius: 25px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.selected-equipment-info {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 12px;
}

.selected-equipment-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
    border-bottom: 1px solid #dee2e6;
}

.selected-equipment-row:last-child {
    border-bottom: none;
}

.selected-equipment-label {
    font-size: 14px;
    color: #6c757d;
    font-weight: 600;
    flex: 1;
}

.selected-equipment-value {
    font-size: 14px;
    color: #495057;
    font-weight: 500;
    text-align: right;
    flex: 1;
}

/* Responsive Design */
@media (max-width: 768px) {
    .react-header {
        height: 100px;
    }
    
    .react-header-title {
        font-size: 18px;
    }
    
    .react-form-group {
        padding: 0 15px;
    }
    
    .react-modal-content {
        width: 90%;
        padding: 15px;
    }
    
    .react-priority-container {
        flex-wrap: wrap;
        gap: 10px;
    }
    
    .react-priority-button {
        width: 70px;
        height: 50px;
    }
    
    .equipment-details,
    .selected-equipment-info {
        grid-template-columns: 1fr;
    }
    
    .equipment-header,
    .selected-equipment-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 8px;
    }
}

@media (max-width: 480px) {
    .react-header {
        height: 80px;
    }
    
    .react-header-title {
        font-size: 16px;
    }
    
    .react-form-group {
        padding: 0 10px;
    }
    
    .react-modal-content {
        width: 95%;
        padding: 10px;
    }
    
    .priority-container {
        grid-template-columns: 1fr;
    }
    
    .priority-button {
        min-height: 60px;
    }
}
</style>
@endsection

@section('content')
<div class="workorder-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('maintenance.workorders') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Request Maintenance</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" onclick="saveMaintenanceRequest()">
                    <i class="fas fa-plus"></i> Create
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="form-container">
            <form id="maintenanceForm" method="POST" action="{{ route('maintenance.request.store') }}">
                @csrf
                
                <!-- Request Number Section -->
                <div class="form-section">
                    <div class="request-number-display">
                        <span class="request-number-label">Request Number</span>
                        <span id="requestNumber" class="request-number-value">WO1001</span>
                    </div>
                </div>

                <!-- Work Order Details Section -->
                <div class="form-section">
                    <div class="section-title">
                        <i class="fas fa-clipboard-list"></i>
                        Work Order Details
                    </div>
                    
                    <!-- Work Order Title -->
                    <div class="form-group">
                        <label class="form-label">Work Order Title <span class="required">*</span></label>
                        <input type="text" name="workOrderTitle" id="workOrderTitle" class="form-input" placeholder="Write Request Title" required>
                        <div class="error-text" id="workOrderTitleError"></div>
                    </div>

                    <!-- Country and Project Row -->
                    <div class="form-group">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                            <div>
                                <label class="form-label">Country <span class="required">*</span></label>
                                <select name="country" id="country" class="form-input" required>
                                    <option value="">Select Country</option>
                                    <option value="Zambia">Zambia</option>
                                    <option value="DRC">DRC</option>
                                </select>
                                <div class="error-text" id="countryError"></div>
                            </div>
                            <div>
                                <label class="form-label">Project <span class="required">*</span></label>
                                <select name="project" id="project" class="form-input" required disabled>
                                    <option value="">Select Project</option>
                                </select>
                                <div class="error-text" id="projectError"></div>
                            </div>
                        </div>
                    </div>

                    <!-- Memo -->
                    <div class="form-group">
                        <label class="form-label">Memo</label>
                        <textarea name="memo" id="memo" class="form-input" rows="4" placeholder="Write Memo" style="height: 80px; resize: vertical;"></textarea>
                    </div>
                </div>

                <!-- Priority Section -->
                <div class="form-section">
                    <div class="section-title">
                        <i class="fas fa-exclamation-triangle"></i>
                        Priority Level
                    </div>
                    
                    <div class="priority-container">
                        <div class="priority-item">
                            <button type="button" class="priority-button" data-priority="Low" onclick="selectPriority('Low')">
                            🟢
                                <span class="priority-text">Low</span>
                            </button>
                        </div>
                        <div class="priority-item">
                            <button type="button" class="priority-button" data-priority="Medium" onclick="selectPriority('Medium')">
                            🟡
                                <span class="priority-text">Medium</span>
                            </button>
                        </div>
                        <div class="priority-item">
                            <button type="button" class="priority-button" data-priority="High" onclick="selectPriority('High')">
                            🟠
                                <span class="priority-text">High</span>
                            </button>
                        </div>
                        <div class="priority-item">
                            <button type="button" class="priority-button" data-priority="Emergency" onclick="selectPriority('Emergency')">
                            🔴
                                <span class="priority-text">Emergency</span>
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Equipment and Assignment Section -->
                <div class="form-section">
                    <div class="section-title">
                        <i class="fas fa-tools"></i>
                        Equipment & Assignment
                    </div>
                    
                    <!-- Equipment -->
                    <div class="form-group">
                        <div class="select-row" onclick="openEquipmentModal(event)">
                            <span class="select-label">Equipment</span>
                            <div class="select-right">
                                <span class="select-text">Select</span>
                                <i class="fas fa-chevron-right select-arrow"></i>
                            </div>
                        </div>
                        <div id="equipmentDetails" class="selected-items-container"></div>
                        <input type="hidden" name="selectedEquipment" id="selectedEquipment">
                    </div>

                    <!-- Assign To -->
                    <div class="form-group">
                        <div class="select-row" onclick="openAssignToModal(event)">
                            <span class="select-label">Assign To</span>
                            <div class="select-right">
                                <span class="select-text">Select</span>
                                <i class="fas fa-chevron-right select-arrow"></i>
                            </div>
                        </div>
                        <div id="assignToList" class="selected-items-container"></div>
                        <input type="hidden" name="assignTo" id="assignTo">
                    </div>

                    <!-- Request Type -->
                    <div class="form-group">
                        <div class="select-row" onclick="openRequestTypeModal(event)">
                            <span class="select-label">Request Type</span>
                            <div class="select-right">
                                <span class="select-text">Select</span>
                                <i class="fas fa-chevron-right select-arrow"></i>
                            </div>
                        </div>
                        <div id="requestTypeSelected" class="selected-items-container"></div>
                        <input type="hidden" name="requestType" id="requestType">
                    </div>
                </div>

                <!-- Tasks Section -->
                <div class="form-section">
                    <div class="section-title">
                        <i class="fas fa-tasks"></i>
                        Tasks
                    </div>
                    
                    <div class="form-group">
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                            <label class="form-label">Task List</label>
                            <button type="button" class="add-task-button" onclick="openAddTaskModal(event)">
                                <i class="fas fa-plus"></i> Add Task
                            </button>
                        </div>
                        <div id="taskList" class="task-list-container"></div>
                        <input type="hidden" name="tasks" id="tasks">
                    </div>
                </div>
        </form>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Equipment</h3>
            <button type="button" class="close-btn" onclick="closeEquipmentModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" class="search-input" placeholder="Search equipment..." onkeyup="filterEquipment()">
                <button type="button" class="search-btn" onclick="filterEquipment()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div id="equipmentList" class="equipment-list">
                <!-- Equipment items will be loaded here -->
            </div>
        </div>
    </div>
</div>

<!-- Assign To Modal -->
<div id="assignToModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Assign To</h3>
            <button type="button" class="close-btn" onclick="closeAssignToModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="assignToSearch" class="search-input" placeholder="Search team members..." onkeyup="filterTeamMembers()">
                <button type="button" class="search-btn" onclick="filterTeamMembers()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div id="assignToListModal" class="user-list">
                <!-- Team members will be loaded here -->
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-primary" onclick="saveAssignTo()">Save</button>
        </div>
    </div>
</div>

<!-- Request Type Modal -->
<div id="requestTypeModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Request Type</h3>
            <button type="button" class="close-btn" onclick="closeRequestTypeModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            @foreach(['General', 'Preventive', 'Breakdown', 'Corrective', 'Safety', 'Upgrade', 'Cleaning', 'Testing'] as $type)
                <div class="equipment-item" onclick="selectRequestType('{{ $type }}')">
                    <div class="equipment-name">{{ $type }}</div>
                    <div style="width: 20px; height: 20px; border: 2px solid #015185; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-left: auto;">
                        <div id="requestType_{{ $type }}" style="width: 10px; height: 10px; border-radius: 50%; background-color: #015185; display: none;"></div>
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</div>

<!-- Add Task Modal -->
<div id="addTaskModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Add Task</h3>
            <button type="button" class="close-btn" onclick="closeAddTaskModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="form-group">
                <label class="form-label">Task Title <span class="required">*</span></label>
                <input type="text" id="taskTitle" class="form-input" placeholder="Task Title" required>
                <div class="error-text" id="taskTitleError"></div>
            </div>
            <div class="form-group">
                <label class="form-label">Type</label>
                <div style="display: flex; gap: 10px; margin-top: 5px;">
                    <button type="button" class="priority-button" data-type="Repair" onclick="selectTaskType('Repair')">Repair</button>
                    <button type="button" class="priority-button" data-type="Replace" onclick="selectTaskType('Replace')">Replace</button>
                </div>
                <input type="hidden" id="taskType">
            </div>
                <div class="form-group">
                    <label class="form-label">Assign To</label>
                    <div style="display: flex; gap: 10px; align-items: center; margin-top: 5px;">
                        <input type="text" id="taskAssignToDisplay" class="form-input" placeholder="Select assignees" readonly style="flex: 1;">
                        <button type="button" class="search-btn" onclick="openTaskAssignToModal(event)">Set</button>
                    </div>
                <div id="taskAssignToList" class="selected-items-container"></div>
            </div>
            <div class="form-group">
                <label class="form-label">Note</label>
                <textarea id="taskNote" class="form-input" rows="3" placeholder="Enter task notes..." style="height: 60px; resize: vertical;"></textarea>
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeAddTaskModal()">Cancel</button>
            <button type="button" class="btn btn-primary" onclick="saveTask()">Save Task</button>
        </div>
    </div>
</div>

<!-- Task Assign To Modal -->
<div id="taskAssignToModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Assign Task To</h3>
            <button type="button" class="close-btn" onclick="closeTaskAssignToModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="taskAssignToSearch" class="search-input" placeholder="Search team members..." onkeyup="filterTaskTeamMembers()">
                <button type="button" class="search-btn" onclick="filterTaskTeamMembers()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div id="taskAssignToListModal" class="user-list">
                <!-- Team members will be loaded here -->
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-primary" onclick="saveTaskAssignTo()">Save</button>
        </div>
    </div>
</div>

<!-- Sync Modal -->
<div id="syncModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content" style="max-width: 400px; text-align: center;">
        <div class="custom-modal-body">
            <div style="font-size: 18px; margin-bottom: 20px; color: #333;">Please Wait</div>
            <div style="display: flex; align-items: center; justify-content: center; gap: 15px;">
                <img src="{{ asset('images/adminlogin/sync.png') }}" alt="Sync" style="width: 30px; height: 30px; animation: spin 1s linear infinite;">
                <div style="font-size: 14px; color: #666;">Saving Maintenance Request...</div>
            </div>
        </div>
    </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content success-modal-content">
        <div class="custom-modal-header" style="text-align: center;">
            <h3 style="margin: 0; width: 100%;">Success</h3>
        </div>
        <div class="custom-modal-body">
            <div style="font-size: 16px; color: #333; margin-bottom: 20px;">Maintenance request created successfully!</div>
            <button class="btn btn-primary" onclick="goBackToRequestMaintenance()" style="background-color: #0055A5; color: white; border: none; padding: 10px 20px; border-radius: 4px; cursor: pointer; font-size: 14px; font-weight: 500;">Back</button>
        </div>
    </div>
</div>

<!-- Bootstrap JS and jQuery -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
// React Native Style JavaScript
let selectedPriority = null;
let selectedEquipment = null;
let selectedAssignees = []; // Main form assignees
let taskAssignees = []; // Task-specific assignees
let selectedRequestType = null;
let tasks = [];
let teamMembers = [];
let equipmentList = [];
let requestNumber = 'WO1001';
let country = null;
let project = null;

// Console logging function for debugging
function logAllData() {
    console.log('=== MAINTENANCE REQUEST DATA LOG ===');
    console.log('Selected Priority:', selectedPriority);
    console.log('Selected Equipment:', selectedEquipment);
    console.log('Selected Assignees (Main):', selectedAssignees);
    console.log('Task Assignees:', taskAssignees);
    console.log('Selected Request Type:', selectedRequestType);
    console.log('Tasks:', tasks);
    console.log('Team Members:', teamMembers);
    console.log('Equipment List:', equipmentList);
    console.log('Request Number:', requestNumber);
    console.log('Country:', country);
    console.log('Project:', project);
    
    // Form field values
    console.log('=== FORM FIELD VALUES ===');
    console.log('Work Order Title:', document.getElementById('workOrderTitle')?.value || 'N/A');
    console.log('Country Select:', document.getElementById('country')?.value || 'N/A');
    console.log('Project Select:', document.getElementById('project')?.value || 'N/A');
    console.log('Memo:', document.getElementById('memo')?.value || 'N/A');
    
    console.log('=== END DATA LOG ===');
}

// Manual logging function - call this from browser console: logAllFormData()
function logAllFormData() {
    console.log('=== COMPREHENSIVE FORM DATA LOG ===');
    logAllData();
    
    // Additional detailed logging
    console.log('=== DETAILED EQUIPMENT DATA ===');
    if (selectedEquipment) {
        Object.keys(selectedEquipment).forEach(key => {
            console.log(`Equipment ${key}:`, selectedEquipment[key]);
        });
    }
    
    console.log('=== DETAILED ASSIGNEE DATA ===');
    selectedAssignees.forEach((assignee, index) => {
        console.log(`Assignee ${index + 1}:`, assignee);
        Object.keys(assignee).forEach(key => {
            console.log(`  ${key}:`, assignee[key]);
        });
    });
    
    console.log('=== DETAILED TASK DATA ===');
    tasks.forEach((task, index) => {
        console.log(`Task ${index + 1}:`, task);
        Object.keys(task).forEach(key => {
            console.log(`  ${key}:`, task[key]);
        });
    });
    
    console.log('=== END COMPREHENSIVE LOG ===');
}

// Make logging function available globally
window.logAllFormData = logAllFormData;

// Dynamic Request Number Generation
function generateRequestNumber(country, project, nextNumber) {
    const countryCode = country === 'Zambia' ? 'ZAM' : country === 'DRC' ? 'DRC' : 'XXX';
    const projectCode = project ? project.slice(0, 3).toUpperCase() : 'XXX';
    return `${countryCode}-${projectCode}-RM-${nextNumber.toString().padStart(3, '0')}`;
}

function extractNumberFromRequestNumber(requestNumber) {
    if (!requestNumber) return 1;
    
    // Try to extract number from new format: ZAM-KAN-RM-001
    const newFormatMatch = requestNumber.match(/-RM-(\d+)$/);
    if (newFormatMatch) {
        const number = parseInt(newFormatMatch[1], 10);
        console.log('Extracted number from new format:', { requestNumber, number });
        return number;
    }
    
    // Try to extract number from old format: RM001
    const oldFormatMatch = requestNumber.match(/RM(\d+)/);
    if (oldFormatMatch) {
        const number = parseInt(oldFormatMatch[1], 10);
        console.log('Extracted number from old format:', { requestNumber, number });
        return number;
    }
    
    // Fallback
    console.log('Using fallback number for:', requestNumber);
    return 1;
}

async function updateRequestNumber() {
    try {
        // Fetch all requests from MongoDB to get the latest request number
        const response = await fetch('https://api.titandrillingzm.com:6009/request-maintenance');
        const result = await response.json();
        
        if (result.success && result.data && result.data.length > 0) {
            // Find the highest request number
            let maxNumber = 0;
            result.data.forEach(req => {
                const num = extractNumberFromRequestNumber(req.requestNumber);
                if (num > maxNumber) maxNumber = num;
            });
            const nextNumber = maxNumber + 1;
            const newRequestNumber = generateRequestNumber(country, project, nextNumber);
            requestNumber = newRequestNumber;
            document.getElementById('requestNumber').textContent = newRequestNumber;
        } else {
            // No existing requests, start from 1
            const newRequestNumber = generateRequestNumber(country, project, 1);
            requestNumber = newRequestNumber;
            document.getElementById('requestNumber').textContent = newRequestNumber;
        }
    } catch (error) {
        console.error('Error updating request number:', error);
        // Fallback to default
        const newRequestNumber = generateRequestNumber(country, project, 1);
        requestNumber = newRequestNumber;
        document.getElementById('requestNumber').textContent = newRequestNumber;
    }
}

// Priority selection
function selectPriority(priority) {
    selectedPriority = priority;
    console.log('Priority selected:', priority);
    
    // Remove selected class from all priority buttons
    document.querySelectorAll('.priority-button').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    // Add selected class to the clicked button
    document.querySelector(`[data-priority="${priority}"]`).classList.add('selected');
    logAllData(); // Log all data after priority selection
}

// Request type selection
function selectRequestType(type) {
    selectedRequestType = type;
    console.log('Request type selected:', type);
    document.querySelectorAll('[id^="requestType_"]').forEach(div => {
        div.style.display = 'none';
    });
    document.getElementById(`requestType_${type}`).style.display = 'block';
    closeRequestTypeModal();
    
    // Update the display text for request type
    const requestTypeSelectRow = document.querySelector('[onclick*="openRequestTypeModal"]');
    if (requestTypeSelectRow) {
        const selectText = requestTypeSelectRow.querySelector('.select-text');
        if (selectText) {
            selectText.textContent = type;
        }
    }
    
    logAllData(); // Log all data after request type selection
}

// Task type selection
function selectTaskType(type) {
    document.getElementById('taskType').value = type;
    document.querySelectorAll('[data-type]').forEach(btn => {
        btn.classList.remove('selected');
    });
    document.querySelector(`[data-type="${type}"]`).classList.add('selected');
}

// Modal functions - Updated to use custom-modal classes
function openEquipmentModal(event) {
    const modal = document.getElementById('equipmentModal');
    modal.style.display = 'block';
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    setTimeout(() => {
        const searchInput = document.getElementById('equipmentSearch');
        if (searchInput) searchInput.focus();
    }, 10);
    
    loadEquipmentFromMongoDB();
}

function closeEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    modal.style.display = 'none';
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function openAssignToModal(event) {
    const modal = document.getElementById('assignToModal');
    modal.style.display = 'block';
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    setTimeout(() => {
        const searchInput = document.getElementById('assignToSearch');
        if (searchInput) searchInput.focus();
    }, 10);
    
    loadUsersFromMongoDB();
}

function closeAssignToModal() {
    const modal = document.getElementById('assignToModal');
    modal.style.display = 'none';
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function closeTaskAssignToModal() {
    const modal = document.getElementById('taskAssignToModal');
    modal.style.display = 'none';
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function openRequestTypeModal(event) {
    const modal = document.getElementById('requestTypeModal');
    modal.style.display = 'block';
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    setTimeout(() => {
        const modalContent = modal.querySelector('.custom-modal-content');
        if (modalContent) modalContent.scrollTop = 0;
    }, 10);
}

function closeRequestTypeModal() {
    const modal = document.getElementById('requestTypeModal');
    modal.style.display = 'none';
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function openAddTaskModal(event) {
    const modal = document.getElementById('addTaskModal');
    modal.style.display = 'block';
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    setTimeout(() => {
        const titleInput = document.getElementById('taskTitle');
        if (titleInput) titleInput.focus();
    }, 10);
    
    // Reset task assignees and update display
    taskAssignees = [];
    updateTaskAssignToDisplay();
}

function closeAddTaskModal() {
    const modal = document.getElementById('addTaskModal');
    modal.style.display = 'none';
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function openTaskAssignToModal(event) {
    const modal = document.getElementById('taskAssignToModal');
    modal.style.display = 'block';
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    setTimeout(() => {
        const modalContent = modal.querySelector('.custom-modal-content');
        if (modalContent) modalContent.scrollTop = 0;
    }, 10);
    loadUsersFromMongoDB();
    // Auto-focus search input
    setTimeout(() => {
        const searchInput = document.getElementById('taskAssignToSearch');
        if (searchInput) searchInput.focus();
    }, 100);
}


// Load equipment from MongoDB - Using same endpoint as Create Workorder
async function loadEquipmentFromMongoDB() {
    try {
        console.log('🔄 Loading equipment from MongoDB...');
        
        // Show loading state
        const equipmentListEl = document.getElementById('equipmentList');
        if (equipmentListEl) {
            equipmentListEl.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;"><div class="loading-spinner" style="width: 30px; height: 30px; border: 3px solid #f3f3f3; border-top: 3px solid #0055a5; border-radius: 50%; animation: spin 1s linear infinite; margin: 0 auto 15px;"></div>Loading equipment...</div>';
        }
        
        // Fetch equipment data from MongoDB API - SAME ENDPOINT AS CREATE WORKORDER
        const response = await fetch('https://api.titandrillingzm.com:6001/EquipmentsDataJSON/GlobalEquipmentJsonData');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('✅ MongoDB equipment response:', result);
        console.log('Success:', result.success);
        console.log('Data count:', result.data?.length || 0);
        
        if (result.success && Array.isArray(result.data)) {
            console.log(`Found ${result.data.length} equipment items`);
            
            equipmentList = result.data.map((equipment, index) => ({
                id: equipment._id || equipment.id || index.toString(),
                name: equipment.vehicleNumber || equipment.equipmentName || equipment.name || equipment.model || 'Unknown Equipment',
                type: equipment.vehicleType || equipment.equipmentCategory || equipment.equipmentType || equipment.category || 'N/A',
                location: equipment.location || equipment.site || equipment.country || 'N/A',
                model: equipment.model || equipment.equipmentModel || 'N/A',
                meterReading: equipment.meterReading || equipment.odometerReading || '0',
                meterUnit: equipment.meterUnit || equipment.meterType || 'KM',
                status: equipment.status || 'Active',
                manufacturer: equipment.make || equipment.manufacturer || 'N/A',
                year: equipment.year || equipment.yearOfManufacture || 'N/A',
                registration: equipment.regNum || equipment.registrationNumber || 'N/A',
                serialNumber: equipment.vin || equipment.serialNumber || equipment.vinNumber || 'N/A',
                engineNumber: equipment.engineNumber || 'N/A',
                category: equipment.mainCategory || equipment.category || 'N/A',
                comment: equipment.comment || 'N/A',
                country: equipment.country || 'N/A',
                project: equipment.project || 'N/A',
                createdBy: equipment.createdBy || 'N/A',
                isActive: equipment.isActive !== undefined ? equipment.isActive : true,
                odometerReading: equipment.odometerReading || '0',
                timestamp: equipment.timestamp || 'N/A',
                updatedAt: equipment.updatedAt || equipment.createdAt || new Date().toISOString()
            }));
            
            console.log(`✅ Mapped ${equipmentList.length} equipment items`);
            console.log('Sample equipment:', equipmentList[0]);
            updateEquipmentList();
        } else {
            console.error('Failed to load equipment data from MongoDB:', result);
            equipmentList = [];
            updateEquipmentList();
        }
    } catch (error) {
        console.error('❌ Error loading equipment from MongoDB:', error);
        console.error('Error details:', error.message);
        equipmentList = [];
        updateEquipmentList();
    }
}

function updateEquipmentList() {
    const equipmentListEl = document.getElementById('equipmentList');
    if (!equipmentListEl) {
        console.error('Equipment list element not found');
        return;
    }
    
    if (!equipmentList || equipmentList.length === 0) {
        equipmentListEl.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;">No equipment found. Please check MongoDB connection.</div>';
        return;
    }
    
    console.log(`Rendering ${equipmentList.length} equipment items`);
    
    equipmentListEl.innerHTML = equipmentList.map((equipment, index) => {
        // Use equipment name or model as display name
        const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
            ? equipment.name 
            : equipment.model;
        
        return `
        <div class="equipment-item" onclick='selectEquipment(${JSON.stringify(equipment).replace(/'/g, "&apos;")})'>
            <div class="equipment-name">${displayName}</div>
            <div class="equipment-details">
                <div style="font-size: 12px; color: #666; margin-bottom: 5px;">
                    <strong>Category:</strong> ${equipment.category} | 
                    <strong>Model:</strong> ${equipment.model} | 
                    <strong>Status:</strong> ${equipment.status}
                </div>
                <div style="font-size: 12px; color: #666;">
                    <strong>Registration:</strong> ${equipment.registration} | 
                    <strong>Meter:</strong> ${equipment.meterReading} ${equipment.meterUnit}
                </div>
            </div>
        </div>
        `;
    }).join('');
    
    console.log('✅ Equipment list rendered successfully');
}

function selectEquipment(equipment) {
    selectedEquipment = equipment;
    
    console.log('Equipment selected:', selectedEquipment);
    
    // Use equipment name or model as display name
    const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
        ? equipment.name 
        : equipment.model;
    
    // Update the equipment select row display
    const equipmentSelectRow = document.querySelector('[onclick*="openEquipmentModal"]');
    if (equipmentSelectRow) {
        const selectText = equipmentSelectRow.querySelector('.select-text');
        if (selectText) {
            selectText.textContent = displayName;
        }
    }
    
    document.getElementById('equipmentDetails').innerHTML = `
        <div class="selected-equipment-card">
            <div class="selected-equipment-header">
                <div class="selected-equipment-name">${displayName}</div>
                <div class="selected-equipment-status ${equipment.status.toLowerCase().replace(/\s+/g, '-')}">${equipment.status}</div>
            </div>
            <div class="selected-equipment-info">
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Category:</div>
                    <div class="selected-equipment-value">${equipment.category}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Model:</div>
                    <div class="selected-equipment-value">${equipment.model}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Manufacturer:</div>
                    <div class="selected-equipment-value">${equipment.manufacturer}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Registration:</div>
                    <div class="selected-equipment-value">${equipment.registration}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Meter Reading:</div>
                    <div class="selected-equipment-value">${equipment.meterReading} ${equipment.meterUnit}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Country:</div>
                    <div class="selected-equipment-value">${equipment.country}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Engine Number:</div>
                    <div class="selected-equipment-value">${equipment.engineNumber}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">VIN Number:</div>
                    <div class="selected-equipment-value">${equipment.serialNumber}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Active:</div>
                    <div class="selected-equipment-value">${equipment.isActive ? 'Yes' : 'No'}</div>
                </div>
                <div class="selected-equipment-row">
                    <div class="selected-equipment-label">Last Updated:</div>
                    <div class="selected-equipment-value">${new Date(equipment.updatedAt).toLocaleDateString()}</div>
                </div>
            </div>
        </div>
    `;
    closeEquipmentModal();
    logAllData(); // Log all data after equipment selection
}

function filterEquipment() {
    const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
    
    if (!searchTerm.trim()) {
        updateEquipmentList();
        return;
    }
    
    const filteredEquipment = equipmentList.filter(equipment => {
        const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
            ? equipment.name 
            : equipment.model;
        
        return displayName.toLowerCase().includes(searchTerm) ||
               equipment.category.toLowerCase().includes(searchTerm) ||
               equipment.model.toLowerCase().includes(searchTerm) ||
               equipment.registration.toLowerCase().includes(searchTerm);
    });
    
    const equipmentListEl = document.getElementById('equipmentList');
    if (filteredEquipment.length === 0) {
        equipmentListEl.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;">No equipment matches your search.</div>';
        return;
    }
    
    equipmentListEl.innerHTML = filteredEquipment.map((equipment, index) => {
        const displayName = equipment.name !== 'Unknown Equipment' && equipment.name && equipment.name.trim() !== '' 
            ? equipment.name 
            : equipment.model;
        
        return `
        <div class="equipment-item" onclick='selectEquipment(${JSON.stringify(equipment).replace(/'/g, "&apos;")})'>
            <div class="equipment-name">${displayName}</div>
            <div class="equipment-details">
                <div style="font-size: 12px; color: #666; margin-bottom: 5px;">
                    <strong>Category:</strong> ${equipment.category} | 
                    <strong>Model:</strong> ${equipment.model} | 
                    <strong>Status:</strong> ${equipment.status}
                </div>
                <div style="font-size: 12px; color: #666;">
                    <strong>Registration:</strong> ${equipment.registration} | 
                    <strong>Meter:</strong> ${equipment.meterReading} ${equipment.meterUnit}
                </div>
            </div>
        </div>
        `;
    }).join('');
}

// Load users from MongoDB - Using same endpoint as Create Workorder
async function loadUsersFromMongoDB() {
    try {
        console.log('🔄 Loading users from MongoDB...');
        
        const response = await fetch('https://api.titandrillingzm.com:6003/users/all');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('✅ MongoDB users response:', result);
        console.log('Success:', result.success);
        console.log('Data count:', result.data?.length || 0);
        
        if (result.success && Array.isArray(result.data)) {
            teamMembers = result.data.map(user => ({
                id: user._id || user.id || user.uid,
                fullName: user.name || user.displayName || user.fullName || user.username || 'Unknown User',
                name: user.name || user.displayName || user.fullName || user.username || 'Unknown User',
                employeeNumber: user.employeeNumber || user.empNumber || 'N/A',
                role: user.role || user.userRole || user.position || 'User',
                profilePic: user.avatar || user.profilePicture || user.photoURL || null,
                email: user.email || user.emailAddress || '',
                mobile: user.phone || user.phoneNumber || user.mobile || '',
                department: user.department || user.dept || '',
                countries: user.countries || [],
                projects: user.projects || [],
                country: user.country || '',
                project: user.project || '',
                status: user.status || user.active || 'Active',
                isBlocked: user.isBlocked || false
            }));
            
            console.log(`✅ Loaded ${teamMembers.length} users from MongoDB`);
            console.log('Sample user:', teamMembers[0]);
            updateAssignToList();
            updateTaskAssignToList();
        } else {
            console.warn('No user data found or invalid format from MongoDB');
            teamMembers = [];
            updateAssignToList();
            updateTaskAssignToList();
        }
    } catch (error) {
        console.error('❌ Error loading users from MongoDB:', error);
        console.error('Error details:', error.message);
        teamMembers = [];
        updateAssignToList();
        updateTaskAssignToList();
    }
}

// Assign to functions
function updateAssignToList() {
    const assignToList = document.getElementById('assignToListModal');
    if (!assignToList) {
        console.error('Assign to list element not found');
        return;
    }
    
    if (!teamMembers || teamMembers.length === 0) {
        assignToList.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;"><i class="fas fa-users" style="font-size: 48px; color: #ddd; margin-bottom: 15px; display: block;"></i><p>No team members found. Check MongoDB connection.</p></div>';
        return;
    }
    
    console.log(`Displaying ${teamMembers.length} team members`);
    
    assignToList.innerHTML = teamMembers.map((member, index) => `
        <div class="user-item" onclick="selectTeamMember('${member.id}', '${member.fullName.replace(/'/g, "&apos;")}')">
            <div class="user-name">${member.fullName}</div>
            <div class="user-details">Emp#: ${member.employeeNumber} | Role: ${member.role} | Dept: ${member.department || 'N/A'}</div>
        </div>
    `).join('');
    
    console.log('✅ Team members list displayed');
}

function selectTeamMember(id, name) {
    console.log('Team member selected:', { id, name });
    
    // Check which modal is open by checking visibility
    const mainModal = document.getElementById('assignToModal');
    const taskModal = document.getElementById('taskAssignToModal');
    
    if (mainModal.style.display === 'block') {
        // Handle main form assignment
        if (!selectedAssignees.some(a => a.id === id)) {
            selectedAssignees.push({ id, fullName: name });
            console.log('Added to main assignees:', selectedAssignees);
            updateAssignToDisplay();
        }
        closeAssignToModal();
    } else if (taskModal.style.display === 'block') {
        // Handle task assignment
        if (!taskAssignees.some(a => a.id === id)) {
            taskAssignees.push({ id, fullName: name });
            console.log('Added to task assignees:', taskAssignees);
            updateTaskAssignToDisplay();
        }
        closeTaskAssignToModal();
    }
    
    logAllData(); // Log all data after team member selection
}

function updateAssignToDisplay() {
    const assignToList = document.getElementById('assignToList');
    if (selectedAssignees.length > 0) {
        assignToList.innerHTML = `
            <div class="selected-items-row">
                ${selectedAssignees.map(item => `
                    <div class="selected-item">
                        <span class="selected-item-text">${item.fullName}</span>
                        <div class="close-button" onclick="removeAssignee('${item.id}')">
                            <div class="close-circle">×</div>
                        </div>
                    </div>
                `).join('')}
            </div>
        `;
        
        // Update the select row display text
        const assignToSelectRow = document.querySelector('[onclick*="openAssignToModal"]');
        if (assignToSelectRow) {
            const selectText = assignToSelectRow.querySelector('.select-text');
            if (selectText) {
                selectText.textContent = `${selectedAssignees.length} selected`;
            }
        }
    } else {
        assignToList.innerHTML = '<div style="color: #666; font-style: italic;">No assignees selected</div>';
        
        // Reset the select row display text
        const assignToSelectRow = document.querySelector('[onclick*="openAssignToModal"]');
        if (assignToSelectRow) {
            const selectText = assignToSelectRow.querySelector('.select-text');
            if (selectText) {
                selectText.textContent = 'Select';
            }
        }
    }
}

function updateTaskAssignToDisplay() {
    const taskAssignToList = document.getElementById('taskAssignToList');
    if (taskAssignees.length > 0) {
        taskAssignToList.innerHTML = `
            <div class="react-selected-items-row">
                ${taskAssignees.map(item => `
                    <div class="react-selected-item">
                        <span class="react-selected-item-text">${item.fullName}</span>
                        <div class="react-close-button" onclick="removeTaskAssignee('${item.id}')">
                            <div class="react-close-circle">×</div>
                        </div>
                    </div>
                `).join('')}
            </div>
        `;
    } else {
        taskAssignToList.innerHTML = '<div class="react-selected-text">No Assignees Selected</div>';
    }
}

function removeAssignee(id) {
    selectedAssignees = selectedAssignees.filter(a => a.id !== id);
    updateAssignToDisplay();
}

function removeTaskAssignee(id) {
    taskAssignees = taskAssignees.filter(a => a.id !== id);
    updateTaskAssignToDisplay();
}

function saveAssignTo() {
    closeAssignToModal();
}

function updateTaskAssignToList() {
    const taskAssignToList = document.getElementById('taskAssignToListModal');
    if (!taskAssignToList) {
        console.error('Task assign to list element not found');
        return;
    }
    
    if (!teamMembers || teamMembers.length === 0) {
        taskAssignToList.innerHTML = '<div style="text-align: center; padding: 40px; color: #888;"><i class="fas fa-users" style="font-size: 48px; color: #ddd; margin-bottom: 15px; display: block;"></i><p>No team members found. Check MongoDB connection.</p></div>';
        return;
    }
    
    console.log(`Displaying ${teamMembers.length} task team members`);
    
    taskAssignToList.innerHTML = teamMembers.map((member, index) => `
        <div class="user-item" onclick="selectTeamMember('${member.id}', '${member.fullName.replace(/'/g, "&apos;")}')">
            <div class="user-name">${member.fullName}</div>
            <div class="user-details">Emp#: ${member.employeeNumber} | Role: ${member.role} | Dept: ${member.department || 'N/A'}</div>
        </div>
    `).join('');
    
    console.log('✅ Task team members list displayed');
}

function saveTaskAssignTo() {
    closeTaskAssignToModal();
    // Return to task modal
    document.getElementById('addTaskModal').style.display = 'block';
}

function filterTeamMembers() {
    const searchTerm = document.getElementById('assignToSearch').value.toLowerCase();
    // Implement filtering logic
}

function filterTaskTeamMembers() {
    const searchTerm = document.getElementById('taskAssignToSearch').value.toLowerCase();
    // Implement filtering logic
}

// Task functions
function saveTask() {
    const taskTitle = document.getElementById('taskTitle').value;
    const taskType = document.getElementById('taskType').value;
    const taskNote = document.getElementById('taskNote').value;
    
    if (!taskTitle.trim()) {
        alert('Task title is required');
        return;
    }
    
    const task = {
        id: Date.now().toString(),
        title: taskTitle,
        type: taskType,
        note: taskNote,
        assignees: [...taskAssignees], // Use task-specific assignees
        status: 'Initiated' // Set initial status
    };
    
    tasks.push(task);
    console.log('Task added to tasks array:', tasks);
    updateTaskList();
    closeAddTaskModal();
    
    // Reset form
    document.getElementById('taskTitle').value = '';
    document.getElementById('taskNote').value = '';
    taskAssignees = []; // Reset task assignees
    updateTaskAssignToDisplay(); // Update display
    
    logAllData(); // Log all data after task save
}

// Update task status
function updateTaskStatus(taskIndex, newStatus) {
    if (tasks[taskIndex]) {
        tasks[taskIndex].status = newStatus;
        console.log(`Task ${taskIndex} status updated to:`, newStatus);
        updateTaskList();
        logAllData(); // Log all data after status update
    }
}

// Delete task
function deleteTask(taskIndex) {
    if (confirm('Are you sure you want to delete this task?')) {
        tasks.splice(taskIndex, 1);
        console.log(`Task ${taskIndex} deleted. Remaining tasks:`, tasks);
        updateTaskList();
        logAllData(); // Log all data after task deletion
    }
}

function updateTaskList() {
    const taskList = document.getElementById('taskList');
    if (tasks.length > 0) {
        taskList.innerHTML = tasks.map((task, index) => `
            <div class="task-card">
                <div class="task-card-header">
                    <div class="task-card-title">${task.title}</div>
                    <div class="task-card-actions">
                        <select class="task-status-dropdown" onchange="updateTaskStatus(${index}, this.value)">
                            <option value="Initiated" ${task.status === 'Initiated' ? 'selected' : ''}>Initiated</option>
                            <option value="On Hold" ${task.status === 'On Hold' ? 'selected' : ''}>On Hold</option>
                            <option value="Completed" ${task.status === 'Completed' ? 'selected' : ''}>Completed</option>
                            <option value="Cancelled" ${task.status === 'Cancelled' ? 'selected' : ''}>Cancelled</option>
                        </select>
                        <button class="task-delete-btn" onclick="deleteTask(${index})" title="Delete Task">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
                <div class="task-card-type">${task.type || 'Repair'}</div>
                ${task.assignees && task.assignees.length > 0 ? 
                    `<div style="font-size: 13px; color: #666; margin-top: 5px;">Assigned to: ${task.assignees.map(u => u.fullName).join(', ')}</div>` : 
                    ''
                }
                ${task.note ? `<div style="font-size: 12px; color: #888; margin-top: 5px; font-style: italic;">${task.note}</div>` : ''}
            </div>
        `).join('');
    } else {
        taskList.innerHTML = '';
    }
}

// Form submission
async function saveMaintenanceRequest() {
    console.log('=== FORM SUBMISSION STARTED ===');
    logAllData(); // Log all data before validation
    
    if (!document.getElementById('workOrderTitle').value.trim()) {
        alert('Work Order Title is required');
        return;
    }
    
    if (!document.getElementById('country').value) {
        alert('Country is required');
        return;
    }
    
    if (!document.getElementById('project').value) {
        alert('Project is required');
        return;
    }
    
    // Update country and project variables
    country = document.getElementById('country').value;
    project = document.getElementById('project').value;
    
    console.log('Form validation passed. Updated country:', country, 'project:', project);
    
    // Update request number
    await updateRequestNumber();
    
    try {
        // Validate required fields
        if (!selectedEquipment) {
            alert('Please select equipment');
            return;
        }
        
        if (selectedAssignees.length === 0) {
            alert('Please assign to at least one person');
            return;
        }
        
        if (!selectedRequestType) {
            alert('Please select a request type');
            return;
        }
        
        console.log('=== VALIDATION PASSED ===');
        logAllData(); // Log all data before saving
        
        // Show sync modal
        showSyncModal();
        
        // Safely get form values with null checks
        const getFormValue = (elementId) => {
            const element = document.getElementById(elementId);
            return element ? element.value : "";
        };

        // Prepare the maintenance request data for MongoDB
        const maintenanceRequestData = {
            _id: Date.now().toString(),
            id: Date.now().toString(),
            adminUid: "cahj3jJqe5Z16pQdLGSWrNgm84o1", // Should be from logged-in user
            requestNumber: requestNumber || "",
            workOrderTitle: getFormValue('workOrderTitle'),
            requestType: selectedRequestType || "",
            priority: selectedPriority || "Medium",
            description: getFormValue('memo'),
            equipment: {
                _id: selectedEquipment ? (selectedEquipment.id || "") : "",
                id: selectedEquipment ? (selectedEquipment.id || "") : "",
                equipmentName: selectedEquipment ? (selectedEquipment.name || "") : "",
                equipmentNumber: selectedEquipment ? (selectedEquipment.registration || "") : "",
                model: selectedEquipment ? (selectedEquipment.model || "") : "",
                serialNumber: selectedEquipment ? (selectedEquipment.serialNumber || "") : "",
                mainCategory: selectedEquipment ? (selectedEquipment.category || "") : "",
                status: selectedEquipment ? (selectedEquipment.status || "") : "",
                country: selectedEquipment ? (selectedEquipment.country || "") : "",
                project: project || ""
            },
            assignTo: selectedAssignees.map(assignee => ({
                _id: assignee.id || "",
                fullName: assignee.fullName || assignee.name || "",
                name: assignee.fullName || assignee.name || "",
                email: assignee.email || "",
                phoneNumber: assignee.mobile || "",
                role: assignee.role || ""
            })),
            country: country || "",
            project: project || "",
            tasks: tasks.map(task => ({
                id: task.id || "",
                title: task.title || "",
                description: task.type || "",
                note: task.note || "",
                status: task.status || "Initiated",
                createdAt: new Date().toISOString(),
                updatedAt: new Date().toISOString(),
                assignees: task.assignees.map(assignee => ({
                    id: assignee.id || "",
                    fullName: assignee.fullName || "",
                    employeeNumber: assignee.employeeNumber || "",
                    role: assignee.role || "",
                    profilePic: {
                        url: assignee.profilePic || "",
                        path: ""
                    }
                }))
            })),
            status: "Requested",
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
            userId: "cahj3jJqe5Z16pQdLGSWrNgm84o1", // Should be from logged-in user
            isArchived: false
        };
        
        console.log('=== SAVING TO MONGODB ===');
        console.log('Maintenance request data:', maintenanceRequestData);
        
        // Save to MongoDB
        const response = await fetch('https://api.titandrillingzm.com:6009/request-maintenance', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(maintenanceRequestData)
        });
        
        const result = await response.json();
        
        if (result.success) {
            console.log('✅ Maintenance request saved successfully to MongoDB');
            console.log('MongoDB response:', result);
            
            // Send email notifications
            console.log('=== SENDING EMAIL NOTIFICATIONS ===');
            try {
                const notificationData = {
                    maintenanceData: {
                        _id: result.data._id,
                        id: result.data.id,
                        requestNumber: result.data.requestNumber,
                        workOrderTitle: result.data.workOrderTitle,
                        equipmentName: selectedEquipment ? selectedEquipment.name : 'N/A',
                        priority: result.data.priority,
                        country: result.data.country,
                        project: result.data.project,
                        tasksCount: tasks.length,
                        description: result.data.description
                    },
                    createdByName: "Admin User", // Should be from logged-in user
                    requesterEmail: "admin@titandrillingzm.com" // Should be from logged-in user
                };
                
                const notifResponse = await fetch('https://api.titandrillingzm.com:6022/api/maintenance-notifications/new-request', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(notificationData)
                });
                
                const notifResult = await notifResponse.json();
                console.log('✅ Notifications sent:', notifResult);
            } catch (notifError) {
                console.error('⚠️ Failed to send notifications (non-critical):', notifError);
            }
            
            // Hide sync modal
            hideSyncModal();
            
            // Show success modal
            showSuccessModal();
            
            // Reset form
            resetForm();
        } else {
            hideSyncModal();
            console.error('❌ MongoDB save failed:', result.error);
            throw new Error(`Failed to save maintenance request: ${result.error}`);
        }
        
    } catch (error) {
        hideSyncModal();
        console.error('❌ Error saving maintenance request:', error);
        alert(`Failed to save maintenance request: ${error.message}`);
    }
}

function closeSuccessModal() {
    document.getElementById('successModal').style.display = 'none';
    // Redirect or reset form
    window.location.href = '{{ route("maintenance.workorders") }}';
}

// Reset form function
function resetForm() {
    console.log('=== RESETTING FORM ===');
    
    // Clear form fields with null checks
    const clearField = (elementId) => {
        const element = document.getElementById(elementId);
        if (element) element.value = '';
    };
    
    clearField('workOrderTitle');
    clearField('country');
    clearField('project');
    clearField('memo');
    clearField('note');
    
    // Reset variables
    selectedPriority = null;
    selectedEquipment = null;
    selectedAssignees = [];
    taskAssignees = [];
    selectedRequestType = null;
    tasks = [];
    country = null;
    project = null;
    requestNumber = 'WO1001';
    
    // Reset UI with null checks
    const clearElement = (elementId) => {
        const element = document.getElementById(elementId);
        if (element) element.innerHTML = '';
    };
    
    clearElement('equipmentDetails');
    clearElement('assignToList');
    clearElement('taskList');
    
    // Reset priority buttons
    document.querySelectorAll('.priority-button').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    // Reset request type buttons
    document.querySelectorAll('.react-request-type-button').forEach(btn => {
        btn.classList.remove('selected');
    });
    
    // Reset project dropdown
    const projectSelect = document.getElementById('project');
    if (projectSelect) {
        projectSelect.innerHTML = '<option value="">Select Project</option>';
        projectSelect.disabled = true;
    }
    
    console.log('Form reset completed');
}

// Country change handler
document.getElementById('country').addEventListener('change', function() {
    const countryValue = this.value;
    const projectSelect = document.getElementById('project');
    
    console.log('Country changed to:', countryValue);
    
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    projectSelect.disabled = true;
    
    // Update country variable
    country = countryValue;
    
    if (countryValue === 'Zambia') {
        const options = ['Kansanshi', 'Kalumbila', 'Kimteto', 'Kobold', 'FQM Exploration', 'Mimosa'];
        options.forEach(opt => {
            projectSelect.innerHTML += `<option value="${opt}">${opt}</option>`;
        });
        projectSelect.disabled = false;
    } else if (countryValue === 'DRC') {
        const options = ['Lubumbashi', 'Musompo', 'IME', 'Kamoa'];
        options.forEach(opt => {
            projectSelect.innerHTML += `<option value="${opt}">${opt}</option>`;
        });
        projectSelect.disabled = false;
    }
    
    // Update request number when country changes
    updateRequestNumber();
    logAllData(); // Log all data after country change
});

// Project change handler
document.getElementById('project').addEventListener('change', function() {
    const projectValue = this.value;
    project = projectValue;
    
    console.log('Project changed to:', projectValue);
    
    // Update request number when project changes
    updateRequestNumber();
    logAllData(); // Log all data after project change
});

// Initialize the form
document.addEventListener('DOMContentLoaded', function() {
    // Set initial request number
    updateRequestNumber();
});

// Close modals when clicking outside
window.onclick = function(event) {
    const modals = ['equipmentModal', 'assignToModal', 'taskAssignToModal', 'requestTypeModal', 'addTaskModal', 'syncModal', 'successModal'];
    modals.forEach(modalId => {
        const modal = document.getElementById(modalId);
        if (event.target === modal) {
            modal.style.display = 'none';
            modal.classList.remove('show');
        }
    });
}

// Sync Modal Functions
function showSyncModal() {
    const modal = document.getElementById('syncModal');
    if (modal) {
        modal.style.display = 'block';
        modal.classList.add('show');
    }
}

function hideSyncModal() {
    const modal = document.getElementById('syncModal');
    if (modal) {
        modal.style.display = 'none';
        modal.classList.remove('show');
    }
}

// Success Modal Functions
function showSuccessModal() {
    const modal = document.getElementById('successModal');
    if (modal) {
        modal.style.display = 'block';
        modal.classList.add('show');
    }
}

function hideSuccessModal() {
    const modal = document.getElementById('successModal');
    if (modal) {
        modal.style.display = 'none';
        modal.classList.remove('show');
    }
}

function closeSuccessModal() {
    hideSuccessModal();
    // Redirect to work orders list
    window.location.href = "{{ route('maintenance.workorders') }}";
}

function goBackToRequestMaintenance() {
    hideSuccessModal();
    // Redirect to Request Maintenance page
    window.location.href = "{{ route('maintenance.request') }}";
}
</script>
@endsection

