@extends('welcome')

@section('title', 'Create New Work Order')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.workorder-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Form Groups */
.form-group {
    margin-bottom: 25px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.required {
    color: var(--danger-color);
    font-weight: bold;
}

.form-input {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-input.error {
    border-color: var(--danger-color);
}

.form-textarea {
    min-height: 100px;
    resize: vertical;
}

.error-message {
    color: var(--danger-color);
    font-size: 12px;
    margin-top: 5px;
    display: none;
}

.error-message.show {
    display: block;
}

/* Request Number Display */
.request-number-display {
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 15px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
}

.request-number-label {
    font-weight: 500;
    color: var(--text-primary);
}

.request-number-value {
    font-weight: 700;
    color: var(--primary-color);
    font-size: 16px;
}

/* Priority Selection */
.priority-container {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 15px;
    margin-top: 10px;
    max-width: 52%;
}

.priority-option {
    display: flex;
    flex-direction: column;
    align-items: center;
    padding: 15px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-base);
    background: white;
}

.priority-option:hover {
    border-color: var(--primary-color);
    transform: translateY(-2px);
    box-shadow: var(--shadow-sm);
}

.priority-option.selected {
    border-color: var(--primary-color);
    background: var(--primary-color);
    color: white;
}

.priority-image {
    width: 36px;
    height: 36px;
    margin-bottom: 8px;
    object-fit: contain;
    display: block;
}

.priority-label {
    font-size: 12px;
    font-weight: 500;
    text-align: center;
}

/* Dropdown Styles */
.dropdown-container {
    position: relative;
}

.dropdown-select {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    background: white;
    cursor: pointer;
    transition: var(--transition-base);
}

.dropdown-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.dropdown-select:disabled {
    background: var(--bg-secondary);
    color: var(--text-secondary);
    cursor: not-allowed;
}

/* Select Row Styles */
.select-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 15px 20px;
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-base);
    margin-bottom: 10px;
}

.select-row:hover {
    background: var(--bg-hover);
    border-color: var(--primary-color);
}

.select-label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
}

.select-right {
    display: flex;
    align-items: center;
}

.select-text {
    color: var(--primary-color);
    font-size: 14px;
    font-weight: 500;
    margin-right: 8px;
}

.select-arrow {
    color: var(--primary-color);
    font-size: 16px;
}

/* Selected Items Display */
.selected-items-container {
    margin-top: 10px;
}

.selected-items-row {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.selected-item {
    display: flex;
    align-items: center;
    background: #e6f0ff;
    border: 1px solid var(--primary-color);
    border-radius: 20px;
    padding: 6px 12px;
    gap: 8px;
}

.selected-item-text {
    font-size: 12px;
    color: var(--primary-color);
    font-weight: 500;
}

.remove-btn {
    background: var(--danger-color);
    color: white;
    border: none;
    border-radius: 50%;
    width: 18px;
    height: 18px;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    font-size: 12px;
    transition: var(--transition-base);
}

.remove-btn:hover {
    background: #c82333;
}

/* Task Management */
.task-list-container {
    margin-top: 20px;
}

.task-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 20px;
    margin-bottom: 15px;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.task-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
}

.task-card-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
}

.task-card-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    flex: 1;
    margin-right: 10px;
}

.task-card-status {
    background: var(--info-color);
    color: white;
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 500;
    margin-right: 10px;
}

.task-card-menu {
    position: relative;
}

.task-menu-btn {
    background: none;
    border: none;
    cursor: pointer;
    padding: 8px;
    border-radius: 4px;
    transition: var(--transition-base);
    color: var(--danger-color);
}

.task-menu-btn:hover {
    background: #ffebee;
    color: #c62828;
}

/* Item form styling */
.item-form {
    animation: fadeIn 0.3s ease-in-out;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

/* Category-specific styling */
.task-card[data-category="Task"] {
    border-left: 4px solid var(--primary-color);
}

.task-card[data-category="Parts"] {
    border-left: 4px solid var(--warning-color);
}

.task-card[data-category="Labor"] {
    border-left: 4px solid var(--info-color);
}

.task-card[data-category="Additional Cost"] {
    border-left: 4px solid var(--success-color);
}

.task-card[data-category="Remarks"] {
    border-left: 4px solid var(--secondary-color);
}

/* Category Tabs */
.category-tabs {
    display: flex;
    gap: 5px;
    margin-bottom: 20px;
    border-bottom: 1px solid var(--border-light);
    overflow-x: auto;
}

.tab-button {
    background: transparent;
    border: none;
    padding: 12px 16px;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    cursor: pointer;
    transition: var(--transition-base);
    border-bottom: 2px solid transparent;
    white-space: nowrap;
    display: flex;
    align-items: center;
    gap: 8px;
}

.tab-button:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-button.active {
    color: var(--primary-color);
    border-bottom-color: var(--primary-color);
    background: var(--bg-light);
}

/* Add Task Button */
.add-task-btn {
    background: var(--primary-color);
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 8px;
}

.add-task-btn:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
}

/* Body scroll lock when modal is open */
body.modal-open {
    overflow: hidden;
}

/* Modal Styles - Matching Inspection2 */
.custom-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    overflow-y: auto;
    align-items: center;
    justify-content: center;
}

.custom-modal.show {
    display: flex;
}

.custom-modal-content {
    background-color: #fefefe;
    margin: auto;
    padding: 0;
    border: 1px solid #888;
    width: 80%;
    max-width: 600px;
    border-radius: 8px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    position: relative;
    max-height: 90vh;
    overflow-y: auto;
    animation: modalSlideIn 0.3s ease;
    align-self: center;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: scale(0.9);
    }
    to {
        opacity: 1;
        transform: scale(1);
    }
}

.custom-modal-header {
    background-color: #0055A5;
    color: white;
    padding: 15px 20px;
    border-radius: 8px 8px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 500;
}

.close-btn {
    color: white;
    font-size: 28px;
    font-weight: bold;
    background: none;
    border: none;
    cursor: pointer;
    padding: 0;
    line-height: 1;
}

.close-btn:hover {
    opacity: 0.7;
}

.custom-modal-body {
    padding: 20px;
}

.search-container {
    display: flex;
    gap: 10px;
    margin-bottom: 20px;
}

.search-input {
    flex: 1;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.search-btn {
    padding: 10px 15px;
    background-color: #0055A5;
    color: white;
    border: none;
    border-radius: 4px;
    cursor: pointer;
}

.search-btn:hover {
    background-color: #003f7d;
}

.equipment-list, .checklist-list, .user-list, .inspection-list, .jobcard-list {
    max-height: 400px;
    overflow-y: auto;
    border: 1px solid #eee;
    border-radius: 4px;
}

.equipment-item, .checklist-item, .user-item, .inspection-item, .jobcard-item {
    padding: 15px;
    border-bottom: 1px solid #eee;
    cursor: pointer;
    transition: background-color 0.2s;
}

.equipment-item:hover, .checklist-item:hover, .user-item:hover, .inspection-item:hover, .jobcard-item:hover {
    background-color: #f8f9fa;
}

.equipment-item:last-child, .checklist-item:last-child, .user-item:last-child, .inspection-item:last-child, .jobcard-item:last-child {
    border-bottom: none;
}

.equipment-name, .user-name, .inspection-name, .jobcard-name {
    font-weight: 600;
    color: #0055A5;
    margin-bottom: 5px;
}

.equipment-details, .checklist-details, .user-details, .inspection-details, .jobcard-details {
    font-size: 12px;
    color: #666;
    line-height: 1.4;
}

/* Modal Footer - Centered Buttons */
.modal-footer {
    margin-top: 20px;
    text-align: center;
    padding: 15px 20px;
    border-top: 1px solid #eee;
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 15px;
}

.modal-footer .btn {
    padding: 12px 24px;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    min-width: 120px;
    border: none;
    color: white;
    background-color: #0055A5;
    text-align: center;
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-footer .btn:hover {
    background-color: #003f7d;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0, 85, 165, 0.3);
}

.modal-footer .btn-primary {
    background-color: #0055A5;
    color: white;
    border: none;
}

.modal-footer .btn-primary:hover {
    background-color: #003f7d;
    transform: translateY(-1px);
}

.modal-footer .btn-secondary {
    background-color: #0055A5;
    color: white;
    border: none;
}

.modal-footer .btn-secondary:hover {
    background-color: #003f7d;
    transform: translateY(-1px);
}

/* Responsive improvements for modals */
@media (max-width: 768px) {
    .custom-modal {
        padding: 20px 10px;
    }
    
    .custom-modal-content {
        width: 95%;
        max-width: 95%;
        margin: auto;
        max-height: 90vh;
    }
    
    .success-modal-container {
        width: 95%;
        max-width: 95%;
        margin: auto;
    }
    
    .search-container {
        flex-direction: column;
    }
    
    .equipment-list, .checklist-list, .user-list, .inspection-list, .jobcard-list {
        max-height: 300px;
    }
    
    .modal-footer {
        flex-direction: column;
        gap: 10px;
    }
    
    .modal-footer .btn {
        margin: 5px;
        width: 100%;
        max-width: 200px;
    }
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 8px;
}

.btn-primary {
    background: var(--primary-color);
    color: white;
}

.btn-primary:hover {
    background: var(--primary-hover);
}

.btn-secondary {
    background: var(--secondary-color);
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
}

.btn-outline {
    background: transparent;
    color: var(--primary-color);
    border: 1px solid var(--primary-color);
}

.btn-outline:hover {
    background: var(--primary-color);
    color: white;
}

/* Loading and Sync States */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-modal:not([style*="display: none"]) {
    display: flex !important;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    from { transform: rotate(0deg); }
    to { transform: rotate(360deg); }
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* Success Modal Styles - Matching Create Incident File */
.success-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    overflow-y: auto;
    z-index: 10000;
}

.success-modal-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
    max-width: 500px;
    width: 90%;
    max-height: 80vh;
    overflow: hidden;
    position: relative;
    margin: auto;
    animation: modalSlideIn 0.3s ease;
}

.success-modal-header {
    background: #0055A5;
    padding: 20px 30px;
    text-align: center;
    border-radius: 12px 12px 0 0;
}

.success-title {
    color: white;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
    font-family: var(--font-family);
}

.success-modal-body {
    padding: 40px 30px;
    text-align: center;
}

.success-message {
    color: #333;
    font-size: 18px;
    font-weight: 500;
    margin: 0 0 30px 0;
    font-family: var(--font-family);
}

.success-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
    margin-top: 20px;
}

.success-btn {
    background: #0055A5;
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 8px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    font-family: var(--font-family);
    min-width: 120px;
}

.success-btn:hover {
    background: #003d7a;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.success-btn:active {
    transform: translateY(0);
}

.back-btn {
    background: #0055A5;
}

.view-pdf-btn {
    background: #0055A5;
}

/* Modal Responsive Styles */
@media (max-width: 768px) {
    .success-modal-container {
        width: 95%;
        margin: 10px;
    }
    
    .success-modal-header {
        padding: 15px 20px;
    }
    
    .success-title {
        font-size: 20px;
    }
    
    .success-modal-body {
        padding: 30px 20px;
    }
    
    .success-message {
        font-size: 16px;
    }
    
    .success-buttons {
        flex-direction: column;
    }
    
    .success-btn {
        width: 100%;
        margin: 5px 0;
    }
}


/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .priority-container {
        grid-template-columns: repeat(2, 1fr);
    }
    
    .modal-content {
        width: 95%;
        padding: 20px;
    }
}

@media (max-width: 480px) {
    .priority-container {
        grid-template-columns: 1fr;
    }
    
    .form-group {
        margin-bottom: 20px;
    }
    
    .modal-content {
        padding: 15px;
    }
}

/* Utility Classes */
.hidden {
    display: none !important;
}

.text-center {
    text-align: center;
}

.text-success {
    color: var(--success-color);
}

.text-danger {
    color: var(--danger-color);
}

.text-warning {
    color: var(--warning-color);
}

.text-info {
    color: var(--info-color);
}

.mb-0 { margin-bottom: 0; }
.mb-1 { margin-bottom: 0.25rem; }
.mb-2 { margin-bottom: 0.5rem; }
.mb-3 { margin-bottom: 1rem; }
.mb-4 { margin-bottom: 1.5rem; }
.mb-5 { margin-bottom: 3rem; }

.mt-0 { margin-top: 0; }
.mt-1 { margin-top: 0.25rem; }
.mt-2 { margin-top: 0.5rem; }
.mt-3 { margin-top: 1rem; }
.mt-4 { margin-top: 1.5rem; }
.mt-5 { margin-top: 3rem; }

/* Preview Styles */
.equipment-preview, .users-preview {
    margin-top: 15px;
    padding: 15px;
    background: var(--bg-light);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
}

.preview-card {
    background: white;
    border-radius: var(--radius-md);
    padding: 15px;
    box-shadow: var(--shadow-sm);
}

.preview-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 1px solid var(--border-light);
}

.preview-header h4 {
    margin: 0;
    color: var(--text-primary);
    font-size: 16px;
}

.preview-status {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 500;
    text-transform: uppercase;
    background: var(--success-color);
    color: white;
}

.preview-details {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 10px;
}

.preview-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 5px 0;
}

.preview-label {
    font-weight: 500;
    color: var(--text-secondary);
    font-size: 12px;
}

.preview-value {
    color: var(--text-primary);
    font-size: 12px;
    text-align: right;
    word-break: break-all;
    overflow-wrap: break-word;
    max-width: 120px;
    font-weight: 500;
}

/* Specific styling for meter reading to keep it compact */
#equipmentPreviewMeter {
    font-size: 11px;
    max-width: 100px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.users-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
    gap: 10px;
    margin-top: 10px;
}

.user-preview-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 12px;
    display: flex;
    align-items: center;
    gap: 10px;
    transition: var(--transition-base);
}

.user-preview-card:hover {
    box-shadow: var(--shadow-sm);
}

.user-preview-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: var(--bg-secondary);
    display: flex;
    align-items: center;
    justify-content: center;
    overflow: hidden;
    flex-shrink: 0;
}

.user-preview-avatar img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.user-preview-info {
    flex: 1;
    min-width: 0;
}

.user-preview-name {
    font-weight: 600;
    color: var(--text-primary);
    font-size: 13px;
    margin-bottom: 2px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.user-preview-role {
    font-size: 11px;
    color: var(--text-secondary);
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Form Row Layout */
.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

@media (max-width: 768px) {
    .form-row {
        grid-template-columns: 1fr;
        gap: 15px;
    }
}

/* Date Input Styles */
.date-input-container {
    position: relative;
    display: flex;
    align-items: center;
}

.date-input {
    padding-right: 40px;
    cursor: pointer;
}

.date-icon {
    position: absolute;
    right: 12px;
    color: var(--primary-color);
    font-size: 16px;
    pointer-events: none;
}

/* Collapsible Sections */
.collapsible-section {
    margin-top: 20px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    overflow: hidden;
}

.collapsible-header {
    background: var(--bg-light);
    padding: 15px 20px;
    cursor: pointer;
    display: flex;
    justify-content: space-between;
    align-items: center;
    transition: var(--transition-base);
    border-bottom: 1px solid var(--border-light);
}

.collapsible-header:hover {
    background: var(--bg-secondary);
}

.collapsible-title {
    font-weight: 600;
    color: var(--primary-color);
    font-size: 16px;
}

.collapsible-arrow {
    color: var(--primary-color);
    font-size: 14px;
    transition: transform 0.3s ease;
}

.collapsible-arrow.rotated {
    transform: rotate(180deg);
}

.collapsible-content {
    padding: 20px;
    background: white;
    display: none;
    animation: slideDown 0.3s ease;
}

.collapsible-content.expanded {
    display: block;
}

@keyframes slideDown {
    from {
        opacity: 0;
        max-height: 0;
    }
    to {
        opacity: 1;
        max-height: 1000px;
    }
}

/* Currency Input */
.currency-input {
    position: relative;
    display: flex;
    align-items: center;
}

.currency-symbol {
    position: absolute;
    left: 12px;
    color: var(--text-secondary);
    font-weight: 500;
    z-index: 1;
}

.currency-input .form-input {
    padding-left: 30px;
}

/* Additional form styling */
.collapsible-content .form-group {
    margin-bottom: 20px;
}

.collapsible-content .form-group:last-child {
    margin-bottom: 0;
}

/* Date Picker Modal */
.date-picker-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10002;
}

.date-picker-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 20px;
    width: 90%;
    max-width: 400px;
    box-shadow: var(--shadow-md);
}

.date-picker-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 1px solid var(--border-light);
}

.date-picker-header h4 {
    margin: 0;
    color: var(--text-primary);
    font-size: 18px;
}

.date-picker-close {
    background: none;
    border: none;
    font-size: 24px;
    color: var(--text-secondary);
    cursor: pointer;
    padding: 5px;
    border-radius: 4px;
    transition: var(--transition-base);
}

.date-picker-close:hover {
    background: var(--bg-secondary);
    color: var(--text-primary);
}

.date-picker-body {
    margin-bottom: 20px;
}

.date-picker-footer {
    display: flex;
    justify-content: flex-end;
    gap: 10px;
    padding-top: 15px;
    border-top: 1px solid var(--border-light);
}

/* Option List Styles */
.option-list {
    max-height: 300px;
    overflow-y: auto;
}

.option-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 16px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    margin-bottom: 8px;
    cursor: pointer;
    transition: var(--transition-base);
    background: white;
}

.option-item:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
}

.option-item.selected {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.option-item.selected .option-check {
    display: block !important;
    color: white;
}

.option-text {
    font-weight: 500;
    color: var(--text-primary);
}

.option-check {
    color: var(--primary-color);
    font-size: 14px;
}

/* Enhanced Inspection Display */
.inspection-item {
    border-left: 4px solid var(--info-color);
    margin-bottom: 12px;
}

.inspection-description {
    margin-top: 8px;
    font-size: 12px;
    color: var(--text-secondary);
    line-height: 1.4;
    background: var(--bg-light);
    padding: 8px;
    border-radius: var(--radius-sm);
    border-left: 3px solid var(--info-color);
}

.inspection-findings {
    margin-top: 6px;
    font-size: 11px;
    color: var(--warning-color);
    background: #fff3cd;
    padding: 6px;
    border-radius: var(--radius-sm);
    border-left: 3px solid var(--warning-color);
}

.inspection-recommendations {
    margin-top: 6px;
    font-size: 11px;
    color: var(--success-color);
    background: #d4edda;
    padding: 6px;
    border-radius: var(--radius-sm);
    border-left: 3px solid var(--success-color);
}

/* Additional Status Styles for Inspections */
.equipment-status.completed {
    background: var(--success-color);
    color: white;
}

.equipment-status.pending {
    background: var(--warning-color);
    color: white;
}

.equipment-status.inprogress {
    background: var(--info-color);
    color: white;
}

.equipment-status.scheduled {
    background: var(--secondary-color);
    color: white;
}

.equipment-status.unknown {
    background: var(--secondary-color);
    color: white;
}

/* Enhanced Equipment Item Styles */
.equipment-item {
    padding: 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    margin-bottom: 12px;
    cursor: pointer;
    transition: all 0.3s ease;
    background: white;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.equipment-item:hover {
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    transform: translateY(-2px);
}

.equipment-info {
    width: 100%;
}

.equipment-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 12px;
    padding-bottom: 8px;
    border-bottom: 1px solid var(--border-color);
}

.equipment-name {
    font-weight: 700;
    color: var(--text-primary);
    font-size: 16px;
    margin: 0;
}

.equipment-details {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.equipment-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 4px 0;
    border-bottom: 1px solid #f0f0f0;
}

.equipment-row:last-child {
    border-bottom: none;
}

.equipment-label {
    font-weight: 600;
    color: var(--text-secondary);
    font-size: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    min-width: 100px;
}

.equipment-value {
    color: var(--text-primary);
    font-size: 13px;
    text-align: right;
    flex: 1;
    margin-left: 12px;
}

.equipment-status.active {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
    padding: 6px 12px;
    border-radius: var(--radius-sm);
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    white-space: nowrap;
}

.equipment-status.inactive {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
    padding: 6px 12px;
    border-radius: var(--radius-sm);
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    white-space: nowrap;
}

.equipment-status.maintenance {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffeaa7;
    padding: 6px 12px;
    border-radius: var(--radius-sm);
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    white-space: nowrap;
}

.equipment-status.maintenancerequired {
    background: #ffeaa7;
    color: #856404;
    border: 1px solid #fdcb6e;
    padding: 6px 12px;
    border-radius: var(--radius-sm);
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    white-space: nowrap;
}


/* Task Status Dropdown Styles */
.task-card-status-container {
    position: relative;
    display: flex;
    align-items: center;
}

.task-status-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 8px 12px;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-sm);
    background: white;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 14px;
    min-width: 120px;
    justify-content: space-between;
}

.task-status-btn:hover {
    border-color: var(--primary-color);
}

.task-status-indicator {
    width: 10px;
    height: 10px;
    border-radius: 50%;
    flex-shrink: 0;
}

.task-status-indicator.initiated {
    background-color: #6c757d;
}

.task-status-indicator.inprogress {
    background-color: #ffc107;
}

.task-status-indicator.onhold {
    background-color: #fd7e14;
}

.task-status-indicator.complete {
    background-color: #28a745;
}

.task-status-btn span {
    font-weight: 500;
    color: var(--text-primary);
    flex: 1;
}

.task-status-arrow {
    color: var(--text-secondary);
    font-size: 12px;
    transition: transform 0.3s ease;
}

.task-status-btn.open .task-status-arrow {
    transform: rotate(180deg);
}

.task-status-dropdown {
    position: absolute;
    bottom: 100%;
    left: 0;
    right: 0;
    background: white;
    border: 2px solid var(--border-color);
    border-radius: var(--radius-sm);
    box-shadow: 0 -4px 12px rgba(0, 0, 0, 0.15);
    z-index: 1000;
    display: none;
    margin-bottom: 4px;
}

.task-status-dropdown.show {
    display: block;
}

.task-status-option {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 12px;
    cursor: pointer;
    transition: background-color 0.2s ease;
    font-size: 14px;
}

.task-status-option:hover {
    background-color: var(--bg-light);
}

.task-status-option:first-child {
    border-radius: var(--radius-sm) var(--radius-sm) 0 0;
}

.task-status-option:last-child {
    border-radius: 0 0 var(--radius-sm) var(--radius-sm);
}

.task-status-option span {
    font-weight: 500;
    color: var(--text-primary);
}
</style>
@endsection

@section('content')
<div class="workorder-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('maintenance.workorder') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">New Job Card</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" onclick="saveWorkOrder()">
                    <i class="fas fa-save"></i>
                    Create
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="form-container">
            <!-- Request Number Section -->
            <div class="form-section">
                <div class="request-number-display">
                    <span class="request-number-label">Job Card #</span>
                    <span class="request-number-value" id="requestNumber">Generating...</span>
                </div>
                <div style="margin-top: 10px; padding: 10px; background: #e6f0ff; border-left: 4px solid #0055A5; border-radius: 4px;">
                    <div style="font-size: 13px; color: #374151; font-weight: 500;">
                        📊 <span id="jobCardCountInfo">Calculating...</span>
                    </div>
                </div>
                
                <!-- Total Job Cards in Database -->
                <div style="margin-top: 15px; padding: 15px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 8px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <div>
                            <div style="color: rgba(255,255,255,0.9); font-size: 12px; font-weight: 500; margin-bottom: 5px;">
                                Total Job Cards in MongoDB Database
                            </div>
                            <div style="color: white; font-size: 32px; font-weight: 700;" id="totalJobCardsDisplay">
                                <i class="fas fa-spinner fa-spin"></i>
                            </div>
                        </div>
                        <div style="background: rgba(255,255,255,0.2); padding: 15px; border-radius: 50%; width: 60px; height: 60px; display: flex; align-items: center; justify-content: center;">
                            <i class="fas fa-clipboard-list" style="color: white; font-size: 28px;"></i>
                        </div>
                    </div>
                    <div style="margin-top: 12px; padding-top: 12px; border-top: 1px solid rgba(255,255,255,0.2);">
                        <button onclick="viewAllJobCards()" style="background: rgba(255,255,255,0.9); color: #667eea; border: none; padding: 8px 16px; border-radius: 6px; font-size: 13px; font-weight: 600; cursor: pointer; transition: all 0.3s;">
                            <i class="fas fa-eye"></i> View All Job Cards
                        </button>
                    </div>
                </div>
            </div>

            <!-- Basic Information Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-info-circle"></i>
                    Basic Information
                </h2>

                <div class="form-group">
                    <label class="form-label">
                        Job Card Title
                        <span class="required">*</span>
                    </label>
                    <input type="text" class="form-input" id="workOrderTitle" placeholder="Enter job card title">
                    <div class="error-message" id="titleError">Job Card Title is required</div>
                </div>

                <div class="form-group">
                    <label class="form-label">
                        Country
                        <span class="required">*</span>
                    </label>
                    <select class="form-input" id="country" onchange="updateProjectOptions()">
                        <option value="">Select Country</option>
                        <option value="Zambia">Zambia</option>
                        <option value="DRC">DRC</option>
                    </select>
                    <div class="error-message" id="countryError">Country is required</div>
                </div>

                <div class="form-group">
                    <label class="form-label">
                        Project
                        <span class="required">*</span>
                    </label>
                    <select class="form-input" id="project" disabled>
                        <option value="">Select Project</option>
                    </select>
                    <div class="error-message" id="projectError">Project is required</div>
                </div>

                <div class="form-group">
                    <label class="form-label">Memo</label>
                    <textarea class="form-input form-textarea" id="memo" placeholder="Enter memo or notes"></textarea>
                </div>

                <!-- Date Fields -->
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Start Date</label>
                        <input type="date" class="form-input" id="startDate">
                    </div>
                    <div class="form-group">
                        <label class="form-label">Due Date</label>
                        <input type="date" class="form-input" id="dueDate">
                    </div>
                </div>

                <div class="form-group">
                    <label class="form-label">Priority</label>
                    <div class="priority-container">
                        <div class="priority-option" data-priority="Low" onclick="selectPriority('Low')">
                            <img class="priority-image" src="{{ asset('images/insvecico/Low_vectorIcon_1.png') }}" alt="Low priority icon">
                            <div class="priority-label">Low</div>
                        </div>
                        <div class="priority-option" data-priority="Medium" onclick="selectPriority('Medium')">
                            <img class="priority-image" src="{{ asset('images/insvecico/Medium_vectorIcon_1.png') }}" alt="Medium priority icon">
                            <div class="priority-label">Medium</div>
                        </div>
                        <div class="priority-option" data-priority="High" onclick="selectPriority('High')">
                            <img class="priority-image" src="{{ asset('images/insvecico/High_vectorIcon_1.png') }}" alt="High priority icon">
                            <div class="priority-label">High</div>
                        </div>
                        <div class="priority-option" data-priority="Emergency" onclick="selectPriority('Emergency')">
                            <img class="priority-image" src="{{ asset('images/insvecico/Emergency_vector_cion.png') }}" alt="Emergency priority icon">
                            <div class="priority-label">Emergency</div>
                        </div>
                    </div>
                </div>


                <!-- Additional Details Section -->
                <div class="collapsible-section">
                    <div class="collapsible-header" onclick="toggleCollapsible('additionalDetails')">
                        <span class="collapsible-title">Additional Details</span>
                        <i class="fas fa-chevron-down collapsible-arrow" id="additionalDetailsArrow"></i>
                    </div>
                    <div class="collapsible-content" id="additionalDetailsContent">
                        <div class="form-group">
                            <div class="select-row" onclick="openJobCardTypeModal(event)">
                                <span class="select-label">Job Card Type</span>
                                <div class="select-right">
                                    <span class="select-text" id="jobCardTypeText">General</span>
                                    <i class="fas fa-chevron-right select-arrow"></i>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label class="form-label">Enter estimated cost</label>
                            <input type="number" class="form-input" id="estimatedCost" placeholder="Enter estimated cost" step="0.01" min="0">
                        </div>

                        <div class="form-group">
                            <label class="form-label">Enter labor time in hours</label>
                            <input type="number" class="form-input" id="laborTimeHours" placeholder="Enter labor time in hours" step="0.5" min="0">
                        </div>

                        <div class="form-group">
                            <div class="select-row" onclick="openLinkInspectionModal(event)">
                                <span class="select-label">Link Inspection</span>
                                <div class="select-right">
                                    <span class="select-text" id="linkInspectionText">Select</span>
                                    <i class="fas fa-chevron-right select-arrow"></i>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <div class="select-row" onclick="openJobCardModal(event)">
                                <span class="select-label">Job Card</span>
                                <div class="select-right">
                                    <span class="select-text" id="jobCardText">Select</span>
                                    <i class="fas fa-chevron-right select-arrow"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Cost Details Section -->
                <div class="collapsible-section">
                    <div class="collapsible-header" onclick="toggleCollapsible('costDetails')">
                        <span class="collapsible-title">Cost Details</span>
                        <i class="fas fa-chevron-up collapsible-arrow" id="costDetailsArrow"></i>
                    </div>
                    <div class="collapsible-content" id="costDetailsContent" style="display: block;">
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Part & Material Cost</label>
                                <div class="currency-input">
                                    <span class="currency-symbol">$</span>
                                    <input type="number" class="form-input" id="partMaterialCost" placeholder="0.00" step="0.01" min="0" value="0.00">
                                </div>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Total Labor Cost</label>
                                <div class="currency-input">
                                    <span class="currency-symbol">$</span>
                                    <input type="number" class="form-input" id="totalLaborCost" placeholder="0.00" step="0.01" min="0" value="0.00">
                                </div>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Additional Cost</label>
                                <div class="currency-input">
                                    <span class="currency-symbol">$</span>
                                    <input type="number" class="form-input" id="additionalCost" placeholder="0.00" step="0.01" min="0" value="0.00">
                                </div>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Tax Amount</label>
                                <div class="currency-input">
                                    <span class="currency-symbol">$</span>
                                    <input type="number" class="form-input" id="taxAmount" placeholder="0.00" step="0.01" min="0" value="0.00">
                                </div>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Tax(%)</label>
                                <div class="currency-input">
                                    <span class="currency-symbol">%</span>
                                    <input type="number" class="form-input" id="taxPercentage" placeholder="0.00" step="0.01" min="0" value="0.00">
                                </div>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Total WO Cost</label>
                                <div class="currency-input">
                                    <span class="currency-symbol">$</span>
                                    <input type="number" class="form-input" id="totalWOCost" placeholder="0.00" step="0.01" min="0" value="0.00" readonly>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Equipment Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-tools"></i>
                    Equipment
                </h2>

                <div class="form-group">
                    <div class="select-row" onclick="openEquipmentModal(event)">
                        <span class="select-label">Equipment</span>
                        <div class="select-right">
                            <span class="select-text" id="equipmentText">Select Equipment</span>
                            <i class="fas fa-chevron-right select-arrow"></i>
                        </div>
                    </div>
                    <div class="error-message" id="equipmentError">Equipment is required</div>
                    
                    <!-- Equipment Preview -->
                    <div class="equipment-preview" id="equipmentPreview" style="display: none;">
                        <div class="preview-card">
                            <div class="preview-header">
                                <h4 id="equipmentPreviewName">Equipment Name</h4>
                                <span class="preview-status" id="equipmentPreviewStatus">Active</span>
                            </div>
                            <div class="preview-details">
                                <div class="preview-row">
                                    <span class="preview-label">Type:</span>
                                    <span class="preview-value" id="equipmentPreviewType">Unknown</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Location:</span>
                                    <span class="preview-value" id="equipmentPreviewLocation">Unknown</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Model:</span>
                                    <span class="preview-value" id="equipmentPreviewModel">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Serial:</span>
                                    <span class="preview-value" id="equipmentPreviewSerial">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Registration:</span>
                                    <span class="preview-value" id="equipmentPreviewReg">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Make:</span>
                                    <span class="preview-value" id="equipmentPreviewMake">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Engine:</span>
                                    <span class="preview-value" id="equipmentPreviewEngine">-</span>
                                </div>
                                <div class="preview-row">
                                    <span class="preview-label">Meter Reading:</span>
                                    <span class="preview-value" id="equipmentPreviewMeter">-</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Assignment Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-users"></i>
                    Assignment
                </h2>

                <div class="form-group">
                    <div class="select-row" onclick="openAssignToModal(event)">
                        <span class="select-label">Assign To</span>
                        <div class="select-right">
                            <span class="select-text" id="assignToText">Select Users</span>
                            <i class="fas fa-chevron-right select-arrow"></i>
                        </div>
                    </div>
                    <div class="selected-items-container" id="selectedUsersContainer" style="display: none;">
                        <div class="selected-items-row" id="selectedUsersList"></div>
                    </div>
                    
                    <!-- Users Preview -->
                    <div class="users-preview" id="usersPreview" style="display: none;">
                        <div class="preview-header">
                            <h4>Selected Users (${selectedUsers.length})</h4>
                        </div>
                        <div class="users-grid" id="usersPreviewGrid">
                            <!-- User preview cards will be added here -->
                        </div>
                    </div>
                </div>
            </div>

            <!-- Tasks & Items Section -->
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-tasks"></i>
                    Tasks & Items
                </h2>

                <!-- Category Tabs -->
                <div class="category-tabs">
                    <button class="tab-button active" data-category="Task" onclick="switchCategory('Task')">
                        <i class="fas fa-tasks"></i> Tasks
                    </button>
                    <button class="tab-button" data-category="Parts" onclick="switchCategory('Parts')">
                        <i class="fas fa-cogs"></i> Parts
                    </button>
                    <button class="tab-button" data-category="Labor" onclick="switchCategory('Labor')">
                        <i class="fas fa-user-tie"></i> Labor
                    </button>
                    <button class="tab-button" data-category="Additional Cost" onclick="switchCategory('Additional Cost')">
                        <i class="fas fa-dollar-sign"></i> Additional Cost
                    </button>
                    <button class="tab-button" data-category="Remarks" onclick="switchCategory('Remarks')">
                        <i class="fas fa-comment"></i> Remarks
                    </button>
                </div>

                <!-- Add Button -->
                <div class="form-group">
                    <button type="button" class="add-task-btn" onclick="openAddItemModal(event)">
                        <i class="fas fa-plus"></i>
                        Add <span id="addButtonText">Task</span>
                    </button>
                </div>

                <!-- Items List -->
                <div class="task-list-container" id="itemsListContainer">
                    <!-- Items will be dynamically added here -->
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Equipment Selection Modal -->
<div id="equipmentModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Equipment</h3>
            <button type="button" class="close-btn" onclick="closeEquipmentModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="equipmentSearch" class="search-input" placeholder="Search equipment..." onkeyup="filterEquipment()">
                <button type="button" class="search-btn" onclick="filterEquipment()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="equipment-list" id="equipmentList">
                <div class="loading-state">
                    <div class="loading-spinner"></div>
                    Loading equipment...
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Assign To Modal -->
<div id="assignToModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Assign To Users</h3>
            <button type="button" class="close-btn" onclick="closeAssignToModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="userSearch" class="search-input" placeholder="Search users..." onkeyup="filterUsers()">
                <button type="button" class="search-btn" onclick="filterUsers()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="user-list" id="userList">
                <div class="loading-state">
                    <div class="loading-spinner"></div>
                    Loading users...
                </div>
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeAssignToModal()">Cancel</button>
            <button class="btn btn-primary" onclick="saveAssignTo()">Save</button>
        </div>
    </div>
</div>

<!-- Add Item Modal -->
<div id="addItemModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3 id="addItemModalTitle">Add Task</h3>
            <button type="button" class="close-btn" onclick="closeAddItemModal()">&times;</button>
        </div>
        <div class="custom-modal-body" id="addItemModalBody">
            <!-- Task Form -->
            <div id="taskForm" class="item-form">
                <div class="form-group">
                    <label class="form-label">Task Description</label>
                    <input type="text" class="form-input" id="taskDescription" placeholder="Enter task description">
                </div>
                <div class="form-group">
                    <label class="form-label">Task Type</label>
                    <select class="form-input" id="taskType">
                        <option value="Repair">Repair</option>
                        <option value="Replace">Replace</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Assign To</label>
                    <div class="select-row" onclick="openTaskAssignToModal(event)">
                        <span class="select-label">Select Users</span>
                        <div class="select-right">
                            <span class="select-text" id="taskAssignToText">Select Users</span>
                            <i class="fas fa-chevron-right select-arrow"></i>
                        </div>
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Notes</label>
                    <textarea class="form-input form-textarea" id="taskNotes" placeholder="Enter task notes"></textarea>
                </div>
            </div>

            <!-- Parts Form -->
            <div id="partsForm" class="item-form" style="display: none;">
                <div class="form-group">
                    <label class="form-label">Item Number</label>
                    <input type="text" class="form-input" id="partsItemNumber" placeholder="Enter item number">
                </div>
                <div class="form-group">
                    <label class="form-label">Part Number</label>
                    <input type="text" class="form-input" id="partsPartNumber" placeholder="Enter part number">
                </div>
                <div class="form-group">
                    <label class="form-label">Item Name</label>
                    <input type="text" class="form-input" id="partsItemName" placeholder="Enter item name">
                </div>
                <div class="form-group">
                    <label class="form-label">Quantity</label>
                    <input type="number" class="form-input" id="partsQuantity" placeholder="Enter quantity">
                </div>
                <div class="form-group">
                    <label class="form-label">Unit</label>
                    <select class="form-input" id="partsUnit">
                        <option value="Piece">Piece</option>
                        <option value="Unit">Unit</option>
                        <option value="Box">Box</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Item Cost</label>
                    <input type="number" class="form-input" id="partsItemCost" placeholder="Enter item cost" step="0.01">
                </div>
                <div class="form-group">
                    <label class="form-label">Site</label>
                    <input type="text" class="form-input" id="partsSite" placeholder="Enter site">
                </div>
                <div class="form-group">
                    <label class="form-label">Notes</label>
                    <textarea class="form-input form-textarea" id="partsNote" placeholder="Enter notes"></textarea>
                </div>
            </div>

            <!-- Labor Form -->
            <div id="laborForm" class="item-form" style="display: none;">
                <div class="form-group">
                    <label class="form-label">Worker Name</label>
                    <input type="text" class="form-input" id="laborWorker" placeholder="Enter worker name">
                </div>
                <div class="form-group">
                    <label class="form-label">Labor Code</label>
                    <input type="text" class="form-input" id="laborCode" placeholder="Enter labor code">
                </div>
                <div class="form-group">
                    <label class="form-label">Rate</label>
                    <input type="number" class="form-input" id="laborRate" placeholder="Enter rate" step="0.01">
                </div>
                <div class="form-group">
                    <label class="form-label">Work Hours</label>
                    <div style="display: flex; gap: 10px;">
                        <input type="number" class="form-input" id="laborHours" placeholder="Hours" min="0" max="24">
                        <input type="number" class="form-input" id="laborMinutes" placeholder="Minutes" min="0" max="59">
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Notes</label>
                    <textarea class="form-input form-textarea" id="laborNote" placeholder="Enter notes"></textarea>
                </div>
            </div>

            <!-- Additional Cost Form -->
            <div id="additionalcostForm" class="item-form" style="display: none;">
                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea class="form-input form-textarea" id="additionalCostDescription" placeholder="Enter cost description"></textarea>
                </div>
                <div class="form-group">
                    <label class="form-label">Amount</label>
                    <input type="number" class="form-input" id="additionalCostAmount" placeholder="Enter amount" step="0.01">
                </div>
            </div>

            <!-- Remarks Form -->
            <div id="remarksForm" class="item-form" style="display: none;">
                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea class="form-input form-textarea" id="remarksDescription" placeholder="Enter remarks description"></textarea>
                </div>
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeAddItemModal()">Cancel</button>
            <button class="btn btn-primary" onclick="saveItem()" id="saveItemBtn">Save</button>
        </div>
    </div>
</div>

<!-- Task Assign To Modal -->
<div id="taskAssignToModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Assign Task To Users</h3>
            <button type="button" class="close-btn" onclick="closeTaskAssignToModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="taskUserSearch" class="search-input" placeholder="Search users..." onkeyup="filterTaskUsers()">
                <button type="button" class="search-btn" onclick="filterTaskUsers()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="user-list" id="taskUserList">
                <!-- User list will be populated here -->
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeTaskAssignToModal()">Cancel</button>
            <button class="btn btn-primary" onclick="saveTaskAssignTo()">Save</button>
        </div>
    </div>
</div>

<!-- Job Card Type Modal -->
<div id="jobCardTypeModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Job Card Type</h3>
            <button type="button" class="close-btn" onclick="closeJobCardTypeModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="option-list">
                <div class="option-item" onclick="selectJobCardType('General')">
                    <span class="option-text">General</span>
                </div>
                <div class="option-item" onclick="selectJobCardType('Maintenance')">
                    <span class="option-text">Maintenance</span>
                </div>
                <div class="option-item" onclick="selectJobCardType('Repair')">
                    <span class="option-text">Repair</span>
                </div>
                <div class="option-item" onclick="selectJobCardType('Installation')">
                    <span class="option-text">Installation</span>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Link Inspection Modal -->
<div id="linkInspectionModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Link Inspection</h3>
            <button type="button" class="close-btn" onclick="closeLinkInspectionModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="inspectionSearch" class="search-input" placeholder="Search inspections..." onkeyup="filterInspections()">
                <button type="button" class="search-btn" onclick="filterInspections()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="inspection-list" id="inspectionList">
                <div class="loading-state">
                    <div class="loading-spinner"></div>
                    Loading inspections...
                </div>
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeLinkInspectionModal()">Cancel</button>
            <button class="btn btn-primary" onclick="saveLinkInspection()">Save</button>
        </div>
    </div>
</div>

<!-- Job Card Modal -->
<div id="jobCardModal" class="custom-modal">
    <div class="custom-modal-content">
        <div class="custom-modal-header">
            <h3>Select Job Card</h3>
            <button type="button" class="close-btn" onclick="closeJobCardModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div class="search-container">
                <input type="text" id="jobCardSearch" class="search-input" placeholder="Search job cards..." onkeyup="filterJobCards()">
                <button type="button" class="search-btn" onclick="filterJobCards()">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <div class="jobcard-list" id="jobCardList">
                <div class="loading-state">
                    <div class="loading-spinner"></div>
                    Loading job cards...
                </div>
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeJobCardModal()">Cancel</button>
            <button class="btn btn-primary" onclick="saveJobCard()">Save</button>
        </div>
    </div>
</div>

<!-- Loading Modal -->
<div class="sync-modal" id="loadingModal" style="display: none;">
    <div class="sync-content">
        <div class="sync-icon">⏳</div>
        <div class="sync-text" id="loadingTitle">Please Wait</div>
        <div class="sync-subtext" id="loadingSubtext">Initializing...</div>
    </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="success-modal-overlay" style="display: none;">
    <div class="success-modal-container">
        <div class="success-modal-header">
            <h2 class="success-title">Success</h2>
        </div>
        <div class="success-modal-body">
            <p class="success-message">Job Card Generated and Saved Successfully!</p>
            <div class="success-buttons">
                <button type="button" class="success-btn back-btn" onclick="goBackToWorkOrders()">
                    Back
                </button>
                <button type="button" class="success-btn view-pdf-btn" onclick="viewPDF()">
                    View PDF
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Include html2pdf library for PDF generation -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>

<!-- All Job Cards Modal -->
<div id="allJobCardsModal" class="custom-modal">
    <div class="custom-modal-content" style="max-width: 1200px; width: 90%; max-height: 90vh;">
        <div class="custom-modal-header">
            <h3><i class="fas fa-clipboard-list"></i> All Job Cards in MongoDB Database</h3>
            <button type="button" class="close-btn" onclick="closeAllJobCardsModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div style="background: #f8f9fa; padding: 15px; border-radius: 6px; margin-bottom: 15px;">
                <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 10px;">
                    <div style="font-size: 18px; font-weight: 600; color: #0055A5;">
                        <i class="fas fa-database"></i> Total Job Cards: <span id="allJobCardsCount">0</span>
                    </div>
                    <div style="display: flex; gap: 10px;">
                        <input type="text" id="jobCardsSearchInput" class="search-input" placeholder="Search job cards..." onkeyup="filterAllJobCards()" style="width: 250px;">
                        <button onclick="refreshAllJobCards()" class="search-btn" style="background: #28a745;">
                            <i class="fas fa-sync-alt"></i> Refresh
                        </button>
                    </div>
                </div>
            </div>
            
            <div id="allJobCardsList" style="max-height: 60vh; overflow-y: auto;">
                <div style="text-align: center; padding: 40px; color: #666;">
                    <i class="fas fa-spinner fa-spin" style="font-size: 32px; margin-bottom: 10px;"></i>
                    <p>Loading job cards from MongoDB...</p>
                </div>
            </div>
        </div>
        <div class="modal-footer">
            <button class="btn btn-secondary" onclick="closeAllJobCardsModal()">Close</button>
        </div>
    </div>
</div>


<script>
// ============================================================================
// CREATE NEW JOB CARD - WEB VERSION
// ============================================================================
// 
// PDF STORAGE FLOW (CORRECTED):
// ------------------------------
// 1. User clicks "Create" button
// 2. Validate form fields (title, country, project, equipment)
// 3. Generate PDF blob locally using html2pdf library
// 4. Upload PDF to backend filesystem:
//    Path: uploads/documents/GlobalJobcardReports/{adminUid}/{fileName}.pdf
// 5. Backend returns download URL:
//    Format: https://api.titandrillingzm.com:6005/jobcards/download-file/{adminUid}/{fileName}
// 6. Create job card in MongoDB WITH pdfDownloadUrl already included
// 7. Show success modal with "Back" and "View PDF" buttons
// 8. View PDF opens the MongoDB URL in new tab (persistent, shareable)
//
// KEY POINTS:
// - PDF file is stored on backend filesystem (not in MongoDB)
// - PDF URL is stored in MongoDB (pdfDownloadUrl field)
// - Job card is created with PDF URL from the start (no update needed)
// - View PDF uses MongoDB URL (works even after page refresh)
//
// ============================================================================

// MongoDB API Configuration
const MONGODB_API_BASE = 'https://api.titandrillingzm.com:6005';
const ADMIN_UID = 'cahj3jJqe5Z16pQdLGSWrNgm84o1'; // Replace with actual logged-in admin UID

// Global variables
let selectedEquipment = null;
let selectedUsers = [];
let selectedTaskUsers = [];
let items = []; // Changed from tasks to items to include all types
let equipmentList = [];
let userList = [];
let taskUserList = [];
let currentCategory = 'Task'; // Track current selected category
let selectedJobCardType = 'General';
let selectedInspection = null;
let selectedJobCard = null;
let inspectionList = [];
let jobCardList = [];
let allJobCardsFromDB = []; // Store all job cards from MongoDB

// Debug helper - Call in console to see current state
window.debugJobCard = function() {
    console.log('🔍 DEBUG: Current Job Card State');
    console.log('   - Selected Equipment:', selectedEquipment);
    console.log('   - Selected Users:', selectedUsers);
    console.log('   - Items:', items);
    console.log('   - PDF Blob:', window.generatedPDFBlob);
    console.log('   - PDF Blob URL:', window.generatedPDFBlobUrl);
    console.log('   - PDF MongoDB URL:', window.generatedPDFUrl);
    console.log('   - Request Number:', document.getElementById('requestNumber')?.textContent);
};

// Project options based on country
const projectOptions = {
    'Zambia': [
        'Kansanshi', 'Kalumbila', 'Kimteto', 'Kobold', 'Koblenz',
        'FQM Exploration', 'Mimosa'
    ],
    'DRC': [
        'Lubumbashi', 'Musompo', 'IME', 'Kamoa'
    ]
};

// Initialize the page
document.addEventListener('DOMContentLoaded', async function() {
    console.log('🚀 Page loaded, initializing...');
    
    // Test API connection first
    const apiOnline = await testAPIConnection();
    if (!apiOnline) {
        showToast('warning', 'Warning', 'MongoDB API may be offline. Some features may not work.');
    }
    
    // Auto-select Zambia and Kobold
    const countrySelect = document.getElementById('country');
    if (countrySelect) {
        countrySelect.value = 'Zambia';
        updateProjectOptions(); // This will auto-select Kobold
    }
    
    generateRequestNumber();
    loadEquipmentFromMongoDB();
    loadUsersFromMongoDB();
    loadInspectionsFromMongoDB();
    initializeCostCalculation();
    fetchTotalJobCardsCount();
    
    console.log('✅ Page initialization complete');
});

// Generate request number from MongoDB
async function generateRequestNumber() {
    const country = document.getElementById('country').value;
    const project = document.getElementById('project').value;
    
    console.log('generateRequestNumber called with:', { country, project });
    
    if (country && project) {
        const countryCode = country === 'Zambia' ? 'ZAM' : (country === 'DRC' ? 'DRC' : 'XXX');
        
        // Handle special project codes
        let projectCode;
        if (project === 'Koblenz' || project === 'Kobold') {
            projectCode = 'KOB-JC';
        } else {
            projectCode = project.slice(0, 3).toUpperCase() + '-JC';
        }
        
        console.log('Generated codes:', { countryCode, projectCode });
        
        try {
            // Get ALL job cards from MongoDB to count them correctly
            const response = await fetch(`${MONGODB_API_BASE}/jobcards/all`);
            
            let nextNumber = 1;
            let totalJobCards = 0;
            
            if (response.ok) {
                const result = await response.json();
                console.log('📊 MongoDB All Job Cards Response:', result);
                
                if (result.success && Array.isArray(result.data)) {
                    totalJobCards = result.data.length;
                    nextNumber = totalJobCards + 1;
                    
                    console.log(`   📁 Total job cards in entire database: ${totalJobCards}`);
                    console.log(`   ➕ Next job card number (Total + 1): ${nextNumber}`);
                } else {
                    console.warn('⚠️ No job cards found in MongoDB, starting from 1');
                }
            } else {
                console.warn('⚠️ Could not fetch from MongoDB, using default number 1');
            }
            
            // Format number with leading zeros (3 digits: 001, 010, 100)
            const formattedNumber = nextNumber.toString().padStart(3, '0');
            const requestNumber = `${countryCode}-${projectCode}-${formattedNumber}`;
            
            console.log('✅ Generated Job Card Number:', requestNumber);
            console.log(`   📊 Formula: Total Job Cards in DB (${totalJobCards}) + 1 = ${nextNumber} → Formatted: ${formattedNumber}`);
            
            document.getElementById('requestNumber').textContent = requestNumber;
            
            // Update count info display
            const countInfo = document.getElementById('jobCardCountInfo');
            if (countInfo) {
                countInfo.innerHTML = `📊 Total job cards in database: <strong style="color: #0055A5;">${totalJobCards}</strong> | Next Number: <strong style="color: #28a745;">${formattedNumber}</strong> (${totalJobCards} + 1)`;
            }
        } catch (error) {
            console.error('Error fetching job cards from MongoDB:', error);
            // Fallback to 001
            const formattedNumber = '001';
            const requestNumber = `${countryCode}-${projectCode}-${formattedNumber}`;
            document.getElementById('requestNumber').textContent = requestNumber;
            
            // Update count info display
            const countInfo = document.getElementById('jobCardCountInfo');
            if (countInfo) {
                countInfo.innerHTML = `⚠️ Could not connect to database. Using default number.`;
            }
        }
    } else {
        document.getElementById('requestNumber').textContent = 'Select Country & Project';
        
        // Update count info display
        const countInfo = document.getElementById('jobCardCountInfo');
        if (countInfo) {
            countInfo.innerHTML = `Please select <strong>Country</strong> and <strong>Project</strong> to generate job card number.`;
        }
    }
}

// Update project options based on country
function updateProjectOptions() {
    const country = document.getElementById('country').value;
    const projectSelect = document.getElementById('project');
    
    // Clear existing options
    projectSelect.innerHTML = '<option value="">Select Project</option>';
    
    if (country && projectOptions[country]) {
        projectSelect.disabled = false;
        projectOptions[country].forEach(project => {
            const option = document.createElement('option');
            option.value = project;
            option.textContent = project;
            projectSelect.appendChild(option);
        });
        
        // Auto-select Kobold for Zambia to match Firebase data
        if (country === 'Zambia') {
            projectSelect.value = 'Kobold';
        }
    } else {
        projectSelect.disabled = true;
    }
    
    // Generate new request number when country/project changes
    generateRequestNumber();
}

// Priority selection
function selectPriority(priority) {
    // Remove selected class from all options
    document.querySelectorAll('.priority-option').forEach(option => {
        option.classList.remove('selected');
    });
    
    // Add selected class to clicked option
    event.currentTarget.classList.add('selected');
    
    // Store selected priority
    window.selectedPriority = priority;
}


// Close dropdown when clicking outside
document.addEventListener('click', function(event) {
    // Close all task status dropdowns when clicking outside
    const taskStatusDropdowns = document.querySelectorAll('.task-status-dropdown');
    taskStatusDropdowns.forEach(dropdown => {
        if (!dropdown.closest('.task-card-status-container').contains(event.target)) {
            dropdown.classList.remove('show');
            const btn = dropdown.previousElementSibling;
            if (btn) {
                btn.classList.remove('open');
                const arrow = btn.querySelector('.task-status-arrow');
                if (arrow) {
                    arrow.style.transform = 'rotate(0deg)';
                }
            }
        }
    });
});

// Task status dropdown functions
function toggleItemStatusDropdown(itemId) {
    const dropdown = document.getElementById(`taskStatusDropdown${itemId}`);
    const btn = dropdown.previousElementSibling;
    const arrow = btn.querySelector('.task-status-arrow');
    
    if (dropdown.classList.contains('show')) {
        dropdown.classList.remove('show');
        btn.classList.remove('open');
        arrow.style.transform = 'rotate(0deg)';
    } else {
        // Close all other dropdowns first
        document.querySelectorAll('.task-status-dropdown').forEach(dd => {
            if (dd.id !== `taskStatusDropdown${itemId}`) {
                dd.classList.remove('show');
                const otherBtn = dd.previousElementSibling;
                if (otherBtn) {
                    otherBtn.classList.remove('open');
                    const otherArrow = otherBtn.querySelector('.task-status-arrow');
                    if (otherArrow) {
                        otherArrow.style.transform = 'rotate(0deg)';
                    }
                }
            }
        });
        
        dropdown.classList.add('show');
        btn.classList.add('open');
        arrow.style.transform = 'rotate(180deg)';
    }
}

function selectItemStatus(itemId, status, statusClass) {
    // Find the item in the items array
    const itemIndex = items.findIndex(item => item.id === itemId);
    if (itemIndex !== -1) {
        items[itemIndex].status = status;
        items[itemIndex].statusClass = statusClass;
        
        // Re-render items to update the display
        renderItems();
    }
}

// Equipment modal functions
function openEquipmentModal(event) {
    const modal = document.getElementById('equipmentModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Auto-focus search input
    setTimeout(() => {
        const searchInput = document.getElementById('equipmentSearch');
        if (searchInput) {
            searchInput.focus();
        }
    }, 100);
}

function closeEquipmentModal() {
    const modal = document.getElementById('equipmentModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function updateEquipmentDisplay() {
    const container = document.getElementById('equipmentList');
    if (!container) return;
    
    container.innerHTML = '';
    
    if (equipmentList.length === 0) {
        container.innerHTML = '<div class="no-data">No equipment available</div>';
        return;
    }
    
    equipmentList.forEach(equipment => {
        const equipmentItem = document.createElement('div');
        equipmentItem.className = 'equipment-item';
        equipmentItem.innerHTML = `
            <div class="equipment-info">
                <div class="equipment-name">${equipment.name}</div>
                <div class="equipment-details">
                    <span class="equipment-type">${equipment.type}</span>
                    <span class="equipment-location">${equipment.location}</span>
                </div>
            </div>
            <div class="equipment-status ${equipment.status.toLowerCase()}">${equipment.status}</div>
        `;
        equipmentItem.onclick = () => selectEquipment(equipment);
        container.appendChild(equipmentItem);
    });
}

function filterEquipment() {
    const searchTerm = document.getElementById('equipmentSearch').value.toLowerCase();
    const equipmentListDiv = document.getElementById('equipmentList');
    
    equipmentListDiv.innerHTML = '';
    
    if (equipmentList.length === 0) {
        equipmentListDiv.innerHTML = '<div class="no-data">Loading equipment...</div>';
        return;
    }
    
    const filteredEquipment = equipmentList.filter(equipment => 
        equipment.name.toLowerCase().includes(searchTerm) ||
        equipment.type.toLowerCase().includes(searchTerm) ||
        equipment.location.toLowerCase().includes(searchTerm) ||
        (equipment.regNum && equipment.regNum.toLowerCase().includes(searchTerm)) ||
        (equipment.model && equipment.model.toLowerCase().includes(searchTerm)) ||
        (equipment.manufacturer && equipment.manufacturer.toLowerCase().includes(searchTerm)) ||
        (equipment.serialNumber && equipment.serialNumber.toLowerCase().includes(searchTerm)) ||
        (equipment.engineNumber && equipment.engineNumber.toLowerCase().includes(searchTerm))
    );
    
    if (filteredEquipment.length === 0) {
        equipmentListDiv.innerHTML = '<div class="no-data">No equipment matches your search</div>';
        return;
    }
    
    filteredEquipment.forEach(equipment => {
        const equipmentItem = document.createElement('div');
        equipmentItem.className = 'equipment-item';
        equipmentItem.innerHTML = `
            <div class="equipment-info">
                <div class="equipment-header">
                    <div class="equipment-name">${equipment.name}</div>
                    <div class="equipment-status ${equipment.status.toLowerCase().replace(/\s+/g, '')}">${equipment.status}</div>
                </div>
                <div class="equipment-details">
                    <div class="equipment-row">
                        <span class="equipment-label">Type:</span>
                        <span class="equipment-value">${equipment.type}</span>
                    </div>
                    <div class="equipment-row">
                        <span class="equipment-label">Location:</span>
                        <span class="equipment-value">${equipment.location}</span>
                    </div>
                    <div class="equipment-row">
                        <span class="equipment-label">Model:</span>
                        <span class="equipment-value">${equipment.model || 'N/A'}</span>
                    </div>
                    <div class="equipment-row">
                        <span class="equipment-label">Registration:</span>
                        <span class="equipment-value">${equipment.regNum || 'N/A'}</span>
                    </div>
                    <div class="equipment-row">
                        <span class="equipment-label">Make:</span>
                        <span class="equipment-value">${equipment.manufacturer || 'N/A'}</span>
                    </div>
                    <div class="equipment-row">
                        <span class="equipment-label">Engine:</span>
                        <span class="equipment-value">${equipment.engineNumber || 'N/A'}</span>
                    </div>
                    <div class="equipment-row">
                        <span class="equipment-label">Meter Reading:</span>
                        <span class="equipment-value">${equipment.meterReading ? parseFloat(equipment.meterReading).toLocaleString() : '0'} ${equipment.meterUnit || ''}</span>
                    </div>
                </div>
            </div>
        `;
        equipmentItem.onclick = () => selectEquipment(equipment);
        equipmentListDiv.appendChild(equipmentItem);
    });
}

function selectEquipment(equipment) {
    selectedEquipment = equipment;
    document.getElementById('equipmentText').textContent = equipment.name;
    document.getElementById('equipmentError').classList.remove('show');
    
    // Show equipment preview
    showEquipmentPreview(equipment);
    
    closeEquipmentModal();
}

function showEquipmentPreview(equipment) {
    const preview = document.getElementById('equipmentPreview');
    preview.style.display = 'block';
    
    document.getElementById('equipmentPreviewName').textContent = equipment.name;
    document.getElementById('equipmentPreviewStatus').textContent = equipment.status;
    document.getElementById('equipmentPreviewType').textContent = equipment.type || 'Unknown';
    document.getElementById('equipmentPreviewLocation').textContent = equipment.location || 'Unknown';
    document.getElementById('equipmentPreviewModel').textContent = equipment.model || '-';
    document.getElementById('equipmentPreviewSerial').textContent = equipment.serialNumber || '-';
    
    // Update additional preview details if elements exist
    const regElement = document.getElementById('equipmentPreviewReg');
    const makeElement = document.getElementById('equipmentPreviewMake');
    const engineElement = document.getElementById('equipmentPreviewEngine');
    const meterElement = document.getElementById('equipmentPreviewMeter');
    
    if (regElement) regElement.textContent = equipment.regNum || '-';
    if (makeElement) makeElement.textContent = equipment.manufacturer || '-';
    if (engineElement) engineElement.textContent = equipment.engineNumber || '-';
    
    // Format meter reading in compact format
    if (meterElement) {
        const meterReading = equipment.meterReading ? parseFloat(equipment.meterReading) : 0;
        const meterUnit = equipment.meterUnit || 'KM';
        
        // Format large numbers in scientific notation for very large values
        let formattedReading;
        if (meterReading >= 1000000) {
            formattedReading = meterReading.toExponential(3);
        } else {
            formattedReading = meterReading.toLocaleString('en-US', {
                minimumFractionDigits: 0,
                maximumFractionDigits: 3
            });
        }
        
        meterElement.textContent = `${formattedReading} ${meterUnit}`;
    }
}

// Load equipment from MongoDB
async function loadEquipmentFromMongoDB() {
    try {
        console.log('🔄 Loading equipment from MongoDB...');
        const response = await fetch('https://api.titandrillingzm.com:6001/EquipmentsDataJSON/GlobalEquipmentJsonData');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('MongoDB Equipment API Response:', result);
        
        // Process equipment data
        if (result.success && Array.isArray(result.data)) {
            equipmentList = result.data.map(equipment => ({
                id: equipment._id || equipment.id,
                name: equipment.vehicleNumber || equipment.equipmentName || equipment.name || equipment.title || 'Unknown Equipment',
                type: equipment.vehicleType || equipment.equipmentCategory || equipment.equipmentType || equipment.type || 'Unknown',
                status: equipment.status || 'Active',
                location: equipment.location || equipment.site || 'Unknown',
                model: equipment.model || equipment.equipmentModel || '',
                serialNumber: equipment.vin || equipment.serialNumber || equipment.serial || '',
                manufacturer: equipment.make || equipment.manufacturer || '',
                year: equipment.year || equipment.yearOfManufacture || '',
                country: equipment.country || '',
                project: equipment.project || '',
                regNum: equipment.regNum || equipment.registrationNumber || '',
                engineNumber: equipment.engineNumber || '',
                meterReading: equipment.meterReading || '',
                meterUnit: equipment.meterUnit || '',
                mainCategory: equipment.mainCategory || equipment.category || '',
                createdAt: equipment.createdAt || '',
                updatedAt: equipment.updatedAt || ''
            }));
            
            console.log(`✅ Loaded ${equipmentList.length} equipment from MongoDB`);
            updateEquipmentDisplay();
        } else {
            console.warn('No equipment data found in MongoDB');
            equipmentList = [];
            updateEquipmentDisplay();
        }
    } catch (error) {
        console.error('❌ Error loading equipment from MongoDB:', error);
        // Fallback to empty list
        equipmentList = [];
        updateEquipmentDisplay();
    }
}

// Load users from MongoDB
async function loadUsersFromMongoDB() {
    try {
        console.log('🔄 Loading users from MongoDB...');
        const response = await fetch('https://api.titandrillingzm.com:6003/users/all');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('MongoDB Users API Response:', result);
        
        // Process users data
        if (result.success && Array.isArray(result.data)) {
            userList = result.data.map(user => ({
                id: user._id || user.id || user.uid,
                name: user.name || user.displayName || user.fullName || user.username || 'Unknown User',
                email: user.email || user.emailAddress || '',
                role: user.role || user.userRole || user.position || 'User',
                department: user.department || user.dept || '',
                phone: user.phone || user.phoneNumber || user.mobile || '',
                country: user.country || '',
                countries: user.countries || [],
                project: user.project || '',
                projects: user.projects || [],
                status: user.status || user.active || 'Active',
                avatar: user.avatar || user.profilePicture || user.photoURL || ''
            }));
            
            // Also populate taskUserList with the same data
            taskUserList = [...userList];
            
            console.log(`✅ Loaded ${userList.length} users from MongoDB`);
            updateUserDisplay();
        } else {
            console.warn('No user data found in MongoDB');
            userList = [];
            taskUserList = [];
            updateUserDisplay();
        }
    } catch (error) {
        console.error('❌ Error loading users from MongoDB:', error);
        // Fallback to empty list
        userList = [];
        taskUserList = [];
        updateUserDisplay();
    }
}

// Load inspections from MongoDB
async function loadInspectionsFromMongoDB() {
    try {
        console.log('🔄 Loading inspections from MongoDB...');
        const response = await fetch('https://api.titandrillingzm.com:6004/inspections/all');
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('MongoDB Inspections API Response:', result);
        
        // Process inspections data
        if (result.success && Array.isArray(result.data)) {
            inspectionList = result.data.map(inspection => ({
                id: inspection._id || inspection.id,
                name: `Inspection ${inspection.requestNumber || inspection._id}`,
                requestNumber: inspection.requestNumber || inspection._id || 'N/A',
                inspector: inspection.inspector || inspection.inspectorName || inspection.inspectedBy || 'Unknown Inspector',
                date: inspection.inspectionDate || inspection.createdAt || 'Unknown Date',
                status: inspection.status || inspection.inspectionStatus || 'Unknown',
                type: inspection.type || inspection.inspectionType || 'General',
                equipment: inspection.equipment?.equipmentName || inspection.equipmentName || 'Unknown Equipment',
                location: inspection.location || inspection.site || 'Unknown Location',
                description: inspection.description || inspection.notes || inspection.remarks || 'No description available',
                priority: inspection.priority || 'Medium',
                country: inspection.country || '',
                project: inspection.project || ''
            }));
            
            console.log(`✅ Loaded ${inspectionList.length} inspections from MongoDB`);
        } else {
            console.warn('No inspection data found in MongoDB');
            inspectionList = [];
        }
    } catch (error) {
        console.error('❌ Error loading inspections from MongoDB:', error);
        // Fallback to empty list
        inspectionList = [];
    }
}

// Assign to modal functions
function openAssignToModal(event) {
    const modal = document.getElementById('assignToModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Auto-focus search input
    setTimeout(() => {
        const searchInput = document.getElementById('userSearch');
        if (searchInput) {
            searchInput.focus();
        }
    }, 100);
}

function closeAssignToModal() {
    const modal = document.getElementById('assignToModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function filterUsers() {
    const searchTerm = document.getElementById('userSearch').value.toLowerCase();
    const userListDiv = document.getElementById('userList');
    
    userListDiv.innerHTML = '';
    
    userList.filter(user => 
        user.name.toLowerCase().includes(searchTerm) ||
        user.role.toLowerCase().includes(searchTerm) ||
        user.department.toLowerCase().includes(searchTerm)
    ).forEach(user => {
        const userItem = document.createElement('div');
        userItem.className = 'user-item';
        userItem.innerHTML = `
            <div class="user-avatar">
                ${user.avatar ? `<img src="${user.avatar}" alt="${user.name}">` : '<i class="fas fa-user"></i>'}
            </div>
            <div class="user-info">
                <div class="user-name">${user.name}</div>
                <div class="user-role">${user.role} - ${user.department}</div>
            </div>
            <div class="user-checkbox">
                <input type="checkbox" value="${user.id}" onchange="toggleUserSelection(this, '${user.id}')" ${selectedUsers.find(u => u.id === user.id) ? 'checked' : ''}>
            </div>
        `;
        userListDiv.appendChild(userItem);
    });
}

function toggleUserSelection(checkbox, userId) {
    const user = userList.find(u => u.id === userId);
    const existingIndex = selectedUsers.findIndex(u => u.id === userId);
    
    if (checkbox.checked) {
        if (existingIndex === -1) {
            selectedUsers.push(user);
        }
    } else {
        if (existingIndex > -1) {
            selectedUsers.splice(existingIndex, 1);
        }
    }
    
    updateSelectedUsersDisplay();
}

function updateSelectedUsersDisplay() {
    const container = document.getElementById('selectedUsersContainer');
    const list = document.getElementById('selectedUsersList');
    const preview = document.getElementById('usersPreview');
    const previewGrid = document.getElementById('usersPreviewGrid');
    
    if (selectedUsers.length > 0) {
        container.style.display = 'block';
        preview.style.display = 'block';
        list.innerHTML = '';
        previewGrid.innerHTML = '';
        
        // Update selected items display
        selectedUsers.forEach(user => {
            const userTag = document.createElement('div');
            userTag.className = 'selected-item';
            userTag.innerHTML = `
                <span class="selected-item-text">${user.name}</span>
                <button class="remove-btn" onclick="removeUser('${user.id}')">&times;</button>
            `;
            list.appendChild(userTag);
        });
        
        // Update preview grid
        selectedUsers.forEach(user => {
            const userCard = document.createElement('div');
            userCard.className = 'user-preview-card';
            userCard.innerHTML = `
                <div class="user-preview-avatar">
                    ${user.avatar ? `<img src="${user.avatar}" alt="${user.name}">` : '<i class="fas fa-user"></i>'}
                </div>
                <div class="user-preview-info">
                    <div class="user-preview-name">${user.name}</div>
                    <div class="user-preview-role">${user.role} - ${user.department}</div>
                </div>
            `;
            previewGrid.appendChild(userCard);
        });
        
        // Update header count
        document.querySelector('#usersPreview .preview-header h4').textContent = `Selected Users (${selectedUsers.length})`;
        document.getElementById('assignToText').textContent = `${selectedUsers.length} Selected`;
    } else {
        container.style.display = 'none';
        preview.style.display = 'none';
        document.getElementById('assignToText').textContent = 'Select Users';
    }
}

function removeUser(userId) {
    selectedUsers = selectedUsers.filter(u => u.id !== userId);
    updateSelectedUsersDisplay();
    filterUsers(); // Refresh the user list to update checkboxes
}

function saveAssignTo() {
    closeAssignToModal();
}

function updateUserDisplay() {
    const container = document.getElementById('userList');
    if (!container) return;
    
    container.innerHTML = '';
    
    if (userList.length === 0) {
        container.innerHTML = '<div class="no-data">No users available</div>';
        return;
    }
    
    userList.forEach(user => {
        const userItem = document.createElement('div');
        userItem.className = 'user-item';
        userItem.innerHTML = `
            <div class="user-avatar">
                ${user.avatar ? `<img src="${user.avatar}" alt="${user.name}">` : '<i class="fas fa-user"></i>'}
            </div>
            <div class="user-info">
                <div class="user-name">${user.name}</div>
                <div class="user-role">${user.role}</div>
            </div>
            <div class="user-checkbox">
                <input type="checkbox" value="${user.id}" onchange="toggleUserSelection(this, '${user.id}')">
            </div>
        `;
        container.appendChild(userItem);
    });
}

// Category switching
function switchCategory(category) {
    currentCategory = category;
    
    // Update tab buttons
    document.querySelectorAll('.tab-button').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-category="${category}"]`).classList.add('active');
    
    // Update add button text
    document.getElementById('addButtonText').textContent = category;
    
    // Render items for current category
    renderItems();
}

// Item modal functions
function openAddItemModal(event) {
    console.log('Opening Add Item Modal for category:', currentCategory);
    const modal = document.getElementById('addItemModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    showItemForm(currentCategory);
}

function closeAddItemModal() {
    const modal = document.getElementById('addItemModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
    clearItemForm();
}

function showItemForm(category) {
    console.log('Showing form for category:', category);
    // Hide all forms
    document.querySelectorAll('.item-form').forEach(form => {
        form.style.display = 'none';
    });
    
    // Show the appropriate form
    const formId = category.toLowerCase().replace(' ', '') + 'Form';
    console.log('Form ID:', formId);
    const formElement = document.getElementById(formId);
    if (formElement) {
        formElement.style.display = 'block';
        console.log('Form displayed successfully');
    } else {
        console.error('Form element not found:', formId);
    }
    
    // Update modal title and button text
    document.getElementById('addItemModalTitle').textContent = `Add ${category}`;
    document.getElementById('saveItemBtn').textContent = `Save ${category}`;
}

function clearItemForm() {
    console.log('Clearing item form...');
    // Clear all form fields
    document.querySelectorAll('#addItemModal input, #addItemModal textarea, #addItemModal select').forEach(field => {
        if (field.type !== 'button') {
            field.value = '';
        }
    });
    selectedTaskUsers = [];
    updateTaskAssignToDisplay();
    console.log('Form cleared successfully');
}

function openTaskAssignToModal(event) {
    const modal = document.getElementById('taskAssignToModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Auto-focus search input
    setTimeout(() => {
        const searchInput = document.getElementById('taskUserSearch');
        if (searchInput) {
            searchInput.focus();
        }
    }, 100);
    
    filterTaskUsers(); // Load users when modal opens
}

function closeTaskAssignToModal() {
    const modal = document.getElementById('taskAssignToModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function filterTaskUsers() {
    const searchTerm = document.getElementById('taskUserSearch').value.toLowerCase();
    const userListDiv = document.getElementById('taskUserList');
    
    userListDiv.innerHTML = '';
    
    if (taskUserList.length === 0) {
        userListDiv.innerHTML = '<div class="no-data">Loading users...</div>';
        return;
    }
    
    const filteredUsers = taskUserList.filter(user => 
        user.name.toLowerCase().includes(searchTerm) ||
        user.role.toLowerCase().includes(searchTerm) ||
        user.email.toLowerCase().includes(searchTerm) ||
        user.department.toLowerCase().includes(searchTerm)
    );
    
    if (filteredUsers.length === 0) {
        userListDiv.innerHTML = '<div class="no-data">No users match your search</div>';
        return;
    }
    
    filteredUsers.forEach(user => {
        const userItem = document.createElement('div');
        userItem.className = 'select-row';
        userItem.style.marginBottom = '10px';
        userItem.innerHTML = `
            <div>
                <div style="font-weight: 600; color: #015185;">${user.name}</div>
                <div style="font-size: 12px; color: #666;">${user.role} - ${user.department}</div>
                <div style="font-size: 11px; color: #999;">${user.email}</div>
            </div>
            <div>
                <input type="checkbox" onchange="toggleTaskUserSelection('${user.id}')" ${selectedTaskUsers.find(u => u.id === user.id) ? 'checked' : ''}>
            </div>
        `;
        userListDiv.appendChild(userItem);
    });
}

function toggleTaskUserSelection(userId) {
    const user = taskUserList.find(u => u.id === userId);
    const existingIndex = selectedTaskUsers.findIndex(u => u.id === userId);
    
    if (existingIndex > -1) {
        selectedTaskUsers.splice(existingIndex, 1);
    } else {
        selectedTaskUsers.push(user);
    }
    
    updateTaskAssignToDisplay();
}

function updateTaskAssignToDisplay() {
    const text = document.getElementById('taskAssignToText');
    if (selectedTaskUsers.length > 0) {
        text.textContent = `${selectedTaskUsers.length} Selected`;
    } else {
        text.textContent = 'Select Users';
    }
}

function saveTaskAssignTo() {
    closeTaskAssignToModal();
}

function saveItem() {
    let item = {
        id: Date.now(),
        category: currentCategory,
        status: 'Initiated',
        statusClass: 'initiated'
    };

    // Validate and collect data based on category
    switch (currentCategory) {
        case 'Task':
            const taskDescription = document.getElementById('taskDescription').value;
            if (!taskDescription.trim()) {
                alert('Please enter a task description');
                return;
            }
            item.description = taskDescription;
            item.type = document.getElementById('taskType').value;
            item.notes = document.getElementById('taskNotes').value;
            item.assignees = [...selectedTaskUsers];
            item.assignedTo = [...selectedTaskUsers]; // For Firebase compatibility
            break;

        case 'Parts':
            const partsItemName = document.getElementById('partsItemName').value;
            if (!partsItemName.trim()) {
                alert('Please enter an item name');
                return;
            }
            item.itemNumber = document.getElementById('partsItemNumber').value;
            item.partNumber = document.getElementById('partsPartNumber').value;
            item.itemName = partsItemName;
            item.quantity = document.getElementById('partsQuantity').value;
            item.unit = document.getElementById('partsUnit').value;
            item.itemCost = document.getElementById('partsItemCost').value;
            item.site = document.getElementById('partsSite').value;
            item.note = document.getElementById('partsNote').value;
            break;

        case 'Labor':
            const laborWorker = document.getElementById('laborWorker').value;
            if (!laborWorker.trim()) {
                alert('Please enter a worker name');
                return;
            }
            item.worker = laborWorker;
            item.code = document.getElementById('laborCode').value;
            item.rate = document.getElementById('laborRate').value;
            item.hours = document.getElementById('laborHours').value;
            item.minutes = document.getElementById('laborMinutes').value;
            item.note = document.getElementById('laborNote').value;
            break;

        case 'Additional Cost':
            const additionalCostDescription = document.getElementById('additionalCostDescription').value;
            const additionalCostAmount = document.getElementById('additionalCostAmount').value;
            
            console.log('=== SAVING ADDITIONAL COST ITEM ===');
            console.log('Description:', additionalCostDescription);
            console.log('Amount:', additionalCostAmount);
            console.log('Current Category:', currentCategory);
            console.log('Current Items Count:', items.length);
            
            if (!additionalCostDescription.trim()) {
                alert('Please enter a cost description');
                return;
            }
            if (!additionalCostAmount || parseFloat(additionalCostAmount) <= 0) {
                alert('Please enter a valid amount');
                return;
            }
            item.description = additionalCostDescription.trim();
            item.amount = parseFloat(additionalCostAmount).toFixed(2);
            break;

        case 'Remarks':
            const remarksDescription = document.getElementById('remarksDescription').value;
            if (!remarksDescription.trim()) {
                alert('Please enter a remarks description');
                return;
            }
            item.description = remarksDescription;
            break;
    }

    items.push(item);
    console.log('New Item Added:', item);
    console.log('Updated Items Array:', items);
    renderItems();
    closeAddItemModal();
    
    // Show success message
    if (currentCategory === 'Additional Cost') {
        showToast('success', 'Success', 'Additional cost added successfully!');
    } else {
        showToast('success', 'Success', 'Item added successfully!');
    }
}

function renderItems() {
    const container = document.getElementById('itemsListContainer');
    container.innerHTML = '';
    
    // Filter items by current category
    const categoryItems = items.filter(item => item.category === currentCategory);
    
    if (categoryItems.length === 0) {
        container.innerHTML = `<div style="text-align: center; color: #666; padding: 20px;">No ${currentCategory.toLowerCase()} added yet.</div>`;
        return;
    }
    
    categoryItems.forEach(item => {
        const itemCard = document.createElement('div');
        itemCard.className = 'task-card';
        itemCard.setAttribute('data-category', item.category);
        
        const statusClass = item.status ? item.status.toLowerCase().replace(/\s+/g, '') : 'initiated';
        const statusText = item.status || 'Initiated';
        
        let cardContent = `
            <div class="task-card-header">
                <div class="task-card-title">${getItemTitle(item)}</div>
                <div class="task-card-status-container">
                    <button class="task-status-btn" onclick="toggleItemStatusDropdown(${item.id})">
                        <div class="task-status-indicator ${statusClass}"></div>
                        <span>${statusText}</span>
                        <i class="fas fa-chevron-down task-status-arrow"></i>
                    </button>
                    <div class="task-status-dropdown" id="taskStatusDropdown${item.id}">
                        <div class="task-status-option" onclick="selectItemStatus(${item.id}, 'Initiated', 'initiated')">
                            <div class="task-status-indicator initiated"></div>
                            <span>Initiated</span>
                        </div>
                        <div class="task-status-option" onclick="selectItemStatus(${item.id}, 'In Progress', 'inprogress')">
                            <div class="task-status-indicator inprogress"></div>
                            <span>In Progress</span>
                        </div>
                        <div class="task-status-option" onclick="selectItemStatus(${item.id}, 'On Hold', 'onhold')">
                            <div class="task-status-indicator onhold"></div>
                            <span>On Hold</span>
                        </div>
                        <div class="task-status-option" onclick="selectItemStatus(${item.id}, 'Complete', 'complete')">
                            <div class="task-status-indicator complete"></div>
                            <span>Complete</span>
                        </div>
                    </div>
                </div>
                <button class="task-menu-btn" onclick="deleteItem(${item.id})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
            <div style="margin-top: 10px;">
                ${getItemDetails(item)}
            </div>
        `;
        
        itemCard.innerHTML = cardContent;
        container.appendChild(itemCard);
    });
}

function getItemTitle(item) {
    switch (item.category) {
        case 'Task':
            return item.description;
        case 'Parts':
            return item.itemName || 'Unnamed Part';
        case 'Labor':
            return item.worker || 'Unnamed Worker';
        case 'Additional Cost':
            return item.description;
        case 'Remarks':
            return item.description;
        default:
            return 'Unknown Item';
    }
}

function getItemDetails(item) {
    let details = [];
    
    switch (item.category) {
        case 'Task':
            details.push(`Type: ${item.type || 'N/A'}`);
            details.push(`Assignees: ${item.assignees ? item.assignees.length : 0}`);
            if (item.notes) details.push(`Notes: ${item.notes}`);
            break;
        case 'Parts':
            if (item.quantity) details.push(`Qty: ${item.quantity} ${item.unit || ''}`);
            if (item.itemCost) details.push(`Cost: $${item.itemCost}`);
            if (item.site) details.push(`Site: ${item.site}`);
            if (item.note) details.push(`Note: ${item.note}`);
            break;
        case 'Labor':
            if (item.hours || item.minutes) details.push(`Time: ${item.hours || 0}h ${item.minutes || 0}m`);
            if (item.rate) details.push(`Rate: $${item.rate}`);
            if (item.note) details.push(`Note: ${item.note}`);
            break;
        case 'Additional Cost':
            if (item.amount) details.push(`Amount: $${item.amount}`);
            break;
        case 'Remarks':
            // No additional details for remarks
            break;
    }
    
    return details.map(detail => `<div style="font-size: 12px; color: #666; margin-bottom: 3px;">${detail}</div>`).join('');
}

function deleteItem(itemId) {
    items = items.filter(item => item.id !== itemId);
    renderItems();
}

// Form validation
function validateForm() {
    let isValid = true;
    
    // Clear previous errors
    document.querySelectorAll('.error-message').forEach(error => {
        error.classList.remove('show');
    });
    
    // Validate title
    const title = document.getElementById('workOrderTitle').value.trim();
    if (!title) {
        document.getElementById('titleError').classList.add('show');
        isValid = false;
    }
    
    // Validate country
    const country = document.getElementById('country').value;
    if (!country) {
        document.getElementById('countryError').classList.add('show');
        isValid = false;
    }
    
    // Validate project
    const project = document.getElementById('project').value;
    if (!project) {
        document.getElementById('projectError').classList.add('show');
        isValid = false;
    }
    
    // Validate equipment
    if (!selectedEquipment) {
        document.getElementById('equipmentError').classList.add('show');
        isValid = false;
    }
    
    return isValid;
}

// Save work order
function saveWorkOrder() {
    if (!validateForm()) {
        return;
    }
    
    // Show loading modal
    document.getElementById('loadingModal').style.display = 'flex';
    
    // Collect all form data - ensure ALL fields from JSON structure are included
    const workOrderData = {
        title: document.getElementById('workOrderTitle').value,
        country: document.getElementById('country').value,
        project: document.getElementById('project').value,
        memo: document.getElementById('memo').value,
        priority: window.selectedPriority || 'Medium',
        status: 'Initiated',
        workOrderType: selectedJobCardType || 'General',
        estimatedCost: document.getElementById('estimatedCost').value,
        estimatedLaborTime: document.getElementById('laborTimeHours').value,
        additionalCost: document.getElementById('additionalCost').value,
        additionalDetails: document.getElementById('memo').value, // Same as memo for now
        partMaterialCost: document.getElementById('partMaterialCost').value,
        totalLaborCost: document.getElementById('totalLaborCost').value,
        taxAmount: document.getElementById('taxAmount').value,
        taxPercentage: document.getElementById('taxPercentage').value,
        totalWoCost: document.getElementById('totalWOCost').value,
        equipment: selectedEquipment,
        requestNumber: document.getElementById('requestNumber').textContent,
        createdAt: new Date().toISOString()
    };
    
    console.log('Work Order Data:', workOrderData);
    
    // Save to MongoDB and generate PDF
    saveToMongoDB(workOrderData);
}

// Save job card to MongoDB
// ============================================================================
// CORRECTED PDF STORAGE FLOW:
// 1. Generate PDF blob locally (frontend)
// 2. Upload PDF to backend filesystem (uploads/documents/GlobalJobcardReports/{adminUid}/{fileName}.pdf)
// 3. Backend returns download URL (https://api.titandrillingzm.com:6005/jobcards/download-file/{adminUid}/{fileName})
// 4. Create job card in MongoDB WITH pdfDownloadUrl field already populated
// 
// This ensures the job card ALWAYS has the PDF URL from creation, no need to update later.
// ============================================================================
async function saveToMongoDB(workOrderData) {
    try {
        console.log('💾 Starting job card creation process...');
        console.log('📋 Flow: Generate PDF → Upload PDF → Get URL → Create Job Card with PDF URL');
        
        // Show loading modal with initial message
        updateLoadingModal('Initializing...', 'Preparing job card data...');
        
        // Generate unique ID for the job card
        const jobCardId = Date.now().toString();
        
        // STEP 1: Generate PDF FIRST (before saving job card)
        console.log('📄 STEP 1: Generating PDF...');
        updateLoadingModal('Generating PDF...', 'Creating job card document...');
        const pdfBlob = await generatePDFBlob(workOrderData);
        console.log('✅ PDF generated, size:', pdfBlob.size, 'bytes');
        
        // STEP 1.5: VERIFY PDF locally (optional download for debugging)
        console.log('🧪 Creating local verification copy...');
        const localBlobUrl = URL.createObjectURL(pdfBlob);
        window.debugPDFBlobUrl = localBlobUrl; // Store for manual testing
        console.log('   - Local PDF URL (paste in browser to test):', localBlobUrl);
        console.log('   - PDF will be auto-downloaded to verify content...');
        
        // Auto-download PDF locally for immediate verification
        const testLink = document.createElement('a');
        testLink.href = localBlobUrl;
        testLink.download = `LOCAL_VERIFY_${workOrderData.requestNumber}.pdf`;
        testLink.style.display = 'none';
        document.body.appendChild(testLink);
        testLink.click();
        document.body.removeChild(testLink);
        console.log('✅ Local verification PDF downloaded - check your Downloads folder!');
        
        // STEP 2: Upload PDF to MongoDB storage
        console.log('📤 STEP 2: Uploading PDF to MongoDB storage...');
        updateLoadingModal('Uploading PDF...', 'Storing document on server...');
        const pdfFileName = `JobCard_${workOrderData.requestNumber || jobCardId}.pdf`;
        const pdfDownloadUrl = await uploadPDFToMongoDB(pdfBlob, pdfFileName, jobCardId);
        console.log('✅ PDF uploaded successfully, URL:', pdfDownloadUrl);
        
        // Verify we have a valid download URL
        if (!pdfDownloadUrl || !pdfDownloadUrl.startsWith('http')) {
            throw new Error('Invalid PDF download URL received from server');
        }
        
        // Store PDF references globally
        window.generatedPDFBlob = pdfBlob;
        window.generatedPDFFileName = pdfFileName;
        window.generatedPDFBlobUrl = URL.createObjectURL(pdfBlob);
        window.generatedPDFUrl = pdfDownloadUrl;
        
        // STEP 3: Create job card WITH PDF URL included
        console.log('💾 STEP 3: Creating job card in MongoDB with PDF URL...');
        updateLoadingModal('Saving Job Card...', 'Creating database record...');
        
        const jobCardData = {
            _id: jobCardId,
            title: workOrderData.title || "",
            status: "Hold",
            priority: workOrderData.priority || "Medium",
            workOrderType: workOrderData.workOrderType || "General",
            
            // Equipment information
            equipment: workOrderData.equipment ? {
                equipmentName: workOrderData.equipment.name || '',
                mainCategory: workOrderData.equipment.mainCategory || workOrderData.equipment.type || '',
                model: workOrderData.equipment.model || '',
                vehicleNumber: workOrderData.equipment.name || '',
                meterReading: parseFloat(workOrderData.equipment.meterReading) || 0,
                status: workOrderData.equipment.status || 'Active'
            } : null,
            
            // Assignment information
            assignTo: selectedUsers.map(user => ({
                name: user.name || '',
                email: user.email || '',
                uid: user.id || ''
            })),
            
            // Dates
            startDate: document.getElementById('startDate')?.value || null,
            dueDate: document.getElementById('dueDate')?.value || null,
            
            // Description and notes
            description: workOrderData.memo || "",
            notes: workOrderData.memo || "",
            
            // Location information
            location: {
                country: workOrderData.country || "",
                project: workOrderData.project || "",
                coordinates: "",
                address: ""
            },
            
            // Request number
            requestNumber: workOrderData.requestNumber || "",
            
            // Tasks/Items
            items: items || [],
            
            // PDF URL - INCLUDED FROM THE START
            pdfDownloadUrl: pdfDownloadUrl,
            
            // Metadata
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString(),
            createdBy: ADMIN_UID,
            lastModifiedBy: ADMIN_UID
        };
        
        console.log('📊 Job card data being sent to MongoDB:');
        console.log('   - ID:', jobCardData._id);
        console.log('   - Title:', jobCardData.title);
        console.log('   - Request Number:', jobCardData.requestNumber);
        console.log('   - PDF URL:', jobCardData.pdfDownloadUrl);
        console.log('   - Items count:', jobCardData.items.length);
        
        // Save to MongoDB
        const response = await fetch(`${MONGODB_API_BASE}/jobcards`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                adminUid: ADMIN_UID,
                jobCardData: jobCardData
            })
        });
        
        console.log('📊 MongoDB save response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ MongoDB save failed:', errorText);
            throw new Error(`Failed to save job card: ${response.status} - ${errorText}`);
        }
        
        const result = await response.json();
        console.log('✅ Job card saved to MongoDB:', result);
        
        // Verify the response
        if (!result.success) {
            throw new Error(result.error || 'Failed to save job card');
        }
        
        // Verify PDF URL was saved correctly
        if (result.data && result.data.pdfDownloadUrl) {
            console.log('✅✅ PDF URL VERIFIED in MongoDB:', result.data.pdfDownloadUrl);
            console.log('📁 PDF stored at: uploads/documents/GlobalJobcardReports/' + ADMIN_UID + '/' + pdfFileName);
        } else {
            console.warn('⚠️ Job card saved but PDF URL not found in response');
        }
        
        // Hide loading modal
        document.getElementById('loadingModal').style.display = 'none';
        
        // Show success modal
        console.log('✅ All steps completed successfully! Showing success modal...');
        showSuccessModal();
        
        showToast('success', 'Success', 'Job card created with PDF successfully!');
        
    } catch (error) {
        console.error('❌ Error in saveToMongoDB:', error);
        console.error('❌ Error stack:', error.stack);
        
        // Hide loading modal
        document.getElementById('loadingModal').style.display = 'none';
        
        // Show error with more details
        const errorMessage = error.message || 'Unknown error occurred';
        showToast('error', 'Error', 'Failed to create job card: ' + errorMessage);
        
        // Show alert for critical errors
        alert('Error creating job card:\n\n' + errorMessage + '\n\nPlease check the console for more details.');
    }
}

// Update loading modal text
function updateLoadingModal(title, subtext) {
    const titleEl = document.getElementById('loadingTitle');
    const subtextEl = document.getElementById('loadingSubtext');
    
    if (titleEl) titleEl.textContent = title;
    if (subtextEl) subtextEl.textContent = subtext;
    
    console.log('📊 Loading: ' + title + ' - ' + subtext);
}

// Show toast notification
function showToast(type, title, message) {
    // Create toast element
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.innerHTML = `
        <div class="toast-content">
            <div class="toast-title">${title}</div>
            <div class="toast-message">${message}</div>
        </div>
        <button class="toast-close" onclick="this.parentElement.remove()">&times;</button>
    `;
    
    // Add toast styles if not already added
    if (!document.getElementById('toast-styles')) {
        const style = document.createElement('style');
        style.id = 'toast-styles';
        style.textContent = `
            .toast {
                position: fixed;
                top: 20px;
                right: 20px;
                background: white;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                padding: 16px;
                min-width: 300px;
                z-index: 10000;
                display: flex;
                align-items: center;
                justify-content: space-between;
                animation: slideIn 0.3s ease;
            }
            .toast-success {
                border-left: 4px solid #28a745;
            }
            .toast-error {
                border-left: 4px solid #dc3545;
            }
            .toast-warning {
                border-left: 4px solid #ffc107;
            }
            .toast-info {
                border-left: 4px solid #17a2b8;
            }
            .toast-content {
                flex: 1;
            }
            .toast-title {
                font-weight: 600;
                color: #333;
                margin-bottom: 4px;
            }
            .toast-message {
                color: #666;
                font-size: 14px;
            }
            .toast-close {
                background: none;
                border: none;
                font-size: 18px;
                color: #999;
                cursor: pointer;
                margin-left: 12px;
            }
            @keyframes slideIn {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `;
        document.head.appendChild(style);
    }
    
    // Add to page
    document.body.appendChild(toast);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (toast.parentElement) {
            toast.remove();
        }
    }, 5000);
}

// Generate PDF blob (returns blob, doesn't upload)
async function generatePDFBlob(workOrderData) {
    try {
        console.log('📄 Generating PDF blob for job card...');
        console.log('📊 Work Order Data:', workOrderData);
        
        const htmlContent = generatePDFHTML(workOrderData);
        console.log('✅ Generated HTML Content Length:', htmlContent.length);
        
        if (htmlContent.length < 500) {
            throw new Error('Generated HTML is too short, might be invalid');
        }
        
        // Create a temporary container with the full HTML
        const tempContainer = document.createElement('div');
        tempContainer.id = 'pdf-content-container';
        tempContainer.innerHTML = htmlContent;
        
        // Style the container - make it VISIBLE and ON-SCREEN for proper capture
        tempContainer.style.cssText = `
            position: absolute;
            left: 0;
            top: 0;
            width: 794px;
            min-height: 1123px;
            background: white;
            padding: 20px;
            font-family: Arial, Helvetica, sans-serif;
            font-size: 12px;
            color: #000000;
            z-index: 999999;
            visibility: visible;
            display: block;
            box-sizing: border-box;
        `;
        
        document.body.appendChild(tempContainer);
        console.log('✅ Temp container appended to body');
        console.log('   - Container width:', tempContainer.offsetWidth);
        console.log('   - Container height:', tempContainer.offsetHeight);
        console.log('   - Container visibility:', window.getComputedStyle(tempContainer).visibility);
        console.log('   - Container display:', window.getComputedStyle(tempContainer).display);
        console.log('   - Container innerHTML length:', tempContainer.innerHTML.length);
        
        // CRITICAL: Wait longer for content to fully render (especially for images)
        console.log('⏳ Waiting for content to render (3 seconds)...');
        await new Promise(resolve => setTimeout(resolve, 3000));
        
        console.log('✅ Content should be rendered now');
        
        // Get actual content dimensions
        const actualHeight = Math.max(
            tempContainer.scrollHeight,
            tempContainer.offsetHeight,
            tempContainer.clientHeight
        );
        
        console.log('📐 Container measurements:');
        console.log('   - scrollHeight:', tempContainer.scrollHeight);
        console.log('   - offsetHeight:', tempContainer.offsetHeight);
        console.log('   - clientHeight:', tempContainer.clientHeight);
        console.log('   - Using height:', actualHeight);
        console.log('   - Width:', tempContainer.offsetWidth);
        
        // Configure html2pdf options - FIXED for width and height
        const opt = {
            margin: [10, 10, 10, 10],
            filename: `JobCard_${workOrderData.requestNumber}.pdf`,
            image: { type: 'jpeg', quality: 0.95 },
            html2canvas: { 
                scale: 2,
                useCORS: true,
                logging: true,
                letterRendering: true,
                allowTaint: false,
                backgroundColor: '#ffffff',
                width: 794,
                height: actualHeight,
                windowWidth: 794,
                windowHeight: actualHeight,
                scrollY: 0,
                scrollX: 0,
                x: 0,
                y: 0,
                onclone: function(clonedDoc) {
                    console.log('📋 Document cloned for PDF generation');
                    const clonedContainer = clonedDoc.getElementById('pdf-content-container');
                    if (clonedContainer) {
                        clonedContainer.style.width = '794px';
                        clonedContainer.style.maxWidth = '794px';
                        clonedContainer.style.minWidth = '794px';
                        clonedContainer.style.height = 'auto';
                        clonedContainer.style.overflow = 'visible';
                        clonedContainer.style.display = 'block';
                        console.log('   - Cloned container fixed to 794px width');
                    }
                }
            },
            jsPDF: { 
                unit: 'mm',
                format: 'a4',
                orientation: 'portrait',
                compress: true
            },
            pagebreak: { mode: ['avoid-all', 'css', 'legacy'], avoid: 'tr' }
        };
        
        console.log('🔄 Converting HTML to PDF with html2pdf...');
        console.log('   - Container dimensions:', tempContainer.offsetWidth, 'x', tempContainer.offsetHeight);
        
        // Generate PDF blob
        const pdfBlob = await html2pdf()
            .set(opt)
            .from(tempContainer)
            .toPdf()
            .output('blob');
        
        console.log('✅ PDF blob generated successfully!');
        console.log('   - Size:', pdfBlob.size, 'bytes');
        console.log('   - Type:', pdfBlob.type);
        
        // CRITICAL: Verify PDF blob content is valid (check PDF header)
        console.log('🔍 Verifying PDF blob content...');
        const arrayBuffer = await pdfBlob.slice(0, 5).arrayBuffer();
        const headerBytes = new Uint8Array(arrayBuffer);
        const headerString = String.fromCharCode(...headerBytes);
        console.log('   - PDF Header:', headerString, '(should be "%PDF-")');
        
        if (!headerString.startsWith('%PDF-')) {
            console.error('❌ Invalid PDF header detected!');
            console.error('   - Expected: %PDF-');
            console.error('   - Got:', headerString);
            throw new Error('Generated blob is not a valid PDF (invalid header)');
        }
        
        console.log('✅ PDF blob is valid (has correct PDF header)');
        
        // Remove temporary container
        if (document.body.contains(tempContainer)) {
            document.body.removeChild(tempContainer);
            console.log('✅ Temp container removed');
        }
        
        // Verify PDF blob size
        if (!pdfBlob || pdfBlob.size === 0) {
            throw new Error('PDF generation failed: Blob is empty');
        }
        
        if (pdfBlob.size < 5000) {
            console.warn('⚠️ PDF blob is small (', pdfBlob.size, 'bytes), might be incomplete');
            // Don't throw error, just warn - some minimal PDFs can be small
        }
        
        console.log('✅ PDF blob validation complete - ready for upload');
        
        return pdfBlob;
        
    } catch (error) {
        console.error('❌ Error generating PDF blob:', error);
        console.error('❌ PDF generation error details:', {
            message: error.message,
            stack: error.stack
        });
        throw new Error('PDF generation failed: ' + error.message);
    }
}

// Upload PDF to MongoDB storage (returns download URL)
async function uploadPDFToMongoDB(pdfBlob, fileName, jobCardId) {
    try {
        console.log('📤 Uploading PDF to MongoDB storage:', fileName);
        console.log('📊 PDF Blob details:');
        console.log('   - Size:', pdfBlob.size, 'bytes');
        console.log('   - Type:', pdfBlob.type);
        console.log('   - File name:', fileName);
        console.log('   - Admin UID:', ADMIN_UID);
        
        // Ensure the blob is a valid PDF
        if (!pdfBlob || pdfBlob.size === 0) {
            throw new Error('PDF blob is empty or invalid. PDF generation may have failed.');
        }
        
        // Verify blob size is reasonable (not too small)
        if (pdfBlob.size < 1000) {
            throw new Error(`PDF blob is too small (${pdfBlob.size} bytes). PDF may be invalid.`);
        }
        
        // CRITICAL: Test the blob before upload by creating a temporary URL
        const testBlobUrl = URL.createObjectURL(pdfBlob);
        console.log('🧪 Test blob URL created:', testBlobUrl);
        console.log('   - You can test this URL in browser to verify PDF content');
        
        // Wait a bit to ensure blob is fully ready
        await new Promise(resolve => setTimeout(resolve, 500));
        
        // Create FormData for file upload - ENSURE BLOB IS APPENDED CORRECTLY
        const formData = new FormData();
        
        // Method 1: Append blob directly (most reliable)
        formData.append('pdf', pdfBlob, fileName);
        
        console.log('📤 FormData prepared:');
        console.log('   - Has pdf field:', formData.has('pdf'));
        console.log('   - FormData entries:', Array.from(formData.entries()).map(([key, value]) => ({
            key,
            type: value instanceof Blob ? 'Blob' : value instanceof File ? 'File' : typeof value,
            size: value.size || 'N/A'
        })));
        
        const uploadUrl = `${MONGODB_API_BASE}/jobcards/upload-pdf/${ADMIN_UID}`;
        console.log('📤 Uploading to:', uploadUrl);
        
        // IMPORTANT: Don't set Content-Type header, let browser set it with boundary
        const response = await fetch(uploadUrl, {
            method: 'POST',
            body: formData
            // DO NOT SET HEADERS - Browser will auto-set multipart/form-data with boundary
        });
        
        console.log('📊 Upload response status:', response.status);
        console.log('📊 Upload response status text:', response.statusText);
        
        // Clean up test blob URL
        URL.revokeObjectURL(testBlobUrl);
        
        if (!response.ok) {
            let errorText = 'Unknown error';
            try {
                errorText = await response.text();
            } catch (e) {
                errorText = response.statusText;
            }
            console.error('❌ Upload failed with error:', errorText);
            throw new Error(`PDF upload failed (${response.status}): ${errorText}`);
        }
        
        const result = await response.json();
        console.log('✅ PDF upload response:', result);
        
        if (result.success && result.downloadUrl) {
            console.log('✅ PDF stored successfully!');
            console.log('   - Download URL:', result.downloadUrl);
            console.log('   - Backend path:', result.filePath);
            console.log('   - File name:', result.fileName);
            
            // VERIFY: Try to access the PDF immediately to check if it's valid
            console.log('🧪 Verifying uploaded PDF is accessible...');
            try {
                const verifyResponse = await fetch(result.downloadUrl, { method: 'HEAD' });
                console.log('   - File exists on server:', verifyResponse.ok);
                console.log('   - Content-Type:', verifyResponse.headers.get('Content-Type'));
                console.log('   - Content-Length:', verifyResponse.headers.get('Content-Length'));
                
                if (!verifyResponse.ok) {
                    console.warn('⚠️ Uploaded file may not be accessible yet');
                }
            } catch (verifyError) {
                console.warn('⚠️ Could not verify uploaded file:', verifyError.message);
            }
            
            // Verify the URL is valid
            if (!result.downloadUrl.startsWith('http://') && !result.downloadUrl.startsWith('https://')) {
                throw new Error('Invalid PDF download URL format: ' + result.downloadUrl);
            }
            
            return result.downloadUrl;
        } else {
            const errorMsg = result.error || result.message || 'No download URL returned';
            throw new Error('PDF upload failed: ' + errorMsg);
        }
        
    } catch (error) {
        console.error('❌ Error uploading PDF to MongoDB storage:', error);
        console.error('❌ Upload error details:', {
            message: error.message,
            stack: error.stack
        });
        throw new Error('PDF upload failed: ' + error.message);
    }
}

// Test API Connection
async function testAPIConnection() {
    try {
        console.log('🧪 Testing MongoDB API connection...');
        const response = await fetch(`${MONGODB_API_BASE}/jobcards`);
        
        if (!response.ok) {
            throw new Error(`API test failed: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('✅ API Connection Test Result:', result);
        
        if (result.status === 'OK') {
            console.log('✅ MongoDB Job Card API is online and responding');
            return true;
        } else {
            console.warn('⚠️ API responded but with unexpected status');
            return false;
        }
    } catch (error) {
        console.error('❌ API Connection Test Failed:', error);
        return false;
    }
}

// Success Modal Functions (matching create incident file)
function showSuccessModal() {
    console.log('🎉 Showing success modal...');
    const successModal = document.getElementById('successModal');
    
    if (!successModal) {
        console.error('❌ Success modal element not found!');
        return;
    }
    
    // Ensure all other modals are hidden
    document.getElementById('loadingModal').style.display = 'none';
    
    // Show success modal
    successModal.style.display = 'block';
    successModal.style.opacity = '0';
    successModal.style.transition = 'opacity 0.3s ease';
    
    // Fade in the success modal
    setTimeout(() => {
        successModal.style.opacity = '1';
        console.log('✅ Success modal displayed');
    }, 10);
}

function hideSuccessModal() {
    const successModal = document.getElementById('successModal');
    if (successModal) {
        successModal.style.display = 'none';
    }
}

// View PDF function
function viewPDF() {
    console.log('👁️ View PDF clicked');
    console.log('   - Blob URL available:', !!window.generatedPDFBlobUrl);
    console.log('   - MongoDB URL available:', !!window.generatedPDFUrl);
    console.log('   - Blob available:', !!window.generatedPDFBlob);
    
    // PRIORITY 1: Try MongoDB URL (persistent, can be shared)
    if (window.generatedPDFUrl) {
        console.log('📂 Opening PDF from MongoDB storage URL');
        console.log('   - URL:', window.generatedPDFUrl);
        
        // Validate URL format
        if (!window.generatedPDFUrl.startsWith('http://') && !window.generatedPDFUrl.startsWith('https://')) {
            console.error('❌ Invalid PDF URL format:', window.generatedPDFUrl);
            showToast('error', 'Error', 'Invalid PDF URL format.');
            return;
        }
        
        try {
            const pdfWindow = window.open(window.generatedPDFUrl, '_blank');
            
            if (!pdfWindow || pdfWindow.closed || typeof pdfWindow.closed == 'undefined') {
                console.warn('⚠️ Popup was blocked');
                showToast('warning', 'Popup Blocked', 'Please allow popups to view PDF');
                
                // Offer download as alternative
                setTimeout(() => {
                    if (confirm('Popup was blocked. Would you like to download the PDF instead?')) {
                        downloadPDF();
                    }
                }, 100);
                return;
            }
            
            pdfWindow.focus();
            console.log('✅ PDF opened successfully from MongoDB URL');
            return;
        } catch (error) {
            console.error('❌ Error opening PDF from MongoDB URL:', error);
            showToast('error', 'Error', 'Failed to open PDF: ' + error.message);
        }
    }
    
    // PRIORITY 2: Try local blob URL (temporary, only current session)
    if (window.generatedPDFBlobUrl) {
        console.log('📂 Opening PDF from local blob URL');
        try {
            const pdfWindow = window.open(window.generatedPDFBlobUrl, '_blank');
            
            if (!pdfWindow || pdfWindow.closed || typeof pdfWindow.closed == 'undefined') {
                showToast('warning', 'Popup Blocked', 'Please allow popups to view PDF');
                setTimeout(() => {
                    if (confirm('Popup was blocked. Would you like to download the PDF instead?')) {
                        downloadPDF();
                    }
                }, 100);
                return;
            }
            
            pdfWindow.focus();
            console.log('✅ PDF opened successfully from blob URL');
            return;
        } catch (error) {
            console.error('❌ Error opening PDF from blob URL:', error);
        }
    }
    
    // PRIORITY 3: Create new blob URL from blob
    if (window.generatedPDFBlob) {
        console.log('📂 Creating new blob URL from PDF blob');
        const blobUrl = URL.createObjectURL(window.generatedPDFBlob);
        const pdfWindow = window.open(blobUrl, '_blank');
        
        if (pdfWindow) {
            pdfWindow.focus();
            // Clean up blob URL after 1 minute
            setTimeout(() => {
                URL.revokeObjectURL(blobUrl);
            }, 60000);
            console.log('✅ PDF opened from newly created blob URL');
            return;
        }
    }
    
    // No PDF available
    console.error('❌ No PDF available to view');
    showToast('error', 'Error', 'PDF not available. Please try creating the job card again.');
}

// Download PDF function
function downloadPDF() {
    try {
        if (window.generatedPDFBlob) {
            // Create a blob URL for download
            const blobUrl = URL.createObjectURL(window.generatedPDFBlob);
            const link = document.createElement('a');
            link.href = blobUrl;
            link.download = window.generatedPDFFileName || 'JobCard.pdf';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            // Clean up the blob URL after download
            setTimeout(() => {
                URL.revokeObjectURL(blobUrl);
            }, 1000);
            
            showToast('success', 'Success', 'PDF download has started');
        } else if (window.generatedPDFUrl) {
            // Fallback to URL download
            const link = document.createElement('a');
            link.href = window.generatedPDFUrl;
            link.download = window.generatedPDFFileName || 'JobCard.pdf';
            link.target = '_blank';
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            showToast('success', 'Success', 'PDF download has started');
        } else {
            showToast('error', 'Error', 'No PDF available for download');
        }
    } catch (error) {
        console.error('❌ Error downloading PDF:', error);
        showToast('error', 'Error', 'Failed to download PDF: ' + error.message);
    }
}

// Go back to Work Orders list
function goBackToWorkOrders() {
    // Clear all session data and revoke blob URLs
    if (window.generatedPDFBlobUrl) {
        URL.revokeObjectURL(window.generatedPDFBlobUrl);
    }
    
    window.generatedPDFUrl = null;
    window.generatedPDFFileName = null;
    window.generatedPDFBlob = null;
    window.generatedPDFBlobUrl = null;
    
    // Navigate to work orders page
    window.location.href = "{{ route('maintenance.workorder') }}";
}

function generatePDFHTML(workOrderData) {
    console.log('📄 Generating PDF HTML for:', workOrderData);
    
    // Get items from the global items array
    const allItems = items || [];
    console.log('📊 All items for PDF:', allItems);
    console.log('📊 Items breakdown:', {
        tasks: allItems.filter(i => i.category === 'Task').length,
        labor: allItems.filter(i => i.category === 'Labor').length,
        parts: allItems.filter(i => i.category === 'Parts').length,
        additionalCosts: allItems.filter(i => i.category === 'Additional Cost').length,
        remarks: allItems.filter(i => i.category === 'Remarks').length
    });
    
    // Get form values safely
    const startDate = document.getElementById('startDate') ? document.getElementById('startDate').value : 'N/A';
    const dueDate = document.getElementById('dueDate') ? document.getElementById('dueDate').value : 'N/A';
    
    // Ensure we have the selected equipment data
    const equipmentData = selectedEquipment || workOrderData.equipment || null;
    console.log('Equipment data for PDF:', equipmentData);
    
    // Ensure we have the selected users data
    const usersData = selectedUsers || workOrderData.assignTo || [];
    console.log('Users data for PDF:', usersData);
    
    // Format current date and time
    const currentDate = new Date().toLocaleDateString('en-US');
    const currentDateTime = new Date().toLocaleString('en-US');
    
    // Generate task rows with INLINE styles - COMPACT with word-wrap for PDF
    const taskRows = allItems.filter(item => item.category === 'Task').map((item, index) => `
        <tr>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word; overflow-wrap: break-word;">${index + 1}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word; overflow-wrap: break-word;">${item.description || item.text || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word; overflow-wrap: break-word;">${item.type || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word; overflow-wrap: break-word;">${item.status || 'Initiated'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word; overflow-wrap: break-word;">${item.notes || item.note || 'N/A'}</td>
        </tr>
    `).join('');
    
    const tasksTableContent = taskRows || '<tr><td colspan="5" style="border: 1px solid #333; padding: 4px; text-align: center; color: #666; background: white; font-style: italic; font-size: 8px;">No tasks added</td></tr>';

    // Generate labor rows with INLINE styles - COMPACT with word-wrap for PDF
    const laborRows = allItems.filter(item => item.category === 'Labor').map((item, index) => `
        <tr>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${index + 1}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.worker || item.description || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.code || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">$${item.rate || '0.00'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.hours || '0'}h ${item.minutes || '0'}m</td>
        </tr>
    `).join('');
    
    const laborTableContent = laborRows || '<tr><td colspan="5" style="border: 1px solid #333; padding: 4px; text-align: center; color: #666; background: white; font-style: italic; font-size: 8px;">No labor items added</td></tr>';

    // Generate parts rows with INLINE styles - COMPACT with word-wrap for PDF
    const partsRows = allItems.filter(item => item.category === 'Parts').map((item, index) => `
        <tr>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${index + 1}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.itemNumber || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.partNumber || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.itemName || item.description || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.quantity || '0'} ${item.unit || 'Piece'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">$${item.itemCost || item.amount || '0.00'}</td>
        </tr>
    `).join('');
    
    const partsTableContent = partsRows || '<tr><td colspan="6" style="border: 1px solid #333; padding: 4px; text-align: center; color: #666; background: white; font-style: italic; font-size: 8px;">No parts added</td></tr>';

    // Generate additional cost rows with INLINE styles - COMPACT with word-wrap for PDF
    const additionalCostRows = allItems.filter(item => item.category === 'Additional Cost').map((item, index) => `
        <tr>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${index + 1}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.description || 'N/A'}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">$${item.amount || '0.00'}</td>
        </tr>
    `).join('');
    
    const additionalCostTableContent = additionalCostRows || '<tr><td colspan="3" style="border: 1px solid #333; padding: 4px; text-align: center; color: #666; background: white; font-style: italic; font-size: 9px;">No additional costs added</td></tr>';

    // Generate remarks rows with INLINE styles - COMPACT with word-wrap for PDF
    const remarksRows = allItems.filter(item => item.category === 'Remarks').map((item, index) => `
        <tr>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${index + 1}</td>
            <td style="border: 1px solid #333; padding: 3px; text-align: left; color: black; background: white; font-size: 8px; word-wrap: break-word;">${item.description || 'N/A'}</td>
        </tr>
    `).join('');
    
    const remarksTableContent = remarksRows || '<tr><td colspan="2" style="border: 1px solid #333; padding: 4px; text-align: center; color: #666; background: white; font-style: italic; font-size: 8px;">No remarks added</td></tr>';

    // Return HTML with ONLY inline styles - OPTIMIZED for full-page PDF rendering
    return `
        <div style="font-family: Arial, Helvetica, sans-serif; padding: 0; margin: 0; font-size: 10px; background: white; color: black; width: 100%; max-width: 794px; height: auto; box-sizing: border-box;">
            <!-- Header Section with Logo -->
            <div style="background: white; padding: 10px; margin-bottom: 10px; border: 1px solid #ccc; box-sizing: border-box;">
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; table-layout: fixed;">
                    <tr>
                        <td rowspan="5" style="border: none; padding: 0; width: 80px; vertical-align: top;">
                            <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" style="width: 75px; height: auto; display: block;">
                        </td>
                        <td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Document Name:</strong> HSE-FOR-8.1.2 Job Card</td>
                    </tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Document ID:</strong> HSE-029</td></tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Version Date:</strong> 05/08/2021</td></tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Version:</strong> 1</td></tr>
                    <tr><td style="border: none; padding: 1px 0; font-size: 9px; color: black;"><strong style="color: black;">Approved:</strong> HSM</td></tr>
                </table>
            </div>

            <!-- Title -->
            <div style="text-align: center; font-weight: bold; font-size: 16px; margin: 10px 0 8px 0; color: #0055A5; letter-spacing: 2px;">
                JOB CARD
            </div>
            
            <!-- Basic Information Section -->
            <div style="background: white; padding: 8px; margin-bottom: 8px; border: 1px solid #ccc; box-sizing: border-box;">
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; table-layout: fixed;">
                    <tr>
                        <td style="padding: 3px 6px; color: black; font-size: 9px; width: 50%; word-wrap: break-word;"><strong style="color: #0055A5;">NO:</strong> <span style="color: black;">${workOrderData.requestNumber || 'N/A'}</span></td>
                        <td style="padding: 3px 6px; color: black; font-size: 9px; width: 50%; word-wrap: break-word;"><strong style="color: #0055A5;">DATE:</strong> <span style="color: black;">${currentDate}</span></td>
                    </tr>
                    <tr>
                        <td style="padding: 3px 6px; color: black; font-size: 9px; word-wrap: break-word;"><strong style="color: #0055A5;">COUNTRY:</strong> <span style="color: black;">${workOrderData.country || 'N/A'}</span></td>
                        <td style="padding: 3px 6px; color: black; font-size: 9px; word-wrap: break-word;"><strong style="color: #0055A5;">PROJECT:</strong> <span style="color: black;">${workOrderData.project || 'N/A'}</span></td>
                    </tr>
                    <tr>
                        <td colspan="2" style="padding: 3px 6px; color: black; font-size: 9px; word-wrap: break-word;"><strong style="color: #0055A5;">EQUIPMENT:</strong> <span style="color: black;">${equipmentData ? (equipmentData.name || equipmentData.equipmentName || 'N/A') : 'N/A'}</span></td>
                    </tr>
                </table>
            </div>

            <!-- Equipment Details Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Equipment Details</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; font-size: 9px; table-layout: fixed;">
                    <tr>
                        <td style="padding: 2px 4px; width: 50%; color: black; word-wrap: break-word;"><strong style="color: black;">Name:</strong> ${equipmentData ? (equipmentData.name || 'N/A') : 'N/A'}</td>
                        <td style="padding: 2px 4px; width: 50%; color: black; word-wrap: break-word;"><strong style="color: black;">Type:</strong> ${equipmentData ? (equipmentData.type || 'N/A') : 'N/A'}</td>
                    </tr>
                    <tr>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Model:</strong> ${equipmentData ? (equipmentData.model || 'N/A') : 'N/A'}</td>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Reg:</strong> ${equipmentData ? (equipmentData.regNum || 'N/A') : 'N/A'}</td>
                    </tr>
                    <tr>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Make:</strong> ${equipmentData ? (equipmentData.manufacturer || 'N/A') : 'N/A'}</td>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Engine:</strong> ${equipmentData ? (equipmentData.engineNumber || 'N/A') : 'N/A'}</td>
                    </tr>
                    <tr>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Meter:</strong> ${equipmentData ? ((equipmentData.meterReading || '0') + ' ' + (equipmentData.meterUnit || 'KM')) : 'N/A'}</td>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Status:</strong> ${equipmentData ? (equipmentData.status || 'N/A') : 'N/A'}</td>
                    </tr>
                </table>
            </div>

            <!-- Job Card Details Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Job Card Details</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; font-size: 9px; table-layout: fixed;">
                    <tr>
                        <td style="padding: 2px 4px; width: 50%; color: black; word-wrap: break-word;"><strong style="color: black;">Title:</strong> ${workOrderData.title || 'N/A'}</td>
                        <td style="padding: 2px 4px; width: 50%; color: black; word-wrap: break-word;"><strong style="color: black;">Priority:</strong> ${workOrderData.priority || 'Medium'}</td>
                    </tr>
                    <tr>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Memo:</strong> ${workOrderData.memo || 'N/A'}</td>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Type:</strong> ${selectedJobCardType || 'General'}</td>
                    </tr>
                    <tr>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Assigned:</strong> ${usersData.length > 0 ? usersData.map(u => u.name || 'Unknown').join(', ') : 'None'}</td>
                        <td style="padding: 2px 4px; color: black; word-wrap: break-word;"><strong style="color: black;">Est. Cost:</strong> $${workOrderData.estimatedCost || '0.00'}</td>
                    </tr>
                </table>
            </div>

            <!-- Cost Details Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Cost Details</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; font-size: 9px; table-layout: fixed;">
                    <tr>
                        <td style="padding: 2px 4px; width: 50%; color: black;"><strong style="color: black;">Parts Cost:</strong> $${workOrderData.partMaterialCost || '0.00'}</td>
                        <td style="padding: 2px 4px; width: 50%; color: black;"><strong style="color: black;">Labor Cost:</strong> $${workOrderData.totalLaborCost || '0.00'}</td>
                    </tr>
                    <tr>
                        <td style="padding: 2px 4px; color: black;"><strong style="color: black;">Add. Cost:</strong> $${workOrderData.additionalCost || '0.00'}</td>
                        <td style="padding: 2px 4px; color: black;"><strong style="color: black;">Tax:</strong> $${workOrderData.taxAmount || '0.00'} (${workOrderData.taxPercentage || '0'}%)</td>
                    </tr>
                    <tr>
                        <td colspan="2" style="padding: 2px 4px; color: black; font-weight: bold;"><strong style="color: black;">Total:</strong> $${workOrderData.totalWoCost || '0.00'}</td>
                    </tr>
                </table>
            </div>

            <!-- Tasks Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Tasks</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; margin-bottom: 4px; font-size: 8px; table-layout: fixed;">
                    <tr>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 5%;">#</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 35%;">Description</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 15%;">Type</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 15%;">Status</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 30%;">Note</th>
                    </tr>
                    ${tasksTableContent}
                </table>
            </div>

            <!-- Labor Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Labor</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; margin-bottom: 4px; font-size: 8px; table-layout: fixed;">
                    <tr>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 5%;">#</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 35%;">Worker</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 20%;">Code</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 20%;">Rate</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 20%;">Time</th>
                    </tr>
                    ${laborTableContent}
                </table>
            </div>

            <!-- Parts Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Parts</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; margin-bottom: 4px; font-size: 8px; table-layout: fixed;">
                    <tr>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 5%;">#</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 15%;">Item No</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 15%;">Part No</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 35%;">Name</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 15%;">Qty</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 15%;">Cost</th>
                    </tr>
                    ${partsTableContent}
                </table>
            </div>

            <!-- Additional Costs Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Additional Costs</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; margin-bottom: 4px; font-size: 8px; table-layout: fixed;">
                    <tr>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 10%;">#</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 60%;">Description</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 30%;">Amount</th>
                    </tr>
                    ${additionalCostTableContent}
                </table>
            </div>

            <!-- Remarks Section -->
            <div style="margin-top: 8px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <h3 style="color: #0055A5; font-size: 11px; margin: 0 0 6px 0; border-bottom: 2px solid #0055A5; padding-bottom: 2px;">Remarks</h3>
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; margin-bottom: 4px; font-size: 8px; table-layout: fixed;">
                    <tr>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 10%;">#</th>
                        <th style="border: 1px solid #333; padding: 3px; text-align: left; background-color: #0055A5; color: white; font-weight: bold; font-size: 8px; width: 90%;">Description</th>
                    </tr>
                    ${remarksTableContent}
                </table>
            </div>

            <!-- Footer / Signature Section -->
            <div style="margin-top: 12px; background: white; padding: 8px; border: 1px solid #ccc; page-break-inside: avoid; box-sizing: border-box;">
                <table style="width: 100%; max-width: 100%; border-collapse: collapse; table-layout: fixed;">
                    <tr>
                        <td style="width: 50%; padding: 4px; vertical-align: top;">
                            <p style="margin: 2px 0; color: black; font-size: 10px;"><strong style="color: black;">Prepared By:</strong></p>
                            <p style="margin: 2px 0; color: black; font-size: 9px;">Titan Drilling Maintenance Team</p>
                            <p style="color: #666; font-size: 8px; margin: 2px 0;">Date: ${currentDate}</p>
                        </td>
                        <td style="width: 50%; padding: 4px; vertical-align: top; text-align: right;">
                            <p style="margin: 2px 0; color: black; font-size: 10px;"><strong style="color: black;">Approved By:</strong></p>
                            <p style="margin: 2px 0; color: black; font-size: 9px;">________________________</p>
                            <p style="color: #666; font-size: 8px; margin: 2px 0;">HSE Manager</p>
                        </td>
                    </tr>
                </table>
            </div>

            <!-- Document Footer -->
            <div style="text-align: center; margin-top: 10px; padding-top: 8px; border-top: 1px solid #0055A5;">
                <p style="font-size: 8px; color: #666; margin: 2px 0;">
                    This is a computer-generated document. For queries, contact Titan Drilling maintenance department.
                </p>
                <p style="font-size: 8px; color: #666; margin: 2px 0;">
                    <strong style="color: black;">Titan Africa Drilling</strong> | www.titanafricadrilling.com
                </p>
            </div>
        </div>
    `;
}

// OLD FUNCTION - DO NOT USE
function generatePDFHTML_OLD_WITH_CLASSES(workOrderData) {
    return `
        <style>
                body { 
                    font-family: 'Segoe UI', Arial, sans-serif; 
                    padding: 24px; 
                    font-size: 13px; 
                    background: #f8f9fa; 
                    color: #222; 
                }
                .header, .section { 
                    border-radius: 8px; 
                    background: #fff; 
                    box-shadow: 0 1px 4px #e0e0e0; 
                    padding: 16px 20px; 
                    margin-bottom: 18px; 
                }
                .header-table, .checklist-table { 
                    width: 100%; 
                    border-collapse: collapse; 
                    margin-bottom: 10px; 
                }
                .header-table td { 
                    border: none; 
                    padding: 3px 0; 
                }
                .checklist-table th, .checklist-table td { 
                    border: 1px solid #e0e0e0; 
                    padding: 7px 6px; 
                    text-align: left; 
                }
                .title { 
                    text-align: center; 
                    font-weight: bold; 
                    font-size: 22px; 
                    margin: 24px 0 18px 0; 
                    color: #015185; 
                    letter-spacing: 1px; 
                }
                .notes { 
                    font-size: 11px; 
                    margin-bottom: 10px; 
                    color: #888; 
                }
                .checklist-table th { 
                    background-color: #015185; 
                    color: #fff; 
                    font-weight: 600; 
                    font-size: 14px; 
                }
                .summary-section { 
                    margin-top: 24px; 
                    background: #fff; 
                    border-radius: 8px; 
                    box-shadow: 0 1px 4px #e0e0e0; 
                    padding: 18px 20px; 
                }
                .summary-section h3 { 
                    color: #015185; 
                    font-size: 16px; 
                    margin-bottom: 10px; 
                    border-bottom: 1.5px solid #e0e0e0; 
                    padding-bottom: 2px; 
                }
                .summary-section p { 
                    margin: 5px 0; 
                }
                .section strong { 
                    color: #015185; 
                }
                .section label { 
                    color: #888; 
                    font-weight: 400; 
                }
            </style>
        
        <div class="pdf-document">
            <!-- Header Section with Logo -->
            <div class="header">
                <table class="header-table">
                    <tr>
                        <td rowspan="5"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" width="100"></td>
                        <td><strong>Document Name:</strong> HSE-FOR-8.1.2 Job Card</td>
                    </tr>
                    <tr><td><strong>Document ID:</strong> HSE-029</td></tr>
                    <tr><td><strong>Version Date:</strong> 05/08/2021</td></tr>
                    <tr><td><strong>Version:</strong> 1</td></tr>
                    <tr><td><strong>Approved:</strong> HSM</td></tr>
                </table>
            </div>

            <!-- Title -->
            <div class="title">JOB CARD</div>
            
            <!-- Basic Information Section -->
            <div class="section">
                <div style="display: flex; flex-wrap: wrap; gap: 18px 32px;">
                    <div><strong>NO:</strong> ${workOrderData.requestNumber || 'N/A'}</div>
                    <div><strong>DATE:</strong> ${currentDate}</div>
                    <div><strong>COUNTRY:</strong> ${workOrderData.country || 'N/A'}</div>
                    <div><strong>PROJECT:</strong> ${workOrderData.project || 'N/A'}</div>
                    <div><strong>EQUIPMENT:</strong> ${equipmentData ? (equipmentData.name || equipmentData.equipmentName || 'N/A') : 'N/A'}</div>
                    <div><strong>CREATED AT:</strong> ${currentDateTime}</div>
                </div>
            </div>

            <!-- Equipment Details Section -->
            <div class="summary-section">
                <h3>Equipment Details</h3>
                <div style="display: flex; flex-wrap: wrap; gap: 12px 32px;">
                    <div><strong>Equipment Name:</strong> ${equipmentData ? (equipmentData.name || equipmentData.equipmentName || 'N/A') : 'N/A'}</div>
                    <div><strong>Type:</strong> ${equipmentData ? (equipmentData.type || equipmentData.vehicleType || equipmentData.mainCategory || 'N/A') : 'N/A'}</div>
                    <div><strong>Model:</strong> ${equipmentData ? (equipmentData.model || 'N/A') : 'N/A'}</div>
                    <div><strong>Registration:</strong> ${equipmentData ? (equipmentData.regNum || equipmentData.registrationNumber || 'N/A') : 'N/A'}</div>
                    <div><strong>Make:</strong> ${equipmentData ? (equipmentData.manufacturer || equipmentData.make || 'N/A') : 'N/A'}</div>
                    <div><strong>Engine Number:</strong> ${equipmentData ? (equipmentData.engineNumber || 'N/A') : 'N/A'}</div>
                    <div><strong>Meter Reading:</strong> ${equipmentData ? ((equipmentData.meterReading || '0') + ' ' + (equipmentData.meterUnit || 'KM')) : 'N/A'}</div>
                    <div><strong>Status:</strong> ${equipmentData ? (equipmentData.status || 'N/A') : 'N/A'}</div>
                </div>
            </div>

            <!-- Job Card Details Section -->
            <div class="summary-section">
                <h3>Job Card Details</h3>
                <div style="display: flex; flex-wrap: wrap; gap: 12px 32px;">
                    <div><strong>Title:</strong> ${workOrderData.title || 'N/A'}</div>
                    <div><strong>Country:</strong> ${workOrderData.country || 'N/A'}</div>
                    <div><strong>Project:</strong> ${workOrderData.project || 'N/A'}</div>
                    <div><strong>Memo:</strong> ${workOrderData.memo || 'N/A'}</div>
                    <div><strong>Priority:</strong> ${workOrderData.priority || 'Medium'}</div>
                    <div><strong>Assigned To:</strong> ${usersData.length > 0 ? usersData.map(u => u.name || u.email || 'Unknown').join(', ') : 'None'}</div>
                    <div><strong>Start Date:</strong> ${startDate !== 'N/A' ? new Date(startDate).toLocaleString('en-US') : 'N/A'}</div>
                    <div><strong>Due Date:</strong> ${dueDate !== 'N/A' ? new Date(dueDate).toLocaleString('en-US') : 'N/A'}</div>
                    <div><strong>Job Card Type:</strong> ${selectedJobCardType || 'General'}</div>
                    <div><strong>Estimated Cost:</strong> $${workOrderData.estimatedCost || '0.00'}</div>
                    <div><strong>Estimated Labor Time:</strong> ${workOrderData.estimatedLaborTime || '0'} hours</div>
                    <div><strong>Linked Inspection:</strong> ${selectedInspection ? (selectedInspection.name + ' - ' + selectedInspection.requestNumber) : 'None'}</div>
                    <div><strong>Linked Job Card:</strong> ${selectedJobCard ? selectedJobCard.name : 'None'}</div>
                </div>
            </div>

            <!-- Cost Details Section -->
            <div class="summary-section">
                <h3>Cost Details</h3>
                <div style="display: flex; flex-wrap: wrap; gap: 12px 32px;">
                    <div><strong>Part & Material Cost:</strong> $${workOrderData.partMaterialCost || '0.00'}</div>
                    <div><strong>Total Labor Cost:</strong> $${workOrderData.totalLaborCost || '0.00'}</div>
                    <div><strong>Additional Cost:</strong> $${workOrderData.additionalCost || '0.00'}</div>
                    <div><strong>Tax Amount:</strong> $${workOrderData.taxAmount || '0.00'} (${workOrderData.taxPercentage || '0'}%)</div>
                    <div><strong>Total WO Cost:</strong> $${workOrderData.totalWoCost || '0.00'}</div>
                </div>
            </div>

            <!-- Tasks Section -->
            <div class="summary-section">
                <h3>Tasks</h3>
                <table class="checklist-table">
                    <tr>
                        <th>#</th>
                        <th>Description</th>
                        <th>Type</th>
                        <th>Status</th>
                        <th>Note</th>
                    </tr>
                    ${tasksTableContent}
                </table>
            </div>

            <!-- Labor Section -->
            <div class="summary-section">
                <h3>Labor</h3>
                <table class="checklist-table">
                    <tr>
                        <th>#</th>
                        <th>Worker</th>
                        <th>Code</th>
                        <th>Rate</th>
                        <th>Time</th>
                    </tr>
                    ${laborTableContent}
                </table>
            </div>

            <!-- Parts Section -->
            <div class="summary-section">
                <h3>Parts</h3>
                <table class="checklist-table">
                    <tr>
                        <th>#</th>
                        <th>Item No</th>
                        <th>Part No</th>
                        <th>Name</th>
                        <th>Quantity</th>
                        <th>Cost</th>
                    </tr>
                    ${partsTableContent}
                </table>
            </div>

            <!-- Additional Costs Section -->
            <div class="summary-section">
                <h3>Additional Costs</h3>
                <table class="checklist-table">
                    <tr>
                        <th>#</th>
                        <th>Description</th>
                        <th>Amount</th>
                    </tr>
                    ${additionalCostTableContent}
                </table>
            </div>

            <!-- Remarks Section -->
            <div class="summary-section">
                <h3>Remarks</h3>
                <table class="checklist-table">
                    <tr>
                        <th>#</th>
                        <th>Description</th>
                    </tr>
                    ${remarksTableContent}
                </table>
            </div>

            <!-- Footer / Signature Section -->
            <div class="summary-section" style="margin-top: 30px;">
                <div style="display: flex; justify-content: space-between; align-items: flex-start;">
                    <div style="flex: 1;">
                        <p><strong>Prepared By:</strong></p>
                        <p>Titan Drilling Maintenance Team</p>
                        <p style="color: #888; font-size: 11px;">Date: ${currentDate}</p>
                    </div>
                    <div style="flex: 1; text-align: right;">
                        <p><strong>Approved By:</strong></p>
                        <p>________________________________</p>
                        <p style="color: #888; font-size: 11px;">HSE Manager</p>
                    </div>
                </div>
            </div>

            <!-- Document Footer -->
            <div style="text-align: center; margin-top: 40px; padding-top: 20px; border-top: 2px solid #015185;">
                <p style="font-size: 11px; color: #888;">
                    This is a computer-generated document. For any queries, contact Titan Drilling maintenance department.
                </p>
                <p style="font-size: 11px; color: #888; margin-top: 5px;">
                    <strong>Titan Africa Drilling</strong> | www.titanafricadrilling.com
                </p>
            </div>
        </div>
    `;
}

function closeSuccessModal() {
    document.getElementById('successModal').style.display = 'none';
    // Redirect to work orders list
    window.location.href = "{{ route('maintenance.workorder') }}";
}

function goBackToWorkOrders() {
    closeSuccessModal();
}

function downloadPDF() {
    if (window.generatedPDFUrl) {
        // Create a temporary link element to trigger download
        const link = document.createElement('a');
        link.href = window.generatedPDFUrl;
        link.download = window.generatedPDFFileName || 'job-card.pdf';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    } else {
        showToast('error', 'Error', 'No PDF available for download');
    }
}

// Collapsible sections functionality
function toggleCollapsible(sectionId) {
    const content = document.getElementById(sectionId + 'Content');
    const arrow = document.getElementById(sectionId + 'Arrow');
    
    if (content.classList.contains('expanded')) {
        content.classList.remove('expanded');
        arrow.classList.remove('rotated');
    } else {
        content.classList.add('expanded');
        arrow.classList.add('rotated');
    }
}


// Cost calculation functionality
function initializeCostCalculation() {
    const partMaterialCostInput = document.getElementById('partMaterialCost');
    const totalLaborCostInput = document.getElementById('totalLaborCost');
    const additionalCostInput = document.getElementById('additionalCost');
    const taxAmountInput = document.getElementById('taxAmount');
    const totalWOCostInput = document.getElementById('totalWOCost');
    
    if (partMaterialCostInput && totalLaborCostInput && additionalCostInput && totalWOCostInput) {
        partMaterialCostInput.addEventListener('input', calculateTotalWOCost);
        totalLaborCostInput.addEventListener('input', calculateTotalWOCost);
        additionalCostInput.addEventListener('input', calculateTotalWOCost);
        taxAmountInput.addEventListener('input', calculateTotalWOCost);
    }
}

function calculateTotalWOCost() {
    const partMaterialCost = parseFloat(document.getElementById('partMaterialCost').value) || 0;
    const totalLaborCost = parseFloat(document.getElementById('totalLaborCost').value) || 0;
    const additionalCost = parseFloat(document.getElementById('additionalCost').value) || 0;
    const taxAmount = parseFloat(document.getElementById('taxAmount').value) || 0;
    
    const totalWOCost = partMaterialCost + totalLaborCost + additionalCost + taxAmount;
    
    document.getElementById('totalWOCost').value = totalWOCost.toFixed(2);
}

// Job Card Type Modal Functions
function openJobCardTypeModal(event) {
    const modal = document.getElementById('jobCardTypeModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    updateJobCardTypeSelection();
}

function closeJobCardTypeModal() {
    const modal = document.getElementById('jobCardTypeModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function selectJobCardType(type) {
    selectedJobCardType = type;
    document.getElementById('jobCardTypeText').textContent = type;
    updateJobCardTypeSelection();
    closeJobCardTypeModal();
}

function updateJobCardTypeSelection() {
    document.querySelectorAll('.option-item').forEach(item => {
        item.classList.remove('selected');
        const checkIcon = item.querySelector('.option-check');
        if (checkIcon) checkIcon.style.display = 'none';
    });
    
    const selectedItem = Array.from(document.querySelectorAll('.option-item')).find(item => 
        item.querySelector('.option-text').textContent === selectedJobCardType
    );
    
    if (selectedItem) {
        selectedItem.classList.add('selected');
        const checkIcon = selectedItem.querySelector('.option-check');
        if (checkIcon) checkIcon.style.display = 'block';
    }
}

// Link Inspection Modal Functions
function openLinkInspectionModal(event) {
    const modal = document.getElementById('linkInspectionModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Auto-focus search input
    setTimeout(() => {
        const searchInput = document.getElementById('inspectionSearch');
        if (searchInput) {
            searchInput.focus();
        }
    }, 100);
    
    loadInspections();
}

function closeLinkInspectionModal() {
    const modal = document.getElementById('linkInspectionModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function loadInspections() {
    // Use MongoDB data that was already loaded
    if (inspectionList.length === 0) {
        // If no data loaded yet, show loading state
        const inspectionListDiv = document.getElementById('inspectionList');
        inspectionListDiv.innerHTML = `
            <div class="loading-state">
                <div class="loading-spinner"></div>
                Loading inspections from MongoDB...
            </div>
        `;
        
        // Try to load data again
        loadInspectionsFromMongoDB().then(() => {
            filterInspections();
        });
    } else {
        filterInspections();
    }
}

function filterInspections() {
    const searchTerm = document.getElementById('inspectionSearch').value.toLowerCase();
    const inspectionListDiv = document.getElementById('inspectionList');
    
    inspectionListDiv.innerHTML = '';
    
    if (inspectionList.length === 0) {
        inspectionListDiv.innerHTML = '<div class="no-data">No inspections available</div>';
        return;
    }
    
    const filteredInspections = inspectionList.filter(inspection => 
        inspection.name.toLowerCase().includes(searchTerm) ||
        inspection.status.toLowerCase().includes(searchTerm) ||
        inspection.requestNumber.toLowerCase().includes(searchTerm) ||
        inspection.inspector.toLowerCase().includes(searchTerm) ||
        inspection.equipment.toLowerCase().includes(searchTerm) ||
        inspection.location.toLowerCase().includes(searchTerm)
    );
    
    if (filteredInspections.length === 0) {
        inspectionListDiv.innerHTML = '<div class="no-data">No inspections match your search</div>';
        return;
    }
    
    filteredInspections.forEach(inspection => {
        const inspectionItem = document.createElement('div');
        inspectionItem.className = 'equipment-item';
        inspectionItem.innerHTML = `
            <div class="equipment-info">
                <div class="equipment-name">${inspection.name}</div>
                <div class="equipment-details">
                    <span class="equipment-type">Request #: ${inspection.requestNumber}</span>
                    <span class="equipment-location">Inspector: ${inspection.inspector}</span>
                </div>
                <div class="equipment-details" style="margin-top: 4px;">
                    <span class="equipment-type">Date: ${inspection.date}</span>
                    <span class="equipment-location">Status: ${inspection.status}</span>
                </div>
                <div class="equipment-details" style="margin-top: 4px;">
                    <span class="equipment-type">Equipment: ${inspection.equipment}</span>
                    <span class="equipment-location">Location: ${inspection.location}</span>
                </div>
            </div>
            <div class="equipment-status ${inspection.status.toLowerCase().replace(/\s+/g, '')}">${inspection.status}</div>
        `;
        inspectionItem.onclick = () => selectInspection(inspection);
        inspectionListDiv.appendChild(inspectionItem);
    });
}

function selectInspection(inspection) {
    selectedInspection = inspection;
    // Display inspection with request number and inspector
    const displayText = `${inspection.name} - ${inspection.requestNumber}`;
    document.getElementById('linkInspectionText').textContent = displayText;
    closeLinkInspectionModal();
}

function saveLinkInspection() {
    closeLinkInspectionModal();
}

// Job Card Modal Functions
function openJobCardModal(event) {
    const modal = document.getElementById('jobCardModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Auto-focus search input
    setTimeout(() => {
        const searchInput = document.getElementById('jobCardSearch');
        if (searchInput) {
            searchInput.focus();
        }
    }, 100);
    
    loadJobCards();
}

function closeJobCardModal() {
    const modal = document.getElementById('jobCardModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

function loadJobCards() {
    // Mock job card data - replace with actual API call
    jobCardList = [
        { id: 1, name: 'JC-001 - Equipment Maintenance', date: '2024-01-15', status: 'Active' },
        { id: 2, name: 'JC-002 - Safety Check', date: '2024-01-20', status: 'Completed' },
        { id: 3, name: 'JC-003 - Emergency Repair', date: '2024-01-25', status: 'In Progress' },
        { id: 4, name: 'JC-004 - Routine Service', date: '2024-02-01', status: 'Scheduled' }
    ];
    
    filterJobCards();
}

function filterJobCards() {
    const searchTerm = document.getElementById('jobCardSearch').value.toLowerCase();
    const jobCardListDiv = document.getElementById('jobCardList');
    
    jobCardListDiv.innerHTML = '';
    
    jobCardList.filter(jobCard => 
        jobCard.name.toLowerCase().includes(searchTerm) ||
        jobCard.status.toLowerCase().includes(searchTerm)
    ).forEach(jobCard => {
        const jobCardItem = document.createElement('div');
        jobCardItem.className = 'equipment-item';
        jobCardItem.innerHTML = `
            <div class="equipment-info">
                <div class="equipment-name">${jobCard.name}</div>
                <div class="equipment-details">
                    <span class="equipment-type">${jobCard.date}</span>
                    <span class="equipment-location">${jobCard.status}</span>
                </div>
            </div>
            <div class="equipment-status ${jobCard.status.toLowerCase().replace(' ', '')}">${jobCard.status}</div>
        `;
        jobCardItem.onclick = () => selectJobCard(jobCard);
        jobCardListDiv.appendChild(jobCardItem);
    });
}

function selectJobCard(jobCard) {
    selectedJobCard = jobCard;
    document.getElementById('jobCardText').textContent = jobCard.name;
    closeJobCardModal();
}

function saveJobCard() {
    closeJobCardModal();
}

// Add event listeners for form changes
document.getElementById('country').addEventListener('change', updateProjectOptions);
document.getElementById('project').addEventListener('change', generateRequestNumber);

// Close modals when clicking outside
window.onclick = function(event) {
    const modals = [
        'equipmentModal',
        'assignToModal', 
        'addItemModal',
        'taskAssignToModal',
        'jobCardTypeModal',
        'linkInspectionModal',
        'jobCardModal',
        'allJobCardsModal'
    ];
    
    modals.forEach(modalId => {
        const modal = document.getElementById(modalId);
        if (event.target === modal) {
            modal.classList.remove('show');
        }
    });
}

// ========== FETCH TOTAL JOB CARDS COUNT FROM MONGODB ==========
async function fetchTotalJobCardsCount() {
    try {
        console.log('🔄 Fetching total job cards count from MongoDB...');
        
        const response = await fetch(`${MONGODB_API_BASE}/jobcards/all`);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('📊 MongoDB Job Cards Response:', result);
        
        if (result.success && Array.isArray(result.data)) {
            const totalCount = result.data.length;
            
            // Update the display
            const displayElement = document.getElementById('totalJobCardsDisplay');
            if (displayElement) {
                displayElement.innerHTML = `${totalCount}`;
            }
            
            // Store all job cards
            allJobCardsFromDB = result.data;
            
            console.log(`✅ Total Job Cards in MongoDB: ${totalCount}`);
            
        } else {
            console.warn('⚠️ No job cards found in MongoDB');
            document.getElementById('totalJobCardsDisplay').innerHTML = '0';
        }
        
    } catch (error) {
        console.error('❌ Error fetching job cards count:', error);
        document.getElementById('totalJobCardsDisplay').innerHTML = '<span style="font-size: 16px;">Error</span>';
    }
}

// ========== VIEW ALL JOB CARDS MODAL ==========
function viewAllJobCards() {
    const modal = document.getElementById('allJobCardsModal');
    modal.classList.add('show');
    document.body.classList.add('modal-open');
    
    // Load all job cards
    loadAllJobCards();
}

function closeAllJobCardsModal() {
    const modal = document.getElementById('allJobCardsModal');
    modal.classList.remove('show');
    document.body.classList.remove('modal-open');
}

async function loadAllJobCards() {
    try {
        console.log('🔄 Loading all job cards from MongoDB...');
        
        const listContainer = document.getElementById('allJobCardsList');
        listContainer.innerHTML = `
            <div style="text-align: center; padding: 40px; color: #666;">
                <i class="fas fa-spinner fa-spin" style="font-size: 32px; margin-bottom: 10px;"></i>
                <p>Loading job cards from MongoDB...</p>
            </div>
        `;
        
        const response = await fetch(`${MONGODB_API_BASE}/jobcards/all`);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('📊 All Job Cards Response:', result);
        
        if (result.success && Array.isArray(result.data)) {
            allJobCardsFromDB = result.data;
            
            // Update count
            document.getElementById('allJobCardsCount').textContent = allJobCardsFromDB.length;
            
            // Display job cards
            displayAllJobCards(allJobCardsFromDB);
            
            console.log(`✅ Loaded ${allJobCardsFromDB.length} job cards from MongoDB`);
            
        } else {
            listContainer.innerHTML = `
                <div style="text-align: center; padding: 40px; color: #999;">
                    <i class="fas fa-inbox" style="font-size: 48px; margin-bottom: 15px; color: #ddd;"></i>
                    <p style="font-size: 16px;">No job cards found in database</p>
                </div>
            `;
        }
        
    } catch (error) {
        console.error('❌ Error loading job cards:', error);
        document.getElementById('allJobCardsList').innerHTML = `
            <div style="text-align: center; padding: 40px; color: #dc3545;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; margin-bottom: 15px;"></i>
                <p style="font-size: 16px;">Error loading job cards</p>
                <p style="font-size: 14px;">${error.message}</p>
            </div>
        `;
    }
}

function displayAllJobCards(jobCards) {
    const listContainer = document.getElementById('allJobCardsList');
    
    if (!jobCards || jobCards.length === 0) {
        listContainer.innerHTML = `
            <div style="text-align: center; padding: 40px; color: #999;">
                <i class="fas fa-inbox" style="font-size: 48px; margin-bottom: 15px; color: #ddd;"></i>
                <p style="font-size: 16px;">No job cards found</p>
            </div>
        `;
        return;
    }
    
    listContainer.innerHTML = '';
    
    jobCards.forEach((jobCard, index) => {
        const cardElement = document.createElement('div');
        cardElement.className = 'jobcard-item';
        cardElement.style.cssText = `
            background: white;
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 12px;
            transition: all 0.3s ease;
            cursor: pointer;
        `;
        
        // Get status color
        const statusColors = {
            'Hold': '#ffc107',
            'assigned': '#17a2b8',
            'in_progress': '#007bff',
            'complete': '#28a745',
            'cancelled': '#dc3545'
        };
        const statusColor = statusColors[jobCard.status] || '#6c757d';
        
        // Format dates
        const createdDate = jobCard.createdAt ? new Date(jobCard.createdAt).toLocaleString() : 'N/A';
        const updatedDate = jobCard.updatedAt ? new Date(jobCard.updatedAt).toLocaleString() : 'N/A';
        
        cardElement.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 12px;">
                <div style="flex: 1;">
                    <div style="font-size: 18px; font-weight: 600; color: #0055A5; margin-bottom: 6px;">
                        ${jobCard.requestNumber || jobCard._id}
                    </div>
                    <div style="font-size: 14px; color: #333; margin-bottom: 4px;">
                        ${jobCard.title || 'Untitled Job Card'}
                    </div>
                </div>
                <div style="background: ${statusColor}; color: white; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600; text-transform: uppercase;">
                    ${jobCard.status || 'Unknown'}
                </div>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 10px; margin-bottom: 12px;">
                <div>
                    <div style="font-size: 11px; color: #888; text-transform: uppercase; margin-bottom: 2px;">Equipment</div>
                    <div style="font-size: 13px; color: #333; font-weight: 500;">${jobCard.equipment?.equipmentName || 'N/A'}</div>
                </div>
                <div>
                    <div style="font-size: 11px; color: #888; text-transform: uppercase; margin-bottom: 2px;">Priority</div>
                    <div style="font-size: 13px; color: #333; font-weight: 500;">${jobCard.priority || 'N/A'}</div>
                </div>
                <div>
                    <div style="font-size: 11px; color: #888; text-transform: uppercase; margin-bottom: 2px;">Work Order Type</div>
                    <div style="font-size: 13px; color: #333; font-weight: 500;">${jobCard.workOrderType || 'N/A'}</div>
                </div>
                <div>
                    <div style="font-size: 11px; color: #888; text-transform: uppercase; margin-bottom: 2px;">Location</div>
                    <div style="font-size: 13px; color: #333; font-weight: 500;">${jobCard.location?.country || 'N/A'} - ${jobCard.location?.project || 'N/A'}</div>
                </div>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 10px; margin-bottom: 12px;">
                <div>
                    <div style="font-size: 11px; color: #888; text-transform: uppercase; margin-bottom: 2px;">Created At</div>
                    <div style="font-size: 12px; color: #666;">${createdDate}</div>
                </div>
                <div>
                    <div style="font-size: 11px; color: #888; text-transform: uppercase; margin-bottom: 2px;">Updated At</div>
                    <div style="font-size: 12px; color: #666;">${updatedDate}</div>
                </div>
            </div>
            
            ${jobCard.pdfDownloadUrl ? `
                <div style="border-top: 1px solid #e0e0e0; padding-top: 12px; margin-top: 12px;">
                    <div style="display: flex; align-items: center; gap: 10px; flex-wrap: wrap;">
                        <div style="flex: 1; min-width: 200px;">
                            <div style="font-size: 11px; color: #888; text-transform: uppercase; margin-bottom: 4px;">
                                <i class="fas fa-link"></i> PDF Download URL (Dynamic)
                            </div>
                            <div style="font-size: 12px; color: #0055A5; word-break: break-all; background: #f8f9fa; padding: 8px; border-radius: 4px;">
                                ${jobCard.pdfDownloadUrl}
                            </div>
                        </div>
                        <div>
                            <button onclick="window.open('${jobCard.pdfDownloadUrl}', '_blank')" style="background: #0055A5; color: white; border: none; padding: 8px 16px; border-radius: 6px; font-size: 13px; font-weight: 600; cursor: pointer; display: flex; align-items: center; gap: 6px;">
                                <i class="fas fa-file-pdf"></i> View PDF
                            </button>
                        </div>
                    </div>
                </div>
            ` : `
                <div style="border-top: 1px solid #e0e0e0; padding-top: 12px; margin-top: 12px;">
                    <div style="font-size: 12px; color: #999; font-style: italic;">
                        <i class="fas fa-info-circle"></i> No PDF available for this job card
                    </div>
                </div>
            `}
        `;
        
        // Hover effect
        cardElement.addEventListener('mouseenter', function() {
            this.style.boxShadow = '0 4px 12px rgba(0,0,0,0.15)';
            this.style.transform = 'translateY(-2px)';
        });
        
        cardElement.addEventListener('mouseleave', function() {
            this.style.boxShadow = 'none';
            this.style.transform = 'translateY(0)';
        });
        
        listContainer.appendChild(cardElement);
    });
}

function filterAllJobCards() {
    const searchTerm = document.getElementById('jobCardsSearchInput').value.toLowerCase();
    
    const filteredJobCards = allJobCardsFromDB.filter(jobCard => {
        const searchableText = [
            jobCard.requestNumber || '',
            jobCard.title || '',
            jobCard.status || '',
            jobCard.priority || '',
            jobCard.workOrderType || '',
            jobCard.equipment?.equipmentName || '',
            jobCard.location?.country || '',
            jobCard.location?.project || ''
        ].join(' ').toLowerCase();
        
        return searchableText.includes(searchTerm);
    });
    
    displayAllJobCards(filteredJobCards);
    document.getElementById('allJobCardsCount').textContent = filteredJobCards.length;
}

function refreshAllJobCards() {
    loadAllJobCards();
    fetchTotalJobCardsCount(); // Also refresh the main count
}

// ============================================================================
// DEBUGGING & TESTING FUNCTIONS
// ============================================================================

// Test the complete job card creation flow
window.testJobCardCreation = async function() {
    console.log('🧪 TESTING JOB CARD CREATION FLOW');
    console.log('================================');
    
    // Step 1: Test API Connection
    console.log('\n📡 Step 1: Testing MongoDB API Connection...');
    const apiTest = await testAPIConnection();
    console.log('   Result:', apiTest ? '✅ PASS' : '❌ FAIL');
    
    // Step 2: Check form data
    console.log('\n📋 Step 2: Checking Form Data...');
    console.log('   - Title:', document.getElementById('workOrderTitle').value || '❌ EMPTY');
    console.log('   - Country:', document.getElementById('country').value || '❌ EMPTY');
    console.log('   - Project:', document.getElementById('project').value || '❌ EMPTY');
    console.log('   - Equipment:', selectedEquipment ? '✅ Selected' : '❌ Not selected');
    console.log('   - Users:', selectedUsers.length > 0 ? `✅ ${selectedUsers.length} selected` : 'ℹ️ None');
    console.log('   - Items:', items.length > 0 ? `✅ ${items.length} items` : 'ℹ️ None');
    
    // Step 3: Test PDF Generation (if form is valid)
    if (document.getElementById('workOrderTitle').value && 
        document.getElementById('country').value && 
        document.getElementById('project').value && 
        selectedEquipment) {
        
        console.log('\n📄 Step 3: Testing PDF Generation...');
        try {
            const testWorkOrderData = {
                title: document.getElementById('workOrderTitle').value,
                country: document.getElementById('country').value,
                project: document.getElementById('project').value,
                memo: document.getElementById('memo').value,
                priority: window.selectedPriority || 'Medium',
                equipment: selectedEquipment,
                requestNumber: document.getElementById('requestNumber').textContent
            };
            
            const testBlob = await generatePDFBlob(testWorkOrderData);
            console.log('   Result: ✅ PDF generated successfully');
            console.log('   - Size:', testBlob.size, 'bytes');
            
            // Clean up test blob
            URL.revokeObjectURL(URL.createObjectURL(testBlob));
        } catch (error) {
            console.log('   Result: ❌ PDF generation failed');
            console.error('   Error:', error.message);
        }
    } else {
        console.log('\n⚠️ Step 3: Skipped (form not complete)');
    }
    
    console.log('\n================================');
    console.log('🧪 TEST COMPLETE');
    console.log('Call debugJobCard() for current state');
};

// Quick test of MongoDB save endpoint
window.testMongoDBSave = async function() {
    console.log('🧪 Testing MongoDB Save Endpoint...');
    try {
        const testData = {
            _id: 'test_' + Date.now(),
            title: 'Test Job Card',
            status: 'Hold',
            requestNumber: 'TEST-001',
            createdAt: new Date().toISOString()
        };
        
        const response = await fetch(`${MONGODB_API_BASE}/jobcards`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                adminUid: ADMIN_UID,
                jobCardData: testData
            })
        });
        
        const result = await response.json();
        console.log('Test result:', result);
        
        if (result.success) {
            console.log('✅ MongoDB save test PASSED');
            // Clean up test data
            console.log('Note: Test job card created with ID:', testData._id);
        } else {
            console.log('❌ MongoDB save test FAILED:', result.error);
        }
    } catch (error) {
        console.log('❌ MongoDB save test FAILED:', error.message);
    }
};

// Test PDF generation and download locally (verify PDF before upload)
window.testPDFGeneration = async function() {
    console.log('🧪 Testing PDF Generation (Local Download)...');
    
    // Check if form has required data
    if (!document.getElementById('workOrderTitle').value || 
        !document.getElementById('country').value || 
        !document.getElementById('project').value || 
        !selectedEquipment) {
        console.error('❌ Form incomplete. Please fill: Title, Country, Project, and Equipment');
        alert('Please fill in Title, Country, Project, and select Equipment first!');
        return;
    }
    
    try {
        // Collect form data
        const testWorkOrderData = {
            title: document.getElementById('workOrderTitle').value,
            country: document.getElementById('country').value,
            project: document.getElementById('project').value,
            memo: document.getElementById('memo').value,
            priority: window.selectedPriority || 'Medium',
            workOrderType: selectedJobCardType || 'General',
            estimatedCost: document.getElementById('estimatedCost').value,
            estimatedLaborTime: document.getElementById('laborTimeHours').value,
            partMaterialCost: document.getElementById('partMaterialCost').value,
            totalLaborCost: document.getElementById('totalLaborCost').value,
            additionalCost: document.getElementById('additionalCost').value,
            taxAmount: document.getElementById('taxAmount').value,
            taxPercentage: document.getElementById('taxPercentage').value,
            totalWoCost: document.getElementById('totalWOCost').value,
            equipment: selectedEquipment,
            requestNumber: document.getElementById('requestNumber').textContent
        };
        
        console.log('📄 Generating PDF blob...');
        const testBlob = await generatePDFBlob(testWorkOrderData);
        
        console.log('✅ PDF Generated Successfully!');
        console.log('   - Size:', testBlob.size, 'bytes');
        console.log('   - Type:', testBlob.type);
        
        // Verify blob content by downloading it
        const blobUrl = URL.createObjectURL(testBlob);
        const link = document.createElement('a');
        link.href = blobUrl;
        link.download = `TEST_JobCard_${testWorkOrderData.requestNumber}.pdf`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        console.log('✅ Test PDF downloaded. Check your Downloads folder!');
        console.log('📊 If PDF is valid, the issue is in upload. If PDF is blank, issue is in generation.');
        
        // Keep blob URL for 30 seconds for testing
        setTimeout(() => {
            URL.revokeObjectURL(blobUrl);
            console.log('🧹 Cleaned up test blob URL');
        }, 30000);
        
        // Store test blob globally for inspection
        window.testPDFBlob = testBlob;
        window.testPDFBlobUrl = blobUrl;
        
        alert('Test PDF downloaded! Check your Downloads folder.\n\nIf PDF is valid → Issue is in upload\nIf PDF is blank → Issue is in generation');
        
    } catch (error) {
        console.error('❌ Test PDF generation failed:', error);
        alert('PDF generation test failed:\n' + error.message);
    }
};

console.log('💡 Debug commands available:');
console.log('   - testPDFGeneration() - Generate & download PDF locally (verify before upload)');
console.log('   - testJobCardCreation() - Test the complete creation flow');
console.log('   - testMongoDBSave() - Test MongoDB save endpoint');
console.log('   - debugJobCard() - Show current job card state');
console.log('   - testAPIConnection() - Test MongoDB API connection');

</script>
@endsection


