@extends('welcome')

@section('title', 'Job Cards')

@php
    function getStatusColor($status) {
        $status = strtolower($status);
        switch ($status) {
            case 'completed':
            case 'closed':
                return 'success';
            case 'in progress':
            case 'processing':
                return 'primary';
            case 'on hold':
                return 'warning';
            case 'initiated':
                return 'info';
            case 'rework':
                return 'danger';
            case 'archived': 
            case 'cancelled':
                return 'secondary';
            default:
                return 'info';
        }
    }

    function getPriorityColor($priority) {
        $priority = strtolower($priority);
        switch ($priority) {
            case 'high':
            case 'urgent':
                return 'danger';
            case 'medium':
                return 'warning';
            case 'low':
                return 'success';
            default:
                return 'secondary';
        }
    }
@endphp

@section('content')
<div class="workorder-container">
    <div class="workorder-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Job Cards</h1>
                <p>Manage and track job cards efficiently</p>
            </div>
        </div>
    </div>
    <div class="workorder-content">
        <div class="tab-navigation">
            <div class="tab-list">
                <button class="tab-btn active" data-tab="open">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                        <polyline points="14,2 14,8 20,8"/>
                        <line x1="16" y1="13" x2="8" y2="13"/>
                        <line x1="16" y1="17" x2="8" y2="17"/>
                        <polyline points="10,9 9,9 8,9"/>
                    </svg>
                    <span>Open</span>
                </button>
                <button class="tab-btn" data-tab="completed">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                        <polyline points="22,4 12,14.01 9,11.01"/>
                    </svg>
                    <span>Completed</span>
                </button>
                <button class="tab-btn" data-tab="archive">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="21,8 21,21 3,21 3,8"/>
                        <rect x="1" y="3" width="22" height="5"/>
                        <line x1="10" y1="12" x2="14" y2="12"/>
                    </svg>
                    <span>Archive</span>
                </button>
            </div>
        </div>
        <div class="tab-content">
            <div id="open" class="tab-pane active">
                <div class="action-bar">
                    <div class="action-content">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search job cards..." id="openSearch">
                            </div>
                        </div>
                        
                        <div class="action-buttons">
                            <button class="btn-create" onclick="window.location.href='{{ route('maintenance.workorders.create') }}'">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <line x1="12" y1="5" x2="12" y2="19"/>
                                    <line x1="5" y1="12" x2="19" y2="12"/>
                                </svg>
                                + Create Job Card
                            </button>
                            <button class="btn-warning" onclick="displayRequestNumberInfo()">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="10"/>
                                    <path d="M12 16v-4"/>
                                    <path d="M12 8h.01"/>
                                </svg>
                                Request Info
                            </button>
                            <button class="btn-secondary" onclick="sortJobCards()">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M3 6h18"/>
                                    <path d="M7 12h14"/>
                                    <path d="M10 18h11"/>
                                </svg>
                                Sorting
                            </button>
                            <button class="btn-warning" onclick="filterJobCards()">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polygon points="22,3 2,3 10,12.46 10,19 14,21 14,12.46 22,3"/>
                                </svg>
                                Filter
                            </button>
                            <button class="btn-success" onclick="exportToExcel()">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                                    <polyline points="7,10 12,15 17,10"/>
                                    <line x1="12" y1="15" x2="12" y2="3"/>
                                </svg>
                                Excel Export
                            </button>
                        </div>
                    </div>
                </div>
                <div class="table-section">
                    <div class="table-container">
                        <table class="workorder-table">
                            <thead class="table-header">
                                <tr>
                                    <th style="width: 50px;">S.No</th>
                                    <th>Action</th>
                                    <th>Job Card Number</th>
                                    <th>Created Date</th>
                                    <th>Vehicle</th>
                                    <th>Due Date</th>
                                    <th>Status</th>
                                    <th>Priority</th>
                                    <th>Assigned</th>
                                    <th>Memo</th>
                                </tr>
                            </thead>
                            <tbody class="table-body" id="openTableBody">
                                <!-- Job cards will be loaded dynamically -->
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination Controls -->
                    <div class="pagination-container" id="openPagination" style="display: none;">
                        <div class="pagination-info">
                            <span id="openPaginationInfo">Showing 1-10 of 0 job cards</span>
                        </div>
                        <div class="pagination-controls">
                            <button class="pagination-btn" id="openPrevBtn" onclick="changePage('open', -1)" disabled>
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="15,18 9,12 15,6"/>
                                </svg>
                                Previous
                            </button>
                            <div class="pagination-pages" id="openPageNumbers">
                                <!-- Page numbers will be generated dynamically -->
                            </div>
                            <button class="pagination-btn" id="openNextBtn" onclick="changePage('open', 1)" disabled>
                                Next
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="9,18 15,12 9,6"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <div id="completed" class="tab-pane">
                <div class="action-bar">
                    <div class="action-content">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search completed job cards..." id="completedSearch">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="table-section">
                    <div class="table-container">
                        <table class="workorder-table">
                            <thead class="table-header">
                                <tr>
                                    <th style="width: 50px;">S.No</th>
                                    <th>Action</th>
                                    <th>Job Card Number</th>
                                    <th>Created Date</th>
                                    <th>Vehicle</th>
                                    <th>Completed Date</th>
                                    <th>Status</th>
                                    <th>Priority</th>
                                    <th>Assigned</th>
                                    <th>Memo</th>
                                </tr>
                            </thead>
                            <tbody class="table-body" id="completedTableBody">
                                <!-- Completed job cards will be loaded dynamically -->
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination Controls -->
                    <div class="pagination-container" id="completedPagination" style="display: none;">
                        <div class="pagination-info">
                            <span id="completedPaginationInfo">Showing 1-10 of 0 job cards</span>
                        </div>
                        <div class="pagination-controls">
                            <button class="pagination-btn" id="completedPrevBtn" onclick="changePage('completed', -1)" disabled>
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="15,18 9,12 15,6"/>
                                </svg>
                                Previous
                            </button>
                            <div class="pagination-pages" id="completedPageNumbers">
                                <!-- Page numbers will be generated dynamically -->
                            </div>
                            <button class="pagination-btn" id="completedNextBtn" onclick="changePage('completed', 1)" disabled>
                                Next
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="9,18 15,12 9,6"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <div id="archive" class="tab-pane">
                <div class="action-bar">
                    <div class="action-content">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search archived job cards..." id="archiveSearch">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="table-section">
                    <div class="table-container">
                        <table class="workorder-table">
                            <thead class="table-header">
                                <tr>
                                    <th style="width: 50px;">S.No</th>
                                    <th>Action</th>
                                    <th>Job Card Number</th>
                                    <th>Created Date</th>
                                    <th>Vehicle</th>
                                    <th>Archived Date</th>
                                    <th>Status</th>
                                    <th>Priority</th>
                                    <th>Assigned</th>
                                    <th>Memo</th>
                                </tr>
                            </thead>
                            <tbody class="table-body" id="archiveTableBody">
                                <!-- Archived job cards will be loaded dynamically -->
                            </tbody>
                        </table>
                    </div>
                    
                    <!-- Pagination Controls -->
                    <div class="pagination-container" id="archivePagination" style="display: none;">
                        <div class="pagination-info">
                            <span id="archivePaginationInfo">Showing 1-10 of 0 job cards</span>
                        </div>
                        <div class="pagination-controls">
                            <button class="pagination-btn" id="archivePrevBtn" onclick="changePage('archive', -1)" disabled>
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="15,18 9,12 15,6"/>
                                </svg>
                                Previous
                            </button>
                            <div class="pagination-pages" id="archivePageNumbers">
                                <!-- Page numbers will be generated dynamically -->
                            </div>
                            <button class="pagination-btn" id="archiveNextBtn" onclick="changePage('archive', 1)" disabled>
                                Next
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="9,18 15,12 9,6"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Copy all styles from Workorder.blade.php */
.workorder-container {
    min-height: 100vh;
    background: #f8fafc;
}

.workorder-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 2rem 0;
}

.header-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 2rem;
}

.page-title h1 {
    font-size: 2.5rem;
    font-weight: 700;
    margin: 0 0 0.5rem 0;
}

.page-title p {
    font-size: 1.1rem;
    opacity: 0.9;
    margin: 0;
}

.workorder-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 2rem;
    background: white;
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    margin-top: -2rem;
    position: relative;
    z-index: 1;
}

.tab-navigation {
    margin-bottom: 2rem;
}

.tab-list {
    display: flex;
    gap: 1rem;
    border-bottom: 2px solid #e2e8f0;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem 1.5rem;
    background: none;
    border: none;
    color: #64748b;
    font-size: 1rem;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
    border-bottom: 3px solid transparent;
}

.tab-btn:hover {
    color: #3b82f6;
    background: #f1f5f9;
}

.tab-btn.active {
    color: #3b82f6;
    border-bottom-color: #3b82f6;
    background: #f8fafc;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 1.5rem;
    border-radius: 8px 8px 0 0;
}

.action-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 1rem;
}

.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.action-buttons {
    display: flex;
    gap: 0.75rem;
    align-items: center;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

.btn-create {
    background: #3b82f6 !important;
    color: white !important;
    border-color: #3b82f6 !important;
}

.btn-create:hover {
    background: #2563eb !important;
    border-color: #2563eb !important;
}

.btn-warning {
    background: #f59e0b !important;
    color: white !important;
    border-color: #f59e0b !important;
}

.btn-warning:hover {
    background: #d97706 !important;
    border-color: #d97706 !important;
}

.btn-secondary {
    background: #6b7280 !important;
    color: white !important;
    border-color: #6b7280 !important;
}

.btn-secondary:hover {
    background: #4b5563 !important;
    border-color: #4b5563 !important;
}

.btn-success {
    background: #10b981 !important;
    color: white !important;
    border-color: #10b981 !important;
}

.btn-success:hover {
    background: #059669 !important;
    border-color: #059669 !important;
}

.table-section {
    background: white;
}

.table-container {
    overflow-x: auto;
}

.workorder-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.table-header {
    background: #f8fafc;
}

.table-header th {
    padding: 1rem;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
    white-space: nowrap;
}

.table-body tr {
    border-bottom: 1px solid #e5e7eb;
    transition: all 0.2s ease;
}

.table-body tr:hover {
    background: #f8fafc;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.table-body td {
    padding: 1rem;
    vertical-align: middle;
}

.btn-view {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 8px 12px;
    background: #3b82f6;
    color: white;
    border: none;
    border-radius: 6px;
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-view:hover {
    background: #2563eb;
    transform: translateY(-1px);
}

/* ===== DROPDOWN STYLES ===== */
.dropdown-container {
    position: relative;
    display: inline-block;
}

.action-dropdown-btn {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 8px 16px;
    background: #0055A5;
    color: white;
    border: none;
    border-radius: 6px;
    font-size: 13px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    position: relative;
}

.action-dropdown-btn:hover {
    background: #003f7d;
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(0, 85, 165, 0.3);
}

.dropdown-container.active .action-dropdown-btn {
    background: #003f7d;
}

.dropdown-menu-custom {
    position: absolute;
    top: 100%;
    left: 0;
    background: white;
    border: 1px solid #e2e8f0;
    border-radius: 6px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    z-index: 1000;
    min-width: 160px;
    display: none;
    margin-top: 4px;
    overflow: hidden;
}

.dropdown-container.active .dropdown-menu-custom {
    display: block;
}

.dropdown-item-custom {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 10px 14px;
    color: #374151;
    background: none;
    border: none;
    width: 100%;
    text-align: left;
    text-decoration: none;
    font-size: 13px;
    font-weight: 500;
    transition: all 0.2s ease;
    border-bottom: 1px solid #f3f4f6;
    cursor: pointer;
}

.dropdown-item-custom:last-child {
    border-bottom: none;
}

.dropdown-item-custom:hover {
    background: #f8fafc;
    color: #1f2937;
}

.dropdown-item-custom.delete:hover {
    background: #fee2e2;
    color: #991b1b;
}

.dropdown-item-custom.delete:hover svg {
    color: #991b1b;
}

.dropdown-item-custom svg {
    color: #6b7280;
}

/* PDF Viewer Modal */
.pdf-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.8);
    display: none;
    justify-content: center;
    align-items: center;
    z-index: 10000;
}

.pdf-modal-overlay.active {
    display: flex;
}

.pdf-modal-container {
    background: white;
    width: 90%;
    height: 90%;
    max-width: 1200px;
    border-radius: 12px;
    overflow: hidden;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
    display: flex;
    flex-direction: column;
}

.pdf-modal-header {
    background: #0055A5;
    color: white;
    padding: 20px 24px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.pdf-modal-title {
    font-size: 20px;
    font-weight: 600;
    margin: 0;
}

.pdf-modal-close {
    background: none;
    border: none;
    color: white;
    font-size: 28px;
    cursor: pointer;
    padding: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 4px;
    transition: background 0.2s ease;
}

.pdf-modal-close:hover {
    background: rgba(255, 255, 255, 0.1);
}

.pdf-modal-body {
    flex: 1;
    padding: 0;
    overflow: hidden;
}

.pdf-frame {
    width: 100%;
    height: 100%;
    border: none;
}

.pdf-loading {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    height: 100%;
    gap: 16px;
}

.pdf-loading-spinner {
    width: 48px;
    height: 48px;
    border: 4px solid #e2e8f0;
    border-top-color: #0055A5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    to { transform: rotate(360deg); }
}

.pdf-loading-text {
    color: #64748b;
    font-size: 16px;
    font-weight: 500;
}

.status-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
}

.status-badge.success {
    background: #dcfce7;
    color: #166534;
}

.status-badge.primary {
    background: #dbeafe;
    color: #1e40af;
}

.status-badge.warning {
    background: #fef3c7;
    color: #92400e;
}

.status-badge.info {
    background: #e0f2fe;
    color: #0c4a6e;
}

.status-badge.danger {
    background: #fee2e2;
    color: #991b1b;
}

.status-badge.secondary {
    background: #f3f4f6;
    color: #374151;
}

.priority-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
}

.priority-badge.danger {
    background: #fee2e2;
    color: #991b1b;
}

.priority-badge.warning {
    background: #fef3c7;
    color: #92400e;
}

.priority-badge.success {
    background: #dcfce7;
    color: #166534;
}

.priority-badge.secondary {
    background: #f3f4f6;
    color: #374151;
}

.no-data-cell {
    text-align: center;
    padding: 3rem !important;
}

.no-data {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 1rem;
    color: #64748b;
}

.no-data svg {
    opacity: 0.5;
}

.no-data h3 {
    margin: 0;
    font-size: 1.25rem;
    font-weight: 600;
}

.no-data p {
    margin: 0;
    font-size: 1rem;
}

/* ===== PAGINATION STYLES ===== */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 24px;
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
}

.pagination-info {
    color: #64748b;
    font-size: 14px;
    font-weight: 500;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 8px;
}

.pagination-btn {
    display: flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.pagination-btn:hover:not(:disabled) {
    background: #f3f4f6;
    border-color: #9ca3af;
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    background: #f9fafb;
}

.pagination-pages {
    display: flex;
    align-items: center;
    gap: 4px;
}

.page-number {
    padding: 8px 12px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    min-width: 40px;
    text-align: center;
}

.page-number:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
}

.page-number.active {
    background: #3b82f6;
    border-color: #3b82f6;
    color: white;
}

.page-number.active:hover {
    background: #2563eb;
    border-color: #2563eb;
}
</style>

<!-- Firebase SDK -->
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>

<script>
// Firebase Configuration
const firebaseConfig = {
    apiKey: "AIzaSyDsA1JohORkg-EdXWTHlFEea8zCqfvig8w",
    authDomain: "titan-drilling-1f8e9.firebaseapp.com",
    databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com",
    projectId: "titan-drilling-1f8e9",
    storageBucket: "titan-drilling-1f8e9.firebasestorage.app",
    messagingSenderId: "749788854975",
    appId: "1:749788854975:web:71224235d9355be09b640a",
    measurementId: "G-R9L21J6GHD"
};

// Initialize Firebase
firebase.initializeApp(firebaseConfig);
const database = firebase.database();

// Global variables
let allJobCards = [];
let currentPage = { open: 1, completed: 1, archive: 1 };
let itemsPerPage = 10;
let filteredJobCards = { open: [], completed: [], archive: [] };

// Fetch job cards from Firebase (same as work orders)
async function fetchJobCards() {
    try {
        console.log('Fetching job cards from Firebase...');
        
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalWorkOrders/admins/cahj3jJqe5Z16pQdLGSWrNgm84o1.json');
        
        if (response.ok) {
            const data = await response.json();
            
            if (data) {
                // Convert object to array
                allJobCards = Object.keys(data).map(key => ({
                    id: key,
                    ...data[key]
                }));
                
                console.log('Fetched job cards:', allJobCards.length);
                
                // Filter job cards by status
                filterJobCardsByStatus();
                
                // Render all tabs
                renderJobCards('open');
                renderJobCards('completed');
                renderJobCards('archive');
                
            } else {
                console.log('No job cards found');
                allJobCards = [];
            }
        } else {
            console.error('Failed to fetch job cards:', response.status);
        }
        
    } catch (error) {
        console.error('Error fetching job cards:', error);
    }
}

// Filter job cards by status
function filterJobCardsByStatus() {
    filteredJobCards.open = allJobCards.filter(jc => {
        const status = jc.status ? jc.status.toLowerCase() : '';
        return !['completed', 'closed', 'archived', 'cancelled'].includes(status);
    });
    
    filteredJobCards.completed = allJobCards.filter(jc => {
        const status = jc.status ? jc.status.toLowerCase() : '';
        return ['completed', 'closed'].includes(status);
    });
    
    filteredJobCards.archive = allJobCards.filter(jc => {
        const status = jc.status ? jc.status.toLowerCase() : '';
        return ['archived', 'cancelled'].includes(status);
    });
    
    console.log('Filtered job cards:', {
        open: filteredJobCards.open.length,
        completed: filteredJobCards.completed.length,
        archive: filteredJobCards.archive.length
    });
}

// Render job cards for a specific tab
function renderJobCards(tabName) {
    const jobCards = filteredJobCards[tabName];
    const startIndex = (currentPage[tabName] - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const pageJobCards = jobCards.slice(startIndex, endIndex);
    
    const tbody = document.getElementById(tabName + 'TableBody');
    const pagination = document.getElementById(tabName + 'Pagination');
    const paginationInfo = document.getElementById(tabName + 'PaginationInfo');
    
    if (!tbody) return;
    
    // Clear existing content
    tbody.innerHTML = '';
    
    if (pageJobCards.length === 0) {
        tbody.innerHTML = `
            <tr>
                <td colspan="10" class="no-data-cell">
                    <div class="no-data">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"/>
                            <path d="M12 6v6l4 2"/>
                        </svg>
                        <h3>No Data Available</h3>
                        <p>No ${tabName} job cards found.</p>
                    </div>
                </td>
            </tr>
        `;
        if (pagination) pagination.style.display = 'none';
    } else {
        // Render job cards
        pageJobCards.forEach((jobCard, index) => {
            const globalIndex = startIndex + index + 1;
            const row = createJobCardRow(jobCard, globalIndex, tabName);
            tbody.appendChild(row);
        });
        
        // Show pagination
        if (pagination) {
            pagination.style.display = 'flex';
            updatePaginationInfo(tabName, jobCards.length);
            updatePaginationControls(tabName, jobCards.length);
        }
    }
}

// Create job card row
function createJobCardRow(jobCard, index, tabName) {
    const row = document.createElement('tr');
    row.className = 'table-row';
    
    const statusColor = getStatusColorClass(jobCard.status);
    const priorityColor = getPriorityColorClass(jobCard.priority);
    
    // Get equipment name
    const equipmentName = jobCard.equipment ? 
        (jobCard.equipment.equipmentName || jobCard.equipment.name || jobCard.equipment.vehicleNumber || 'N/A') : 
        'N/A';
    
    // Get assigned users
    const assignedUsers = jobCard.assignTo ? 
        (Array.isArray(jobCard.assignTo) ? jobCard.assignTo.length : 1) : 
        0;
    
    // Format dates
    const createdDate = jobCard.createdAt ? 
        new Date(jobCard.createdAt).toLocaleDateString() : 
        'N/A';
    
    const dueDate = jobCard.dueDate ? 
        new Date(jobCard.dueDate).toLocaleDateString() : 
        'N/A';
    
    // Escape strings for HTML attributes
    const safeJobCardId = (jobCard.id || '').replace(/'/g, "\\'").replace(/"/g, '&quot;');
    const safePdfUrl = (jobCard.pdfDownloadUrl || '').replace(/'/g, "\\'").replace(/"/g, '&quot;');
    const safeRequestNumber = (jobCard.requestNumber || jobCard.id || '').replace(/'/g, "\\'").replace(/"/g, '&quot;');
    
    row.innerHTML = `
        <td style="text-align: center; font-weight: 500; color: #64748b;">${index}</td>
        <td>
            <div class="dropdown-container" id="dropdown-container-${index}">
                <button class="action-dropdown-btn" onclick="toggleDropdown(${index})" type="button">
                    Action
                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <polyline points="6,9 12,15 18,9"/>
                    </svg>
                </button>
                <div class="dropdown-menu-custom">
                    <button class="dropdown-item-custom" onclick="viewPDFModal('${safePdfUrl}', '${safeRequestNumber}'); event.stopPropagation();">
                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                            <polyline points="14,2 14,8 20,8"/>
                        </svg>
                        View PDF
                    </button>
                    <button class="dropdown-item-custom delete" onclick="deleteJobCard('${safeJobCardId}', '${safeRequestNumber}'); event.stopPropagation();">
                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <polyline points="3,6 5,6 21,6"/>
                            <path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"/>
                        </svg>
                        Delete Report
                    </button>
                </div>
            </div>
        </td>
        <td>
            <div class="wo-number">
                <strong>${jobCard.requestNumber || jobCard.id}</strong>
            </div>
        </td>
        <td>
            <div class="date-info">
                <span class="date">${createdDate}</span>
            </div>
        </td>
        <td>
            <div class="vehicle-info">
                <span class="vehicle-name">${equipmentName}</span>
            </div>
        </td>
        <td>
            <div class="date-info">
                <span class="date">${dueDate}</span>
            </div>
        </td>
        <td>
            <span class="status-badge ${statusColor}">${jobCard.status || 'N/A'}</span>
        </td>
        <td>
            <span class="priority-badge ${priorityColor}">${jobCard.priority || 'N/A'}</span>
        </td>
        <td>
            <div class="assigned-info">
                <span class="assigned-count">${assignedUsers}</span>
            </div>
        </td>
        <td>
            <div class="memo-info">
                <span class="memo-text">${jobCard.memo || 'N/A'}</span>
            </div>
        </td>
    `;
    
    return row;
}

// Get status color class
function getStatusColorClass(status) {
    if (!status) return 'info';
    const s = status.toLowerCase();
    if (['completed', 'closed'].includes(s)) return 'success';
    if (['in progress', 'processing'].includes(s)) return 'primary';
    if (['on hold'].includes(s)) return 'warning';
    if (['initiated'].includes(s)) return 'info';
    if (['rework'].includes(s)) return 'danger';
    if (['archived', 'cancelled'].includes(s)) return 'secondary';
    return 'info';
}

// Get priority color class
function getPriorityColorClass(priority) {
    if (!priority) return 'secondary';
    const p = priority.toLowerCase();
    if (['high', 'urgent'].includes(p)) return 'danger';
    if (['medium'].includes(p)) return 'warning';
    if (['low'].includes(p)) return 'success';
    return 'secondary';
}

// Update pagination info
function updatePaginationInfo(tabName, totalItems) {
    const paginationInfo = document.getElementById(tabName + 'PaginationInfo');
    if (!paginationInfo) return;
    
    const startIndex = (currentPage[tabName] - 1) * itemsPerPage + 1;
    const endIndex = Math.min(currentPage[tabName] * itemsPerPage, totalItems);
    
    paginationInfo.textContent = `Showing ${startIndex}-${endIndex} of ${totalItems} job cards`;
}

// Update pagination controls
function updatePaginationControls(tabName, totalItems) {
    const totalPages = Math.ceil(totalItems / itemsPerPage);
    const prevBtn = document.getElementById(tabName + 'PrevBtn');
    const nextBtn = document.getElementById(tabName + 'NextBtn');
    const pageNumbers = document.getElementById(tabName + 'PageNumbers');
    
    // Update prev/next buttons
    if (prevBtn) prevBtn.disabled = currentPage[tabName] <= 1;
    if (nextBtn) nextBtn.disabled = currentPage[tabName] >= totalPages;
    
    // Generate page numbers
    if (pageNumbers && totalPages > 1) {
        pageNumbers.innerHTML = '';
        
        const startPage = Math.max(1, currentPage[tabName] - 2);
        const endPage = Math.min(totalPages, startPage + 4);
        
        for (let i = startPage; i <= endPage; i++) {
            const pageBtn = document.createElement('button');
            pageBtn.className = `page-number ${i === currentPage[tabName] ? 'active' : ''}`;
            pageBtn.textContent = i;
            pageBtn.onclick = () => goToPage(tabName, i);
            pageNumbers.appendChild(pageBtn);
        }
    }
}

// Change page
function changePage(tabName, direction) {
    const totalPages = Math.ceil(filteredJobCards[tabName].length / itemsPerPage);
    const newPage = currentPage[tabName] + direction;
    
    if (newPage >= 1 && newPage <= totalPages) {
        currentPage[tabName] = newPage;
        renderJobCards(tabName);
    }
}

// Go to specific page
function goToPage(tabName, page) {
    const totalPages = Math.ceil(filteredJobCards[tabName].length / itemsPerPage);
    
    if (page >= 1 && page <= totalPages) {
        currentPage[tabName] = page;
        renderJobCards(tabName);
    }
}

// Tab switching functionality
function switchTab(tabName) {
    // Update tab buttons
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');
    
    // Update tab panes
    document.querySelectorAll('.tab-pane').forEach(pane => {
        pane.classList.remove('active');
    });
    document.getElementById(tabName).classList.add('active');
    
    // Reset to first page for the new tab
    currentPage[tabName] = 1;
    renderJobCards(tabName);
}

// Search functionality
function setupSearch() {
    const searchInputs = ['openSearch', 'completedSearch', 'archiveSearch'];
    
    searchInputs.forEach(inputId => {
        const input = document.getElementById(inputId);
        if (input) {
            input.addEventListener('input', function() {
                const tabName = inputId.replace('Search', '');
                searchJobCards(tabName, this.value);
            });
        }
    });
}

// Search job cards
function searchJobCards(tabName, searchTerm) {
    const allJobCardsForTab = filteredJobCards[tabName];
    
    if (!searchTerm.trim()) {
        // Reset to original filtered job cards
        filteredJobCards[tabName] = allJobCardsForTab;
    } else {
        // Filter based on search term
        filteredJobCards[tabName] = allJobCardsForTab.filter(jobCard => {
            const searchLower = searchTerm.toLowerCase();
            return (
                (jobCard.requestNumber && jobCard.requestNumber.toLowerCase().includes(searchLower)) ||
                (jobCard.title && jobCard.title.toLowerCase().includes(searchLower)) ||
                (jobCard.memo && jobCard.memo.toLowerCase().includes(searchLower)) ||
                (jobCard.status && jobCard.status.toLowerCase().includes(searchLower)) ||
                (jobCard.priority && jobCard.priority.toLowerCase().includes(searchLower))
            );
        });
    }
    
    // Reset to first page and render
    currentPage[tabName] = 1;
    renderJobCards(tabName);
}

// Toggle dropdown function
function toggleDropdown(jobCardId) {
    // Close all other dropdowns
    document.querySelectorAll('.dropdown-menu').forEach(menu => {
        if (menu.id !== `dropdown-${jobCardId}`) {
            menu.classList.remove('show');
        }
    });
    
    // Toggle current dropdown
    const dropdown = document.getElementById(`dropdown-${jobCardId}`);
    if (dropdown) {
        dropdown.classList.toggle('show');
    }
}

// Close dropdowns when clicking outside
document.addEventListener('click', function(event) {
    if (!event.target.closest('.dropdown')) {
        document.querySelectorAll('.dropdown-menu').forEach(menu => {
            menu.classList.remove('show');
        });
    }
});

// View job card function
function viewJobCard(jobCardId) {
    console.log('Viewing job card details:', jobCardId);
    // Close dropdown
    const dropdown = document.getElementById(`dropdown-${jobCardId}`);
    if (dropdown) {
        dropdown.classList.remove('show');
    }
    
    // You can implement the view details functionality here
    // For now, just show an alert
    alert(`Viewing job card details: ${jobCardId}`);
}

// View PDF function
function viewPDF(jobCardId, pdfUrl) {
    console.log('Viewing PDF for job card:', jobCardId, 'URL:', pdfUrl);
    
    // Close dropdown
    const dropdown = document.getElementById(`dropdown-${jobCardId}`);
    if (dropdown) {
        dropdown.classList.remove('show');
    }
    
    if (pdfUrl && pdfUrl.trim() !== '') {
        // Validate PDF URL format
        if (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://')) {
            alert('Invalid PDF URL format. The PDF may not be available yet.');
            return;
        }
        
        // Open PDF from Firebase Storage in new tab
        console.log('Opening PDF from Firebase Storage:', pdfUrl);
        
        try {
            const pdfWindow = window.open(pdfUrl, '_blank');
            
            // Check if popup was blocked
            if (!pdfWindow || pdfWindow.closed || typeof pdfWindow.closed == 'undefined') {
                alert('Popup blocked! Please allow popups for this site and try again.');
                return;
            }
            
            // Optional: Focus the new window
            pdfWindow.focus();
            
            console.log('PDF opened successfully in new tab');
            
        } catch (error) {
            console.error('Error opening PDF:', error);
            alert('Failed to open PDF. Please check your browser settings and try again.');
        }
    } else {
        alert('PDF not available for this job card. It may not have been generated yet.');
    }
}

// Additional functions for job card specific actions
function displayRequestNumberInfo() {
    // This function will show the next request number info
    alert('Request Number Info: This will show the next available request number for job cards.');
}

function sortJobCards() {
    alert('Sort Job Cards: This will open sorting options for job cards.');
}

function filterJobCards() {
    alert('Filter Job Cards: This will open filtering options for job cards.');
}

function exportToExcel() {
    alert('Export to Excel: This will export all job cards to an Excel file.');
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initializing job card page...');
    
    // Setup search functionality
    setupSearch();
    
    // Setup tab switching
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const tabName = this.getAttribute('data-tab');
            switchTab(tabName);
        });
    });
    
    // Fetch job cards from Firebase
    fetchJobCards();
});

// Add functions to global scope
window.switchTab = switchTab;
window.changePage = changePage;
window.goToPage = goToPage;
window.viewJobCard = viewJobCard;
window.toggleDropdown = toggleDropdown;
window.viewPDF = viewPDF;
window.displayRequestNumberInfo = displayRequestNumberInfo;
window.sortJobCards = sortJobCards;
window.filterJobCards = filterJobCards;
window.exportToExcel = exportToExcel;
</script>

<!-- PDF Viewer Modal -->
<div id="pdfModal" class="pdf-modal-overlay">
    <div class="pdf-modal-container">
        <div class="pdf-modal-header">
            <h2 class="pdf-modal-title" id="pdfModalTitle">Job Card PDF</h2>
            <button class="pdf-modal-close" onclick="closePDFModal()">&times;</button>
        </div>
        <div class="pdf-modal-body">
            <div id="pdfLoadingState" class="pdf-loading">
                <div class="pdf-loading-spinner"></div>
                <p class="pdf-loading-text">Loading PDF...</p>
            </div>
            <iframe id="pdfFrame" class="pdf-frame" style="display: none;"></iframe>
        </div>
    </div>
</div>

<script>
// Toggle dropdown function
function toggleDropdown(index) {
    // Close all other dropdowns
    document.querySelectorAll('.dropdown-container').forEach(container => {
        if (container.id !== `dropdown-container-${index}`) {
            container.classList.remove('active');
        }
    });
    
    // Toggle current dropdown
    const dropdown = document.getElementById(`dropdown-container-${index}`);
    if (dropdown) {
        dropdown.classList.toggle('active');
    }
}

// Close dropdowns when clicking outside
document.addEventListener('click', function(event) {
    if (!event.target.closest('.dropdown-container')) {
        document.querySelectorAll('.dropdown-container').forEach(container => {
            container.classList.remove('active');
        });
    }
});

// View PDF in modal
function viewPDFModal(pdfUrl, requestNumber) {
    console.log('📂 Opening PDF modal for:', requestNumber);
    
    if (!pdfUrl || pdfUrl.trim() === '' || pdfUrl === 'undefined') {
        showNotification('error', 'PDF not available for this job card. It may not have been generated yet.');
        return;
    }
    
    // Validate PDF URL format
    if (!pdfUrl.startsWith('http://') && !pdfUrl.startsWith('https://')) {
        showNotification('error', 'Invalid PDF URL format. The PDF may not be available yet.');
        return;
    }
    
    const modal = document.getElementById('pdfModal');
    const pdfFrame = document.getElementById('pdfFrame');
    const pdfLoadingState = document.getElementById('pdfLoadingState');
    const pdfModalTitle = document.getElementById('pdfModalTitle');
    
    // Update title
    pdfModalTitle.textContent = `Job Card: ${requestNumber}`;
    
    // Show modal and loading state
    modal.classList.add('active');
    pdfLoadingState.style.display = 'flex';
    pdfFrame.style.display = 'none';
    
    // Load PDF in iframe
    pdfFrame.src = pdfUrl;
    
    // Handle iframe load
    pdfFrame.onload = function() {
        pdfLoadingState.style.display = 'none';
        pdfFrame.style.display = 'block';
        console.log('✅ PDF loaded successfully');
    };
    
    // Handle iframe error
    pdfFrame.onerror = function() {
        pdfLoadingState.innerHTML = '<p class="pdf-loading-text" style="color: #dc3545;">Failed to load PDF. Please try again.</p>';
        console.error('❌ Error loading PDF');
    };
}

// Close PDF modal
function closePDFModal() {
    const modal = document.getElementById('pdfModal');
    const pdfFrame = document.getElementById('pdfFrame');
    
    modal.classList.remove('active');
    pdfFrame.src = '';
    
    console.log('🚪 PDF modal closed');
}

// Close modal when clicking outside
document.getElementById('pdfModal')?.addEventListener('click', function(event) {
    if (event.target === this) {
        closePDFModal();
    }
});

// Delete job card function
async function deleteJobCard(jobCardId, requestNumber) {
    console.log('🗑️ Delete job card:', jobCardId, requestNumber);
    
    // Close all dropdowns
    document.querySelectorAll('.dropdown-container').forEach(container => {
        container.classList.remove('active');
    });
    
    // Show confirmation dialog
    if (!confirm(`⚠️ Are you sure you want to DELETE this job card?\n\nJob Card: ${requestNumber}\n\n⚠️ This action CANNOT be undone!`)) {
        console.log('❌ Delete cancelled by user');
        return;
    }
    
    try {
        console.log('🔄 Deleting job card from MongoDB...');
        
        // Call MongoDB delete API
        const response = await fetch(`https://api.titandrillingzm.com:6005/jobcards/${jobCardId}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json'
            }
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('✅ Delete response:', result);
        
        if (result.success) {
            showNotification('success', 'Job card deleted successfully!');
            
            // Refresh the job cards list
            setTimeout(() => {
                fetchJobCards();
            }, 1000);
        } else {
            throw new Error(result.error || 'Failed to delete job card');
        }
        
    } catch (error) {
        console.error('❌ Error deleting job card:', error);
        showNotification('error', 'Failed to delete job card: ' + error.message);
    }
}

// Show notification
function showNotification(type, message) {
    // Create notification element
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: ${type === 'success' ? '#10b981' : '#ef4444'};
        color: white;
        padding: 16px 24px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        z-index: 10001;
        font-size: 14px;
        font-weight: 500;
        max-width: 400px;
        animation: slideInRight 0.3s ease;
    `;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        notification.style.animation = 'slideOutRight 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 5000);
}

// Add animation styles
const style = document.createElement('style');
style.textContent = `
    @keyframes slideInRight {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    @keyframes slideOutRight {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(100%);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

// Add functions to global scope
window.viewPDFModal = viewPDFModal;
window.closePDFModal = closePDFModal;
window.deleteJobCard = deleteJobCard;
</script>
@endsection
