@extends('welcome')

@section('title', 'Service Schedule')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

<meta name="csrf-token" content="{{ csrf_token() }}">

@php
    function getStatusColor($status) {
        $status = strtolower($status);
        switch ($status) {
            case 'completed':
            case 'closed':
            case 'done':
                return 'success';
            case 'in progress':
            case 'processing':
            case 'pending':
                return 'primary';
            case 'on hold':
            case 'overdue':
                return 'danger';
            case 'initiated':
            case 'scheduled':
                return 'info';
            case 'cancelled':
                return 'secondary';
            default:
                return 'info';
        }
    }

    function getDueStatusColor($dueDate) {
        if ($dueDate === 'N/A') return 'secondary';
        
        $due = \Carbon\Carbon::parse($dueDate);
        $now = \Carbon\Carbon::now();
        
        if ($due->isPast()) return 'danger'; // Overdue
        if ($due->diffInDays($now) <= 7) return 'warning'; // Due soon
        return 'success'; // On time
    }
@endphp

@section('content')
<div class="service-schedule-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="javascript:history.back()" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Service Schedule</h1>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="search-icon">
                        <circle cx="11" cy="11" r="8"/>
                        <path d="M21 21l-4.35-4.35"/>
                    </svg>
                    <input type="text" class="search-input" placeholder="Search service schedules..." id="scheduleSearch">
                </div>
                <a href="{{ route('maintenance.create_schedule') }}" class="create-btn">
                    <i class="fas fa-plus"></i>
                    Create Schedule
                </a>
            </div>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="main-content">
        <div class="service-schedule-content">
        <div class="table-section">
            <div class="table-container">
                <table class="service-schedule-table">
                    <thead class="table-header">
                        <tr>
                            <th style="width: 60px;">S.No</th>
                            <th style="width: 120px;">Action</th>
                            <th>Schedule ID</th>
                            <th>Vehicle Number</th>
                            <th>Due Date</th>
                            <th>Status</th>
                            <th>Workorder</th>
                            <th>Schedule Title</th>
                            <th>Last Service Date</th>
                        </tr>
                    </thead>
                    <tbody class="table-body" id="serviceScheduleTableBody">
                        <tr class="loading-row">
                            <td colspan="9" class="loading-state">
                                <div class="loading-content">
                                    <div class="loading-spinner"></div>
                                    <h3>Loading Service Schedules...</h3>
                                    <p>Please wait while we fetch your data from MongoDB.</p>
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination Controls -->
            <div class="pagination-container" id="paginationContainer" style="display: none;">
                <div class="pagination-info">
                    Showing <span id="showingStart">0</span> to <span id="showingEnd">0</span> of <span id="totalRecords">0</span> entries
                </div>
                <div class="pagination-buttons" id="paginationButtons">
                    <!-- Pagination buttons will be inserted here -->
                </div>
            </div>
        </div>
    </div>
</div>

<!-- View Schedule Details Modal -->
<div id="viewScheduleModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content" style="max-width: 900px;">
        <div class="custom-modal-header">
            <h3>Service Schedule Details</h3>
            <button type="button" class="close-btn" onclick="closeViewScheduleModal()">&times;</button>
        </div>
        <div class="custom-modal-body" id="viewScheduleBody" style="max-height: 70vh; overflow-y: auto;">
            <!-- Details will be populated here -->
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-primary" onclick="closeViewScheduleModal()">Close</button>
        </div>
    </div>
</div>

<!-- Delete Schedule Confirmation Modal -->
<div id="deleteScheduleModal" class="custom-modal" style="display: none;">
    <div class="custom-modal-content" style="max-width: 500px;">
        <div class="custom-modal-header">
            <h3>Confirm Delete</h3>
            <button type="button" class="close-btn" onclick="closeDeleteScheduleModal()">&times;</button>
        </div>
        <div class="custom-modal-body">
            <div style="text-align: center; padding: 20px;">
                <i class="fas fa-exclamation-triangle" style="font-size: 48px; color: #dc3545; margin-bottom: 20px;"></i>
                <h4 style="margin-bottom: 10px; color: #374151;">Are you sure?</h4>
                <p style="color: #6b7280; margin-bottom: 0;">This service schedule will be permanently deleted. This action cannot be undone.</p>
            </div>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-secondary" onclick="closeDeleteScheduleModal()">Cancel</button>
            <button type="button" class="btn btn-danger" onclick="confirmDeleteSchedule()">Delete</button>
        </div>
    </div>
</div>

<script>
(function() {
    'use strict';

    // Configuration
    const CONFIG = {
        pagination: {
            itemsPerPage: 10,
            maxVisiblePages: 5
        }
    };

    // Pagination variables
    let allSchedules = [];
    let currentPage = 1;
    let totalPages = 0;

    // Initialize
    document.addEventListener('DOMContentLoaded', function() {
        loadServiceSchedules();
        
        // Search functionality
        const scheduleSearch = document.getElementById('scheduleSearch');
        if (scheduleSearch) {
            scheduleSearch.addEventListener('input', function() {
                filterTable(this.value);
            });
        }
    });

    // Load service schedules from MongoDB
    async function loadServiceSchedules() {
        try {
            console.log('🔄 Loading service schedules from MongoDB...');
            
            const response = await fetch('https://api.titandrillingzm.com:6010/service-schedules');
            const result = await response.json();
            
            console.log('Raw MongoDB data:', result);
            
            if (result.success && result.data && Array.isArray(result.data)) {
                allSchedules = result.data;
                
                console.log(`Found ${allSchedules.length} total service schedules`);
                
                // Sort by creation date (newest first)
                allSchedules.sort((a, b) => {
                    const dateA = new Date(a.createdAt || 0).getTime();
                    const dateB = new Date(b.createdAt || 0).getTime();
                    return dateB - dateA;
                });
                
                // Calculate total pages
                totalPages = Math.ceil(allSchedules.length / CONFIG.pagination.itemsPerPage);
                currentPage = 1;
                
                // Display first page
                displayCurrentPage();
                
                // Show pagination
                updatePagination();
                
                if (allSchedules.length > 0) {
                    showNotification(`Loaded ${allSchedules.length} service schedules successfully`, 'success');
                } else {
                    showNotification('No service schedules found', 'info');
                }
            } else {
                console.log('No data found in MongoDB or invalid format');
                displayEmptyState();
            }
        } catch (error) {
            console.error('❌ Error loading service schedules:', error);
            showNotification('Failed to load service schedules: ' + error.message, 'error');
            displayEmptyState();
        }
    }

    // Display current page of schedules
    function displayCurrentPage() {
        const tableBody = document.getElementById('serviceScheduleTableBody');
        if (!tableBody) return;
        
        if (allSchedules.length === 0) {
            displayEmptyState();
            return;
        }
        
        const startIndex = (currentPage - 1) * CONFIG.pagination.itemsPerPage;
        const endIndex = Math.min(startIndex + CONFIG.pagination.itemsPerPage, allSchedules.length);
        const pageSchedules = allSchedules.slice(startIndex, endIndex);
        
        const rows = pageSchedules.map((schedule, index) => {
            const serialNumber = startIndex + index + 1;
            
            // Format dates
            let formattedLastService = 'N/A';
            if (schedule.lastServiceDate) {
                try {
                    const date = new Date(schedule.lastServiceDate);
                    formattedLastService = date.toLocaleDateString('en-GB', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric'
                    });
                } catch (error) {
                    console.error('Error formatting date:', error);
                }
            }
            
            // Extract equipment name
            const vehicleName = schedule.equipment?.equipmentName || 'N/A';
            
            // Calculate due date (basic calculation - can be enhanced)
            let dueDate = 'N/A';
            if (schedule.lastServiceDate && schedule.intervals && schedule.intervals.length > 0) {
                // Simple calculation - add interval to last service date
                const lastService = new Date(schedule.lastServiceDate);
                const firstInterval = schedule.intervals[0];
                if (firstInterval && firstInterval.value) {
                    const intervalValue = parseInt(firstInterval.value);
                    if (schedule.intervalUnit === 'Day') {
                        lastService.setDate(lastService.getDate() + intervalValue);
                    } else if (schedule.intervalUnit === 'Week') {
                        lastService.setDate(lastService.getDate() + (intervalValue * 7));
                    } else if (schedule.intervalUnit === 'Month') {
                        lastService.setMonth(lastService.getMonth() + intervalValue);
                    } else if (schedule.intervalUnit === 'Year') {
                        lastService.setFullYear(lastService.getFullYear() + intervalValue);
                    }
                    dueDate = lastService.toLocaleDateString('en-GB', {
                        day: '2-digit',
                        month: '2-digit',
                        year: 'numeric'
                    });
                }
            }
            
            return `
                                    <tr class="table-row">
                    <td style="text-align: center; font-weight: 600;">${serialNumber}</td>
                    <td>
                        <div class="action-dropdown-container">
                            <button class="btn-action-dropdown" onclick="toggleActionDropdown(event, '${schedule.id || schedule._id}')">
                                <i class="fas fa-ellipsis-v"></i>
                                            </button>
                            <div class="action-dropdown-menu" id="dropdown-${schedule.id || schedule._id}" style="display: none;">
                                <button class="action-dropdown-item" onclick="event.stopPropagation(); viewScheduleDetails('${schedule.id || schedule._id}')">
                                    <i class="fas fa-eye"></i> View Report
                                </button>
                                <button class="action-dropdown-item delete-item" onclick="event.stopPropagation(); openDeleteScheduleModal('${schedule.id || schedule._id}')">
                                    <i class="fas fa-trash"></i> Delete Report
                                </button>
                            </div>
                                        </div>
                                    </td>
                    <td class="schedule-id">${schedule.scheduleId || 'N/A'}</td>
                    <td>${vehicleName}</td>
                    <td><span class="due-date-badge">${dueDate}</span></td>
                    <td><span class="status-badge status-${getStatusColorClass(schedule.status)}">${schedule.status || 'upcoming'}</span></td>
                    <td>${schedule.jobCardRequestNumber || 'N/A'}</td>
                    <td class="schedule-title">${schedule.title || 'N/A'}</td>
                    <td>${formattedLastService}</td>
                                </tr>
            `;
        }).join('');
        
        tableBody.innerHTML = rows;
    }

    // Display empty state
    function displayEmptyState() {
        const tableBody = document.getElementById('serviceScheduleTableBody');
        if (!tableBody) return;
        
        tableBody.innerHTML = `
                            <tr class="empty-row">
                <td colspan="9" class="empty-state">
                                    <div class="empty-content">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                                            <line x1="16" y1="2" x2="16" y2="6"/>
                                            <line x1="8" y1="2" x2="8" y2="6"/>
                                            <line x1="3" y1="10" x2="21" y2="10"/>
                                        </svg>
                                        <h3>No Service Schedules</h3>
                                        <p>There are currently no service schedules to display.</p>
                                    </div>
                                </td>
                            </tr>
        `;
        
        // Hide pagination
        const paginationContainer = document.getElementById('paginationContainer');
        if (paginationContainer) {
            paginationContainer.style.display = 'none';
        }
    }
    
    // Toggle action dropdown
    function toggleActionDropdown(event, scheduleId) {
        event.stopPropagation();
        
        // Close all other dropdowns
        document.querySelectorAll('.action-dropdown-menu').forEach(menu => {
            if (menu.id !== `dropdown-${scheduleId}`) {
                menu.style.display = 'none';
            }
        });
        
        // Toggle current dropdown
        const dropdown = document.getElementById(`dropdown-${scheduleId}`);
        if (dropdown) {
            dropdown.style.display = dropdown.style.display === 'none' ? 'block' : 'none';
        }
    }
    
    // Close dropdowns when clicking outside
    document.addEventListener('click', function(event) {
        if (!event.target.closest('.action-dropdown-container')) {
            document.querySelectorAll('.action-dropdown-menu').forEach(menu => {
                menu.style.display = 'none';
            });
        }
    });
    
    // Modal state variable
    let currentDeleteScheduleId = null;
    
    // Delete Schedule Modal Functions
    function openDeleteScheduleModal(scheduleId) {
        currentDeleteScheduleId = scheduleId;
        const modal = document.getElementById('deleteScheduleModal');
        modal.style.display = 'block';
        document.body.classList.add('modal-open');
        
        // Close any open dropdowns
        document.querySelectorAll('.action-dropdown-menu').forEach(menu => {
            menu.style.display = 'none';
        });
    }
    
    function closeDeleteScheduleModal() {
        const modal = document.getElementById('deleteScheduleModal');
        modal.style.display = 'none';
        document.body.classList.remove('modal-open');
        currentDeleteScheduleId = null;
    }
    
    async function confirmDeleteSchedule() {
        if (!currentDeleteScheduleId) return;
        
        try {
            const response = await fetch(`https://api.titandrillingzm.com:6010/service-schedules/${currentDeleteScheduleId}`, {
                method: 'DELETE'
            });
            
            const result = await response.json();
            
            if (result.success) {
                showNotification('Service schedule deleted successfully', 'success');
                closeDeleteScheduleModal();
                // Reload schedules
                await loadServiceSchedules();
            } else {
                showNotification('Failed to delete schedule: ' + result.error, 'error');
            }
        } catch (error) {
            console.error('Error deleting schedule:', error);
            showNotification('Failed to delete schedule: ' + error.message, 'error');
        }
    }

    // Update pagination
    function updatePagination() {
        const paginationContainer = document.getElementById('paginationContainer');
        const paginationButtons = document.getElementById('paginationButtons');
        
        if (!paginationContainer || !paginationButtons) return;
        
        if (totalPages <= 1) {
            paginationContainer.style.display = 'none';
            return;
        }
        
        paginationContainer.style.display = 'flex';
        
        // Update info
        const startIndex = (currentPage - 1) * CONFIG.pagination.itemsPerPage + 1;
        const endIndex = Math.min(currentPage * CONFIG.pagination.itemsPerPage, allSchedules.length);
        
        document.getElementById('showingStart').textContent = startIndex;
        document.getElementById('showingEnd').textContent = endIndex;
        document.getElementById('totalRecords').textContent = allSchedules.length;
        
        // Generate pagination buttons
        let buttonsHTML = '';
        
        // Previous button
        buttonsHTML += `
            <button class="pagination-btn ${currentPage === 1 ? 'disabled' : ''}" 
                    onclick="goToPage(${currentPage - 1})" 
                    ${currentPage === 1 ? 'disabled' : ''}>
                <i class="fas fa-chevron-left"></i>
            </button>
        `;
        
        // Page number buttons
        const maxVisible = CONFIG.pagination.maxVisiblePages;
        let startPage = Math.max(1, currentPage - Math.floor(maxVisible / 2));
        let endPage = Math.min(totalPages, startPage + maxVisible - 1);
        
        if (endPage - startPage < maxVisible - 1) {
            startPage = Math.max(1, endPage - maxVisible + 1);
        }
        
        if (startPage > 1) {
            buttonsHTML += `<button class="pagination-btn" onclick="goToPage(1)">1</button>`;
            if (startPage > 2) {
                buttonsHTML += `<span class="pagination-ellipsis">...</span>`;
            }
        }
        
        for (let i = startPage; i <= endPage; i++) {
            buttonsHTML += `
                <button class="pagination-btn ${i === currentPage ? 'active' : ''}" 
                        onclick="goToPage(${i})">
                    ${i}
                </button>
            `;
        }
        
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                buttonsHTML += `<span class="pagination-ellipsis">...</span>`;
            }
            buttonsHTML += `<button class="pagination-btn" onclick="goToPage(${totalPages})">${totalPages}</button>`;
        }
        
        // Next button
        buttonsHTML += `
            <button class="pagination-btn ${currentPage === totalPages ? 'disabled' : ''}" 
                    onclick="goToPage(${currentPage + 1})" 
                    ${currentPage === totalPages ? 'disabled' : ''}>
                <i class="fas fa-chevron-right"></i>
            </button>
        `;
        
        paginationButtons.innerHTML = buttonsHTML;
    }

    function goToPage(page) {
        if (page < 1 || page > totalPages || page === currentPage) return;
        
        currentPage = page;
        displayCurrentPage();
        updatePagination();
        
        // Scroll to top of table
        const tableSection = document.querySelector('.table-section');
        if (tableSection) {
            tableSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
    }

    function viewScheduleDetails(scheduleId) {
        const schedule = allSchedules.find(s => s.id === scheduleId || s._id === scheduleId);
        
        if (!schedule) {
            showNotification('Schedule not found', 'error');
            return;
        }
        
        // Format creation date
        let formattedCreatedDate = 'N/A';
        if (schedule.createdAt) {
            try {
                const date = new Date(schedule.createdAt);
                formattedCreatedDate = date.toLocaleString('en-GB', {
                    day: '2-digit',
                    month: '2-digit',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
            } catch (error) {
                console.error('Error formatting date:', error);
            }
        }
        
        // Format last service date
        let formattedLastService = 'N/A';
        if (schedule.lastServiceDate) {
            try {
                const date = new Date(schedule.lastServiceDate);
                formattedLastService = date.toLocaleDateString('en-GB', {
                    day: '2-digit',
                    month: '2-digit',
                    year: 'numeric'
                });
            } catch (error) {
                console.error('Error formatting date:', error);
            }
        }
        
        // Calculate due date
        let dueDate = 'N/A';
        if (schedule.lastServiceDate && schedule.intervals && schedule.intervals.length > 0) {
            const lastService = new Date(schedule.lastServiceDate);
            const firstInterval = schedule.intervals[0];
            if (firstInterval && firstInterval.value) {
                const intervalValue = parseInt(firstInterval.value);
                if (schedule.intervalUnit === 'Day') {
                    lastService.setDate(lastService.getDate() + intervalValue);
                } else if (schedule.intervalUnit === 'Week') {
                    lastService.setDate(lastService.getDate() + (intervalValue * 7));
                } else if (schedule.intervalUnit === 'Month') {
                    lastService.setMonth(lastService.getMonth() + intervalValue);
                } else if (schedule.intervalUnit === 'Year') {
                    lastService.setFullYear(lastService.getFullYear() + intervalValue);
                }
                dueDate = lastService.toLocaleDateString('en-GB', {
                    day: '2-digit',
                    month: '2-digit',
                    year: 'numeric'
                });
            }
        }
        
        // Equipment details
        let equipmentDetails = '<div style="color: #6b7280;">No equipment assigned</div>';
        if (schedule.equipment) {
            equipmentDetails = `
                <div style="background: #f9fafb; padding: 15px; border-radius: 8px; margin-top: 10px;">
                    <div style="margin-bottom: 8px;"><strong>Equipment Name:</strong> ${schedule.equipment.equipmentName || 'N/A'}</div>
                    <div style="margin-bottom: 8px;"><strong>Equipment Number:</strong> ${schedule.equipment.equipmentNumber || 'N/A'}</div>
                    <div style="margin-bottom: 8px;"><strong>Entry Number:</strong> ${schedule.equipment.entryNumber || 'N/A'}</div>
                    <div style="margin-bottom: 8px;"><strong>Meter Reading:</strong> ${schedule.equipment.meterReading || 0} ${schedule.equipment.meterUnit || ''}</div>
            </div>
            `;
        }
        
        // Tasks details
        let tasksDetails = '<div style="color: #6b7280;">No tasks assigned</div>';
        if (schedule.tasks && Array.isArray(schedule.tasks) && schedule.tasks.length > 0) {
            tasksDetails = schedule.tasks.map((task, index) => `
                <div style="background: #f9fafb; padding: 12px; border-radius: 6px; margin-bottom: 8px; border-left: 4px solid #0055a5;">
                    <strong>${index + 1}. ${task}</strong>
                    </div>
            `).join('');
        }
        
        // Intervals details
        let intervalsDetails = '<div style="color: #6b7280;">No intervals configured</div>';
        if (schedule.intervals && Array.isArray(schedule.intervals) && schedule.intervals.length > 0) {
            intervalsDetails = schedule.intervals.map((interval, index) => `
                <div style="background: #f9fafb; padding: 10px; border-radius: 6px; margin-bottom: 8px;">
                    <strong>Interval ${index + 1}:</strong> Every ${interval.value} ${interval.unit}
                </div>
            `).join('');
        }
        
        // Notify before details
        let notifyBeforeDetails = 'Not configured';
        if (schedule.notifyBefore && schedule.notifyBefore.value) {
            notifyBeforeDetails = `${schedule.notifyBefore.value} ${schedule.notifyBefore.unit || ''} - ${schedule.notifyBefore.time || ''}`;
        }
        
        const detailsHTML = `
            <div style="padding: 20px;">
                <!-- Header Info Grid -->
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 30px; padding: 20px; background: linear-gradient(135deg, #f8fafc 0%, #fff 100%); border-radius: 8px; border: 1px solid #e2e8f0;">
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px; text-transform: uppercase;">Schedule ID</div>
                        <div style="font-size: 18px; font-weight: 700; color: #0055a5;">${schedule.scheduleId || 'N/A'}</div>
                    </div>
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px; text-transform: uppercase;">Created Date</div>
                        <div style="font-size: 15px; color: #374151;">${formattedCreatedDate}</div>
                    </div>
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px; text-transform: uppercase;">Status</div>
                        <div><span class="status-badge status-${getStatusColorClass(schedule.status)}">${schedule.status || 'upcoming'}</span></div>
                    </div>
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px; text-transform: uppercase;">Repeat Type</div>
                        <div style="font-size: 15px; color: #374151;">${schedule.repeatType || 'N/A'}</div>
                    </div>
                </div>
                
                <!-- Title and Country/Project -->
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-calendar-alt" style="color: #0055a5;"></i> Schedule Title
                    </div>
                    <div style="color: #374151; font-size: 15px; background: #f9fafb; padding: 12px; border-radius: 6px;">${schedule.title || 'N/A'}</div>
                        </div>
                        
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 25px;">
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 8px; font-size: 14px;">Country</div>
                        <div style="color: #374151; font-size: 15px; background: #f9fafb; padding: 10px; border-radius: 6px;">${schedule.country || 'N/A'}</div>
                    </div>
                    <div>
                        <div style="font-weight: 600; color: #6b7280; margin-bottom: 8px; font-size: 14px;">Project</div>
                        <div style="color: #374151; font-size: 15px; background: #f9fafb; padding: 10px; border-radius: 6px;">${schedule.project || 'N/A'}</div>
                </div>
        </div>
                
                <!-- Note -->
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-sticky-note" style="color: #0055a5;"></i> Note
    </div>
                    <div style="color: #6b7280; background: #f9fafb; padding: 15px; border-radius: 8px; line-height: 1.6;">${schedule.note || 'No note provided'}</div>
</div>

                <!-- Equipment Details -->
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-tools" style="color: #0055a5;"></i> Equipment Details
                    </div>
                    ${equipmentDetails}
                </div>
                
                <!-- Service Details -->
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-cog" style="color: #0055a5;"></i> Service Details
                    </div>
                    <div style="background: #f9fafb; padding: 15px; border-radius: 8px;">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                            <div>
                                <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Last Service Date</div>
                                <div style="color: #374151; font-size: 15px;">${formattedLastService}</div>
                            </div>
                            <div>
                                <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Due Date</div>
                                <div style="color: #374151; font-size: 15px;">${dueDate}</div>
                            </div>
                            <div>
                                <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Meter Reading</div>
                                <div style="color: #374151; font-size: 15px;">${schedule.meterReading || 0} ${schedule.meterUnit || ''}</div>
                            </div>
                            <div>
                                <div style="font-weight: 600; color: #6b7280; margin-bottom: 5px; font-size: 13px;">Interval Unit</div>
                                <div style="color: #374151; font-size: 15px;">${schedule.intervalUnit || 'N/A'}</div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Tasks -->
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-tasks" style="color: #0055a5;"></i> Tasks (${schedule.tasks ? schedule.tasks.length : 0})
                    </div>
                    ${tasksDetails}
                </div>
                
                <!-- Intervals -->
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-repeat" style="color: #0055a5;"></i> Intervals Configuration
                    </div>
                    ${intervalsDetails}
                </div>
                
                <!-- Notify Before -->
                <div style="margin-bottom: 25px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-bell" style="color: #0055a5;"></i> Notify Before
                    </div>
                    <div style="background: #f9fafb; padding: 12px; border-radius: 6px; color: #374151;">${notifyBeforeDetails}</div>
                </div>
                
                <!-- Job Card Info (if linked) -->
                ${schedule.jobCardId ? `
                <div style="margin-bottom: 0; border-top: 2px solid #e2e8f0; padding-top: 20px;">
                    <div style="font-weight: 600; color: #374151; margin-bottom: 10px; font-size: 16px; display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-link" style="color: #0055a5;"></i> Linked Job Card
                    </div>
                    <div style="background: #f0f9ff; padding: 15px; border-radius: 8px; border-left: 4px solid #0055a5;">
                        <div style="margin-bottom: 8px;"><strong>Request Number:</strong> ${schedule.jobCardRequestNumber || 'N/A'}</div>
                        <div style="margin-bottom: 8px;"><strong>Title:</strong> ${schedule.jobCardTitle || 'N/A'}</div>
                        <div style="margin-bottom: 8px;"><strong>Status:</strong> <span class="status-badge status-${getStatusColorClass(schedule.jobCardStatus)}">${schedule.jobCardStatus || 'N/A'}</span></div>
                        <div><strong>Created:</strong> ${schedule.jobCardCreatedAt ? new Date(schedule.jobCardCreatedAt).toLocaleString('en-GB') : 'N/A'}</div>
                    </div>
                </div>
                ` : ''}
            </div>
        `;
        
        document.getElementById('viewScheduleBody').innerHTML = detailsHTML;
        
        const modal = document.getElementById('viewScheduleModal');
        modal.style.display = 'block';
        document.body.classList.add('modal-open');
        
        // Close any open dropdowns
        document.querySelectorAll('.action-dropdown-menu').forEach(menu => {
            menu.style.display = 'none';
        });
    }
    
    function closeViewScheduleModal() {
        const modal = document.getElementById('viewScheduleModal');
        modal.style.display = 'none';
        document.body.classList.remove('modal-open');
    }

    // Helper function for status colors
    function getStatusColorClass(status) {
        if (!status) return 'info';
        const statusLower = status.toLowerCase();
        if (['completed', 'closed', 'done'].includes(statusLower)) return 'success';
        if (['in progress', 'processing', 'pending'].includes(statusLower)) return 'primary';
        if (['on hold', 'overdue'].includes(statusLower)) return 'danger';
        if (['initiated', 'scheduled', 'upcoming'].includes(statusLower)) return 'info';
        if (['cancelled'].includes(statusLower)) return 'secondary';
        return 'info';
    }

        function filterTable(searchTerm) {
            const tableBody = document.getElementById('serviceScheduleTableBody');
            if (!tableBody) return;
            
            const rows = tableBody.getElementsByTagName('tr');
            
            for (let row of rows) {
                const cells = row.getElementsByTagName('td');
                let found = false;
                
                for (let cell of cells) {
                    if (cell.textContent.toLowerCase().includes(searchTerm.toLowerCase())) {
                        found = true;
                        break;
                    }
                }
                
                row.style.display = found ? '' : 'none';
            }
        }

    function toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        if (sidebar) {
            sidebar.classList.toggle('collapsed');
        }
    }

        // Notification system
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `service-notification service-notification-${type}`;
            notification.textContent = message;
            
            notification.style.cssText = `
                position: fixed;
                top: 80px;
                right: 20px;
                padding: 12px 20px;
                border-radius: 0;
                color: white;
                font-weight: 500;
                font-size: 14px;
                z-index: 10001;
                opacity: 0;
                transform: translateX(100%);
                transition: all 0.3s ease;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                font-family: 'Inter', sans-serif;
            `;
            
            if (type === 'success') {
                notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
            } else if (type === 'error') {
                notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
            } else if (type === 'warning') {
                notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
            } else {
                notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
            }
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.opacity = '1';
                notification.style.transform = 'translateX(0)';
            }, 100);
            
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        document.body.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }

    // Make functions globally accessible
    window.goToPage = goToPage;
    window.viewScheduleDetails = viewScheduleDetails;
    window.closeViewScheduleModal = closeViewScheduleModal;
    window.toggleSidebar = toggleSidebar;
    window.showNotification = showNotification;
    window.toggleActionDropdown = toggleActionDropdown;
    window.openDeleteScheduleModal = openDeleteScheduleModal;
    window.closeDeleteScheduleModal = closeDeleteScheduleModal;
    window.confirmDeleteSchedule = confirmDeleteSchedule;
    
    // Close modals when clicking outside
    window.onclick = function(event) {
        const viewModal = document.getElementById('viewScheduleModal');
        const deleteModal = document.getElementById('deleteScheduleModal');
        
        if (event.target === viewModal) {
            closeViewScheduleModal();
        } else if (event.target === deleteModal) {
            closeDeleteScheduleModal();
        }
    };
})();
</script>
<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== SERVICE SCHEDULE CONTAINER ===== */
.service-schedule-container {
    min-height: 100vh;
    background: #f8fafc;
}

/* ===== HEADER STYLES ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: all 0.2s ease;
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: #0055a5;
    border: none;
    padding: 12px 24px;
    border-radius: 6px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    color: #0055a5;
    text-decoration: none;
}

/* Search Container */
.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #9ca3af;
    z-index: 1;
}

.search-input {
    padding: 10px 12px 10px 40px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    background: white;
    width: 300px;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #9ca3af;
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* ===== CONTENT SECTION ===== */
.service-schedule-content {
    background: white;
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    overflow: hidden;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.action-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 20px;
}

.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.action-buttons {
    display: flex;
    align-items: center;
    gap: 12px;
}

.btn-create {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: #0055a5;
    color: white;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
}

.btn-create:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-2px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    text-decoration: none;
    color: black !important;
}

/* ===== TABLE SECTION ===== */
.table-section {
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    overflow: hidden;
    margin-bottom: 24px;
}

.table-container {
    overflow-x: auto;
}

.service-schedule-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
}

.table-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.table-header th {
    padding: 16px 12px;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e5e7eb;
    border-radius: 0;
}

.table-body {
    background: white;
}

.table-row {
    border-bottom: 1px solid #f3f4f6;
    transition: all 0.2s ease;
}

.table-row:hover {
    background: #f8fafc;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.table-row td {
    padding: 16px 12px;
    font-size: 14px;
    color: #374151;
    vertical-align: middle;
}

/* ===== BUTTON STYLES ===== */
.btn-view {
    display: flex;
    align-items: center;
    gap: 6px;
    padding: 6px 12px;
    background: #0055a5;
    color: white;
    border: none;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(0, 85, 165, 0.2);
}

.btn-view:hover {
    background: linear-gradient(135deg, #003f7d 0%, #002a5c 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
}

/* ===== STATUS BADGES ===== */
.status-badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-success {
    color: green;
}

.status-primary {
    /* background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); */
    /* color: white; */
}

.status-warning {
    /* background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); */
    color: red;
}

.status-info {
    /* background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%); */
    color: #0055a5;
}

.status-danger {
    /* background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); */
    color: red;
}

.status-secondary {
    /* background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); */
    /* color: white; */
}

/* ===== DUE DATE BADGES ===== */
.due-date-badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.due-success {
    color: green;
}

.due-warning {
    color: red;
}

.due-danger {
    color: red;
}

.due-secondary {
    /* background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); */
    /* color: white; */
}

/* ===== LOADING STATE ===== */
.loading-row {
    background: white;
}

.loading-state {
    text-align: center;
    padding: 48px 24px;
    background: white;
}

.loading-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.loading-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* ===== PAGINATION BUTTONS ===== */
.pagination-buttons {
    display: flex;
    gap: 5px;
    align-items: center;
}

.pagination-btn {
    min-width: 36px;
    height: 36px;
    padding: 0 12px;
    border: 1px solid #d1d5db;
    background: white;
    color: #374151;
    border-radius: 6px;
    cursor: pointer;
    transition: all 0.2s ease;
    font-size: 14px;
    font-weight: 500;
    display: flex;
    align-items: center;
    justify-content: center;
}

.pagination-btn:hover:not(.disabled) {
    background: #f9fafb;
    border-color: #0055a5;
    color: #0055a5;
}

.pagination-btn.active {
    background: #0055a5;
    color: white;
    border-color: #0055a5;
}

.pagination-btn.disabled {
    opacity: 0.5;
    cursor: not-allowed;
    background: #f9fafb;
}

.pagination-ellipsis {
    padding: 0 8px;
    color: #9ca3af;
    font-size: 14px;
}

/* ===== EMPTY STATE ===== */
.empty-row {
    background: white;
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    background: white;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-content svg {
    color: #9ca3af;
}

.empty-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.empty-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* ===== PAGINATION STYLES ===== */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 24px;
    background: white;
    border-top: 1px solid #e5e7eb;
}

.pagination-info {
    font-size: 14px;
    color: #6b7280;
}


/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .main-content {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .search-input {
        width: 250px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .header-left {
        justify-content: center;
    }
    
    .header-right {
        justify-content: center;
        flex-direction: column;
        gap: 10px;
    }
    
    .search-input {
        width: 100%;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
    
    .pagination-container {
        flex-direction: column;
        gap: 15px;
        padding: 15px;
    }
    
    .pagination-info {
        font-size: 13px;
    }
    
    .pagination-btn {
        min-width: 32px;
        height: 32px;
        padding: 0 10px;
        font-size: 13px;
    }
}

@media (max-width: 992px) {
    .service-schedule-container {
        padding: 16px;
    }
    
    .service-schedule-header {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 26px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create {
        padding: 10px 16px;
        font-size: 13px;
        flex: 1;
        min-width: 150px;
    }
    
    .table-container {
        overflow-x: auto;
    }
    
    .service-schedule-table {
        min-width: 800px;
    }
    
}

@media (max-width: 768px) {
    .service-schedule-container {
        padding: 12px;
    }
    
    .service-schedule-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 12px;
        align-items: stretch;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-create {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
        justify-content: center;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .service-schedule-table {
        min-width: 700px;
    }
    
    .table-header th,
    .table-row td {
        padding: 10px 8px;
        font-size: 13px;
    }
    
    .table-header th {
        font-size: 12px;
        font-weight: 600;
    }
    
    .status-badge,
    .due-date-badge {
        padding: 4px 8px;
        font-size: 11px;
    }
    
    .btn-view {
        padding: 6px 10px;
        font-size: 11px;
    }
    
    .schedule-title {
        max-width: 150px;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }
    
}

@media (max-width: 576px) {
    .service-schedule-container {
        padding: 8px;
    }
    
    .service-schedule-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .btn-create {
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .search-input {
        padding: 12px 12px 12px 40px;
        font-size: 14px;
    }
    
    .search-icon {
        left: 12px;
        width: 16px;
        height: 16px;
    }
    
    .table-container {
        margin-bottom: 16px;
    }
    
    .service-schedule-table {
        min-width: 600px;
    }
    
    .table-header th,
    .table-row td {
        padding: 8px 6px;
        font-size: 12px;
    }
    
    .table-header th {
        font-size: 11px;
        padding: 10px 6px;
    }
    
    .status-badge,
    .due-date-badge {
        padding: 3px 6px;
        font-size: 10px;
    }
    
    .btn-view {
        padding: 5px 8px;
        font-size: 10px;
    }
    
    .schedule-title {
        max-width: 120px;
    }
    
    .empty-state {
        padding: 32px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
}

@media (max-width: 480px) {
    .service-schedule-container {
        padding: 4px;
    }
    
    .service-schedule-header {
        padding: 8px;
        margin-bottom: 8px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-bar {
        padding: 6px 8px;
    }
    
    .btn-create {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .search-input {
        padding: 14px 14px 14px 44px;
        font-size: 15px;
        min-height: 48px;
    }
    
    .search-icon {
        left: 14px;
        width: 18px;
        height: 18px;
    }
    
    .service-schedule-table {
        min-width: 500px;
    }
    
    .table-header th,
    .table-row td {
        padding: 6px 4px;
        font-size: 11px;
    }
    
    .table-header th {
        font-size: 10px;
        padding: 8px 4px;
    }
    
    .status-badge,
    .due-date-badge {
        padding: 2px 4px;
        font-size: 9px;
    }
    
    .btn-view {
        padding: 4px 6px;
        font-size: 9px;
    }
    
    .schedule-title {
        max-width: 100px;
    }
    
    .empty-state {
        padding: 24px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
    
}

/* ===== TABLET LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 1024px) and (orientation: landscape) {
    .service-schedule-container {
        padding: 16px;
    }
    
    .action-content {
        flex-direction: row;
        align-items: center;
    }
    
    .search-section {
        max-width: 300px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 6px;
    }
    
    .btn-create {
        padding: 8px 12px;
        font-size: 12px;
    }
}

/* ===== MOBILE LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 896px) and (orientation: landscape) {
    .service-schedule-header {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .btn-create {
        padding: 10px 12px;
        font-size: 13px;
    }
}

/* ===== TABLE RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .service-schedule-table {
        font-size: 12px;
    }
    
    .table-header th:first-child,
    .table-row td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 10;
        border-right: 1px solid #e2e8f0;
    }
    
    .table-header th:first-child {
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    }
    
    .schedule-id {
        font-weight: 600;
        color: #0055a5;
        font-size: 12px;
    }
}


/* ===== ACCESSIBILITY IMPROVEMENTS ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

/* ===== HIGH DPI DISPLAYS ===== */
@media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
    .search-icon,
    .btn-create svg,
    .btn-view svg,
    .add-vehicle-btn svg,
    .cancel-btn svg {
        image-rendering: -webkit-optimize-contrast;
        image-rendering: crisp-edges;
    }
}

/* ===== FOCUS STATES ===== */
.search-input:focus,
.btn-create:focus,
.btn-view:focus {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== SMOOTH ANIMATIONS ===== */
* {
    transition: all 0.2s ease;
}

/* ===== HOVER EFFECTS ===== */
.btn-create:active,
.btn-view:active {
    transform: translateY(0);
}

/* ===== LOADING STATES ===== */
.btn-create:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.btn-create:disabled:hover {
    background: #0055a5;
    transform: none;
}

/* ===== RESPONSIVE PAGINATION ===== */
@media (max-width: 768px) {
    .pagination-container {
        flex-direction: column;
        gap: 15px;
        padding: 15px;
    }
    
    .pagination-info {
        font-size: 13px;
    }
    
    .pagination-btn {
        min-width: 32px;
        height: 32px;
        padding: 0 10px;
        font-size: 13px;
    }
}

@media (max-width: 480px) {
    .pagination-buttons {
        flex-wrap: wrap;
        justify-content: center;
    }
    
    .pagination-btn {
        min-width: 28px;
        height: 28px;
        padding: 0 8px;
        font-size: 12px;
    }
}

/* ===== ACTION DROPDOWN ===== */
.action-dropdown-container {
    position: relative;
    display: inline-block;
}

.btn-action-dropdown {
    background: #0055a5;
    color: white;
    border: none;
    padding: 8px 12px;
    border-radius: 4px;
    cursor: pointer;
    transition: all 0.2s ease;
    font-size: 16px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-action-dropdown:hover {
    background: #003f7d;
    transform: translateY(-1px);
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
}

.action-dropdown-menu {
    position: absolute;
    top: 100%;
    right: -190px;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
    min-width: 180px;
    z-index: 1000;
    margin-top: 5px;
    overflow: hidden;
}

.action-dropdown-item {
    display: flex;
    align-items: center;
    gap: 10px;
    width: 100%;
    padding: 12px 16px;
    background: white;
    border: none;
    text-align: left;
    cursor: pointer;
    transition: all 0.2s ease;
    font-size: 14px;
    color: #374151;
    border-bottom: 1px solid #f3f4f6;
}

.action-dropdown-item:last-child {
    border-bottom: none;
}

.action-dropdown-item:hover {
    background: #f9fafb;
    color: #0055a5;
}

.action-dropdown-item.delete-item {
    color: #dc3545;
}

.action-dropdown-item.delete-item:hover {
    background: #fee;
    color: #c82333;
}

.action-dropdown-item i {
    width: 16px;
    text-align: center;
}

/* ===== MODAL STYLES ===== */
body.modal-open {
    overflow: hidden;
}

.custom-modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100vw;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    overflow-y: auto;
}

.custom-modal-content {
    background-color: #ffffff;
    margin: 50px auto;
    padding: 0;
    border: 1px solid #e5e7eb;
    border-radius: 12px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
    animation: modalSlideIn 0.3s ease;
    position: relative;
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: scale(0.95) translateY(-20px);
    }
    to {
        opacity: 1;
        transform: scale(1) translateY(0);
    }
}

.custom-modal-header {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
    padding: 20px 24px;
    border-radius: 12px 12px 0 0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.custom-modal-header h3 {
    margin: 0;
    font-size: 18px;
    font-weight: 600;
}

.close-btn {
    color: white;
    font-size: 28px;
    font-weight: bold;
    background: none;
    border: none;
    cursor: pointer;
    padding: 0;
    line-height: 1;
    transition: all 0.2s ease;
}

.close-btn:hover {
    opacity: 0.7;
    transform: scale(1.1);
}

.custom-modal-body {
    padding: 24px;
}

.modal-footer {
    padding: 16px 24px;
    background-color: #f9fafb;
    border-top: 1px solid #e5e7eb;
    display: flex;
    justify-content: flex-end;
    align-items: center;
    gap: 12px;
    border-radius: 0 0 12px 12px;
}

.modal-footer .btn {
    padding: 10px 20px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 8px;
}

.modal-footer .btn-primary {
    background: linear-gradient(135deg, #0055a5 0%, #003f7d 100%);
    color: white;
}

.modal-footer .btn-primary:hover {
    background: linear-gradient(135deg, #003f7d 0%, #002a5c 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.modal-footer .btn-secondary {
    background: #f3f4f6;
    color: #374151;
    border: 1px solid #d1d5db;
}

.modal-footer .btn-secondary:hover {
    background: #e5e7eb;
    border-color: #9ca3af;
}

.modal-footer .btn-danger {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
    color: white;
}

.modal-footer .btn-danger:hover {
    background: linear-gradient(135deg, #c82333 0%, #a71d2a 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
}

/* ===== RESPONSIVE MODAL ===== */
@media (max-width: 768px) {
    .custom-modal-content {
        width: 95%;
        margin: 20px auto;
    }
    
    .custom-modal-header {
        padding: 16px 20px;
    }
    
    .custom-modal-header h3 {
        font-size: 16px;
    }
    
    .custom-modal-body {
        padding: 20px;
    }
    
    .modal-footer {
        padding: 12px 20px;
        flex-direction: column-reverse;
    }
    
    .modal-footer .btn {
        width: 100%;
    }
}

@media (max-width: 480px) {
    .custom-modal-content {
        width: 98%;
        margin: 10px auto;
    }
    
    .custom-modal-header {
        padding: 12px 16px;
    }
    
    .custom-modal-body {
        padding: 16px;
    }
}
</style>

@endsection