@extends('welcome')

@section('title', 'Task Master')

@section('content')
<div class="task-master-container">
    <div class="task-master-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Task Master</h1>
                <p>Manage and organize tasks efficiently across different categories and vehicles</p>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="task-master-content">
        <!-- Action Bar -->
        <div class="action-bar">
            <div class="action-content">
                <!-- Search Section -->
                <div class="search-section">
                    <div class="search-container">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="11" cy="11" r="8"/>
                            <path d="m21 21-4.35-4.35"/>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search tasks..." id="taskSearch">
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="action-buttons">
                    <button class="btn-create" data-bs-toggle="modal" data-bs-target="#add-checklist">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M12 5v14m-7-7h14"/>
                        </svg>
                        Add Task
                    </button>
                </div>
            </div>
        </div>

        <!-- Table Section -->
        <div class="table-section">
            <div class="table-container">
                <table class="task-master-table">
                    <thead class="table-header">
                        <tr>
                            <th>Category</th>
                            <th>Vehicle</th>
                            <th>Task</th>
                        </tr>
                    </thead>
                    <tbody class="table-body">
                        <!-- Empty State -->
                        <tr class="empty-row">
                            <td colspan="3">
                                <div class="empty-state">
                                    <div class="empty-content">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M9 11H5a2 2 0 0 0-2 2v3c0 1.1.9 2 2 2h4m0-7v7m0-7h6m-6 0V9a2 2 0 0 1 2-2h2a2 2 0 0 1 2 2v2m-6 0h6"/>
                                        </svg>
                                        <h3>No Tasks Found</h3>
                                        <p>There are currently no tasks to display. Create your first task to get started.</p>
                                    </div>
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <script>
        (function() {
            'use strict';
            
            // Configuration object
            const CONFIG = {
                selectors: {
                    sidebar: '.sidebar',
                    taskSearch: '#taskSearch',
                    vehicleSearch: '#vehicleSearch',
                    tableBody: '.table-body',
                    emptyRow: '.empty-row',
                    taskForm: '#taskForm',
                    modals: '.modal',
                    backdrop: '.modal-backdrop'
                },
                classes: {
                    collapsed: 'collapsed',
                    emptyRow: 'empty-row',
                    show: 'show',
                    modalOpen: 'modal-open'
                },
                messages: {
                    noResults: 'No tasks found matching your search criteria.',
                    searchPlaceholder: 'Search tasks...',
                    fillRequired: 'Please fill in all required fields',
                    taskAdded: 'Task added successfully!'
                },
                timing: {
                    notificationDelay: 100,
                    notificationDuration: 3000,
                    animationDuration: 300
                },
                zIndex: {
                    notification: 10001
                }
            };
            
            // DOM elements cache
            let elements = {};
            
            // Initialize DOM elements
            function initElements() {
                elements = {
                    sidebar: document.querySelector(CONFIG.selectors.sidebar),
                    taskSearch: document.querySelector(CONFIG.selectors.taskSearch),
                    vehicleSearch: document.querySelector(CONFIG.selectors.vehicleSearch),
                    tableBody: document.querySelector(CONFIG.selectors.tableBody),
                    emptyRow: document.querySelector(CONFIG.selectors.emptyRow),
                    taskForm: document.querySelector(CONFIG.selectors.taskForm),
                    modals: document.querySelectorAll(CONFIG.selectors.modals),
                    backdrop: document.querySelector(CONFIG.selectors.backdrop)
                };
            }
            
            // Toggle sidebar
            function toggleSidebar() {
                if (elements.sidebar) {
                    elements.sidebar.classList.toggle(CONFIG.classes.collapsed);
                }
            }
            
            // Filter table based on search term
            function filterTable() {
                if (!elements.taskSearch || !elements.tableBody) return;
                
                const searchTerm = elements.taskSearch.value.toLowerCase();
                const rows = elements.tableBody.querySelectorAll(`tr:not(.${CONFIG.classes.emptyRow})`);
                
                let hasResults = false;
                
                rows.forEach(row => {
                    const text = row.textContent.toLowerCase();
                    const found = text.includes(searchTerm);
                    row.style.display = found ? '' : 'none';
                    if (found) hasResults = true;
                });
                
                // Show/hide empty state
                if (elements.emptyRow) {
                    elements.emptyRow.style.display = hasResults ? 'none' : '';
                }
            }
            
            // Close modal
            function closeModal() {
                elements.modals.forEach(modal => {
                    modal.classList.remove(CONFIG.classes.show);
                    modal.style.display = 'none';
                });
                
                // Remove backdrop if it exists
                if (elements.backdrop) {
                    elements.backdrop.remove();
                }
                
                // Remove modal-open class from body
                document.body.classList.remove(CONFIG.classes.modalOpen);
                document.body.style.overflow = '';
                document.body.style.paddingRight = '';
            }
            
            // Show notification
            function showNotification(message, type = 'info') {
                const notification = document.createElement('div');
                notification.className = `task-notification task-notification-${type}`;
                notification.textContent = message;
                
                notification.style.cssText = `
                    position: fixed;
                    top: 80px;
                    right: 20px;
                    padding: 12px 20px;
                    border-radius: 0;
                    color: white;
                    font-weight: 500;
                    font-size: 14px;
                    z-index: ${CONFIG.zIndex.notification};
                    opacity: 0;
                    transform: translateX(100%);
                    transition: all ${CONFIG.timing.animationDuration}ms ease;
                    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                    font-family: var(--font-family);
                `;
                
                // Set background based on type
                const backgrounds = {
                    success: 'linear-gradient(135deg, #10b981 0%, #059669 100%)',
                    error: 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)',
                    warning: 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)',
                    info: 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)'
                };
                
                notification.style.background = backgrounds[type] || backgrounds.info;
                
                document.body.appendChild(notification);
                
                // Animate in
                setTimeout(() => {
                    notification.style.opacity = '1';
                    notification.style.transform = 'translateX(0)';
                }, CONFIG.timing.notificationDelay);
                
                // Animate out and remove
                setTimeout(() => {
                    notification.style.opacity = '0';
                    notification.style.transform = 'translateX(100%)';
                    setTimeout(() => {
                        if (notification.parentNode) {
                            document.body.removeChild(notification);
                        }
                    }, CONFIG.timing.animationDuration);
                }, CONFIG.timing.notificationDuration);
            }
            
            // Initialize search functionality
            function initSearch() {
                if (elements.taskSearch) {
                    elements.taskSearch.addEventListener('input', filterTable);
                }
                
                if (elements.vehicleSearch) {
                    elements.vehicleSearch.addEventListener('input', function() {
                        // Vehicle search functionality can be added here
                        console.log('Vehicle search:', this.value);
                    });
                }
            }
            
            // Initialize form handling
            function initFormHandlers() {
                if (elements.taskForm) {
                    elements.taskForm.addEventListener('submit', function(e) {
                        e.preventDefault();
                        
                        const formData = new FormData(this);
                        const category = formData.get('category');
                        const taskName = formData.get('taskName');
                        const vehicleNumber = formData.get('vehicleNumber');
                        
                        // Validate required fields
                        if (!category || !taskName) {
                            showNotification(CONFIG.messages.fillRequired, 'error');
                            return;
                        }
                        
                        // Simulate form submission
                        showNotification(CONFIG.messages.taskAdded, 'success');
                        
                        // Close modal
                        closeModal();
                        
                        // Reset form
                        this.reset();
                    });
                }
            }
            
            // Initialize the application
            function init() {
                initElements();
                initSearch();
                initFormHandlers();
            }
            
            // Make functions globally accessible
            window.toggleSidebar = toggleSidebar;
            window.filterTable = filterTable;
            window.closeModal = closeModal;
            window.showNotification = showNotification;
            
            // Initialize when DOM is ready
            document.addEventListener('DOMContentLoaded', init);
        })();
    </script>
</div>
<!-- Add Task Modal -->
<div class="modal fade" id="add-checklist" tabindex="-1" aria-labelledby="addTaskModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="addTaskModalLabel">Add Task</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">&times;</button>
            </div>
            <div class="modal-body">
                <form class="task-form" id="taskForm">
                    <div class="form-group">
                        <label for="category">Category * :</label>
                        <input type="text" class="form-control" placeholder="Enter category" id="category" name="category" required>
                    </div>
                    <div class="form-group">
                        <label for="taskName">Task Name * :</label>
                        <input type="text" class="form-control" placeholder="Enter task name" id="taskName" name="taskName" required>
                    </div>
                    <div class="form-group">
                        <label for="vehicleNumber">Vehicle Number :</label>
                        <div class="input-with-button">
                            <input type="text" class="form-control" placeholder="Select vehicle from inventory list" id="vehicleNumber" name="vehicleNumber" readonly>
                            <button type="button" class="btn-select" data-bs-toggle="modal" data-bs-target="#add-vehicle-number">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M7 7h.01M7 3h5c.512 0 1.024.195 1.414.586l7 7a2 2 0 0 1 0 2.828l-7 7a2 2 0 0 1-2.828 0l-7-7A1.994 1.994 0 0 1 3 12V7a4 4 0 0 1 4-4z"/>
                                </svg>
                            </button>
                        </div>
                    </div>
                    <div class="modal-actions">
                        <button type="submit" class="btn-save">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M12 5v14m-7-7h14"/>
                            </svg>
                            Add Task
                        </button>
                        <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M18 6L6 18M6 6l12 12"/>
                            </svg>
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
<!-- Select Vehicle Modal -->
<div class="modal fade" id="add-vehicle-number" tabindex="-1" aria-labelledby="selectVehicleModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="selectVehicleModalLabel">Select Vehicle</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">&times;</button>
            </div>
            <div class="modal-body">
                <div class="search-section">
                    <div class="search-container">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="11" cy="11" r="8"/>
                            <path d="m21 21-4.35-4.35"/>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search vehicles..." id="vehicleSearch">
                    </div>
                </div>
                
                <div class="vehicle-list">
                    <div class="empty-state">
                        <div class="empty-content">
                            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9L18.7 8.3c-.3-.8-1-1.3-1.9-1.3H7.2c-.9 0-1.6.5-1.9 1.3L3.5 11.1C2.7 11.3 2 12.1 2 13v3c0 .6.4 1 1 1h2m14 0H5m14 0v3c0 .6-.4 1-1 1H6c-.6 0-1-.4-1-1v-3"/>
                            </svg>
                            <h3>No Vehicles Found</h3>
                            <p>No vehicles available for selection.</p>
                        </div>
                    </div>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M18 6L6 18M6 6l12 12"/>
                        </svg>
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --secondary-hover: #5a6268;
    --success-color: #28a745;
    --success-hover: #218838;
    --warning-color: #ffc107;
    --warning-hover: #e0a800;
    --danger-color: #dc3545;
    --danger-hover: #c82333;
    --info-color: #17a2b8;
    --info-hover: #138496;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #6c757d;
    --text-light: #ffffff;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-hover: #e9ecef;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --transition-slow: all 0.3s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-modal: 1050;
    --z-index-dropdown: 1000;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== MAIN CONTAINER ===== */
.task-master-container {
    background: var(--bg-light);
}

/* ===== HEADER STYLES ===== */
.task-master-header {
    background: var(--primary-color);
    border-radius: 0;
    padding: 32px;
    margin-bottom: 24px;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.page-title h1 {
    color: var(--text-light);
    font-size: 32px;
    font-weight: 700;
    margin: 0 0 8px 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.page-title p {
    color: rgba(255, 255, 255, 0.9);
    font-size: 16px;
    font-weight: 400;
    margin: 0;
}

/* ===== MAIN CONTENT ===== */
.task-master-content {
    background: var(--bg-primary);
    border-radius: 0;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: var(--bg-light);
    border-bottom: 1px solid var(--border-light);
    padding: 20px 24px;
}

.action-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 20px;
}

/* ===== SEARCH SECTION ===== */
.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: var(--text-muted);
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid var(--border-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: var(--bg-primary);
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

/* ===== ACTION BUTTONS ===== */
.action-buttons {
    display: flex;
    align-items: center;
    gap: 12px;
}

.btn-create {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid var(--border-color);
    border-radius: 0;
    background: var(--success-color);
    color: var(--text-light);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-create:hover {
    background: var(--success-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    text-decoration: none;
}

/* ===== TABLE SECTION ===== */
.table-section {
    padding: 0;
}

.table-container {
    overflow-x: auto;
}

.task-master-table {
    width: 100%;
    border-collapse: collapse;
    font-family: var(--font-family);
}

.table-header {
    background: var(--bg-light);
}

.table-header th {
    padding: 16px 12px;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    color: var(--text-primary);
    border-bottom: 2px solid var(--border-light);
    border-radius: 0;
}

.table-body {
    background: var(--bg-primary);
}

.table-row {
    border-bottom: 1px solid var(--border-light);
    transition: var(--transition-base);
}

.table-row:hover {
    background: var(--bg-light);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.table-row td {
    padding: 16px 12px;
    font-size: 14px;
    color: var(--text-primary);
    vertical-align: middle;
}

/* ===== EMPTY STATE ===== */
.empty-row {
    background: var(--bg-primary);
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    background: var(--bg-primary);
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-content svg {
    color: var(--text-muted);
}

.empty-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.empty-content p {
    font-size: 14px;
    color: var(--text-muted);
    margin: 0;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 768px) {
    .task-master-container {
        padding: 16px;
    }
    
    .task-master-header {
        padding: 24px 20px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 16px;
    }
    
    .search-section {
        max-width: none;
    }
    
    .action-buttons {
        justify-content: center;
    }
    
    .table-header th,
    .table-row td {
        padding: 12px 8px;
        font-size: 13px;
    }
    
    .empty-state {
        padding: 32px 16px;
    }
}

@media (max-width: 480px) {
    .task-master-container {
        padding: 12px;
    }
    
    .task-master-header {
        padding: 20px 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-bar {
        padding: 16px;
    }
    
    .btn-create {
        padding: 8px 12px;
        font-size: 13px;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* Focus styles for better accessibility */
.btn-create:focus,
.search-input:focus {
    outline: 2px solid var(--primary-color);
    outline-offset: 2px;
}

/* Screen reader only content */
.sr-only {
    position: absolute;
    width: 1px;
    height: 1px;
    padding: 0;
    margin: -1px;
    overflow: hidden;
    clip: rect(0, 0, 0, 0);
    white-space: nowrap;
    border: 0;
}

/* ===== MODAL STYLES ===== */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    z-index: 1055;
    display: none;
    justify-content: center;
    align-items: center;
}

.modal.show {
    display: flex !important;
}

.modal.fade .modal-dialog {
    transition: transform 0.3s ease-out;
    transform: translate(0, -50px);
}

.modal.show .modal-dialog {
    transform: none;
}

.modal-dialog {
    width: 90%;
    max-width: 500px;
    margin: 0 auto;
}

.modal-content {
    background: #ffffff;
    border: none;
    border-radius: 0;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
}

.modal-header {
    padding: 20px 24px;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #0055a5;
    color: white;
}

.modal-title {
    margin: 0;
    color: white;
    font-size: 20px;
    font-weight: 600;
}

.modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: white;
    padding: 8px;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 0;
    transition: background-color 0.2s;
}

.modal-close:hover {
    background-color: rgba(255, 255, 255, 0.2);
}

.modal-body {
    padding: 24px;
}

/* ===== FORM STYLES ===== */
.task-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.form-group label {
    font-weight: 500;
    color: #374151;
    font-size: 14px;
}

.form-control {
    border: 1px solid #d1d5db;
    border-radius: 0;
    padding: 12px 16px;
    font-size: 14px;
    transition: all 0.2s ease;
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.input-with-button {
    display: flex;
    gap: 0.5rem;
}

.input-with-button .form-control {
    flex: 1;
}

.btn-select {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 0.75rem;
    background: #f8fafc;
    border: 2px solid #e2e8f0;
    border-radius: 0;
    color: #64748b;
    cursor: pointer;
    transition: all 0.2s ease;
    min-width: 3rem;
}

.btn-select:hover {
    background: #e2e8f0;
    border-color: #cbd5e1;
    color: #475569;
}

/* ===== MODAL ACTIONS ===== */
.modal-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
    margin-top: 2rem;
}

.btn-save {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: #00B894;
    color: white;
    border: 1px solid #00B894;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-save:hover {
    background: #00a085;
    border-color: #00a085;
    transform: translateY(-1px);
}

.btn-cancel {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: #545d6b;
    color: white;
    border: 1px solid #545d6b;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel:hover {
    background: #3d4449;
    border-color: #3d4449;
    transform: translateY(-1px);
}

/* ===== VEHICLE LIST ===== */
.vehicle-list {
    margin: 1.5rem 0;
    min-height: 200px;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .task-master-container {
        padding: 20px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 16px;
    }
    
    .search-section {
        max-width: none;
    }
    
    .action-buttons {
        justify-content: center;
    }
    
    .modal-dialog {
        width: 90%;
        max-width: 600px;
    }
}

@media (max-width: 992px) {
    .task-master-container {
        padding: 16px;
    }
    
    .task-master-header {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 26px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create {
        padding: 10px 16px;
        font-size: 13px;
        flex: 1;
        min-width: 150px;
    }
    
    .modal-dialog {
        width: 95%;
        max-width: 700px;
    }
}

@media (max-width: 768px) {
    .task-master-container {
        padding: 12px;
    }
    
    .task-master-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 12px;
        align-items: stretch;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-create {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
        justify-content: center;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .task-master-table {
        min-width: 500px;
    }
    
    .table-header th,
    .table-row td {
        padding: 10px 8px;
        font-size: 13px;
    }
    
    .table-header th {
        font-size: 12px;
        font-weight: 600;
    }
    
    .modal-dialog {
        width: 95%;
        margin: 10px auto;
    }
    
    .modal-content {
        border-radius: 0;
    }
    
    .modal-body {
        padding: 20px;
    }
    
    .modal-actions {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-save,
    .btn-cancel {
        width: 100%;
        justify-content: center;
    }
    
    .form-group {
        margin-bottom: 12px;
    }
    
    .input-with-button {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-select {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .task-master-container {
        padding: 8px;
    }
    
    .task-master-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .btn-create {
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .search-input {
        padding: 12px 12px 12px 40px;
        font-size: 14px;
    }
    
    .search-icon {
        left: 12px;
        width: 16px;
        height: 16px;
    }
    
    .table-container {
        margin-bottom: 16px;
    }
    
    .task-master-table {
        min-width: 400px;
    }
    
    .table-header th,
    .table-row td {
        padding: 8px 6px;
        font-size: 12px;
    }
    
    .table-header th {
        font-size: 11px;
        padding: 10px 6px;
    }
    
    .empty-state {
        padding: 32px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
    .modal-dialog {
        width: 98%;
        margin: 5px auto;
    }
    
    .modal-header {
        padding: 12px 16px;
    }
    
    .modal-title {
        font-size: 18px;
    }
    
    .modal-body {
        padding: 16px;
    }
    
    .modal-actions {
        margin-top: 16px;
    }
    
    .form-group {
        margin-bottom: 10px;
    }
    
    .form-control {
        padding: 10px 12px;
        font-size: 14px;
    }
    
    .btn-save,
    .btn-cancel {
        padding: 12px 16px;
        font-size: 14px;
    }
}

@media (max-width: 480px) {
    .task-master-container {
        padding: 4px;
    }
    
    .task-master-header {
        padding: 8px;
        margin-bottom: 8px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-bar {
        padding: 6px 8px;
    }
    
    .btn-create {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .search-input {
        padding: 14px 14px 14px 44px;
        font-size: 15px;
        min-height: 48px;
    }
    
    .search-icon {
        left: 14px;
        width: 18px;
        height: 18px;
    }
    
    .task-master-table {
        min-width: 350px;
    }
    
    .table-header th,
    .table-row td {
        padding: 6px 4px;
        font-size: 11px;
    }
    
    .table-header th {
        font-size: 10px;
        padding: 8px 4px;
    }
    
    .empty-state {
        padding: 24px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
    
    .modal-dialog {
        width: 100%;
        margin: 0;
        height: 100vh;
    }
    
    .modal-content {
        height: 100vh;
        border-radius: 0;
    }
    
    .modal-header {
        padding: 10px 12px;
    }
    
    .modal-title {
        font-size: 16px;
    }
    
    .modal-body {
        padding: 12px;
        overflow-y: auto;
    }
    
    .modal-actions {
        margin-top: 12px;
    }
    
    .form-control {
        padding: 8px 10px;
        font-size: 13px;
    }
    
    .btn-save,
    .btn-cancel {
        padding: 10px 12px;
        font-size: 13px;
    }
}

/* ===== TABLET LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 1024px) and (orientation: landscape) {
    .task-master-container {
        padding: 16px;
    }
    
    .action-content {
        flex-direction: row;
        align-items: center;
    }
    
    .search-section {
        max-width: 300px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 6px;
    }
    
    .btn-create {
        padding: 8px 12px;
        font-size: 12px;
    }
}

/* ===== MOBILE LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 896px) and (orientation: landscape) {
    .task-master-header {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .btn-create {
        padding: 10px 12px;
        font-size: 13px;
    }
}

/* ===== TABLE RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .task-master-table {
        font-size: 12px;
    }
    
    .table-header th:first-child,
    .table-row td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 10;
        border-right: 1px solid #e2e8f0;
    }
    
    .table-header th:first-child {
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    }
}

/* ===== MODAL RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .modal {
        align-items: flex-start;
        padding-top: 20px;
    }
    
    .modal-dialog {
        margin: 0;
    }
    
    .form-group label {
        font-size: 13px;
        margin-bottom: 6px;
    }
    
    .form-control {
        font-size: 14px;
        padding: 10px 12px;
    }
}

/* ===== ACCESSIBILITY IMPROVEMENTS ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

/* ===== HIGH DPI DISPLAYS ===== */
@media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
    .search-icon,
    .btn-create svg,
    .btn-save svg,
    .btn-cancel svg,
    .btn-select svg {
        image-rendering: -webkit-optimize-contrast;
        image-rendering: crisp-edges;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== ADDITIONAL REQUEST_MAINTENANCE-STYLE IMPROVEMENTS ===== */
.task-master-container {
    background: #f8fafc;
}

.task-master-content {
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
}

.table-header th {
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

/* ===== PRINT STYLES ===== */
@media print {
    .action-bar,
    .modal {
        display: none !important;
    }
    
    .task-master-container {
        background: white;
    }
    
    .task-master-header {
        background: white;
        color: black;
    }
}
</style>
@endsection