@extends('welcome')

@section('title', 'Track Job Card')

@section('styles')
<style>
    @import url('https://fonts.googleapis.com/css2?family=Jost:wght@400;500;600;700&display=swap');

    :root {
        --primary: #0055A5;
        --primary-dark: #003a73;
        --accent: #FFD205;
        --surface: #ffffff;
        --surface-soft: #f4f7fb;
        --text-strong: #0b1d33;
        --text-muted: #6b7d92;
    }

    .track-wrapper {
        max-width: 100%;
        margin: 0 auto;
        padding: 28px 18px 120px;
        min-height: 100vh;
        background: var(--surface-soft);
        font-family: 'Jost', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
    }

    .jobcard-shell {
        position: relative;
        border-radius: 28px;
        padding: 26px 22px 30px;
        background: linear-gradient(180deg, rgba(0, 85, 165, 0.95) 0%, rgba(0, 57, 120, 0.98) 100%);
        color: #fff;
        box-shadow: 0 28px 56px rgba(0, 55, 120, 0.28);
        overflow: hidden;
        margin-bottom: 20px;
    }

    .jobcard-shell::after {
        content: '';
        position: absolute;
        inset: 0;
        background: radial-gradient(circle at top right, rgba(255, 210, 5, 0.35), transparent 60%);
        pointer-events: none;
    }

    .header-nav {
        position: relative;
        z-index: 1;
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 16px;
        margin-bottom: 18px;
    }

    .nav-button {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 10px 14px;
        border-radius: 16px;
        border: 1px solid rgba(255, 255, 255, 0.35);
        color: #fff;
        text-decoration: none;
        font-weight: 600;
        font-size: 14px;
        transition: background 0.2s ease, transform 0.2s ease;
    }

    .nav-button:hover {
        background: rgba(0, 0, 0, 0.12);
        transform: translateY(-1px);
    }

    .header-center {
        flex: 1;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 8px;
        text-align: center;
    }

    .track-status-chip {
        padding: 6px 16px;
        border-radius: 999px;
        font-size: 12px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.08em;
        border: 1px solid rgba(255, 255, 255, 0.35);
        background: rgba(0, 0, 0, 0.16);
    }

    .status-hold { background: rgba(255, 210, 5, 0.18); color: #fff0ad; border-color: rgba(255, 210, 5, 0.48); }
    .status-in-progress { background: rgba(8, 189, 255, 0.2); color: #cbefff; border-color: rgba(8, 189, 255, 0.55); }
    .status-assigned { background: rgba(161, 194, 255, 0.22); color: #e5edff; border-color: rgba(161, 194, 255, 0.55); }
    .status-complete { background: rgba(45, 209, 144, 0.24); color: #d5ffeb; border-color: rgba(45, 209, 144, 0.55); }
    .status-cancelled { background: rgba(255, 94, 94, 0.24); color: #ffe2e2; border-color: rgba(255, 94, 94, 0.55); }

    .track-title {
        margin: 0;
        font-size: 26px;
        font-weight: 700;
        letter-spacing: -0.01em;
    }

    .track-subtitle {
        margin: 0;
        font-size: 14px;
        color: rgba(255, 255, 255, 0.78);
        font-weight: 500;
    }

    .download-btn {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        padding: 10px 14px;
        border-radius: 14px;
        background: rgba(255, 255, 255, 0.12);
        color: #fff;
        font-weight: 600;
        font-size: 14px;
        text-decoration: none;
        border: 1px solid rgba(255, 255, 255, 0.35);
        transition: transform 0.2s ease, background 0.2s ease;
    }

    .download-btn:hover {
        background: rgba(255, 255, 255, 0.2);
        transform: translateY(-1px);
    }

    .header-meta {
        position: relative;
        z-index: 1;
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 14px;
        padding: 16px;
        border-radius: 18px;
        background: rgba(0, 0, 0, 0.18);
        margin-bottom: 18px;
    }

    .meta-item {
        display: flex;
        flex-direction: column;
        gap: 4px;
        text-align: left;
    }

    .meta-label {
        font-size: 12px;
        text-transform: uppercase;
        letter-spacing: 0.08em;
        color: rgba(255, 255, 255, 0.68);
        font-weight: 600;
    }

    .meta-value {
        font-size: 16px;
        font-weight: 600;
    }

    .meta-subtext {
        display: block;
        margin-top: 4px;
        font-size: 12px;
        font-weight: 500;
        color: rgba(255, 255, 255, 0.75);
    }

    .track-badges {
        position: relative;
        z-index: 1;
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 16px;
    }

    .track-badge {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 7px 14px;
        border-radius: 16px;
        background: rgba(255, 255, 255, 0.14);
        border: 1px solid rgba(255, 255, 255, 0.25);
        font-size: 13px;
        font-weight: 600;
        color: #fff;
    }

    .status-toggle-row {
        position: relative;
        z-index: 1;
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
    }

    .status-toggle {
        border: 1px solid rgba(255, 255, 255, 0.28);
        background: rgba(0, 0, 0, 0.18);
        color: #fff;
        padding: 8px 16px;
        border-radius: 999px;
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s ease, background 0.2s ease;
    }

    .status-toggle:hover {
        background: rgba(0, 0, 0, 0.28);
        transform: translateY(-1px);
    }

    .status-toggle.active {
        background: #fff;
        color: var(--primary);
        border-color: #fff;
    }

    .tab-bar {
        display: flex;
        background: var(--surface);
        border-radius: 16px;
        padding: 6px;
        gap: 6px;
        box-shadow: 0 10px 28px rgba(13, 53, 98, 0.12);
        margin-bottom: 18px;
    }

    .tab-btn {
        flex: 1;
        border: none;
        background: transparent;
        color: var(--text-muted);
        font-weight: 600;
        font-size: 15px;
        border-radius: 12px;
        padding: 10px 14px;
        cursor: pointer;
        transition: background 0.2s ease, color 0.2s ease, transform 0.2s ease;
    }

    .tab-btn.active {
        background: rgba(0, 85, 165, 0.1);
        color: var(--primary);
        transform: translateY(-1px);
    }

    .tab-panel {
        display: none;
    }

    .tab-panel.active {
        display: block;
    }

    .section-card {
        background: var(--surface);
        border-radius: 18px;
        padding: 18px 20px;
        box-shadow: 0 16px 32px rgba(15, 37, 64, 0.08);
        border: 1px solid rgba(0, 50, 102, 0.06);
        margin-bottom: 16px;
    }

    .section-card:last-child {
        margin-bottom: 0;
    }

    .section-title {
        margin: 0 0 12px;
        font-size: 18px;
        font-weight: 700;
        color: var(--text-strong);
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .section-subtitle {
        margin: -6px 0 12px;
        font-size: 13px;
        color: var(--text-muted);
        font-weight: 500;
    }

    .overview-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
        gap: 14px;
    }

    .overview-item {
        background: rgba(0, 85, 165, 0.06);
        border-radius: 14px;
        padding: 12px;
        border: 1px solid rgba(0, 85, 165, 0.12);
        display: flex;
        flex-direction: column;
        gap: 4px;
    }

    .overview-label {
        font-size: 12px;
        letter-spacing: 0.08em;
        text-transform: uppercase;
        color: var(--text-muted);
        font-weight: 600;
    }

    .overview-value {
        font-size: 15px;
        font-weight: 600;
        color: var(--text-strong);
    }

    .track-controls {
        display: flex;
        flex-wrap: wrap;
        gap: 12px;
        margin: 0;
    }

    .track-btn-secondary {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 10px 16px;
        border-radius: 14px;
        border: 1px solid rgba(0, 85, 165, 0.2);
        background: rgba(0, 85, 165, 0.08);
        color: var(--primary);
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s ease, box-shadow 0.2s ease, background 0.2s ease;
    }

    .track-btn-secondary:hover {
        background: rgba(0, 85, 165, 0.14);
        transform: translateY(-1px);
        box-shadow: 0 12px 24px rgba(0, 55, 120, 0.16);
    }

    .section-actions {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
    }

    .items-filters {
        display: flex;
        gap: 8px;
        flex-wrap: wrap;
    }

    .items-filter-btn {
        border-radius: 999px;
        padding: 6px 14px;
        border: 1px solid rgba(0, 85, 165, 0.14);
        background: rgba(0, 85, 165, 0.04);
        color: var(--primary);
        font-size: 12px;
        font-weight: 600;
        cursor: pointer;
        transition: background 0.2s ease, transform 0.2s ease;
    }

    .items-filter-btn.active {
        background: var(--primary);
        color: #fff;
        transform: translateY(-1px);
    }

    .track-items {
        display: grid;
        gap: 12px;
        margin-top: 16px;
    }

    .item-card {
        border-radius: 16px;
        border: 1px solid rgba(0, 85, 165, 0.12);
        background: linear-gradient(130deg, rgba(0, 85, 165, 0.08) 0%, #fff 70%);
        padding: 16px;
        display: flex;
        flex-direction: column;
        gap: 12px;
        box-shadow: 0 12px 26px rgba(9, 45, 82, 0.08);
    }

    .item-card-header {
        display: flex;
        align-items: flex-start;
        justify-content: space-between;
        gap: 12px;
    }

    .item-title {
        font-size: 16px;
        font-weight: 600;
        color: var(--text-strong);
    }

    .item-badges {
        display: flex;
        gap: 8px;
        flex-wrap: wrap;
        margin-top: 6px;
    }

    .item-category-badge,
    .track-tag {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        padding: 5px 12px;
        border-radius: 18px;
        background: rgba(0, 85, 165, 0.1);
        color: var(--primary);
        font-size: 12px;
        font-weight: 600;
    }

    .track-tag.badge-muted {
        background: rgba(107, 125, 146, 0.14);
        color: var(--text-strong);
    }

    .item-actions {
        display: flex;
        align-items: center;
        gap: 8px;
        flex-wrap: wrap;
    }

    .item-action-btn {
        border: none;
        background: rgba(0, 85, 165, 0.08);
        color: var(--primary);
        font-size: 13px;
        font-weight: 600;
        padding: 6px 10px;
        border-radius: 12px;
        cursor: pointer;
        transition: background 0.2s ease, transform 0.2s ease;
    }

    .item-action-btn:hover {
        background: rgba(0, 85, 165, 0.16);
        transform: translateY(-1px);
    }

    .status-chip {
        display: inline-flex;
        align-items: center;
        gap: 4px;
        padding: 6px 12px;
        border-radius: 999px;
        font-size: 12px;
        font-weight: 600;
    }

    .chip-initiated { background: rgba(0, 85, 165, 0.12); color: var(--primary); }
    .chip-in-progress { background: rgba(255, 210, 5, 0.18); color: #8a6a00; }
    .chip-completed { background: rgba(45, 209, 144, 0.18); color: #046644; }
    .chip-pending { background: rgba(255, 122, 0, 0.16); color: #a24b00; }

    .item-meta-grid {
        display: grid;
        gap: 6px;
        font-size: 13px;
        color: var(--text-muted);
    }

    .item-meta-grid strong {
        color: var(--text-strong);
        font-weight: 600;
    }

    .item-note {
        background: rgba(0, 85, 165, 0.08);
        color: var(--text-strong);
        padding: 10px 12px;
        border-radius: 12px;
        font-size: 13px;
        line-height: 1.5;
    }

    .item-assignees {
        display: flex;
        gap: 6px;
        flex-wrap: wrap;
    }

    .item-assignee {
        background: rgba(0, 85, 165, 0.12);
        color: var(--primary);
        padding: 4px 10px;
        border-radius: 999px;
        font-size: 12px;
        font-weight: 600;
    }

    .item-footer {
        display: flex;
        align-items: center;
        justify-content: space-between;
        font-size: 11px;
        color: var(--text-muted);
    }

    .track-list {
        list-style: none;
        margin: 0;
        padding: 0;
        display: flex;
        flex-direction: column;
        gap: 10px;
    }

    .track-list-item {
        display: flex;
        align-items: center;
        gap: 12px;
        padding: 12px 14px;
        border-radius: 14px;
        border: 1px solid rgba(0, 85, 165, 0.12);
        background: rgba(0, 85, 165, 0.06);
    }

    .track-avatar {
        width: 40px;
        height: 40px;
        border-radius: 12px;
        background: linear-gradient(135deg, #0055A5, #002f63);
        color: #fff;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        font-weight: 700;
        font-size: 16px;
    }

    .remark-card {
        display: flex;
        gap: 12px;
        padding: 16px 0;
        border-bottom: 1px solid rgba(0, 85, 165, 0.08);
    }

    .remark-card:last-child {
        border-bottom: none;
    }

    .remark-avatar {
        width: 44px;
        height: 44px;
        border-radius: 50%;
        background: linear-gradient(135deg, #0055A5, #002f63);
        color: #fff;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 700;
        font-size: 18px;
    }

    .remark-body {
        flex: 1;
        display: flex;
        flex-direction: column;
        gap: 6px;
    }

    .remark-meta {
        display: flex;
        align-items: center;
        justify-content: space-between;
        font-size: 12px;
        color: var(--text-muted);
    }

    .remark-text {
        font-size: 14px;
        color: var(--text-strong);
        line-height: 1.6;
    }

    .empty-placeholder,
    .empty-state {
        padding: 32px 18px;
        border-radius: 16px;
        border: 1px dashed rgba(0, 85, 165, 0.18);
        background: rgba(0, 85, 165, 0.04);
        color: var(--text-muted);
        font-weight: 500;
        text-align: center;
    }

    .track-table {
        width: 100%;
        border-collapse: collapse;
        border-radius: 16px;
        overflow: hidden;
        border: 1px solid rgba(0, 85, 165, 0.12);
    }

    .track-table thead {
        background: rgba(0, 85, 165, 0.08);
        color: var(--primary);
        font-weight: 600;
    }

    .track-table th,
    .track-table td {
        padding: 12px 14px;
        font-size: 13px;
        text-align: left;
        border-bottom: 1px solid rgba(0, 85, 165, 0.08);
    }

    .track-table tbody tr:last-child td {
        border-bottom: none;
    }

    .attachment-link {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 8px 14px;
        border-radius: 12px;
        background: rgba(0, 85, 165, 0.1);
        color: var(--primary);
        font-weight: 600;
        text-decoration: none;
        transition: background 0.2s ease, transform 0.2s ease;
    }

    .attachment-link:hover {
        background: rgba(0, 85, 165, 0.18);
        transform: translateY(-1px);
    }

    .loader-overlay {
        position: fixed;
        inset: 0;
        background: rgba(255, 255, 255, 0.86);
        backdrop-filter: blur(6px);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 1050;
    }

    .loader-spinner {
        width: 70px;
        height: 70px;
        border-radius: 50%;
        border: 6px solid rgba(0, 85, 165, 0.14);
        border-top-color: var(--primary);
        animation: spin 0.9s linear infinite;
    }

    .loader-text {
        margin-top: 14px;
        font-size: 15px;
        font-weight: 600;
        color: var(--text-strong);
        text-align: center;
    }

    @keyframes spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }

    .track-error {
        background: rgba(255, 94, 94, 0.12);
        border: 1px solid rgba(255, 94, 94, 0.35);
        border-radius: 18px;
        padding: 18px;
        color: #b91c1c;
        margin-bottom: 18px;
    }

    .track-modal-overlay {
        position: fixed;
        inset: 0;
        background: rgba(13, 37, 64, 0.55);
        backdrop-filter: blur(2px);
        display: none;
        align-items: center;
        justify-content: center;
        z-index: 12000;
        padding: 20px;
    }

    .track-modal-overlay.active {
        display: flex;
    }

    .track-modal {
        background: #fff;
        border-radius: 22px;
        box-shadow: 0 32px 56px rgba(13, 37, 64, 0.28);
        width: 100%;
        max-width: 520px;
        max-height: 90vh;
        overflow-y: auto;
        padding: 28px;
        position: relative;
    }

    .modal-subtitle {
        font-size: 13px;
        color: var(--text-muted);
        margin-top: -4px;
        margin-bottom: 10px;
    }

    .form-field label {
        font-size: 12px;
        letter-spacing: 0.08em;
        text-transform: uppercase;
        color: var(--text-muted);
        font-weight: 600;
        margin-bottom: 6px;
        display: block;
    }

    .form-field input,
    .form-field select,
    .form-field textarea {
        width: 100%;
        border-radius: 12px;
        border: 1px solid rgba(0, 85, 165, 0.18);
        background: rgba(0, 85, 165, 0.06);
        padding: 12px 14px;
        font-size: 14px;
        color: var(--text-strong);
        transition: border-color 0.2s ease, box-shadow 0.2s ease;
    }

    .form-field textarea {
        min-height: 90px;
        resize: vertical;
    }

    .form-field input:focus,
    .form-field select:focus,
    .form-field textarea:focus {
        outline: none;
        border-color: var(--primary);
        box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.15);
    }

    .modal-actions {
        display: flex;
        justify-content: flex-end;
        gap: 10px;
        margin-top: 10px;
    }

    .modal-btn {
        border: none;
        border-radius: 12px;
        padding: 11px 20px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s ease, box-shadow 0.2s ease;
    }

    .modal-btn.secondary {
        background: rgba(0, 85, 165, 0.08);
        color: var(--primary);
    }

    .modal-btn.secondary:hover {
        transform: translateY(-1px);
        box-shadow: 0 12px 20px rgba(0, 55, 120, 0.12);
    }

    .modal-btn.primary {
        background: linear-gradient(135deg, var(--primary), var(--primary-dark));
        color: #fff;
        box-shadow: 0 16px 32px rgba(0, 55, 120, 0.28);
    }

    .modal-btn.primary:hover {
        transform: translateY(-1px);
    }

    .modal-close {
        position: absolute;
        top: 18px;
        right: 18px;
        width: 34px;
        height: 34px;
        border-radius: 50%;
        border: none;
        background: rgba(0, 85, 165, 0.08);
        color: var(--primary);
        font-size: 20px;
        cursor: pointer;
        transition: background 0.2s ease;
    }

    .modal-close:hover {
        background: rgba(0, 85, 165, 0.16);
    }

    .assign-search {
        border-radius: 12px;
        border: 1px solid rgba(0, 85, 165, 0.18);
        background: rgba(0, 85, 165, 0.06);
        padding: 12px 14px;
        font-size: 14px;
        color: var(--text-strong);
        margin-bottom: 14px;
    }

    .assign-list {
        border: 1px solid rgba(0, 85, 165, 0.12);
        border-radius: 14px;
        overflow: hidden;
        max-height: 320px;
        overflow-y: auto;
    }

    .assign-item {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 12px 16px;
        border-bottom: 1px solid rgba(0, 85, 165, 0.08);
        cursor: pointer;
        transition: background 0.2s ease;
    }

    .assign-item:last-child {
        border-bottom: none;
    }

    .assign-item:hover {
        background: rgba(0, 85, 165, 0.08);
    }

    .assign-item.active {
        background: rgba(0, 85, 165, 0.12);
        border-left: 4px solid var(--primary);
    }

    .assign-indicator {
        width: 20px;
        height: 20px;
        border-radius: 50%;
        border: 2px solid rgba(0, 85, 165, 0.2);
    }

    .assign-item.active .assign-indicator {
        background: var(--primary);
        border-color: transparent;
        box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.18);
    }

    .assign-meta {
        display: grid;
        gap: 4px;
        font-size: 13px;
    }

    .assign-name {
        font-weight: 600;
        color: var(--text-strong);
    }

    .assign-sub {
        color: var(--text-muted);
        font-size: 12px;
    }

    .form-flex {
        display: flex;
        gap: 12px;
        flex-wrap: wrap;
    }

    .form-flex .form-field {
        flex: 1 1 160px;
    }

    .timeline {
        position: relative;
        padding-left: 26px;
        display: grid;
        gap: 14px;
    }

    .timeline::before {
        content: '';
        position: absolute;
        top: 6px;
        bottom: 6px;
        left: 10px;
        width: 2px;
        background: linear-gradient(180deg, rgba(0, 85, 165, 0.18), rgba(0, 85, 165, 0.02));
    }

    .timeline-item {
        background: rgba(0, 85, 165, 0.06);
        border: 1px solid rgba(0, 85, 165, 0.12);
        border-radius: 14px;
        padding: 12px 14px;
        position: relative;
        box-shadow: 0 8px 20px rgba(9, 45, 82, 0.12);
    }

    .timeline-item::before {
        content: '';
        position: absolute;
        left: -22px;
        top: 14px;
        width: 12px;
        height: 12px;
        border-radius: 50%;
        background: var(--accent);
        border: 3px solid #fff;
        box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.18);
    }

    .timeline-meta {
        font-size: 12px;
        color: var(--text-muted);
        margin-bottom: 6px;
        display: flex;
        gap: 12px;
    }

    .d-none {
        display: none !important;
    }

    @media (max-width: 540px) {
        .track-wrapper {
            padding: 20px 14px 96px;
        }

        .jobcard-shell {
            padding: 22px 18px 26px;
            border-radius: 24px;
        }

        .header-meta {
            grid-template-columns: repeat(2, minmax(0, 1fr));
        }

        .overview-grid {
            grid-template-columns: repeat(2, minmax(0, 1fr));
        }
    }

    @media (max-width: 400px) {
        .header-meta,
        .overview-grid {
            grid-template-columns: repeat(1, minmax(0, 1fr));
        }

        .track-btn-secondary {
            flex: 1 1 100%;
            justify-content: center;
        }
    }

    .section-actions {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
    }

    .items-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 18px;
        margin-bottom: 18px;
    }

    .items-header .section-title {
        display: flex;
        align-items: center;
        gap: 10px;
        margin-bottom: 4px;
    }

    .items-count {
        margin: 0;
        font-size: 13px;
        color: var(--text-muted);
        font-weight: 500;
    }

    .items-toolbar {
        display: flex;
        align-items: center;
        gap: 12px;
    }

    .add-item-btn {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        background: var(--primary);
        color: #fff;
        border: none;
        border-radius: 12px;
        padding: 10px 18px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s ease, box-shadow 0.2s ease, background 0.2s ease;
    }

    .add-item-btn:hover {
        background: var(--primary-dark);
        transform: translateY(-1px);
        box-shadow: 0 12px 24px rgba(0, 55, 120, 0.16);
    }

    .add-item-btn i {
        font-size: 14px;
    }

    .items-tabs {
        display: flex;
        gap: 10px;
        flex-wrap: wrap;
        margin-bottom: 18px;
    }

    .items-filter-btn {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        border-radius: 999px;
        padding: 8px 14px;
        border: 1px solid rgba(0, 85, 165, 0.15);
        background: rgba(0, 85, 165, 0.05);
        color: var(--text-muted);
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        transition: background 0.2s ease, color 0.2s ease, transform 0.2s ease, box-shadow 0.2s ease;
    }

    .items-filter-btn i {
        font-size: 13px;
    }

    .items-filter-btn.active {
        background: var(--primary);
        color: #fff;
        box-shadow: 0 8px 18px rgba(0, 55, 120, 0.22);
        transform: translateY(-1px);
    }

    .items-filter-btn.active i {
        color: #fff;
    }

    .items-tabs::-webkit-scrollbar {
        display: none;
    }

    .items-filter-btn span.tab-label {
        font-size: 13px;
        font-weight: 600;
    }
</style>
@endsection

@section('content')
<div class="track-wrapper">
    <div id="trackError" class="track-error">
        <h3>Unable to load job card</h3>
        <p id="trackErrorMessage">Please check the job card information and try again.</p>
    </div>

    <div class="jobcard-shell d-none" id="trackHeader">
        <div class="header-nav">
            <a href="{{ route('maintenance.workorder') }}" class="nav-button">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="15 18 9 12 15 6"></polyline><line x1="9" y1="12" x2="21" y2="12"></line></svg>
                Back
            </a>
            <div class="header-center">
                <div class="track-status-chip status-hold" id="trackStatusChip">Status</div>
                <h1 class="track-title" id="trackTitle">Job Card</h1>
                <p class="track-subtitle" id="trackSubtitle">Request number • Created</p>
            </div>
            <a id="downloadPdfBtn" href="#" target="_blank" rel="noopener" class="download-btn d-none">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7 10 12 15 17 10"/><line x1="12" y1="15" x2="12" y2="3"/></svg>
                Download
            </a>
        </div>

        <div class="header-meta">
            <div class="meta-item">
                <span class="meta-label">Priority</span>
                <span class="meta-value" id="trackPriority">-</span>
            </div>
            <div class="meta-item">
                <span class="meta-label">Progress</span>
                <span class="meta-value" id="trackProgress">0%</span>
                <span class="meta-subtext" id="trackProgressBreakdown">No tasks tracked yet</span>
            </div>
            <div class="meta-item">
                <span class="meta-label">Start</span>
                <span class="meta-value" id="trackStartDate">-</span>
            </div>
            <div class="meta-item">
                <span class="meta-label">Due</span>
                <span class="meta-value" id="trackDueDate">-</span>
            </div>
        </div>

        <div class="track-badges" id="trackBadges"></div>

        <div class="status-toggle-row" id="statusChipRow">
            <button class="status-toggle" data-action="set-status" data-status-value="Hold">Hold</button>
            <button class="status-toggle" data-action="set-status" data-status-value="assigned">Assigned</button>
            <button class="status-toggle" data-action="set-status" data-status-value="in_progress">In Progress</button>
            <button class="status-toggle" data-action="set-status" data-status-value="complete">Complete</button>
            <button class="status-toggle" data-action="set-status" data-status-value="cancelled">Cancelled</button>
        </div>
    </div>

    <div class="tab-bar" id="trackTabs">
        <button class="tab-btn active" data-tab-target="overviewTab">Overview</button>
        <button class="tab-btn" data-tab-target="remarksTab">Remarks</button>
    </div>

    <div class="tab-panel active" id="overviewTab">
        <div class="section-card d-none" id="trackOverview">
            <div class="section-title">Summary</div>
            <div class="overview-grid">
                <div class="overview-item">
                    <span class="overview-label">Assigned To</span>
                    <span class="overview-value" id="trackAssigneeCount">0 team members</span>
                </div>
                <div class="overview-item">
                    <span class="overview-label">Created By</span>
                    <span class="overview-value" id="trackCreatedBy">-</span>
                </div>
                <div class="overview-item">
                    <span class="overview-label">Created</span>
                    <span class="overview-value" id="trackCreatedAt">-</span>
                </div>
                <div class="overview-item">
                    <span class="overview-label">Updated</span>
                    <span class="overview-value" id="trackUpdatedAt">-</span>
                </div>
            </div>
            <div class="overview-grid">
                <div class="overview-item">
                    <span class="overview-label">Completed</span>
                    <span class="overview-value" id="trackCompletedDate">-</span>
                </div>
                <div class="overview-item">
                    <span class="overview-label">Equipment</span>
                    <span class="overview-value" id="trackEquipment">-</span>
                </div>
                <div class="overview-item">
                    <span class="overview-label">Category</span>
                    <span class="overview-value" id="trackCategory">-</span>
                </div>
                <div class="overview-item">
                    <span class="overview-label">Work Type</span>
                    <span class="overview-value" id="trackType">-</span>
                </div>
            </div>
        </div>

        <div class="section-card d-none" id="descriptionCard">
            <div class="section-title">Description &amp; Notes</div>
            <p id="trackDescription" style="margin:0;"></p>
        </div>

        <div class="section-card d-none" id="trackControls">
            <div class="section-title">Quick Actions</div>
            <div class="track-controls">
                <button class="track-btn-secondary" data-action="change-status">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="3"/><path d="M12 3v1"/><path d="M12 20v1"/><path d="M20 12h1"/><path d="M3 12h1"/><path d="M18.36 5.64l.7-.7"/><path d="M5.64 18.36l-.7.7"/><path d="M18.36 18.36l.7.7"/><path d="M5.64 5.64l-.7-.7"/></svg>
                    Update Status
                </button>
                <button class="track-btn-secondary" data-action="add-item">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="12" y1="5" x2="12" y2="19"/><line x1="5" y1="12" x2="19" y2="12"/></svg>
                    Add Item
                </button>
                <button class="track-btn-secondary" data-action="add-remark">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg>
                    Add Remark
                </button>
                <button class="track-btn-secondary" data-action="refresh">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="23 4 23 10 17 10"/><polyline points="1 20 1 14 7 14"/><path d="M3.51 9A9 9 0 0 1 21 8"/><path d="M20.49 15A9 9 0 0 1 3 16"/></svg>
                    Refresh
                </button>
            </div>
        </div>

        <div class="section-card d-none" id="itemsSection">
            <div class="items-header">
                <div>
                    <div class="section-title">
                        <i class="fas fa-tasks"></i>
                        Tasks &amp; Items
                    </div>
                    <p class="items-count" id="itemsCountLabel"></p>
                </div>
                <div class="items-toolbar">
                    <button class="add-item-btn" id="itemsAddButton" data-action="add-item">
                        <i class="fas fa-plus"></i>
                        <span id="itemsAddLabel">Add Item</span>
                    </button>
                </div>
            </div>
            <div class="items-tabs" id="itemsFilters"></div>
            <div class="track-items" id="itemsList">
                <div class="empty-placeholder" id="itemsEmpty">No work items have been recorded yet.</div>
            </div>
        </div>

        <div class="section-card d-none" id="trackDetails">
            <div class="section-title">Assigned Team</div>
            <ul class="track-list" id="trackAssignees"></ul>
            <div class="empty-state d-none" id="assigneesEmpty">No team members assigned to this job card.</div>

            <div class="section-title" style="margin-top:20px;">Tracking Timeline</div>
            <div class="timeline" id="trackTimeline"></div>
            <div class="empty-state d-none" id="timelineEmpty">Tracking information will appear here as progress is updated.</div>

            <div class="section-title" style="margin-top:20px;">Job Card Tasks</div>
            <div class="empty-state d-none" id="tasksEmpty">No work items have been added to this job card yet.</div>
            <div class="table-responsive">
                <table class="track-table d-none" id="tasksTable">
                    <thead>
                        <tr>
                            <th style="width: 38px;">#</th>
                            <th>Task</th>
                            <th>Assigned To</th>
                            <th>Status</th>
                            <th>Category</th>
                            <th>Notes</th>
                        </tr>
                    </thead>
                    <tbody id="tasksTableBody"></tbody>
                </table>
            </div>

            <div class="section-title" style="margin-top:20px;">Attachments</div>
            <div id="attachmentsList"></div>
            <div class="empty-state d-none" id="attachmentsEmpty">No attachments have been uploaded for this job card.</div>

            <div class="section-title" style="margin-top:20px;">Checklist Items</div>
            <div id="checklistItems"></div>
            <div class="empty-state d-none" id="checklistEmpty">No checklist items recorded for this job card.</div>
        </div>
    </div>

    <div class="tab-panel" id="remarksTab">
        <div class="section-card d-none" id="remarksSection">
            <div class="track-section-header" style="margin-bottom:0;">
                <div class="section-title" style="margin-bottom:0;">Remarks</div>
                <div class="section-actions">
                    <button class="track-btn-secondary" data-action="add-remark">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg>
                        Add Remark
                    </button>
                </div>
            </div>
            <div id="remarksList">
                <div class="empty-placeholder" id="remarksEmpty">No remarks yet.</div>
            </div>
        </div>
    </div>
</div>

<div class="loader-overlay" id="trackLoader">
    <div style="text-align:center;">
        <div class="loader-spinner"></div>
        <div class="loader-text">Loading job card details...</div>
    </div>
</div>

<!-- Status Modal -->
<div class="track-modal-overlay" id="statusModal" aria-hidden="true">
    <div class="track-modal" role="dialog" aria-modal="true" aria-labelledby="statusModalTitle">
        <button class="modal-close" data-modal-close>&times;</button>
        <h3 id="statusModalTitle">Update Job Card Status</h3>
        <p class="modal-subtitle">Adjust the overall status of this job card. All watchers will see the new state immediately.</p>
        <form id="statusForm">
            <div class="form-field">
                <label for="statusSelect">Status</label>
                <select id="statusSelect" required>
                    <option value="Hold">Hold</option>
                    <option value="assigned">Assigned</option>
                    <option value="in_progress">In Progress</option>
                    <option value="complete">Complete</option>
                    <option value="cancelled">Cancelled</option>
                </select>
            </div>
            <div class="form-field">
                <label for="statusNote">Internal Note (optional)</label>
                <textarea id="statusNote" placeholder="Describe the reason for this status change..."></textarea>
            </div>
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" data-modal-close>Cancel</button>
                <button type="submit" class="modal-btn primary">Update Status</button>
            </div>
        </form>
    </div>
</div>

<!-- Item Modal -->
<div class="track-modal-overlay" id="itemModal" aria-hidden="true">
    <div class="track-modal" role="dialog" aria-modal="true" aria-labelledby="itemModalTitle">
        <button class="modal-close" data-modal-close>&times;</button>
        <h3 id="itemModalTitle">Add Item</h3>
        <p class="modal-subtitle" id="itemModalSubtitle">Create a new work item for this job card.</p>
        <form id="itemForm">
            <input type="hidden" id="itemId">
            <div class="form-flex">
                <div class="form-field">
                    <label for="itemCategory">Category</label>
                    <select id="itemCategory" required>
                        <option value="Task">Task</option>
                        <option value="Parts">Parts</option>
                        <option value="Labor">Labor</option>
                        <option value="Additional Cost">Additional Cost</option>
                        <option value="Attach">Attach</option>
                    </select>
                </div>
                <div class="form-field" id="itemTypeField">
                    <label for="itemType">Type</label>
                    <select id="itemType">
                        <option value="Repair">Repair</option>
                        <option value="Replace">Replace</option>
                    </select>
                </div>
            </div>
            <div class="form-field" id="itemTitleField">
                <label for="itemTitle">Title / Description</label>
                <input type="text" id="itemTitle" placeholder="Enter a short descriptive title" required>
            </div>
            <div class="form-flex" id="partsRow" hidden>
                <div class="form-field">
                    <label for="itemNumber">Item #</label>
                    <input type="text" id="itemNumber">
                </div>
                <div class="form-field">
                    <label for="partNumber">Part #</label>
                    <input type="text" id="partNumber">
                </div>
            </div>
            <div class="form-flex" id="quantityRow" hidden>
                <div class="form-field">
                    <label for="quantity">Quantity</label>
                    <input type="number" id="quantity" min="0" step="1">
                </div>
                <div class="form-field">
                    <label for="unit">Unit</label>
                    <input type="text" id="unit">
                </div>
            </div>
            <div class="form-flex" id="costRow" hidden>
                <div class="form-field">
                    <label for="itemCost">Cost</label>
                    <input type="number" id="itemCost" step="0.01" min="0">
                </div>
                <div class="form-field">
                    <label for="itemSite">Site / Location</label>
                    <input type="text" id="itemSite">
                </div>
            </div>
            <div class="form-flex" id="laborRow" hidden>
                <div class="form-field">
                    <label for="laborCode">Labor Code</label>
                    <input type="text" id="laborCode">
                </div>
                <div class="form-field">
                    <label for="laborRate">Rate</label>
                    <input type="number" id="laborRate" step="0.01" min="0">
                </div>
            </div>
            <div class="form-flex" id="laborTimeRow" hidden>
                <div class="form-field">
                    <label for="laborHours">Hours</label>
                    <input type="number" id="laborHours" min="0" step="1">
                </div>
                <div class="form-field">
                    <label for="laborMinutes">Minutes</label>
                    <input type="number" id="laborMinutes" min="0" step="1">
                </div>
            </div>
            <div class="form-field" id="amountRow" hidden>
                <label for="additionalAmount">Amount</label>
                <input type="number" id="additionalAmount" step="0.01" min="0">
            </div>
            <div class="form-field" id="fileRow" hidden>
                <label for="itemFileName">Attachment Name</label>
                <input type="text" id="itemFileName" placeholder="E.g. inspection-report.pdf">
            </div>
            <div class="form-field">
                <label for="itemNote">Notes (optional)</label>
                <textarea id="itemNote" placeholder="Add additional context, acceptance criteria or instructions..."></textarea>
            </div>
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" data-modal-close>Cancel</button>
                <button type="submit" class="modal-btn primary">Save Item</button>
            </div>
        </form>
    </div>
</div>

<!-- Assign Modal -->
<div class="track-modal-overlay" id="assignModal" aria-hidden="true">
    <div class="track-modal" role="dialog" aria-modal="true" aria-labelledby="assignModalTitle">
        <button class="modal-close" data-modal-close>&times;</button>
        <h3 id="assignModalTitle">Assign Team Members</h3>
        <p class="modal-subtitle">Select one or more team members to be responsible for this item.</p>
        <input type="search" class="assign-search" id="assignSearch" placeholder="Search by name, employee number or role...">
        <div class="assign-list" id="assignList">
            <div class="empty-placeholder" id="assignEmpty">No team members found.</div>
        </div>
        <div class="modal-actions" style="margin-top:18px;">
            <button type="button" class="modal-btn secondary" data-modal-close>Cancel</button>
            <button type="button" class="modal-btn primary" id="saveAssignBtn">Save Selection</button>
        </div>
    </div>
</div>

<!-- Item Status Modal -->
<div class="track-modal-overlay" id="itemStatusModal" aria-hidden="true">
    <div class="track-modal" role="dialog" aria-modal="true" aria-labelledby="itemStatusModalTitle">
        <button class="modal-close" data-modal-close>&times;</button>
        <h3 id="itemStatusModalTitle">Update Item Status</h3>
        <form id="itemStatusForm">
            <input type="hidden" id="itemStatusId">
            <div class="form-field">
                <label for="itemStatusSelect">Status</label>
                <select id="itemStatusSelect" required>
                    <option value="Initiated">Initiated</option>
                    <option value="assigned">Assigned</option>
                    <option value="in_progress">In Progress</option>
                    <option value="Complete">Complete</option>
                    <option value="Reject">Reject</option>
                </select>
            </div>
            <div class="form-field">
                <label for="itemStatusNote">Update Note (optional)</label>
                <textarea id="itemStatusNote" placeholder="Describe this progress update..."></textarea>
            </div>
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" data-modal-close>Cancel</button>
                <button type="submit" class="modal-btn primary">Update Item</button>
            </div>
        </form>
    </div>
</div>

<!-- Remark Modal -->
<div class="track-modal-overlay" id="remarkModal" aria-hidden="true">
    <div class="track-modal" role="dialog" aria-modal="true" aria-labelledby="remarkModalTitle">
        <button class="modal-close" data-modal-close>&times;</button>
        <h3 id="remarkModalTitle">Add Remark</h3>
        <form id="remarkForm">
            <div class="form-field">
                <label for="remarkText">Remark</label>
                <textarea id="remarkText" required placeholder="Share an update or context with the team..."></textarea>
            </div>
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" data-modal-close>Cancel</button>
                <button type="submit" class="modal-btn primary">Add Remark</button>
            </div>
        </form>
    </div>
</div>
@endsection

@section('scripts')
<script>
    const JOB_CARD_API_BASE = 'https://api.titandrillingzm.com:6005';
    const USERS_API_BASE = 'https://api.titandrillingzm.com:6003';
    const TEAM_ENDPOINT = '/users/team-management';

    const INTERACTIVE_STATE = {
        jobCardId: null,
        jobCard: null,
        adminUid: null,
        items: [],
        progressMetrics: null,
        selectedCategory: 'All',
        assignableMembers: [],
        selectedAssignees: new Map(),
        activeItemId: null,
        assignSearch: '',
        eventsBound: false,
        ui: {},
    };

    function getQueryParam(param, defaultValue = '') {
        const urlParams = new URLSearchParams(window.location.search);
        return urlParams.get(param) || defaultValue;
    }

    function formatDateTime(value, fallback = '—') {
        if (!value) return fallback;
        const date = new Date(value);
        if (Number.isNaN(date.getTime())) return fallback;
        return date.toLocaleString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    function formatDate(value, fallback = '—') {
        if (!value) return fallback;
        const date = new Date(value);
        if (Number.isNaN(date.getTime())) return fallback;
        return date.toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric'
        });
    }

    function formatStatus(status = '') {
        if (!status) return { label: 'Unknown', className: 'status-hold' };
        const normalized = status.toLowerCase();
        switch (normalized) {
            case 'hold':
            case 'initiated':
                return { label: 'Hold', className: 'status-hold' };
            case 'assigned':
                return { label: 'Assigned', className: 'status-assigned' };
            case 'in_progress':
            case 'in progress':
                return { label: 'In Progress', className: 'status-in-progress' };
            case 'complete':
            case 'completed':
                return { label: 'Completed', className: 'status-complete' };
            case 'cancelled':
                return { label: 'Cancelled', className: 'status-cancelled' };
            default:
                return { label: status, className: 'status-hold' };
        }
    }

    function formatTaskStatus(status = '') {
        const normalized = status.toLowerCase();
        if (['initiated', 'hold', 'pending'].includes(normalized)) return { label: 'Initiated', className: 'chip-pending' };
        if (['assigned', 'in_progress', 'in progress'].includes(normalized)) return { label: 'In Progress', className: 'chip-in-progress' };
        if (['completed', 'done', 'resolved'].includes(normalized)) return { label: 'Completed', className: 'chip-completed' };
        return { label: status || 'Pending', className: 'chip-pending' };
    }

    function showLoader(show) {
        const loader = document.getElementById('trackLoader');
        if (loader) loader.classList.toggle('d-none', !show);
    }

    function togglePageLoader(show) {
        const hasGlobalShow = typeof window.showPageLoader === 'function';
        const hasGlobalHide = typeof window.hidePageLoader === 'function';
        const pageLoader = document.getElementById('pageLoader');

        if (show) {
            if (hasGlobalShow) {
                window.showPageLoader();
            } else if (pageLoader) {
                pageLoader.classList.remove('hidden');
            }
        } else {
            if (hasGlobalHide) {
                window.hidePageLoader();
            } else if (pageLoader) {
                pageLoader.classList.add('hidden');
            }
        }
    }

    function showError(message) {
        showLoader(false);
        const errorBox = document.getElementById('trackError');
        const messageEl = document.getElementById('trackErrorMessage');
        if (messageEl) messageEl.textContent = message;
        if (errorBox) errorBox.style.display = 'block';
    }

    function hideError() {
        const errorBox = document.getElementById('trackError');
        if (errorBox) errorBox.style.display = 'none';
    }

    function showSection(id, show) {
        const element = document.getElementById(id);
        if (element) element.classList.toggle('d-none', !show);
    }

    function initializeTabNavigation() {
        const tabButtons = document.querySelectorAll('[data-tab-target]');
        const tabPanels = document.querySelectorAll('.tab-panel');
        if (!tabButtons.length || !tabPanels.length) return;

        tabButtons.forEach((button) => {
            button.addEventListener('click', () => {
                const target = button.dataset.tabTarget;
                tabButtons.forEach((btn) => btn.classList.toggle('active', btn === button));
                tabPanels.forEach((panel) => panel.classList.toggle('active', panel.id === target));
            });
        });
    }

    function updateStatusChip(statusInfo) {
        const chip = document.getElementById('trackStatusChip');
        if (!chip) return;
        chip.textContent = statusInfo.label;
        chip.className = `track-status-chip ${statusInfo.className}`;
    }

    function updateStatusToggles(status = '') {
        const normalized = normalizeStatusValue(status);
        document.querySelectorAll('[data-action="set-status"]').forEach((button) => {
            const value = normalizeStatusValue(button.dataset.statusValue);
            button.classList.toggle('active', value === normalized);
        });
    }

    function renderAssignees(assignTo = []) {
        const list = document.getElementById('trackAssignees');
        const emptyState = document.getElementById('assigneesEmpty');
        const countEl = document.getElementById('trackAssigneeCount');

        if (!list || !emptyState || !countEl) return;

        list.innerHTML = '';

        if (!Array.isArray(assignTo) || assignTo.length === 0) {
            list.classList.add('d-none');
            emptyState.classList.remove('d-none');
            countEl.textContent = 'Not assigned';
            return;
        }

        countEl.textContent = `${assignTo.length} team member${assignTo.length > 1 ? 's' : ''}`;
        list.classList.remove('d-none');
        emptyState.classList.add('d-none');

        assignTo.forEach((member, index) => {
            const initials = (member.name || member.email || `Team Member ${index + 1}`)
                .split(' ')
                .map(part => part[0])
                .join('')
                .toUpperCase()
                .slice(0, 2);

            const item = document.createElement('li');
            item.className = 'track-list-item';
            item.innerHTML = `
                <span class="track-avatar">${initials}</span>
                <div class="track-assignee-info">
                    <span class="track-assignee-name">${member.name || member.email || `Team Member ${index + 1}`}</span>
                    ${member.email ? `<span class="track-assignee-email">${member.email}</span>` : ''}
                </div>
            `;
            list.appendChild(item);
        });
    }

    function renderTasks(items = []) {
        const table = document.getElementById('tasksTable');
        const tbody = document.getElementById('tasksTableBody');
        const emptyState = document.getElementById('tasksEmpty');
        if (!table || !tbody || !emptyState) return;

        tbody.innerHTML = '';

        if (!Array.isArray(items) || items.length === 0) {
            table.classList.add('d-none');
            emptyState.classList.remove('d-none');
            return;
        }

        table.classList.remove('d-none');
        emptyState.classList.add('d-none');

        items.forEach((item, index) => {
            const statusInfo = formatTaskStatus(item.status || item.currentStatus);
            const assignees = Array.isArray(item.assignTo) && item.assignTo.length > 0
                ? item.assignTo.map(person => person.name || person.email || 'Team Member').join(', ')
                : 'Not assigned';

            const row = document.createElement('tr');
            row.innerHTML = `
                <td>${index + 1}</td>
                <td>
                    <div style="font-weight:600;color:#0f172a;">${item.title || item.taskTitle || 'Task'}</div>
                    ${item.description ? `<div style="color:#64748b;font-size:13px;margin-top:4px;">${item.description}</div>` : ''}
                </td>
                <td>${assignees}</td>
                <td><span class="status-chip ${statusInfo.className}">${statusInfo.label}</span></td>
                <td>${item.category || 'General'}</td>
                <td>${item.notes || item.internalNotes || '—'}</td>
            `;
            tbody.appendChild(row);
        });
    }

    function renderAttachments(attachments = [], pdfUrl = '') {
        const container = document.getElementById('attachmentsList');
        const emptyState = document.getElementById('attachmentsEmpty');
        if (!container || !emptyState) return;

        container.innerHTML = '';
        let hasAttachments = false;

        if (Array.isArray(attachments) && attachments.length > 0) {
            attachments.forEach((attachment) => {
                if (!attachment) return;
                const url = attachment.url || attachment.downloadUrl;
                if (!url) return;

                hasAttachments = true;
                const link = document.createElement('a');
                link.href = url;
                link.className = 'attachment-link';
                link.target = '_blank';
                link.rel = 'noopener';
                link.innerHTML = `
                    <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 16V7a2 2 0 0 0-2-2h-4a4 4 0 0 1-8 0H3a2 2 0 0 0-2 2v9"/><path d="M5 21h14"/><path d="M9 21v-7a3 3 0 0 1 6 0v7"/></svg>
                    ${attachment.originalName || attachment.filename || 'Attachment'}
                `;
                container.appendChild(link);
            });
        }

        if (pdfUrl) {
            hasAttachments = true;
            const pdfLink = document.createElement('a');
            pdfLink.href = pdfUrl;
            pdfLink.className = 'attachment-link';
            pdfLink.target = '_blank';
            pdfLink.rel = 'noopener';
            pdfLink.innerHTML = `
                <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/><polyline points="14 2 14 8 20 8"/></svg>
                Job Card PDF
            `;
            container.appendChild(pdfLink);
        }

        if (!hasAttachments) {
            emptyState.classList.remove('d-none');
        } else {
            emptyState.classList.add('d-none');
        }
    }

    function renderTimeline(tracking = {}) {
        const container = document.getElementById('trackTimeline');
        const emptyState = document.getElementById('timelineEmpty');
        if (!container || !emptyState) return;

        container.innerHTML = '';
        const history = Array.isArray(tracking.history) ? tracking.history : [];

        if (history.length === 0) {
            container.parentElement.querySelector('.timeline')?.classList.add('d-none');
            emptyState.classList.remove('d-none');
            return;
        }

        emptyState.classList.add('d-none');
        container.parentElement.querySelector('.timeline')?.classList.remove('d-none');

        history
            .sort((a, b) => new Date(b.timestamp || b.date) - new Date(a.timestamp || a.date))
            .forEach((entry) => {
                const item = document.createElement('div');
                item.className = 'timeline-item';
                const timestamp = formatDateTime(entry.timestamp || entry.date, '');
                item.innerHTML = `
                    <div class="timeline-meta">
                        <span>${timestamp || 'Timestamp unavailable'}</span>
                        ${entry.updatedBy ? `<span>• ${entry.updatedBy}</span>` : ''}
                    </div>
                    <div style="font-weight:600;color:#0f172a;">${entry.step || 'Update'}</div>
                    ${entry.notes ? `<div style="color:#475569;font-size:14px;margin-top:6px;">${entry.notes}</div>` : ''}
                `;
                container.appendChild(item);
            });
    }

    function renderChecklist(checklist = []) {
        const container = document.getElementById('checklistItems');
        const emptyState = document.getElementById('checklistEmpty');
        if (!container || !emptyState) return;

        container.innerHTML = '';

        if (!Array.isArray(checklist) || checklist.length === 0) {
            emptyState.classList.remove('d-none');
            return;
        }

        emptyState.classList.add('d-none');

        checklist.forEach((item, index) => {
            const statusInfo = formatTaskStatus(item.status || item.result);
            const wrapper = document.createElement('div');
            wrapper.className = 'track-list-item';
            wrapper.innerHTML = `
                <span class="track-avatar" style="background:linear-gradient(135deg,#0ea5e9,#0284c7);">${String(index + 1).padStart(2, '0')}</span>
                <div style="flex:1;">
                    <div style="font-weight:600;color:#0f172a;">${item.title || item.item || 'Checklist Item'}</div>
                    ${item.notes ? `<div style="color:#64748b;font-size:13px;margin-top:4px;">${item.notes}</div>` : ''}
                </div>
                <span class="status-chip ${statusInfo.className}">${statusInfo.label}</span>
            `;
            container.appendChild(wrapper);
        });
    }

    const PROGRESS_STAGE_WEIGHTS = {
        hold: 0,
        assigned: 33,
        in_progress: 67,
        complete: 100,
        cancelled: 0,
        other: 33,
    };

    const PROGRESS_STATUS_LABELS = {
        hold: 'Hold',
        assigned: 'Assigned',
        in_progress: 'In Progress',
        complete: 'Complete',
        cancelled: 'Cancelled',
        other: 'Other',
    };

    function normalizeProgressStatus(status = '') {
        const normalized = (status || '').toString().trim().toLowerCase();
        if (!normalized) return 'hold';
        if (['complete', 'completed', 'done', 'finished', 'resolved', 'closed'].includes(normalized)) return 'complete';
        if (['in progress', 'in_progress', 'progress', 'ongoing', 'working', 'processing', 'started'].includes(normalized)) return 'in_progress';
        if (['assigned', 'scheduled', 'initiated', 'open', 'pending start', 'pending-start'].includes(normalized)) return 'assigned';
        if (['hold', 'on hold', 'paused', 'pending', 'awaiting', 'waiting'].includes(normalized)) return 'hold';
        if (['cancelled', 'canceled', 'void', 'aborted', 'rejected'].includes(normalized)) return 'cancelled';
        return 'other';
    }

    function extractProgressItems(jobCard = {}) {
        const collections = [
            Array.isArray(jobCard.items) ? jobCard.items : [],
            Array.isArray(jobCard.checklistItems) ? jobCard.checklistItems : [],
            Array.isArray(jobCard.tracking?.tasks) ? jobCard.tracking.tasks : [],
        ];

        const excludedCategories = new Set(['remarks', 'remark', 'note', 'notes', 'attachment', 'attachments', 'attach', 'document']);
        const seen = new Set();
        const actionable = [];

        collections.forEach((list, collectionIndex) => {
            list.forEach((item, index) => {
                if (!item) return;
                const category = (item.category || item.type || '').toString().trim().toLowerCase();
                if (excludedCategories.has(category)) return;

                const identifier =
                    item.id ||
                    item._id ||
                    item.itemId ||
                    item.taskId ||
                    item.checklistId ||
                    item.referenceId ||
                    `${collectionIndex}_${index}_${item.title || item.description || item.text || 'item'}`;

                const key = String(identifier);
                if (!key || seen.has(key)) return;

                seen.add(key);
                actionable.push(item);
            });
        });

        return actionable;
    }

    function calculateProgressMetrics(jobCard = {}) {
        const items = extractProgressItems(jobCard);
        const counts = {
            hold: 0,
            assigned: 0,
            in_progress: 0,
            complete: 0,
            cancelled: 0,
            other: 0,
        };

        let weightedSum = 0;

        items.forEach((item) => {
            const normalized = normalizeProgressStatus(item.status || item.currentStatus || item.result || item.state || item.progressStatus);
            counts[normalized] = (counts[normalized] || 0) + 1;
            weightedSum += PROGRESS_STAGE_WEIGHTS[normalized] ?? PROGRESS_STAGE_WEIGHTS.other;
        });

        const total = items.length;
        const percentages = {};
        Object.keys(counts).forEach((key) => {
            percentages[key] = total ? Math.round((counts[key] / total) * 100) : 0;
        });

        const overallPercent = total ? Math.round(weightedSum / total) : 0;

        return {
            total,
            counts,
            percentages,
            overallPercent,
        };
    }

    function buildProgressBreakdown(metrics, fallbackProgress) {
        if (!metrics || !metrics.total) {
            if (typeof fallbackProgress === 'number' && !Number.isNaN(fallbackProgress)) {
                return `Progress reported at ${Math.round(fallbackProgress)}%`;
            }
            return 'No tasks tracked yet';
        }

        const order = ['complete', 'in_progress', 'assigned', 'hold', 'cancelled'];
        const segments = order
            .map((key) => {
                const count = metrics.counts[key] || 0;
                if (!count) return null;
                const percentage = metrics.percentages[key] ?? 0;
                return `${percentage}% ${PROGRESS_STATUS_LABELS[key]} (${count})`;
            })
            .filter(Boolean);

        return segments.length ? segments.join(' • ') : 'No actionable tasks tracked yet';
    }

    function populateBadges(jobCard) {
        const badgesContainer = document.getElementById('trackBadges');
        if (!badgesContainer) return;

        badgesContainer.innerHTML = '';

        const badges = [
            { label: jobCard.requestNumber || 'No Request Number', icon: '📄' },
            { label: jobCard.priority ? `Priority: ${jobCard.priority}` : null, icon: '⚡' },
            { label: jobCard.location?.country || jobCard.location?.project ? `${jobCard.location?.country || ''}${jobCard.location?.project ? ' • ' + jobCard.location.project : ''}` : null, icon: '📍' },
            { label: jobCard.workOrderType ? `Type: ${jobCard.workOrderType}` : null, icon: '🛠️' },
        ].filter(Boolean);

        badges.forEach((badge) => {
            if (!badge.label) return;
            const badgeEl = document.createElement('span');
            badgeEl.className = 'track-badge';
            badgeEl.textContent = `${badge.icon} ${badge.label}`;
            badgesContainer.appendChild(badgeEl);
        });
    }

    async function fetchJobCardDetails(jobCardId) {
        const response = await fetch(`${JOB_CARD_API_BASE}/jobcards/report/${encodeURIComponent(jobCardId)}`);
        if (!response.ok) {
            throw new Error(`API responded with ${response.status}`);
        }
        const result = await response.json();
        if (!result.success || !result.data) {
            throw new Error(result.error || 'Job card not found');
        }
        return result.data;
    }

    function applyJobCardData(jobCard) {
        const titleEl = document.getElementById('trackTitle');
        const subtitleEl = document.getElementById('trackSubtitle');
        const createdByEl = document.getElementById('trackCreatedBy');
        const createdAtEl = document.getElementById('trackCreatedAt');
        const updatedAtEl = document.getElementById('trackUpdatedAt');
        const startDateEl = document.getElementById('trackStartDate');
        const dueDateEl = document.getElementById('trackDueDate');
        const completedDateEl = document.getElementById('trackCompletedDate');
        const progressEl = document.getElementById('trackProgress');
        const progressBreakdownEl = document.getElementById('trackProgressBreakdown');
        const equipmentEl = document.getElementById('trackEquipment');
        const categoryEl = document.getElementById('trackCategory');
        const typeEl = document.getElementById('trackType');
        const priorityEl = document.getElementById('trackPriority');
        const descriptionEl = document.getElementById('trackDescription');
        const downloadBtn = document.getElementById('downloadPdfBtn');

        const statusInfo = formatStatus(jobCard.status);
        updateStatusChip(statusInfo);
        updateStatusToggles(jobCard.status);

        if (titleEl) titleEl.textContent = jobCard.title || jobCard.requestNumber || 'Job Card';
        if (subtitleEl) subtitleEl.textContent = `${jobCard.requestNumber || 'No request number'} • Created ${formatDateTime(jobCard.createdAt)}`;
        if (createdByEl) createdByEl.textContent = jobCard.createdBy || jobCard.adminUid || '—';
        if (createdAtEl) createdAtEl.textContent = formatDateTime(jobCard.createdAt);
        if (updatedAtEl) updatedAtEl.textContent = formatDateTime(jobCard.updatedAt);
        if (startDateEl) startDateEl.textContent = formatDate(jobCard.startDate);
        if (dueDateEl) dueDateEl.textContent = formatDate(jobCard.dueDate);
        if (completedDateEl) completedDateEl.textContent = formatDate(jobCard.completedDate);
        const trackingProgressValue = typeof jobCard.tracking?.progress === 'number' ? jobCard.tracking.progress : null;
        const progressMetrics = calculateProgressMetrics(jobCard);
        const calculatedProgress = progressMetrics.total ? progressMetrics.overallPercent : (trackingProgressValue ?? 0);
        if (progressEl) {
            progressEl.textContent = `${calculatedProgress}%`;
            progressEl.title = buildProgressBreakdown(progressMetrics, trackingProgressValue);
        }
        if (progressBreakdownEl) {
            progressBreakdownEl.textContent = buildProgressBreakdown(progressMetrics, trackingProgressValue);
        }
        INTERACTIVE_STATE.progressMetrics = progressMetrics;
        if (equipmentEl) equipmentEl.textContent = jobCard.equipment?.equipmentName || jobCard.equipment?.vehicleNumber || '—';
        if (categoryEl) categoryEl.textContent = jobCard.equipment?.mainCategory || jobCard.equipment?.status || '—';
        if (typeEl) typeEl.textContent = jobCard.workOrderType || '—';
        if (priorityEl) priorityEl.textContent = jobCard.priority || 'Medium';
        if (descriptionEl) descriptionEl.textContent = jobCard.description || jobCard.notes || 'No description recorded for this job card.';

        if (downloadBtn) {
            const pdfUrl = jobCard.pdfDownloadUrl || jobCard.pdfUrl || jobCard.documentUrl;
            if (pdfUrl) {
                downloadBtn.href = pdfUrl;
                downloadBtn.classList.remove('d-none');
            } else {
                downloadBtn.classList.add('d-none');
            }
        }

        populateBadges(jobCard);
        renderAssignees(jobCard.assignTo || []);
        renderTasks(jobCard.items || jobCard.checklistItems || []);
        renderAttachments(jobCard.attachments || [], jobCard.pdfDownloadUrl || jobCard.pdfUrl || jobCard.documentUrl);
        renderTimeline(jobCard.tracking || {});
        renderChecklist(jobCard.checklistItems || []);
    }

    function initializeInteractiveJobCard(jobCard) {
        INTERACTIVE_STATE.jobCard = jobCard;
        INTERACTIVE_STATE.adminUid = jobCard.adminUid || jobCard.adminUID || jobCard.createdBy || null;
        INTERACTIVE_STATE.items = normalizeInteractiveItems(Array.isArray(jobCard.items) ? jobCard.items : []);
        INTERACTIVE_STATE.selectedCategory = INTERACTIVE_STATE.selectedCategory || 'All';
        cacheInteractiveElements();
        if (!INTERACTIVE_STATE.eventsBound) {
            bindInteractiveEvents();
            INTERACTIVE_STATE.eventsBound = true;
        }
        renderInteractiveSections();
    }

    function normalizeInteractiveItems(items) {
        return items.map((item, index) => {
            const clone = { ...item };
            clone.id = clone.id || clone._id || `item_${Date.now()}_${index}`;
            clone.category = clone.category || 'Task';
            clone.status = clone.status || clone.currentStatus || 'Initiated';
            clone.assignTo = Array.isArray(clone.assignTo) ? clone.assignTo : (clone.assignTo ? [clone.assignTo] : []);
            clone.createdAt = clone.createdAt || Date.now();
            return clone;
        });
    }

    function cacheInteractiveElements() {
        const ui = INTERACTIVE_STATE.ui;
        ui.controls = document.getElementById('trackControls');
        ui.itemsSection = document.getElementById('itemsSection');
        ui.itemsList = document.getElementById('itemsList');
        ui.itemsEmpty = document.getElementById('itemsEmpty');
        ui.itemsFilters = document.getElementById('itemsFilters');
        ui.itemsCountLabel = document.getElementById('itemsCountLabel');
        ui.itemsAddButton = document.getElementById('itemsAddButton');
        ui.itemsAddLabel = document.getElementById('itemsAddLabel');
        ui.remarksSection = document.getElementById('remarksSection');
        ui.remarksList = document.getElementById('remarksList');
        ui.remarksEmpty = document.getElementById('remarksEmpty');
        ui.statusForm = document.getElementById('statusForm');
        ui.statusSelect = document.getElementById('statusSelect');
        ui.statusNote = document.getElementById('statusNote');
        ui.itemForm = document.getElementById('itemForm');
        ui.itemIdInput = document.getElementById('itemId');
        ui.itemCategory = document.getElementById('itemCategory');
        ui.itemType = document.getElementById('itemType');
        ui.itemTitle = document.getElementById('itemTitle');
        ui.itemNote = document.getElementById('itemNote');
        ui.itemNumber = document.getElementById('itemNumber');
        ui.partNumber = document.getElementById('partNumber');
        ui.quantity = document.getElementById('quantity');
        ui.unit = document.getElementById('unit');
        ui.itemCost = document.getElementById('itemCost');
        ui.itemSite = document.getElementById('itemSite');
        ui.laborCode = document.getElementById('laborCode');
        ui.laborRate = document.getElementById('laborRate');
        ui.laborHours = document.getElementById('laborHours');
        ui.laborMinutes = document.getElementById('laborMinutes');
        ui.additionalAmount = document.getElementById('additionalAmount');
        ui.itemFileName = document.getElementById('itemFileName');
        ui.partsRow = document.getElementById('partsRow');
        ui.quantityRow = document.getElementById('quantityRow');
        ui.costRow = document.getElementById('costRow');
        ui.laborRow = document.getElementById('laborRow');
        ui.laborTimeRow = document.getElementById('laborTimeRow');
        ui.amountRow = document.getElementById('amountRow');
        ui.fileRow = ui.itemFileName ? ui.itemFileName.closest('.form-field') : null;
        ui.assignModal = document.getElementById('assignModal');
        ui.assignList = document.getElementById('assignList');
        ui.assignEmpty = document.getElementById('assignEmpty');
        ui.assignSearch = document.getElementById('assignSearch');
        ui.saveAssignBtn = document.getElementById('saveAssignBtn');
        ui.itemStatusForm = document.getElementById('itemStatusForm');
        ui.itemStatusId = document.getElementById('itemStatusId');
        ui.itemStatusSelect = document.getElementById('itemStatusSelect');
        ui.itemStatusNote = document.getElementById('itemStatusNote');
        ui.remarkForm = document.getElementById('remarkForm');
        ui.remarkText = document.getElementById('remarkText');
        ui.statusModal = document.getElementById('statusModal');
        ui.itemModal = document.getElementById('itemModal');
        ui.itemModalTitle = document.getElementById('itemModalTitle');
        ui.itemModalSubtitle = document.getElementById('itemModalSubtitle');
        ui.remarkModal = document.getElementById('remarkModal');
        ui.itemStatusModal = document.getElementById('itemStatusModal');
    }

    function bindInteractiveEvents() {
        document.addEventListener('click', handleInteractiveClick);

        const ui = INTERACTIVE_STATE.ui;
        if (ui.statusForm) ui.statusForm.addEventListener('submit', handleStatusFormSubmit);
        if (ui.itemForm) ui.itemForm.addEventListener('submit', handleItemFormSubmit);
        if (ui.itemCategory) ui.itemCategory.addEventListener('change', (event) => updateItemModalFields(event.target.value));
        if (ui.assignSearch) ui.assignSearch.addEventListener('input', (event) => {
            INTERACTIVE_STATE.assignSearch = event.target.value.toLowerCase();
            renderAssignList();
        });
        if (ui.saveAssignBtn) ui.saveAssignBtn.addEventListener('click', handleAssignSave);
        if (ui.itemStatusForm) ui.itemStatusForm.addEventListener('submit', handleItemStatusFormSubmit);
        if (ui.remarkForm) ui.remarkForm.addEventListener('submit', handleRemarkFormSubmit);

        document.querySelectorAll('[data-modal-close]').forEach((btn) => {
            btn.addEventListener('click', () => {
                const overlay = btn.closest('.track-modal-overlay');
                if (overlay) closeModal(overlay.id);
            });
        });

        document.querySelectorAll('.track-modal-overlay').forEach((overlay) => {
            overlay.addEventListener('click', (event) => {
                if (event.target === overlay) {
                    closeModal(overlay.id);
                }
            });
        });

        document.addEventListener('keydown', (event) => {
            if (event.key === 'Escape') {
                closeAllModals();
            }
        });
    }

    function renderInteractiveSections() {
        const ui = INTERACTIVE_STATE.ui;
        if (ui.controls) ui.controls.classList.remove('d-none');
        if (ui.itemsSection) ui.itemsSection.classList.remove('d-none');
        if (ui.remarksSection) ui.remarksSection.classList.remove('d-none');
        renderInteractiveItems();
        renderInteractiveRemarks();
        renderTasks(INTERACTIVE_STATE.items);
    }

    function handleInteractiveClick(event) {
        const actionBtn = event.target.closest('[data-action]');
        if (!actionBtn) return;
        const action = actionBtn.dataset.action;
        const itemId = actionBtn.dataset.itemId;
        performInteractiveAction(action, itemId, actionBtn.dataset);
    }

    function performInteractiveAction(action, itemId, data = {}) {
        switch (action) {
            case 'change-status':
                openStatusModalInteractive();
                break;
            case 'add-item':
                if ((data?.category || '').toLowerCase() === 'remarks') {
                    openRemarkModalInteractive();
                } else {
                    openItemModalInteractive(null, data?.category);
                }
                break;
            case 'add-remark':
                openRemarkModalInteractive();
                break;
            case 'refresh':
                if (INTERACTIVE_STATE.jobCardId) {
                    loadJobCard(true);
                }
                break;
            case 'edit-item':
                openItemModalInteractive(itemId);
                break;
            case 'item-status':
                openItemStatusModalInteractive(itemId);
                break;
            case 'assign-item':
                openAssignModalInteractive(itemId);
                break;
            case 'delete-item':
                deleteItemInteractive(itemId);
                break;
            case 'view-attachment':
                viewAttachmentInteractive(itemId);
                break;
            case 'set-status':
                openStatusModalInteractive(data.statusValue || itemId);
                break;
            case 'filter-category':
                INTERACTIVE_STATE.selectedCategory = itemId || 'All';
                renderInteractiveItems();
                break;
            default:
                break;
        }
    }

    function renderInteractiveItems() {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.itemsList) return;

        renderInteractiveFilters();
        updateItemsAddButton();

        const allItems = Array.isArray(INTERACTIVE_STATE.items) ? [...INTERACTIVE_STATE.items] : [];
        const activeCategory = INTERACTIVE_STATE.selectedCategory || 'All';
        const normalizedActive = activeCategory.toLowerCase();

        const allVisible = allItems.filter((item) => (item.category || '').toLowerCase() !== 'remarks');
        const filtered = activeCategory === 'All'
            ? allVisible
            : allItems.filter((item) => (item.category || '').toLowerCase() === normalizedActive);

        ui.itemsList.innerHTML = '';
        const emptyStateEl = document.createElement('div');
        emptyStateEl.id = 'itemsEmpty';
        emptyStateEl.className = 'empty-placeholder';
        emptyStateEl.textContent = 'No work items have been recorded yet.';
        ui.itemsEmpty = emptyStateEl;

        if (!filtered.length) {
            ui.itemsList.appendChild(emptyStateEl);
        } else {
            filtered
                .sort((a, b) => (b.updatedAt || b.createdAt || 0) - (a.updatedAt || a.createdAt || 0))
                .forEach((item) => {
                    ui.itemsList.appendChild(buildInteractiveItemCard(item));
                });
        }

        if (ui.itemsCountLabel) {
            const count = filtered.length;
            let labelText;
            if (activeCategory === 'All') {
                labelText = `${count} All Items`;
            } else if (normalizedActive === 'remarks') {
                labelText = `${count} Remark${count === 1 ? '' : 's'}`;
            } else {
                labelText = `${count} ${activeCategory} Item${count === 1 ? '' : 's'} · ${allVisible.length} overall`;
            }
            ui.itemsCountLabel.textContent = labelText;
        }
    }

    function renderInteractiveFilters() {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.itemsFilters) return;

        const baseCategories = ['Task', 'Parts', 'Labor', 'Additional Cost', 'Remarks'];

        ui.itemsFilters.innerHTML = '';

        const createButton = (label) => {
            const btn = document.createElement('button');
            btn.type = 'button';
            btn.className = `items-filter-btn ${INTERACTIVE_STATE.selectedCategory === label ? 'active' : ''}`;
            btn.dataset.action = 'filter-category';
            btn.dataset.itemId = label;
            btn.innerHTML = `${getCategoryIcon(label)}<span class="tab-label">${label}</span>`;
            btn.addEventListener('click', () => {
                INTERACTIVE_STATE.selectedCategory = label;
                renderInteractiveItems();
            });
            return btn;
        };

        ui.itemsFilters.appendChild(createButton('All'));
        baseCategories.forEach((category) => {
            ui.itemsFilters.appendChild(createButton(category));
        });
    }

    function buildInteractiveItemCard(item) {
        const card = document.createElement('div');
        card.className = 'item-card';

        const statusInfo = formatTaskStatus(item.status || 'Initiated');
        const statusClass = getStatusChipClass(statusInfo.label);

        card.innerHTML = `
            <div class="item-card-header">
                <div>
                    <div class="item-title">${sanitizeHTML(getItemTitleInteractive(item))}</div>
                    <div class="item-badges">
                        <span class="item-category-badge">${sanitizeHTML(item.category || 'Task')}</span>
                        <span class="status-chip ${statusClass}">${sanitizeHTML(statusInfo.label)}</span>
                        ${item.type ? `<span class="track-tag badge-muted">Type: ${sanitizeHTML(item.type)}</span>` : ''}
                    </div>
                </div>
                <div class="item-actions">
                    ${item.fileUrl ? `<button class="item-action-btn" data-action="view-attachment" data-item-id="${item.id}">View</button>` : ''}
                    <button class="item-action-btn" data-action="item-status" data-item-id="${item.id}">Status</button>
                    <button class="item-action-btn" data-action="assign-item" data-item-id="${item.id}">Assign</button>
                    <button class="item-action-btn" data-action="edit-item" data-item-id="${item.id}">Edit</button>
                    <button class="item-action-btn" data-action="delete-item" data-item-id="${item.id}">Delete</button>
                </div>
            </div>
            ${buildItemMetaRows(item)}
            ${formatAssigneeChips(item.assignTo)}
            ${formatItemNote(item)}
            <div class="item-footer">
                <span>Created ${formatDateTime(item.createdAt)}</span>
                ${item.updatedAt ? `<span class="track-tag">Updated ${formatDateTime(item.updatedAt)}</span>` : ''}
            </div>
        `;

        return card;
    }

    function renderInteractiveRemarks() {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.remarksList || !ui.remarksEmpty) return;

        const remarks = INTERACTIVE_STATE.items.filter((item) => (item.category || '').toLowerCase() === 'remarks');
        ui.remarksList.innerHTML = '';

        if (!remarks.length) {
            ui.remarksEmpty.classList.remove('d-none');
            return;
        }

        ui.remarksEmpty.classList.add('d-none');
        remarks
            .sort((a, b) => (b.createdAt || 0) - (a.createdAt || 0))
            .forEach((remark) => {
                ui.remarksList.appendChild(buildRemarkCardInteractive(remark));
            });
    }

    function buildRemarkCardInteractive(remark) {
        const wrapper = document.createElement('div');
        wrapper.className = 'remark-card';
        const name = remark.userEmail || remark.userRole || 'Team Member';
        wrapper.innerHTML = `
            <div class="remark-avatar">${getInitials(name)}</div>
            <div class="remark-body">
                <div class="remark-meta">
                    <span>${sanitizeHTML(name)}</span>
                    <span>${formatDateTime(remark.createdAt)}</span>
                </div>
                <div class="remark-text">${sanitizeHTML(remark.description || remark.text || '')}</div>
            </div>
        `;
        return wrapper;
    }

    function getItemTitleInteractive(item) {
        const category = (item.category || '').toLowerCase();
        if (category === 'parts') return item.itemName || item.text || item.description || 'Part';
        if (category === 'labor') return item.worker || item.text || item.description || 'Labor';
        if (category === 'additional cost') return item.description || item.text || 'Additional Cost';
        if (category === 'attach') return item.fileName || item.text || item.description || 'Attachment';
        return item.text || item.title || item.description || 'Task';
    }

    function formatAssigneeChips(assignTo) {
        if (!Array.isArray(assignTo) || !assignTo.length) return '';
        return `
            <div class="item-assignees">
                ${assignTo.map((assignee) => `<span class="item-assignee">${sanitizeHTML(assignee.name || assignee.fullName || assignee.email || assignee)}</span>`).join('')}
            </div>
        `;
    }

    function formatItemNote(item) {
        const note = item.note || item.description || '';
        if (!note) return '';
        return `<div class="item-note">${sanitizeHTML(note)}</div>`;
    }

    function openStatusModalInteractive(prefillStatus) {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.statusSelect) return;
        const desiredStatus = prefillStatus || INTERACTIVE_STATE.jobCard?.status || 'Hold';
        ui.statusSelect.value = mapStatusToOption(desiredStatus);
        if (ui.statusNote) ui.statusNote.value = '';
        openModal('statusModal');
    }

    async function handleStatusFormSubmit(event) {
        event.preventDefault();
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.statusSelect) return;
        const statusValue = ui.statusSelect.value;
        const note = ui.statusNote ? ui.statusNote.value.trim() : '';
        const payload = { status: statusValue, updatedAt: new Date().toISOString() };

        if (note) {
            INTERACTIVE_STATE.items.push(createRemark(note, 'Admin Portal'));
            payload.items = sanitizeItems(INTERACTIVE_STATE.items);
        }

        const success = await updateJobCardInteractive(payload, 'Job card status updated.');
        if (success) {
            closeModal('statusModal');
            await loadJobCard(false);
        }
    }

    function openItemModalInteractive(itemId, defaultCategory) {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.itemForm) return;
        resetItemFormInteractive();

        if (itemId) {
            const item = INTERACTIVE_STATE.items.find((entry) => entry.id === itemId);
            if (!item) {
                showNotification('error', 'Item not found.');
                return;
            }
            if (ui.itemModalTitle) ui.itemModalTitle.textContent = 'Edit Item';
            if (ui.itemModalSubtitle) ui.itemModalSubtitle.textContent = 'Update the selected work item.';
            fillItemFormInteractive(item);
        } else {
            if (ui.itemModalTitle) ui.itemModalTitle.textContent = 'Add Item';
            if (ui.itemModalSubtitle) ui.itemModalSubtitle.textContent = 'Create a new work item.';
            if (defaultCategory && ui.itemCategory) {
                ui.itemCategory.value = defaultCategory;
                updateItemModalFields(defaultCategory);
            }
        }

        openModal('itemModal');
    }

    function resetItemFormInteractive() {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.itemForm) return;
        ui.itemForm.reset();
        if (ui.itemIdInput) ui.itemIdInput.value = '';
        if (ui.itemCategory) ui.itemCategory.value = 'Task';
        if (ui.itemType) ui.itemType.value = 'Repair';
        updateItemModalFields('Task');
    }

    function fillItemFormInteractive(item) {
        const ui = INTERACTIVE_STATE.ui;
        if (ui.itemIdInput) ui.itemIdInput.value = item.id;
        if (ui.itemCategory) ui.itemCategory.value = item.category || 'Task';
        if (ui.itemType) ui.itemType.value = item.type || 'Repair';
        if (ui.itemTitle) ui.itemTitle.value = getItemTitleInteractive(item);
        if (ui.itemNote) ui.itemNote.value = item.note || item.description || '';
        if (ui.itemNumber) ui.itemNumber.value = item.itemNumber || '';
        if (ui.partNumber) ui.partNumber.value = item.partNumber || '';
        if (ui.quantity) ui.quantity.value = item.quantity || '';
        if (ui.unit) ui.unit.value = item.unit || '';
        if (ui.itemCost) ui.itemCost.value = item.itemCost || item.amount || '';
        if (ui.itemSite) ui.itemSite.value = item.site || item.location || '';
        if (ui.laborCode) ui.laborCode.value = item.laborCode || '';
        if (ui.laborRate) ui.laborRate.value = item.rate || '';
        if (ui.laborHours) ui.laborHours.value = item.hours || '';
        if (ui.laborMinutes) ui.laborMinutes.value = item.minutes || '';
        if (ui.additionalAmount) ui.additionalAmount.value = item.amount || '';
        if (ui.itemFileName) ui.itemFileName.value = item.fileName || item.filename || '';
        updateItemModalFields(item.category || 'Task');
    }

    function updateItemModalFields(category) {
        const ui = INTERACTIVE_STATE.ui;
        const cat = (category || 'Task').toLowerCase();
        toggleElement(ui.itemType?.closest('.form-field'), cat === 'task');
        toggleElement(ui.partsRow, cat === 'parts');
        toggleElement(ui.quantityRow, cat === 'parts');
        toggleElement(ui.costRow, cat === 'parts' || cat === 'labor');
        toggleElement(ui.laborRow, cat === 'labor');
        toggleElement(ui.laborTimeRow, cat === 'labor');
        toggleElement(ui.amountRow, cat === 'additional cost');
        toggleElement(ui.fileRow, cat === 'attach');

        if (ui.itemTitle) {
            if (cat === 'task') ui.itemTitle.placeholder = 'Describe the task to perform...';
            else if (cat === 'parts') ui.itemTitle.placeholder = 'Part description';
            else if (cat === 'labor') ui.itemTitle.placeholder = 'Personnel / contractor name';
            else if (cat === 'additional cost') ui.itemTitle.placeholder = 'Cost description';
            else if (cat === 'attach') ui.itemTitle.placeholder = 'Attachment title';
            else ui.itemTitle.placeholder = 'Item description';
        }
    }

    async function handleItemFormSubmit(event) {
        event.preventDefault();
        const ui = INTERACTIVE_STATE.ui;
        const existingId = ui.itemIdInput?.value || '';
        const existingItem = existingId ? INTERACTIVE_STATE.items.find((entry) => entry.id === existingId) : null;
        const updatedItem = collectItemFormDataInteractive(existingItem);
        if (!updatedItem) return;

        if (existingItem) {
            const index = INTERACTIVE_STATE.items.findIndex((item) => item.id === existingId);
            if (index !== -1) INTERACTIVE_STATE.items[index] = updatedItem;
        } else {
            INTERACTIVE_STATE.items.push(updatedItem);
        }

        const success = await persistInteractiveItems(existingItem ? 'Item updated successfully.' : 'Item added successfully.');
        if (success) {
            closeModal('itemModal');
            await loadJobCard(false);
        }
    }

    function collectItemFormDataInteractive(existingItem) {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.itemCategory || !ui.itemTitle) return null;

        const category = ui.itemCategory.value || 'Task';
        const now = Date.now();
        const base = existingItem ? { ...existingItem } : {};

        base.id = existingItem?.id || `item_${now}`;
        base.category = category;
        base.type = category === 'Task' ? (ui.itemType?.value || 'Repair') : existingItem?.type;
        base.text = ui.itemTitle.value.trim();
        base.description = ui.itemTitle.value.trim();
        base.title = ui.itemTitle.value.trim();
        base.note = ui.itemNote?.value.trim() || '';
        base.status = existingItem?.status || 'Initiated';
        base.assignTo = Array.isArray(existingItem?.assignTo) ? existingItem.assignTo : [];
        base.createdAt = existingItem?.createdAt || now;
        base.updatedAt = now;
        base.createdBy = existingItem?.createdBy || (INTERACTIVE_STATE.jobCard?.createdBy || 'Admin Portal');

        switch (category.toLowerCase()) {
            case 'parts':
                base.itemNumber = ui.itemNumber?.value.trim() || '';
                base.partNumber = ui.partNumber?.value.trim() || '';
                base.itemName = ui.itemTitle.value.trim();
                base.quantity = ui.quantity?.value ? Number(ui.quantity.value) : '';
                base.unit = ui.unit?.value.trim() || '';
                base.itemCost = ui.itemCost?.value ? Number(ui.itemCost.value) : '';
                base.site = ui.itemSite?.value.trim() || '';
                break;
            case 'labor':
                base.worker = ui.itemTitle.value.trim();
                base.laborCode = ui.laborCode?.value.trim() || '';
                base.rate = ui.laborRate?.value ? Number(ui.laborRate.value) : '';
                base.hours = ui.laborHours?.value ? Number(ui.laborHours.value) : '';
                base.minutes = ui.laborMinutes?.value ? Number(ui.laborMinutes.value) : '';
                base.itemCost = ui.itemCost?.value ? Number(ui.itemCost.value) : '';
                break;
            case 'additional cost':
                base.description = ui.itemTitle.value.trim();
                base.amount = ui.additionalAmount?.value ? Number(ui.additionalAmount.value) : '';
                break;
            case 'attach':
                base.fileName = ui.itemFileName?.value.trim() || ui.itemTitle.value.trim();
                base.fileUrl = existingItem?.fileUrl || '';
                break;
            default:
                break;
        }

        return base;
    }

    function openAssignModalInteractive(itemId) {
        const item = INTERACTIVE_STATE.items.find((entry) => entry.id === itemId);
        if (!item) {
            showNotification('error', 'Item not found.');
            return;
        }
        INTERACTIVE_STATE.activeItemId = itemId;
        INTERACTIVE_STATE.selectedAssignees = new Map();
        (item.assignTo || []).forEach((assignee) => {
            const memberId = assignee.id || assignee.uid || assignee.employeeNumber || assignee.email || assignee.name;
            if (memberId) {
                INTERACTIVE_STATE.selectedAssignees.set(memberId, assignee);
            }
        });
        openModal('assignModal');
        loadTeamMembers();
    }

    async function loadTeamMembers() {
        if (INTERACTIVE_STATE.assignableMembers.length) {
            renderAssignList();
            return;
        }

        try {
            const ui = INTERACTIVE_STATE.ui;
            if (ui.assignList) ui.assignList.innerHTML = '<div class="empty-placeholder">Loading team members...</div>';
            const response = await fetch(`${USERS_API_BASE}${TEAM_ENDPOINT}?page=1&limit=500`);
            if (!response.ok) throw new Error(`Team API responded with ${response.status}`);
            const result = await response.json();
            const members = Array.isArray(result.data) ? result.data : [];
            INTERACTIVE_STATE.assignableMembers = members.map(normalizeMemberInteractive);
            renderAssignList();
        } catch (error) {
            console.error('❌ Failed to load team members:', error);
            INTERACTIVE_STATE.assignableMembers = [];
            renderAssignList();
            showNotification('error', 'Unable to load team members.');
        }
    }

    function normalizeMemberInteractive(member) {
        return {
            id: member._id || member.id || member.uid || member.employeeNumber || member.email,
            name: member.name || member.fullName || member.displayName || 'Team Member',
            fullName: member.fullName || member.name || member.displayName || 'Team Member',
            email: member.email || '',
            employeeNumber: member.employeeNumber || '',
            role: member.role || member.position || '',
        };
    }

    function renderAssignList() {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.assignList || !ui.assignEmpty) return;

        const search = INTERACTIVE_STATE.assignSearch.toLowerCase();
        const members = INTERACTIVE_STATE.assignableMembers.filter((member) => {
            if (!search) return true;
            const haystack = `${member.fullName} ${member.email} ${member.employeeNumber} ${member.role}`.toLowerCase();
            return haystack.includes(search);
        });

        ui.assignList.innerHTML = '';

        if (!members.length) {
            ui.assignEmpty.classList.remove('d-none');
            return;
        }

        ui.assignEmpty.classList.add('d-none');

        members.forEach((member) => {
            const isActive = INTERACTIVE_STATE.selectedAssignees.has(member.id);
            const item = document.createElement('div');
            item.className = `assign-item ${isActive ? 'active' : ''}`;
            item.dataset.memberId = member.id;
            item.innerHTML = `
                <div class="assign-meta">
                    <span class="assign-name">${sanitizeHTML(member.fullName)}</span>
                    ${member.employeeNumber ? `<span class="assign-sub">Emp#: ${sanitizeHTML(member.employeeNumber)}</span>` : ''}
                    ${member.role ? `<span class="assign-sub">Role: ${sanitizeHTML(member.role)}</span>` : ''}
                </div>
                <div class="assign-indicator"></div>
            `;
            item.addEventListener('click', () => toggleMemberSelectionInteractive(member));
            ui.assignList.appendChild(item);
        });
    }

    function toggleMemberSelectionInteractive(member) {
        if (!member.id) return;
        if (INTERACTIVE_STATE.selectedAssignees.has(member.id)) {
            INTERACTIVE_STATE.selectedAssignees.delete(member.id);
        } else {
            INTERACTIVE_STATE.selectedAssignees.set(member.id, {
                id: member.id,
                name: member.name,
                fullName: member.fullName,
                email: member.email,
                employeeNumber: member.employeeNumber,
                role: member.role,
            });
        }
        renderAssignList();
    }

    async function handleAssignSave() {
        if (!INTERACTIVE_STATE.activeItemId) {
            closeModal('assignModal');
            return;
        }
        const item = INTERACTIVE_STATE.items.find((entry) => entry.id === INTERACTIVE_STATE.activeItemId);
        if (!item) {
            showNotification('error', 'Item not found.');
            return;
        }
        item.assignTo = Array.from(INTERACTIVE_STATE.selectedAssignees.values());
        const success = await persistInteractiveItems('Assignments updated.');
        if (success) {
            closeModal('assignModal');
            await loadJobCard(false);
        }
    }

    function openItemStatusModalInteractive(itemId) {
        const ui = INTERACTIVE_STATE.ui;
        const item = INTERACTIVE_STATE.items.find((entry) => entry.id === itemId);
        if (!item) {
            showNotification('error', 'Item not found.');
            return;
        }
        if (ui.itemStatusId) ui.itemStatusId.value = itemId;
        if (ui.itemStatusSelect) ui.itemStatusSelect.value = mapItemStatusToOption(item.status || 'Initiated');
        if (ui.itemStatusNote) ui.itemStatusNote.value = '';
        openModal('itemStatusModal');
    }

    async function handleItemStatusFormSubmit(event) {
        event.preventDefault();
        const ui = INTERACTIVE_STATE.ui;
        const itemId = ui.itemStatusId?.value;
        if (!itemId) return;
        const item = INTERACTIVE_STATE.items.find((entry) => entry.id === itemId);
        if (!item) {
            showNotification('error', 'Item not found.');
            return;
        }
        item.status = ui.itemStatusSelect?.value || item.status;
        const note = ui.itemStatusNote?.value.trim();
        if (note) item.note = note;
        item.updatedAt = Date.now();
        const success = await persistInteractiveItems('Item status updated.');
        if (success) {
            closeModal('itemStatusModal');
            await loadJobCard(false);
        }
    }

    function openRemarkModalInteractive() {
        const ui = INTERACTIVE_STATE.ui;
        if (ui.remarkText) ui.remarkText.value = '';
        openModal('remarkModal');
    }

    async function handleRemarkFormSubmit(event) {
        event.preventDefault();
        const ui = INTERACTIVE_STATE.ui;
        const text = ui.remarkText?.value.trim();
        if (!text) return;

        INTERACTIVE_STATE.items.push(createRemark(text, 'Admin Portal'));
        const success = await persistInteractiveItems('Remark added.');
        if (success) {
            closeModal('remarkModal');
            await loadJobCard(false);
        }
    }

    async function deleteItemInteractive(itemId) {
        const index = INTERACTIVE_STATE.items.findIndex((item) => item.id === itemId);
        if (index === -1) {
            showNotification('error', 'Item not found.');
            return;
        }
        if (!confirm('Delete this item? This action cannot be undone.')) return;
        INTERACTIVE_STATE.items.splice(index, 1);
        const success = await persistInteractiveItems('Item deleted.');
        if (success) {
            await loadJobCard(false);
        }
    }

    function viewAttachmentInteractive(itemId) {
        const item = INTERACTIVE_STATE.items.find((entry) => entry.id === itemId);
        if (!item || !item.fileUrl) {
            showNotification('error', 'Attachment not available.');
            return;
        }
        window.open(item.fileUrl, '_blank', 'noopener');
    }

    async function persistInteractiveItems(successMessage) {
        const payload = { items: sanitizeItems(INTERACTIVE_STATE.items) };
        return updateJobCardInteractive(payload, successMessage);
    }

    async function updateJobCardInteractive(updatePayload, successMessage) {
        if (!INTERACTIVE_STATE.adminUid || !INTERACTIVE_STATE.jobCardId) {
            showNotification('error', 'Missing job card identifiers.');
            return false;
        }
        try {
            showLoader(true);
            togglePageLoader(true);
            const response = await fetch(`${JOB_CARD_API_BASE}/jobcards/${encodeURIComponent(INTERACTIVE_STATE.adminUid)}/${encodeURIComponent(INTERACTIVE_STATE.jobCardId)}`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify(updatePayload),
            });
            if (!response.ok) {
                const errorText = await response.text();
                throw new Error(errorText || `HTTP ${response.status}`);
            }
            await response.json().catch(() => ({}));
            if (successMessage) showNotification('success', successMessage);
            return true;
        } catch (error) {
            console.error('❌ Failed to update job card:', error);
            showNotification('error', error.message || 'Failed to update job card.');
            return false;
        } finally {
            showLoader(false);
            togglePageLoader(false);
        }
    }

    function sanitizeItems(items) {
        return items.map((item) => {
            const cleaned = JSON.parse(JSON.stringify(item));
            cleaned.assignTo = Array.isArray(cleaned.assignTo)
                ? cleaned.assignTo.map((assignee) => {
                      if (typeof assignee === 'string') return assignee;
                      return {
                          id: assignee.id || assignee.uid || '',
                          name: assignee.name || assignee.fullName || '',
                          fullName: assignee.fullName || assignee.name || '',
                          email: assignee.email || '',
                          employeeNumber: assignee.employeeNumber || '',
                          role: assignee.role || '',
                      };
                  })
                : [];
            return cleaned;
        });
    }

    function buildItemMetaRows(item) {
        const rows = [];
        const category = (item.category || '').toLowerCase();
        if (category === 'parts') {
            if (item.itemNumber) rows.push(`<div><strong>Item #:</strong> ${sanitizeHTML(item.itemNumber)}</div>`);
            if (item.partNumber) rows.push(`<div><strong>Part #:</strong> ${sanitizeHTML(item.partNumber)}</div>`);
            if (item.quantity) rows.push(`<div><strong>Quantity:</strong> ${sanitizeHTML(String(item.quantity))}${item.unit ? ` ${sanitizeHTML(item.unit)}` : ''}</div>`);
            if (item.itemCost) rows.push(`<div><strong>Cost:</strong> ${formatCurrency(item.itemCost)}</div>`);
            if (item.site) rows.push(`<div><strong>Site:</strong> ${sanitizeHTML(item.site)}</div>`);
        } else if (category === 'labor') {
            if (item.worker) rows.push(`<div><strong>Worker:</strong> ${sanitizeHTML(item.worker)}</div>`);
            if (item.laborCode) rows.push(`<div><strong>Labor Code:</strong> ${sanitizeHTML(item.laborCode)}</div>`);
            if (item.rate) rows.push(`<div><strong>Rate:</strong> ${formatCurrency(item.rate)}</div>`);
            if (item.hours || item.minutes) rows.push(`<div><strong>Time:</strong> ${sanitizeHTML(String(item.hours || 0))}h ${sanitizeHTML(String(item.minutes || 0))}m</div>`);
        } else if (category === 'additional cost') {
            if (item.amount) rows.push(`<div><strong>Amount:</strong> ${formatCurrency(item.amount)}</div>`);
        } else if (category === 'attach') {
            if (item.fileName) rows.push(`<div><strong>File:</strong> ${sanitizeHTML(item.fileName)}</div>`);
        }
        if (!rows.length) return '';
        return `<div class="item-meta-grid">${rows.join('')}</div>`;
    }

    function formatCurrency(value) {
        if (value === null || value === undefined || value === '') return '—';
        const number = Number(value);
        if (Number.isNaN(number)) return sanitizeHTML(String(value));
        return `$${number.toFixed(2)}`;
    }

    function sanitizeHTML(input) {
        if (input === null || input === undefined) return '';
        return String(input)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;');
    }

    function toggleElement(element, show) {
        if (!element) return;
        if (show) {
            element.removeAttribute('hidden');
            element.style.display = '';
        } else {
            element.setAttribute('hidden', 'hidden');
            element.style.display = 'none';
        }
    }

    function openModal(id) {
        const overlay = document.getElementById(id);
        if (!overlay) return;
        overlay.classList.add('active');
        overlay.setAttribute('aria-hidden', 'false');
    }

    function closeModal(id) {
        const overlay = document.getElementById(id);
        if (!overlay) return;
        overlay.classList.remove('active');
        overlay.setAttribute('aria-hidden', 'true');
    }

    function closeAllModals() {
        document.querySelectorAll('.track-modal-overlay').forEach((overlay) => {
            overlay.classList.remove('active');
            overlay.setAttribute('aria-hidden', 'true');
        });
    }

    function getInitials(name) {
        const parts = (name || '').split(' ').filter(Boolean);
        if (parts.length === 0) return 'TM';
        if (parts.length === 1) return parts[0].slice(0, 2).toUpperCase();
        return (parts[0][0] + parts[parts.length - 1][0]).toUpperCase();
    }

    function showNotification(type, message) {
        const colors = {
            success: '#10b981',
            error: '#ef4444',
            warning: '#f59e0b',
            info: '#3b82f6',
        };
        const notification = document.createElement('div');
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${colors[type] || colors.info};
            color: white;
            padding: 16px 24px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            z-index: 15000;
            font-size: 14px;
            font-weight: 500;
            max-width: 320px;
            opacity: 0;
            transition: opacity 0.3s ease, transform 0.3s ease;
            transform: translateY(-10px);
        `;
        notification.textContent = message;
        document.body.appendChild(notification);
        requestAnimationFrame(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateY(0)';
        });
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateY(-10px)';
            setTimeout(() => notification.remove(), 300);
        }, 4000);
    }

    function normalizeStatusValue(status = '') {
        const normalized = (status || '').toString().trim().toLowerCase();
        if (['assigned'].includes(normalized)) return 'assigned';
        if (['in progress', 'in_progress'].includes(normalized)) return 'in_progress';
        if (['complete', 'completed'].includes(normalized)) return 'complete';
        if (['cancelled', 'canceled'].includes(normalized)) return 'cancelled';
        return 'hold';
    }

    function mapStatusToOption(statusLabel) {
        switch ((statusLabel || '').toLowerCase()) {
            case 'assigned':
                return 'assigned';
            case 'in progress':
                return 'in_progress';
            case 'completed':
                return 'complete';
            case 'cancelled':
                return 'cancelled';
            default:
                return 'Hold';
        }
    }

    function mapItemStatusToOption(statusLabel) {
        const normalized = (statusLabel || '').toLowerCase();
        if (normalized === 'in progress' || normalized === 'in_progress') return 'in_progress';
        if (normalized === 'assigned') return 'assigned';
        if (normalized === 'complete' || normalized === 'completed') return 'Complete';
        if (normalized === 'reject' || normalized === 'rejected') return 'Reject';
        return 'Initiated';
    }

    function getStatusChipClass(statusLabel) {
        const normalized = (statusLabel || '').toLowerCase();
        if (normalized.includes('progress')) return 'status-in-progress-chip';
        if (normalized.includes('assigned')) return 'status-assigned-chip';
        if (normalized.includes('complete')) return 'status-complete-chip';
        if (normalized.includes('cancel')) return 'status-cancelled-chip';
        return 'status-hold-chip';
    }

    function createRemark(text, author) {
        const now = Date.now();
        return {
            id: `remark_${now}`,
            category: 'Remarks',
            description: text,
            userEmail: author,
            userRole: 'Admin',
            createdAt: now,
            updatedAt: now,
            status: 'Complete',
        };
    }

    async function loadJobCard(showLoading = false) {
        if (!INTERACTIVE_STATE.jobCardId) {
            showError('Job card identifier is missing.');
            return;
        }

        try {
            if (showLoading) {
                showLoader(true);
                togglePageLoader(true);
            }
            hideError();

            const jobCard = await fetchJobCardDetails(INTERACTIVE_STATE.jobCardId);

            applyJobCardData(jobCard);
            initializeInteractiveJobCard(jobCard);

            showSection('trackHeader', true);
            showSection('trackOverview', true);
            showSection('trackDetails', true);
            if (jobCard.description || jobCard.notes) {
                showSection('descriptionCard', true);
            } else {
                showSection('descriptionCard', false);
            }
        } catch (error) {
            console.error('❌ Failed to load job card details:', error);
            showError(error.message || 'Failed to load job card details. Please try again later.');
        } finally {
            showLoader(false);
            if (showLoading) togglePageLoader(false);
        }
    }

    document.addEventListener('DOMContentLoaded', () => {
        const jobCardId = getQueryParam('job_card_id');
        if (!jobCardId) {
            showError('Missing job card identifier in the URL.');
            return;
        }

        INTERACTIVE_STATE.jobCardId = jobCardId.trim();
        initializeTabNavigation();
        loadJobCard(true);
    });

    function updateItemsAddButton() {
        const ui = INTERACTIVE_STATE.ui;
        if (!ui.itemsAddButton || !ui.itemsAddLabel) return;
        const selected = INTERACTIVE_STATE.selectedCategory || 'All';
        const effectiveCategory = selected === 'All' ? 'Task' : selected;
        const friendly = effectiveCategory === 'Additional Cost' ? 'Additional Cost' : effectiveCategory === 'Remarks' ? 'Remark' : effectiveCategory;
        ui.itemsAddLabel.textContent = `Add ${friendly}`;
        ui.itemsAddButton.dataset.category = effectiveCategory;
    }

    function getCategoryIcon(category = '') {
        const map = {
            All: 'fa-layer-group',
            Task: 'fa-tasks',
            Parts: 'fa-cogs',
            Labor: 'fa-user-tie',
            'Additional Cost': 'fa-dollar-sign',
            Remarks: 'fa-comment'
        };
        const icon = map[category] || map.Task;
        return `<i class="fas ${icon}"></i>`;
    }
</script>
@endsection