const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const cors = require('cors');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
require('dotenv').config();

const app = express();
const PORT = process.env.DAILY_REPORTS_PORT || 5008;
// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const userId = req.params.userId || 'default';
    const uploadPath = `/root/node-mongo-api/uploads/documents/DailyInsepectinDox/${userId}`;
    // Create directory if it doesn't exist
    if (!fs.existsSync(uploadPath)) {
      fs.mkdirSync(uploadPath, { recursive: true });
    }
    cb(null, uploadPath);
  },
  filename: function (req, file, cb) {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({ 
  storage: storage,
  limits: {
    fileSize: 50 * 1024 * 1024 // 50MB limit
  },
  fileFilter: function (req, file, cb) {
    // Allow PDF and image files
    const allowedTypes = /jpeg|jpg|png|gif|pdf/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);
    
    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Only PDF and image files are allowed!'));
    }
  }
});

// Middleware
app.use(cors({
  origin: '*',
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization']
}));
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Serve uploaded files statically
app.use('/uploads', express.static('/root/node-mongo-api/uploads'));

// Request logging middleware
app.use((req, res, next) => {
  console.log(`📡 ${req.method} ${req.url} - ${new Date().toISOString()}`);
  next();
});

// Connect to MongoDB with better error handling
const connectDB = async () => {
  try {
    // Using shared MongoDB connection from dbConnection.js
    console.log('✅ MongoDB connected successfully for Daily Reports Handler!');
  } catch (err) {
    console.error('❌ MongoDB connection error:', err);
    console.error('⚠️ Will retry connection...');
    setTimeout(() => {
      connectDB();
    }, 5000);
  }
};

connectDB();

// Handle MongoDB connection events
mongoose.connection.on('error', (err) => {
  console.error('❌ MongoDB connection error:', err);
});

mongoose.connection.on('disconnected', () => {
  console.log('⚠️ MongoDB disconnected');
});

mongoose.connection.on('connected', () => {
  console.log('✅ MongoDB connected');
});

// Define Daily Report Schema
const DailyReportSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  id: { type: String, required: true },
  adminUid: { type: String, required: true },
  requestNumber: { type: String, required: true },
  selectedEquipment: Object,
  step1: Object,
  step2: Object,
  step3: Object,
  createdAt: { type: Date, default: Date.now },
  isArchived: { type: Boolean, default: false },
  pdfDownloadUrl: String,
  uploadTimestamp: { type: Date, default: Date.now }
}, {
  strict: false,
  collection: 'DailyReports'
});

// Define Daily Checklist JSON Schema
const DailyChecklistJSONSchema = new mongoose.Schema({
  _id: { type: String, default: 'GLOBAL_CHECKLIST' },
  adminUid: { type: String, default: 'GLOBAL' },
  checklistData: {
    AllDailyReportChecklistsJson: mongoose.Schema.Types.Mixed
  },
  __v: { type: Number, default: 0 },
  updatedAt: { type: Date, default: Date.now }
}, {
  strict: false,
  collection: 'DailyChecklistJSON'
});

// Define Inspection Schema for storing inspection results
const InspectionSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  userId: { type: String, required: true },
  inspectionId: { type: String, required: true },
  step1: Object,
  step2: Object,
  step3: Object,
  checklistDetails: Object,
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now }
}, {
  strict: false,
  collection: 'Inspections'
});

// Indexes for fast searching
DailyReportSchema.index({ adminUid: 1 });
DailyReportSchema.index({ requestNumber: 1 });
DailyReportSchema.index({ isArchived: 1 });
DailyReportSchema.index({ createdAt: -1 });

const DailyReport = mongoose.model('DailyReport', DailyReportSchema);
const DailyChecklistJSON = mongoose.model('DailyChecklistJSON', DailyChecklistJSONSchema);
const Inspection = mongoose.model('Inspection', InspectionSchema);

// Health Check Route
app.get('/', (req, res) => {
  res.json({
    status: 'Daily Reports Server is running',
    port: PORT,
    collections: ['DailyReports', 'DailyChecklistJSON', 'Inspections'],
    endpoints: {
      dailyReports: '/daily-reports',
      checklists: '/checklists',
      inspections: '/inspections',
      pdfUpload: '/daily-reports/upload-pdf/:userId'
    }
  });
});

// Test PDF upload endpoint
app.get('/test-upload/:userId', (req, res) => {
  const { userId } = req.params;
  res.json({
    success: true,
    message: 'Upload endpoint is accessible',
    userId: userId,
    uploadPath: `/root/node-mongo-api/uploads/documents/DailyInsepectinDox/${userId}/`
  });
});

// Get all daily reports for a specific user
app.get('/daily-reports/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    console.log(`📊 Fetching daily reports for admin: ${adminUid}`);
    
    const reports = await DailyReport.find({ adminUid })
      .sort({ createdAt: -1 })
      .lean();

    console.log(`✅ Successfully fetched ${reports.length} daily reports`);
    
    res.json({
      success: true,
      count: reports.length,
      data: reports
    });
  } catch (err) {
    console.error('❌ Error fetching daily reports:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Get all daily reports (for admin with full access)
app.get('/daily-reports', async (req, res) => {
  try {
    console.log('📊 Fetching all daily reports...');
    
    const reports = await DailyReport.find({})
      .sort({ createdAt: -1 })
      .lean();

    console.log(`✅ Successfully fetched ${reports.length} daily reports`);
    
    res.json({
      success: true,
      count: reports.length,
      data: reports
    });
  } catch (err) {
    console.error('❌ Error fetching daily reports:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Get single daily report by ID
app.get('/daily-reports/:adminUid/:reportId', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    console.log(`📄 Fetching daily report ${reportId} for admin ${adminUid}`);
    
    const report = await DailyReport.findOne({ 
      _id: reportId,
      adminUid 
    }).lean();

    if (!report) {
      return res.status(404).json({ 
        success: false,
        error: 'Daily report not found' 
      });
    }

    console.log(`✅ Successfully fetched daily report ${reportId}`);
    
    res.json({
      success: true,
      data: report
    });
  } catch (err) {
    console.error('❌ Error fetching daily report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Create or update daily report (idempotent)
app.post('/daily-reports', async (req, res) => {
  try {
    const reportData = req.body;
    console.log(`📝 Upsert daily report: ${reportData.requestNumber}`);

    // Normalize IDs
    if (!reportData._id) {
      reportData._id = reportData.id || `DR_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    }
    if (!reportData.id) {
      reportData.id = reportData._id;
    }

    // Upsert to avoid E11000 duplicate errors on retry
    const upserted = await DailyReport.findOneAndUpdate(
      { _id: reportData._id },
      { $set: reportData },
      { new: true, upsert: true, setDefaultsOnInsert: true }
    );

    console.log(`✅ Upserted daily report: ${reportData.requestNumber}`);

    res.status(201).json({
      success: true,
      message: 'Daily report created/updated successfully',
      data: upserted
    });
  } catch (err) {
    console.error('❌ Error creating/updating daily report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Update daily report
app.put('/daily-reports/:adminUid/:reportId', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    const updateData = req.body;
    console.log(`✏️ Updating daily report ${reportId} for admin ${adminUid}`);
    
    const updatedReport = await DailyReport.findOneAndUpdate(
      { _id: reportId, adminUid },
      { $set: updateData },
      { new: true, runValidators: true }
    );

    if (!updatedReport) {
      return res.status(404).json({ 
        success: false,
        error: 'Daily report not found' 
      });
    }

    console.log(`✅ Successfully updated daily report ${reportId}`);
    
    res.json({
      success: true,
      message: 'Daily report updated successfully',
      data: updatedReport
    });
  } catch (err) {
    console.error('❌ Error updating daily report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Archive/Unarchive daily report
app.patch('/daily-reports/:adminUid/:reportId/archive', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    const { isArchived } = req.body;
    console.log(`📦 ${isArchived ? 'Archiving' : 'Unarchiving'} daily report ${reportId}`);
    
    const updatedReport = await DailyReport.findOneAndUpdate(
      { _id: reportId, adminUid },
      { $set: { isArchived } },
      { new: true }
    );

    if (!updatedReport) {
      return res.status(404).json({ 
        success: false,
        error: 'Daily report not found' 
      });
    }

    console.log(`✅ Successfully ${isArchived ? 'archived' : 'unarchived'} daily report ${reportId}`);
    
    res.json({
      success: true,
      message: `Daily report ${isArchived ? 'archived' : 'unarchived'} successfully`,
      data: updatedReport
    });
  } catch (err) {
    console.error(`❌ Error ${req.body.isArchived ? 'archiving' : 'unarchiving'} daily report:`, err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Delete daily report
app.delete('/daily-reports/:adminUid/:reportId', async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    console.log(`🗑️ Deleting daily report ${reportId} for admin ${adminUid}`);
    
    const deletedReport = await DailyReport.findOneAndDelete({ 
      _id: reportId,
      adminUid 
    });

    if (!deletedReport) {
      return res.status(404).json({ 
        success: false,
        error: 'Daily report not found' 
      });
    }

    console.log(`✅ Successfully deleted daily report ${reportId}`);
    
    res.json({
      success: true,
      message: 'Daily report deleted successfully',
      data: deletedReport
    });
  } catch (err) {
    console.error('❌ Error deleting daily report:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Upload PDF for daily report
app.post('/daily-reports/upload-pdf/:userId', upload.single('pdf'), async (req, res) => {
  try {
    const { userId } = req.params;
    console.log(`📤 PDF upload request for user: ${userId}`);
    console.log('📤 Request body:', req.body);
    console.log('📤 Request file:', req.file);
    
    if (!req.file) {
      console.log('❌ No file received in upload request');
      return res.status(400).json({
        success: false,
        error: 'No PDF file uploaded'
      });
    }

    const pdfUrl = `https://api.titandrillingzm.com:5008/uploads/documents/DailyInsepectinDox/${userId}/${req.file.filename}`;
    
    console.log(`✅ PDF uploaded successfully for user ${userId}`);
    console.log(`📁 File saved to: ${req.file.path}`);
    console.log(`🔗 PDF URL: ${pdfUrl}`);
    
    res.json({
      success: true,
      message: 'PDF uploaded successfully',
      pdfUrl: pdfUrl,
      fileName: req.file.filename,
      filePath: req.file.path
    });
  } catch (err) {
    console.error('❌ Error uploading PDF:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Upload images for daily report
app.post('/daily-reports/:adminUid/:reportId/upload-images', upload.array('images', 9), async (req, res) => {
  try {
    const { adminUid, reportId } = req.params;
    
    if (!req.files || req.files.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'No image files uploaded'
      });
    }

    const imageUrls = req.files.map(file => 
      `https://api.titandrillingzm.com:5008/uploads/documents/DailyReportsdocx/${file.filename}`
    );
    
    // Update the report with image URLs
    const updatedReport = await DailyReport.findOneAndUpdate(
      { _id: reportId, adminUid },
      { $set: { 'step2.images': imageUrls } },
      { new: true }
    );

    if (!updatedReport) {
      return res.status(404).json({
        success: false,
        error: 'Daily report not found'
      });
    }

    console.log(`✅ ${imageUrls.length} images uploaded for daily report ${reportId}`);
    
    res.json({
      success: true,
      message: 'Images uploaded successfully',
      imageUrls: imageUrls,
      data: updatedReport
    });
  } catch (err) {
    console.error('❌ Error uploading images:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Get daily report statistics
app.get('/daily-reports/stats/:adminUid', async (req, res) => {
  try {
    const { adminUid } = req.params;
    console.log(`📊 Fetching statistics for admin ${adminUid}`);
    
    const stats = await DailyReport.aggregate([
      { $match: { adminUid } },
      {
        $group: {
          _id: null,
          total: { $sum: 1 },
          archived: { $sum: { $cond: ['$isArchived', 1, 0] } },
          active: { $sum: { $cond: ['$isArchived', 0, 1] } }
        }
      }
    ]);

    const result = stats.length > 0 ? stats[0] : { total: 0, archived: 0, active: 0 };
    
    console.log(`✅ Statistics fetched successfully`);
    
    res.json({
      success: true,
      data: result
    });
  } catch (err) {
    console.error('❌ Error fetching statistics:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// ==================== CHECKLIST MANAGEMENT ENDPOINTS ====================

// Get all daily checklists
app.get('/checklists', async (req, res) => {
  try {
    console.log('📋 Fetching all daily checklists...');
    
    const checklistDoc = await DailyChecklistJSON.findOne({ _id: 'GLOBAL_CHECKLIST' });
    
    if (!checklistDoc || !checklistDoc.checklistData || !checklistDoc.checklistData.AllDailyReportChecklistsJson) {
      return res.json({
        success: true,
        count: 0,
        data: []
      });
    }

    const checklists = Object.entries(checklistDoc.checklistData.AllDailyReportChecklistsJson).map(([id, checklist]) => ({
      id,
      ...checklist
    }));

    console.log(`✅ Successfully fetched ${checklists.length} checklists`);
    
    res.json({
      success: true,
      count: checklists.length,
      data: checklists
    });
  } catch (err) {
    console.error('❌ Error fetching checklists:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Get single checklist by ID
app.get('/checklists/:checklistId', async (req, res) => {
  try {
    const { checklistId } = req.params;
    console.log(`📋 Fetching checklist ${checklistId}`);
    
    const checklistDoc = await DailyChecklistJSON.findOne({ _id: 'GLOBAL_CHECKLIST' });
    
    if (!checklistDoc || !checklistDoc.checklistData || !checklistDoc.checklistData.AllDailyReportChecklistsJson) {
      return res.status(404).json({ 
        success: false,
        error: 'Checklist not found' 
      });
    }

    const checklist = checklistDoc.checklistData.AllDailyReportChecklistsJson[checklistId];
    
    if (!checklist) {
      return res.status(404).json({ 
        success: false,
        error: 'Checklist not found' 
      });
    }

    console.log(`✅ Successfully fetched checklist ${checklistId}`);
    
    res.json({
      success: true,
      data: { id: checklistId, ...checklist }
    });
  } catch (err) {
    console.error('❌ Error fetching checklist:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Upload/Update checklists JSON
app.post('/checklists/upload', async (req, res) => {
  try {
    const { checklistsData } = req.body;
    console.log('📤 Uploading checklists to MongoDB...');
    
    if (!checklistsData || typeof checklistsData !== 'object') {
      return res.status(400).json({
        success: false,
        error: 'Invalid checklists data format'
      });
    }

    // Validate checklist structure
    console.log('🔍 Validating checklist structure...');
    for (const [id, checklist] of Object.entries(checklistsData)) {
      console.log(`📋 Validating checklist ${id}:`, {
        hasFormTitle: !!checklist.formTitle,
        hasInspectionItems: Array.isArray(checklist.inspectionItems),
        inspectionItemsCount: Array.isArray(checklist.inspectionItems) ? checklist.inspectionItems.length : 0,
        hasInspectionMatrix: Array.isArray(checklist.inspectionMatrix),
        hasDocument: !!checklist.document
      });
      
      if (!checklist.formTitle || !Array.isArray(checklist.inspectionItems)) {
        console.error(`❌ Invalid checklist structure for ${id}:`, {
          formTitle: checklist.formTitle,
          inspectionItems: Array.isArray(checklist.inspectionItems) ? 'Array' : typeof checklist.inspectionItems
        });
        return res.status(400).json({
          success: false,
          error: `Invalid checklist structure for ${id}: missing formTitle or inspectionItems`
        });
      }
    }

    // Upsert the global checklist document
    const result = await DailyChecklistJSON.findOneAndUpdate(
      { _id: 'GLOBAL_CHECKLIST' },
      { 
        $set: { 
          'checklistData.AllDailyReportChecklistsJson': checklistsData,
          updatedAt: new Date()
        }
      },
      { new: true, upsert: true, setDefaultsOnInsert: true }
    );

    const uploadedCount = Object.keys(checklistsData).length;
    console.log(`✅ Successfully uploaded ${uploadedCount} checklists to MongoDB`);
    console.log(`📊 Uploaded checklist IDs:`, Object.keys(checklistsData));
    
    res.json({
      success: true,
      message: 'Checklists uploaded successfully',
      count: uploadedCount,
      checklistIds: Object.keys(checklistsData),
      data: result
    });
  } catch (err) {
    console.error('❌ Error uploading checklists:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Add single checklist
app.post('/checklists/add', async (req, res) => {
  try {
    const { checklistId, checklistData } = req.body;
    console.log(`📋 Adding checklist ${checklistId}`);
    
    if (!checklistId || !checklistData) {
      return res.status(400).json({
        success: false,
        error: 'checklistId and checklistData are required'
      });
    }

    if (!checklistData.formTitle || !Array.isArray(checklistData.inspectionItems)) {
      return res.status(400).json({
        success: false,
        error: 'Invalid checklist structure: missing formTitle or inspectionItems'
      });
    }

    // Update the specific checklist in the global document
    const result = await DailyChecklistJSON.findOneAndUpdate(
      { _id: 'GLOBAL_CHECKLIST' },
      { 
        $set: { 
          [`checklistData.AllDailyReportChecklistsJson.${checklistId}`]: checklistData,
          updatedAt: new Date()
        }
      },
      { new: true, upsert: true, setDefaultsOnInsert: true }
    );

    console.log(`✅ Successfully added checklist ${checklistId}`);
    
    res.json({
      success: true,
      message: 'Checklist added successfully',
      data: { id: checklistId, ...checklistData }
    });
  } catch (err) {
    console.error('❌ Error adding checklist:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Delete checklist
app.delete('/checklists/:checklistId', async (req, res) => {
  try {
    const { checklistId } = req.params;
    console.log(`🗑️ Deleting checklist ${checklistId}`);
    
    const result = await DailyChecklistJSON.findOneAndUpdate(
      { _id: 'GLOBAL_CHECKLIST' },
      { 
        $unset: { [`checklistData.AllDailyReportChecklistsJson.${checklistId}`]: 1 },
        $set: { updatedAt: new Date() }
      },
      { new: true }
    );

    if (!result) {
      return res.status(404).json({ 
        success: false,
        error: 'Global checklist document not found' 
      });
    }

    console.log(`✅ Successfully deleted checklist ${checklistId}`);
    
    res.json({
      success: true,
      message: 'Checklist deleted successfully'
    });
  } catch (err) {
    console.error('❌ Error deleting checklist:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// ==================== INSPECTION MANAGEMENT ENDPOINTS ====================

// Get all inspections for a user
app.get('/inspections/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    console.log(`🔍 Fetching inspections for user: ${userId}`);
    
    const inspections = await Inspection.find({ userId })
      .sort({ createdAt: -1 })
      .lean();

    console.log(`✅ Successfully fetched ${inspections.length} inspections`);
    
    res.json({
      success: true,
      count: inspections.length,
      data: inspections
    });
  } catch (err) {
    console.error('❌ Error fetching inspections:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Get single inspection by ID
app.get('/inspections/:userId/:inspectionId', async (req, res) => {
  try {
    const { userId, inspectionId } = req.params;
    console.log(`🔍 Fetching inspection ${inspectionId} for user ${userId}`);
    
    const inspection = await Inspection.findOne({ 
      userId,
      inspectionId 
    }).lean();

    if (!inspection) {
      return res.status(404).json({ 
        success: false,
        error: 'Inspection not found' 
      });
    }

    console.log(`✅ Successfully fetched inspection ${inspectionId}`);
    
    res.json({
      success: true,
      data: inspection
    });
  } catch (err) {
    console.error('❌ Error fetching inspection:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Create or update inspection (idempotent)
app.post('/inspections', async (req, res) => {
  try {
    const { userId, inspectionId, step1, step2, step3, checklistDetails } = req.body;
    console.log(`📝 Upsert inspection: ${inspectionId}`);

    if (!userId || !inspectionId) {
      return res.status(400).json({
        success: false,
        error: 'userId and inspectionId are required'
      });
    }

    const inspectionData = {
      _id: `${userId}_${inspectionId}`,
      userId,
      inspectionId,
      step1,
      step2,
      step3,
      checklistDetails,
      updatedAt: new Date()
    };

    // Upsert to avoid E11000 duplicate errors on retry
    const upserted = await Inspection.findOneAndUpdate(
      { _id: `${userId}_${inspectionId}` },
      { $set: inspectionData },
      { new: true, upsert: true, setDefaultsOnInsert: true }
    );

    console.log(`✅ Upserted inspection: ${inspectionId}`);

    res.status(201).json({
      success: true,
      message: 'Inspection created/updated successfully',
      data: upserted
    });
  } catch (err) {
    console.error('❌ Error creating/updating inspection:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Update inspection step
app.put('/inspections/:userId/:inspectionId/step/:stepNumber', async (req, res) => {
  try {
    const { userId, inspectionId, stepNumber } = req.params;
    const stepData = req.body;
    console.log(`✏️ Updating inspection ${inspectionId} step ${stepNumber} for user ${userId}`);
    
    const updateField = `step${stepNumber}`;
    const updatedInspection = await Inspection.findOneAndUpdate(
      { userId, inspectionId },
      { 
        $set: { 
          [updateField]: stepData,
          updatedAt: new Date()
        }
      },
      { new: true, runValidators: true }
    );

    if (!updatedInspection) {
      return res.status(404).json({ 
        success: false,
        error: 'Inspection not found' 
      });
    }

    console.log(`✅ Successfully updated inspection ${inspectionId} step ${stepNumber}`);
    
    res.json({
      success: true,
      message: `Inspection step ${stepNumber} updated successfully`,
      data: updatedInspection
    });
  } catch (err) {
    console.error('❌ Error updating inspection step:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Delete inspection
app.delete('/inspections/:userId/:inspectionId', async (req, res) => {
  try {
    const { userId, inspectionId } = req.params;
    console.log(`🗑️ Deleting inspection ${inspectionId} for user ${userId}`);
    
    const deletedInspection = await Inspection.findOneAndDelete({ 
      userId,
      inspectionId 
    });

    if (!deletedInspection) {
      return res.status(404).json({ 
        success: false,
        error: 'Inspection not found' 
      });
    }

    console.log(`✅ Successfully deleted inspection ${inspectionId}`);
    
    res.json({
      success: true,
      message: 'Inspection deleted successfully',
      data: deletedInspection
    });
  } catch (err) {
    console.error('❌ Error deleting inspection:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Start server
app.listen(PORT, () => {
  console.log(`🚀 Daily Reports Handler server running on port ${PORT}`);
});

module.exports = app;
