const express = require('express');
const { mongoose } = require('./dbConnection'); // Use shared connection
const admin = require('firebase-admin');
const path = require('path');

const app = express();
const PORT = process.env.PUSH_NOTIFICATION_PORT || 5016;

// Middleware
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// CORS
app.use((req, res, next) => {
  res.header('Access-Control-Allow-Origin', '*');
  res.header('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS');
  res.header('Access-Control-Allow-Headers', 'Origin, X-Requested-With, Content-Type, Accept, Authorization');
  if (req.method === 'OPTIONS') {
    return res.sendStatus(200);
  }
  next();
});

// MongoDB Connection
const MONGO_URI = 'mongodb://api.titandrillingzm.com:27017/titan_drilling';

// Using shared MongoDB connection from dbConnection.js
console.log('✅ Using shared MongoDB connection');

// Use existing GlobalUser model (avoid overwriting)
let GlobalUser;
try {
  // Try to get existing model
  GlobalUser = mongoose.model('GlobalUser');
} catch (error) {
  // If model doesn't exist, create it
  const GlobalUserSchema = new mongoose.Schema({
    _id: String,
    uid: String,
    name: String,
    email: String,
    fcmToken: String,
    fcmTokens: [{
      token: String,
      deviceId: String,
      platform: String,
      updatedAt: Date
    }],
    lastFCMUpdate: Date,
  }, { strict: false, collection: 'GlobalUsers' });
  
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
}

// Initialize Firebase Admin SDK
const serviceAccountPath = path.join(__dirname, 'scripts/titan-drilling-1f8e9-firebase-adminsdk-fbsvc-cafcda6695.json');

try {
  const serviceAccount = require(serviceAccountPath);
  
  admin.initializeApp({
    credential: admin.credential.cert(serviceAccount),
  });
  
  console.log('✅ Firebase Admin SDK initialized for push notifications');
  console.log('📁 Service account loaded from:', serviceAccountPath);
} catch (error) {
  console.error('❌ Error initializing Firebase Admin SDK:', error);
  console.error('📁 Tried to load from:', serviceAccountPath);
}

// ==================== ROUTES ====================

// Health Check
app.get('/push-notifications', (req, res) => {
  res.json({ 
    success: true, 
    message: 'Push Notification Service is running',
    port: PORT,
    usesFirebase: true,
    readFromMongoDB: true
  });
});

// Send push notification to user (reads FCM from MongoDB, sends via Firebase)
app.post('/push-notifications/send', async (req, res) => {
  try {
    const { userId, title, body, data } = req.body;
    
    if (!userId || !title || !body) {
      return res.status(400).json({
        success: false,
        error: 'userId, title, and body are required'
      });
    }

    console.log('📤 Sending push notification to user:', userId);
    console.log('📋 Notification:', { title, body });

    // Get user from MongoDB
    const user = await GlobalUser.findById(userId).select('name fcmToken fcmTokens');
    
    if (!user) {
      return res.status(404).json({
        success: false,
        error: 'User not found in MongoDB'
      });
    }

    // Get all FCM tokens for this user
    const tokens = [];
    
    // Add tokens from fcmTokens array
    if (user.fcmTokens && user.fcmTokens.length > 0) {
      user.fcmTokens.forEach(tokenObj => {
        if (tokenObj.token) {
          tokens.push(tokenObj.token);
        }
      });
    }
    
    // Add legacy fcmToken if exists
    if (user.fcmToken && !tokens.includes(user.fcmToken)) {
      tokens.push(user.fcmToken);
    }

    if (tokens.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No FCM tokens found for this user in MongoDB'
      });
    }

    console.log(`📱 Found ${tokens.length} FCM token(s) for user ${user.name}`);

    // Prepare data payload - stringify nested objects for iOS compatibility
    const dataPayload = {};
    if (data) {
      Object.keys(data).forEach(key => {
        dataPayload[key] = typeof data[key] === 'object' ? JSON.stringify(data[key]) : String(data[key]);
      });
    }

    // Send notification via Firebase Admin SDK with iOS/Android specific settings
    const message = {
      notification: {
        title: title,
        body: body,
      },
      data: dataPayload,
      // iOS specific settings - improved for better delivery
      apns: req.body.apns || {
        headers: {
          'apns-priority': '10', // High priority
        },
        payload: {
          aps: {
            alert: {
              title: title,
              body: body
            },
            sound: 'default',
            badge: 1,
          },
          // Custom data for iOS
          ...dataPayload
        }
      },
      // Android specific settings  
      android: req.body.android || {
        priority: 'high',
        notification: {
          sound: 'default',
          clickAction: 'FLUTTER_NOTIFICATION_CLICK',
        }
      },
      tokens: tokens, // Send to all user's devices
    };

    console.log('📨 Sending via Firebase Admin SDK...');
    
    const response = await admin.messaging().sendEachForMulticast(message);

    console.log(`✅ Notification sent: ${response.successCount} success, ${response.failureCount} failed`);

    // Log failures if any
    if (response.failureCount > 0) {
      response.responses.forEach((resp, idx) => {
        if (!resp.success) {
          console.error(`❌ Failed to send to token ${idx}:`, resp.error);
        }
      });
    }

    res.json({
      success: true,
      successCount: response.successCount,
      failureCount: response.failureCount,
      totalTokens: tokens.length,
      message: `Notification sent to ${response.successCount} device(s)`
    });

  } catch (error) {
    console.error('❌ Error sending push notification:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Send notification to multiple users
app.post('/push-notifications/send-multiple', async (req, res) => {
  try {
    const { userIds, title, body, data } = req.body;
    
    if (!userIds || !Array.isArray(userIds) || userIds.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'userIds array is required'
      });
    }

    if (!title || !body) {
      return res.status(400).json({
        success: false,
        error: 'title and body are required'
      });
    }

    console.log(`📤 Sending push notification to ${userIds.length} users`);

    // Get all users from MongoDB
    const users = await GlobalUser.find({ _id: { $in: userIds } }).select('name fcmToken fcmTokens');
    
    if (users.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No users found in MongoDB'
      });
    }

    // Collect all FCM tokens
    const tokens = [];
    users.forEach(user => {
      if (user.fcmTokens && user.fcmTokens.length > 0) {
        user.fcmTokens.forEach(tokenObj => {
          if (tokenObj.token && !tokens.includes(tokenObj.token)) {
            tokens.push(tokenObj.token);
          }
        });
      }
      if (user.fcmToken && !tokens.includes(user.fcmToken)) {
        tokens.push(user.fcmToken);
      }
    });

    if (tokens.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No FCM tokens found for any users in MongoDB'
      });
    }

    console.log(`📱 Found ${tokens.length} total FCM token(s) from ${users.length} user(s)`);

    // Prepare data payload - stringify nested objects for iOS compatibility
    const dataPayload = {};
    if (data) {
      Object.keys(data).forEach(key => {
        dataPayload[key] = typeof data[key] === 'object' ? JSON.stringify(data[key]) : String(data[key]);
      });
    }

    // Send notification via Firebase Admin SDK with iOS/Android specific settings
    const message = {
      notification: {
        title: title,
        body: body,
      },
      data: dataPayload,
      // iOS specific settings - improved for better delivery
      apns: req.body.apns || {
        headers: {
          'apns-priority': '10', // High priority
        },
        payload: {
          aps: {
            alert: {
              title: title,
              body: body
            },
            sound: 'default',
            badge: 1,
          },
          // Custom data for iOS
          ...dataPayload
        }
      },
      // Android specific settings  
      android: req.body.android || {
        priority: 'high',
        notification: {
          sound: 'default',
          clickAction: 'FLUTTER_NOTIFICATION_CLICK',
        }
      },
      tokens: tokens,
    };

    const response = await admin.messaging().sendEachForMulticast(message);

    console.log(`✅ Notification sent: ${response.successCount} success, ${response.failureCount} failed`);
    
    // Log failures if any
    if (response.failureCount > 0) {
      response.responses.forEach((resp, idx) => {
        if (!resp.success) {
          console.error(`❌ Failed to send to token ${idx}:`, resp.error);
        }
      });
    }

    res.json({
      success: true,
      successCount: response.successCount,
      failureCount: response.failureCount,
      totalTokens: tokens.length,
      totalUsers: users.length,
      message: `Notification sent to ${response.successCount} device(s) across ${users.length} user(s)`
    });

  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Send notification to users with specific roles
app.post('/push-notifications/send-to-roles', async (req, res) => {
  try {
    const { roles, title, body, data } = req.body;
    
    if (!roles || !Array.isArray(roles) || roles.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'roles array is required'
      });
    }
    
    if (!title || !body) {
      return res.status(400).json({
        success: false,
        error: 'title and body are required'
      });
    }

    console.log(`📤 Sending push notification to users with roles: ${roles.join(', ')}`);

    // Get users with specified roles and FCM tokens
    const users = await GlobalUser.find({
      role: { $in: roles },
      $or: [
        { fcmToken: { $exists: true, $ne: null } },
        { 'fcmTokens.0': { $exists: true } }
      ]
    }).select('name email role fcmToken fcmTokens');
    
    if (users.length === 0) {
      return res.status(404).json({
        success: false,
        error: `No users with roles ${roles.join(', ')} and FCM tokens found in MongoDB`
      });
    }

    console.log(`📱 Found ${users.length} user(s) with roles ${roles.join(', ')}`);

    // Collect all FCM tokens
    const tokens = [];
    users.forEach(user => {
      if (user.fcmTokens && user.fcmTokens.length > 0) {
        user.fcmTokens.forEach(tokenObj => {
          if (tokenObj.token && !tokens.includes(tokenObj.token)) {
            tokens.push(tokenObj.token);
          }
        });
      }
      if (user.fcmToken && !tokens.includes(user.fcmToken)) {
        tokens.push(user.fcmToken);
      }
    });

    if (tokens.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No FCM tokens found for users with specified roles'
      });
    }

    console.log(`📱 Sending to ${tokens.length} total device(s) from ${users.length} user(s)`);

    // Prepare data payload - stringify nested objects for iOS compatibility
    const dataPayload = {};
    if (data) {
      Object.keys(data).forEach(key => {
        dataPayload[key] = typeof data[key] === 'object' ? JSON.stringify(data[key]) : String(data[key]);
      });
    }

    // Send notification via Firebase Admin SDK with iOS/Android specific settings
    const message = {
      notification: {
        title: title,
        body: body,
      },
      data: dataPayload,
      // iOS specific settings - improved for better delivery
      apns: req.body.apns || {
        headers: {
          'apns-priority': '10', // High priority
        },
        payload: {
          aps: {
            alert: {
              title: title,
              body: body
            },
            sound: 'default',
            badge: 1,
          },
          // Custom data for iOS
          ...dataPayload
        }
      },
      // Android specific settings  
      android: req.body.android || {
        priority: 'high',
        notification: {
          sound: 'default',
          clickAction: 'FLUTTER_NOTIFICATION_CLICK',
        }
      },
      tokens: tokens,
    };

    const response = await admin.messaging().sendEachForMulticast(message);

    console.log(`✅ Notification sent: ${response.successCount} success, ${response.failureCount} failed`);

    // Log failures if any
    if (response.failureCount > 0) {
      response.responses.forEach((resp, idx) => {
        if (!resp.success) {
          console.error(`❌ Failed to send to token ${idx}:`, resp.error);
        }
      });
    }

    res.json({
      success: true,
      successCount: response.successCount,
      failureCount: response.failureCount,
      totalTokens: tokens.length,
      totalUsers: users.length,
      roles: roles,
      message: `Notification sent to ${response.successCount} device(s) of ${users.length} ${roles.join('/')}(s)`
    });

  } catch (error) {
    console.error('❌ Error sending push notifications to roles:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Send notification to specific users by userIds
app.post('/push-notifications/send-to-users', async (req, res) => {
  try {
    const { userIds, title, body, data, apns, android } = req.body;
    
    if (!userIds || !Array.isArray(userIds) || userIds.length === 0) {
      return res.status(400).json({
        success: false,
        error: 'userIds array is required'
      });
    }
    
    if (!title || !body) {
      return res.status(400).json({
        success: false,
        error: 'title and body are required'
      });
    }

    console.log(`📤 Sending push notification to ${userIds.length} specific users`);

    // Get users with specified IDs and FCM tokens
    const users = await GlobalUser.find({
      _id: { $in: userIds },
      $or: [
        { fcmToken: { $exists: true, $ne: null } },
        { 'fcmTokens.0': { $exists: true } }
      ]
    }).select('name email role fcmToken fcmTokens');
    
    if (users.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No users with FCM tokens found in MongoDB'
      });
    }

    console.log(`📱 Found ${users.length} user(s) with FCM tokens`);

    // Collect all FCM tokens
    const tokens = [];
    users.forEach(user => {
      if (user.fcmTokens && user.fcmTokens.length > 0) {
        user.fcmTokens.forEach(tokenObj => {
          if (tokenObj.token && !tokens.includes(tokenObj.token)) {
            tokens.push(tokenObj.token);
          }
        });
      }
      if (user.fcmToken && !tokens.includes(user.fcmToken)) {
        tokens.push(user.fcmToken);
      }
    });

    if (tokens.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No FCM tokens found for specified users'
      });
    }

    console.log(`📱 Sending to ${tokens.length} total device(s) from ${users.length} user(s)`);

    // Prepare data payload - stringify nested objects for iOS compatibility
    const dataPayload = {};
    if (data) {
      Object.keys(data).forEach(key => {
        dataPayload[key] = typeof data[key] === 'object' ? JSON.stringify(data[key]) : String(data[key]);
      });
    }
    
    // Send notification via Firebase Admin SDK with iOS/Android specific settings
    const message = {
      notification: {
        title: title,
        body: body,
      },
      data: dataPayload,
      // iOS specific settings - improved for better delivery
      apns: apns || {
        headers: {
          'apns-priority': '10', // High priority
        },
        payload: {
          aps: {
            alert: {
              title: title,
              body: body
            },
            sound: 'default',
            badge: 1,
          },
          // Custom data for iOS
          ...dataPayload
        }
      },
      // Android specific settings  
      android: android || {
        priority: 'high',
        notification: {
          sound: 'default',
          clickAction: 'FLUTTER_NOTIFICATION_CLICK',
        }
      },
      tokens: tokens,
    };

    const response = await admin.messaging().sendEachForMulticast(message);

    console.log(`✅ Notification sent: ${response.successCount} success, ${response.failureCount} failed`);

    // Log failures if any
    if (response.failureCount > 0) {
      response.responses.forEach((resp, idx) => {
        if (!resp.success) {
          console.error(`❌ Failed to send to token ${idx}:`, resp.error);
        }
      });
    }

    res.json({
      success: true,
      successCount: response.successCount,
      failureCount: response.failureCount,
      totalTokens: tokens.length,
      totalUsers: users.length,
      message: `Notification sent to ${response.successCount} device(s) of ${users.length} user(s)`
    });

  } catch (error) {
    console.error('❌ Error sending push notifications to users:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Send notification to ALL users
app.post('/push-notifications/send-to-all', async (req, res) => {
  try {
    const { title, body, data } = req.body;
    
    if (!title || !body) {
      return res.status(400).json({
        success: false,
        error: 'title and body are required'
      });
    }

    console.log(`📤 Sending push notification to ALL users`);

    // Get all users from MongoDB with FCM tokens
    const users = await GlobalUser.find({
      $or: [
        { fcmToken: { $exists: true, $ne: null } },
        { 'fcmTokens.0': { $exists: true } }
      ]
    }).select('name fcmToken fcmTokens');
    
    if (users.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No users with FCM tokens found in MongoDB'
      });
    }

    console.log(`📱 Found ${users.length} users with FCM tokens`);

    // Collect all FCM tokens
    const tokens = [];
    users.forEach(user => {
      if (user.fcmTokens && user.fcmTokens.length > 0) {
        user.fcmTokens.forEach(tokenObj => {
          if (tokenObj.token && !tokens.includes(tokenObj.token)) {
            tokens.push(tokenObj.token);
          }
        });
      }
      if (user.fcmToken && !tokens.includes(user.fcmToken)) {
        tokens.push(user.fcmToken);
      }
    });

    if (tokens.length === 0) {
      return res.status(404).json({
        success: false,
        error: 'No FCM tokens found for any users in MongoDB'
      });
    }

    console.log(`📱 Sending to ${tokens.length} total device(s) from ${users.length} user(s)`);

    // Send notification via Firebase Admin SDK with iOS/Android specific settings
    const message = {
      notification: {
        title: title,
        body: body,
      },
      data: data || {},
      // iOS specific settings
      apns: req.body.apns || {
        payload: {
          aps: {
            alert: {
              title: title,
              body: body
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings  
      android: req.body.android || {
        priority: 'high',
        notification: {
          sound: 'default',
          clickAction: 'FLUTTER_NOTIFICATION_CLICK',
        }
      },
      tokens: tokens,
    };

    const response = await admin.messaging().sendEachForMulticast(message);

    console.log(`✅ Notification sent: ${response.successCount} success, ${response.failureCount} failed`);

    // Log failures if any
    if (response.failureCount > 0) {
      response.responses.forEach((resp, idx) => {
        if (!resp.success) {
          console.error(`❌ Failed to send to token ${idx}:`, resp.error);
        }
      });
    }

    res.json({
      success: true,
      successCount: response.successCount,
      failureCount: response.failureCount,
      totalTokens: tokens.length,
      totalUsers: users.length,
      message: `Notification sent to ${response.successCount} device(s) across ${users.length} user(s)`
    });

  } catch (error) {
    console.error('❌ Error sending push notifications to all users:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Start Server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`✅ Push Notification Service running on port ${PORT}`);
  console.log(`📡 API URL: https://api.titandrillingzm.com:${PORT}`);
  console.log(`🔔 Reads FCM tokens from MongoDB`);
  console.log(`📨 Sends notifications via Firebase Admin SDK`);
});

module.exports = app;

