#!/bin/bash

# HTTPS Setup Script for api.titandrillingzm.com
# Run this script on your server (168.231.113.219) as root or with sudo

set -e  # Exit on error

echo "╔══════════════════════════════════════════════════════════╗"
echo "║  Titan Drilling - HTTPS API Setup Script               ║"
echo "║  Setting up: api.titandrillingzm.com                    ║"
echo "╚══════════════════════════════════════════════════════════╝"
echo ""

# Color codes
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    echo -e "${RED}❌ Please run this script as root or with sudo${NC}"
    exit 1
fi

# Configuration
DOMAIN="api.titandrillingzm.com"
EMAIL="admin@titandrillingzm.com"  # Change this to your email
NGINX_AVAILABLE="/etc/nginx/sites-available"
NGINX_ENABLED="/etc/nginx/sites-enabled"
CERTBOT_ROOT="/var/www/certbot"

echo -e "${BLUE}📋 Step 1: Detecting OS and Installing Dependencies${NC}"

# Detect OS
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS=$NAME
    echo -e "${GREEN}✓ Detected OS: $OS${NC}"
else
    echo -e "${RED}❌ Cannot detect OS${NC}"
    exit 1
fi

# Install Nginx and Certbot based on OS
if [[ "$OS" == *"Ubuntu"* ]] || [[ "$OS" == *"Debian"* ]]; then
    echo -e "${YELLOW}Installing packages for Ubuntu/Debian...${NC}"
    apt update
    apt install -y nginx certbot python3-certbot-nginx
    NGINX_CONF_DIR="/etc/nginx/sites-available"
    USE_SITES_ENABLED=true
elif [[ "$OS" == *"CentOS"* ]] || [[ "$OS" == *"Red Hat"* ]] || [[ "$OS" == *"Rocky"* ]]; then
    echo -e "${YELLOW}Installing packages for CentOS/RHEL...${NC}"
    yum install -y nginx certbot python3-certbot-nginx || dnf install -y nginx certbot python3-certbot-nginx
    NGINX_CONF_DIR="/etc/nginx/conf.d"
    USE_SITES_ENABLED=false
else
    echo -e "${RED}❌ Unsupported OS. Please install Nginx and Certbot manually.${NC}"
    exit 1
fi

echo -e "${GREEN}✓ Nginx and Certbot installed${NC}"

# Create Nginx configuration
echo -e "${BLUE}📋 Step 2: Creating Nginx Configuration${NC}"

CONFIG_FILE="$NGINX_CONF_DIR/$DOMAIN.conf"

cat > "$CONFIG_FILE" << 'NGINX_CONFIG'
# Nginx Configuration for api.titandrillingzm.com
# Redirect HTTP to HTTPS
server {
    listen 80;
    server_name api.titandrillingzm.com;
    
    location /.well-known/acme-challenge/ {
        root /var/www/certbot;
    }
    
    location / {
        return 301 https://$server_name$request_uri;
    }
}

# HTTPS Server
server {
    listen 443 ssl http2;
    server_name api.titandrillingzm.com;
    
    ssl_certificate /etc/letsencrypt/live/api.titandrillingzm.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/api.titandrillingzm.com/privkey.pem;
    
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers HIGH:!aNULL:!MD5;
    ssl_prefer_server_ciphers on;
    ssl_session_cache shared:SSL:10m;
    ssl_session_timeout 10m;
    
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    
    client_max_body_size 100M;
    
    add_header 'Access-Control-Allow-Origin' '*' always;
    add_header 'Access-Control-Allow-Methods' 'GET, POST, PUT, DELETE, OPTIONS' always;
    add_header 'Access-Control-Allow-Headers' 'Content-Type, Authorization' always;
    
    if ($request_method = 'OPTIONS') {
        return 204;
    }
    
    location / {
        return 200 '{"status":"ok","message":"Titan Drilling API Gateway"}';
        add_header Content-Type application/json;
    }
    
    location /equipment/ {
        proxy_pass http://127.0.0.1:5001/;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
        proxy_read_timeout 300s;
        proxy_connect_timeout 75s;
    }
    
    location /documents/ {
        proxy_pass http://127.0.0.1:5002/;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
    
    location /admin-users/ {
        proxy_pass http://127.0.0.1:5003/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /inspection/ {
        proxy_pass http://127.0.0.1:5004/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /jobcard/ {
        proxy_pass http://127.0.0.1:5005/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /dvir/ {
        proxy_pass http://127.0.0.1:5006/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /incident/ {
        proxy_pass http://127.0.0.1:5007/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /daily-reports/ {
        proxy_pass http://127.0.0.1:5008/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /maintenance/ {
        proxy_pass http://127.0.0.1:5009/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /service-schedule/ {
        proxy_pass http://127.0.0.1:5010/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /fuel-log/ {
        proxy_pass http://127.0.0.1:5011/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /logbook/ {
        proxy_pass http://127.0.0.1:5012/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /hr-forms/ {
        proxy_pass http://127.0.0.1:5013/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /hse-inspection/ {
        proxy_pass http://127.0.0.1:5014/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /notice-board/ {
        proxy_pass http://127.0.0.1:5015/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /push-notifications/ {
        proxy_pass http://127.0.0.1:5016/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /notifications/ {
        proxy_pass http://127.0.0.1:5017/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    location /admin/ {
        proxy_pass http://127.0.0.1:5050/;
        proxy_http_version 1.1;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
    
    access_log /var/log/nginx/api.titandrillingzm.com-access.log;
    error_log /var/log/nginx/api.titandrillingzm.com-error.log;
}
NGINX_CONFIG

echo -e "${GREEN}✓ Nginx configuration created${NC}"

# Create symlink for sites-enabled (Ubuntu/Debian only)
if [ "$USE_SITES_ENABLED" = true ]; then
    echo -e "${BLUE}📋 Creating symbolic link${NC}"
    ln -sf "$CONFIG_FILE" "$NGINX_ENABLED/$DOMAIN.conf"
    echo -e "${GREEN}✓ Symbolic link created${NC}"
fi

# Create certbot directory
echo -e "${BLUE}📋 Step 3: Creating Certbot Directory${NC}"
mkdir -p $CERTBOT_ROOT
echo -e "${GREEN}✓ Certbot directory created${NC}"

# Test Nginx configuration
echo -e "${BLUE}📋 Step 4: Testing Nginx Configuration${NC}"
nginx -t
if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Nginx configuration is valid${NC}"
else
    echo -e "${RED}❌ Nginx configuration test failed${NC}"
    exit 1
fi

# Start Nginx
echo -e "${BLUE}📋 Step 5: Starting Nginx${NC}"
systemctl start nginx
systemctl enable nginx
echo -e "${GREEN}✓ Nginx started and enabled${NC}"

# Configure firewall
echo -e "${BLUE}📋 Step 6: Configuring Firewall${NC}"
if command -v ufw &> /dev/null; then
    echo -e "${YELLOW}Configuring UFW...${NC}"
    ufw allow 80/tcp
    ufw allow 443/tcp
    echo -e "${GREEN}✓ UFW configured${NC}"
elif command -v firewall-cmd &> /dev/null; then
    echo -e "${YELLOW}Configuring firewalld...${NC}"
    firewall-cmd --permanent --add-service=http
    firewall-cmd --permanent --add-service=https
    firewall-cmd --reload
    echo -e "${GREEN}✓ Firewalld configured${NC}"
else
    echo -e "${YELLOW}⚠ No firewall detected. Please configure manually if needed.${NC}"
fi

# Get SSL certificate
echo -e "${BLUE}📋 Step 7: Getting SSL Certificate${NC}"
echo -e "${YELLOW}⚠ Make sure DNS for $DOMAIN points to this server first!${NC}"
read -p "Press Enter to continue with SSL certificate setup (or Ctrl+C to cancel)..."

certbot certonly --webroot -w $CERTBOT_ROOT -d $DOMAIN --non-interactive --agree-tos --email $EMAIL

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ SSL certificate obtained${NC}"
else
    echo -e "${RED}❌ Failed to obtain SSL certificate${NC}"
    echo -e "${YELLOW}⚠ Make sure:${NC}"
    echo -e "  1. DNS for $DOMAIN points to this server"
    echo -e "  2. Port 80 is accessible from the internet"
    echo -e "  3. Nginx is running"
    exit 1
fi

# Reload Nginx with SSL
echo -e "${BLUE}📋 Step 8: Reloading Nginx with SSL${NC}"
systemctl reload nginx
echo -e "${GREEN}✓ Nginx reloaded with SSL${NC}"

# Test HTTPS
echo -e "${BLUE}📋 Step 9: Testing HTTPS${NC}"
sleep 2
if curl -k -s https://$DOMAIN | grep -q "Titan Drilling"; then
    echo -e "${GREEN}✓ HTTPS is working!${NC}"
else
    echo -e "${YELLOW}⚠ HTTPS test inconclusive, please test manually${NC}"
fi

# Setup auto-renewal
echo -e "${BLUE}📋 Step 10: Setting up Certificate Auto-Renewal${NC}"
systemctl enable certbot.timer || echo -e "${YELLOW}⚠ Certbot timer not available, please setup cron manually${NC}"
echo -e "${GREEN}✓ Auto-renewal configured${NC}"

echo ""
echo -e "${GREEN}╔══════════════════════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║                   🎉 SETUP COMPLETE! 🎉                  ║${NC}"
echo -e "${GREEN}╚══════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${BLUE}Your APIs are now accessible via HTTPS:${NC}"
echo ""
echo -e "  📡 Equipment:    ${GREEN}https://api.titandrillingzm.com/equipment/${NC}"
echo -e "  📡 Documents:    ${GREEN}https://api.titandrillingzm.com/documents/${NC}"
echo -e "  📡 Admin Users:  ${GREEN}https://api.titandrillingzm.com/admin-users/${NC}"
echo -e "  📡 Inspection:   ${GREEN}https://api.titandrillingzm.com/inspection/${NC}"
echo -e "  📡 Job Card:     ${GREEN}https://api.titandrillingzm.com/jobcard/${NC}"
echo -e "  📡 DVIR:         ${GREEN}https://api.titandrillingzm.com/dvir/${NC}"
echo -e "  📡 And more...   ${GREEN}https://api.titandrillingzm.com/${NC}"
echo ""
echo -e "${YELLOW}Test your setup:${NC}"
echo -e "  curl https://api.titandrillingzm.com"
echo ""
echo -e "${YELLOW}View logs:${NC}"
echo -e "  sudo tail -f /var/log/nginx/api.titandrillingzm.com-access.log"
echo -e "  sudo tail -f /var/log/nginx/api.titandrillingzm.com-error.log"
echo ""
echo -e "${BLUE}Certificate will auto-renew before expiry.${NC}"
echo ""

