#!/bin/bash

# Firewall Setup Script for Titan Drilling Backend
# This script blocks direct external access to Node.js ports (5001-5050)
# Only localhost (127.0.0.1) can access these ports

echo "🔒 Setting up firewall rules for Titan Drilling Backend..."
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
  echo "❌ Error: This script must be run as root"
  echo "Please run: sudo bash setup_firewall.sh"
  exit 1
fi

echo "📋 Blocking external access to ports 5001-5050..."
echo "   Only 127.0.0.1 (localhost) will be allowed"

# MongoDB port (27017) - Also block external access
echo ""
echo "🔐 Blocking MongoDB port 27017 from external access..."
iptables -A INPUT -p tcp --dport 27017 -s 127.0.0.1 -j ACCEPT
iptables -A INPUT -p tcp --dport 27017 -j DROP
echo "   ✅ MongoDB port 27017 secured"

# Node.js API ports (5001-5050)
echo ""
echo "🔐 Blocking Node.js API ports 5001-5050 from external access..."
for port in {5001..5050}; do
    # Allow localhost
    iptables -A INPUT -p tcp --dport $port -s 127.0.0.1 -j ACCEPT
    # Drop all other connections
    iptables -A INPUT -p tcp --dport $port -j DROP
    echo "   ✅ Port $port secured"
done

# Save iptables rules
echo ""
echo "💾 Saving firewall rules..."

if command -v iptables-save > /dev/null; then
    # For CentOS/RHEL
    if [ -f /etc/sysconfig/iptables ]; then
        iptables-save > /etc/sysconfig/iptables
        echo "   ✅ Rules saved to /etc/sysconfig/iptables"
    # For Debian/Ubuntu
    elif command -v iptables-persistent > /dev/null; then
        netfilter-persistent save
        echo "   ✅ Rules saved using netfilter-persistent"
    else
        iptables-save > /etc/iptables.rules
        echo "   ✅ Rules saved to /etc/iptables.rules"
    fi
fi

# Display current rules
echo ""
echo "📊 Current firewall rules for Node.js ports:"
iptables -L -n | grep -E "500[0-9]|5010|5011|5012|5013|5014|5015|5016|5017|5018|5019|5020|502[0-9]|5050|27017"

echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "✅ Firewall setup complete!"
echo ""
echo "🔒 Ports 5001-5050 and 27017 are now only accessible from localhost"
echo "🌍 External access must go through Apache proxy (HTTPS)"
echo ""
echo "To test:"
echo "  Internal: curl http://127.0.0.1:5001/data (should work)"
echo "  External: curl http://168.231.113.219:5001/data (should timeout)"
echo "  Via domain: curl https://admin.titandrillingzm.com/api/equipment/data (should work)"
echo ""
echo "To remove these rules:"
echo "  iptables -D INPUT -p tcp --dport 5001 -s 127.0.0.1 -j ACCEPT"
echo "  iptables -D INPUT -p tcp --dport 5001 -j DROP"
echo "  (repeat for each port)"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

